///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/PathManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/NavigationModes.h>
#include <core/viewport/SceneRenderer.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/ObjectNode.h>
#include <core/data/DataSetManager.h>

namespace Core {

/******************************************************************************
* This is called by the system after the input handler is
* no longer the active handler.
******************************************************************************/
void NavigationMode::onDeactivated()
{
	if(viewport) {
		viewport->releaseMouse();
		// Retore old view transformation.
		viewport->setViewMatrix(oldViewMatrix);
		// Retore old projection.
		viewport->setFieldOfView(oldZoom);
		// Refresh viewport.
		viewport->updateViewport(true);
		viewport = NULL;
	}
	ViewportInputHandler::onDeactivated();
}

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	if(event->button() == Qt::RightButton) {
		ViewportInputHandler::onMouseDown(vp, event);
	}
	else {
		viewport = &vp;
		vp.grabMouse();
		// Store starting point.
		startPoint = event->pos();
		// Store old matrices.
		oldViewMatrix = viewport->viewMatrix();
		oldZoom = viewport->fieldOfView();
	}
}

/******************************************************************************
* Handles the mouse up event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseUp(Viewport& vp, QMouseEvent* event)
{
	if(viewport != NULL) {
		viewport->releaseMouse();
		viewport = NULL;
	}
}

/******************************************************************************
* Handles the mouse move event for the given viewport.
******************************************************************************/
void NavigationMode::onMouseMove(Viewport& vp, QMouseEvent* event)
{
	if(viewport) {
		modifyViewMatrix(*viewport, event->pos());
		modifyZoom(*viewport, event->pos());
		viewport->updateViewport(true);
		VIEWPORT_MANAGER.processViewportUpdates();
	}
}

///////////////////////////////////////////////////////////////////////////////
////////////////////////////////// Pan Mode ///////////////////////////////////

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void PanMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(vp.isPerspectiveProjection()) {
		FloatType scaling = 50.0 / (FloatType)vp.viewportRectangle().height();
		FloatType deltaX =  scaling * (FloatType)(currentPos.x() - startPoint.x());
		FloatType deltaY = -scaling * (FloatType)(currentPos.y() - startPoint.y());
		vp.settings()->setViewMatrix(AffineTransformation::translation(Vector3(deltaX, deltaY, 0)) * oldViewMatrix);
	}
	else {
		FloatType deltaX =  2.0 * (FloatType)(currentPos.x() - startPoint.x()) / (FloatType)vp.viewportRectangle().width();
		FloatType deltaY = -2.0 * (FloatType)(currentPos.y() - startPoint.y()) / (FloatType)vp.viewportRectangle().height();
		vp.settings()->setViewMatrix(AffineTransformation::translation(vp.inverseProjectionMatrix() * Vector3(deltaX, deltaY, 0)) * oldViewMatrix);
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////// Zoom Mode ///////////////////////////////////

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void ZoomMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(vp.isPerspectiveProjection()) {
		FloatType amount = (FloatType)(startPoint.y() - currentPos.y()) * 0.5;
		vp.settings()->setViewMatrix(AffineTransformation::translation(Vector3(0, 0, amount)) * oldViewMatrix);
	}
}

/******************************************************************************
* Computes the new view zoom based on the new mouse position.
******************************************************************************/
void ZoomMode::modifyZoom(Viewport& vp, const QPoint& currentPos)
{
	if(!vp.isPerspectiveProjection()) {
		FloatType scaling = (FloatType)exp((FloatType)(currentPos.y() - startPoint.y()) * 0.006);
		vp.settings()->setFieldOfView(oldZoom * scaling);
	}
}

/******************************************************************************
* Zooms the viewport in or out.
******************************************************************************/
void ZoomMode::Zoom(Viewport& vp, FloatType steps)
{
	startPoint = QPoint(0, (int)(steps * 0.2));
	oldViewMatrix = vp.viewMatrix();
	oldZoom = vp.fieldOfView();

	modifyZoom(vp, QPoint(0, 0));
	modifyViewMatrix(vp, QPoint(0, 0));
	vp.updateViewport(true);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////// FOV Mode ///////////////////////////////////

/******************************************************************************
* Computes the new field of view based on the new mouse position.
******************************************************************************/
void FOVMode::modifyZoom(Viewport& vp, const QPoint& currentPos)
{
	if(vp.isPerspectiveProjection()) {
		FloatType newFOV = oldZoom + (FloatType)(currentPos.y() - startPoint.y()) * 0.002;
		newFOV = max(newFOV, (FloatType)(5.0 * FLOATTYPE_PI / 180.0));
		newFOV = min(newFOV, (FloatType)(170.0 * FLOATTYPE_PI / 180.0));
		vp.settings()->setFieldOfView(newFOV);
	}
	else {
		FloatType scaling = (FloatType)exp((FloatType)(currentPos.y() - startPoint.y()) * 0.006);
		vp.settings()->setFieldOfView(oldZoom * scaling);
	}
}

///////////////////////////////////////////////////////////////////////////////
//////////////////////////////// Orbit Mode ///////////////////////////////////

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void OrbitMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	NavigationMode::onMouseDown(vp, event);
	if(viewport) {
		// Update orbiting center.
		if(centerMode() == ORBIT_CONSTRUCTION_PLANE) {
			// Use click point on construction plane.
			_orbitCenter = ORIGIN;
			viewport->grid().viewportComputePlaneIntersection(ORIGIN, _orbitCenter);
			_orbitCenter = viewport->grid().gridMatrix() * _orbitCenter;
		}
		else if(centerMode() == ORBIT_SELECTION_CENTER) {
			// Use center of selected objects.
			_orbitCenter = ORIGIN;
			Box3 bb;
			// Compute bounding box of selection.
		    SceneRenderer* renderer = SceneRenderer::activeRenderer();
		    CHECK_POINTER(renderer);
			bb = renderer->sceneExtents(viewport->settings().get(), ANIM_MANAGER.time(), SceneRenderer::SELECTED_OBJECTS);
			if(bb.isEmpty())
	            bb = renderer->sceneExtents(viewport->settings().get(), ANIM_MANAGER.time(), SceneRenderer::ALL);
			if(!bb.isEmpty())
				_orbitCenter = bb.center();
		}
	}
}

/******************************************************************************
* Computes the new view matrix based on the new mouse position.
******************************************************************************/
void OrbitMode::modifyViewMatrix(Viewport& vp, const QPoint& currentPos)
{
	if(!vp.isPerspectiveProjection())
		vp.settings()->setViewType(Viewport::VIEW_ORTHO);

	FloatType scaling = 4.0 / (FloatType)vp.viewportRectangle().height();
	FloatType deltaX = scaling * (FloatType)(currentPos.x() - startPoint.x());
	FloatType deltaY = scaling * (FloatType)(currentPos.y() - startPoint.y());
	Vector3 t = (oldViewMatrix * orbitCenter()) - ORIGIN;
	vp.settings()->setViewMatrix(
		AffineTransformation::translation(t) *
		AffineTransformation::rotationX(deltaY) *
		AffineTransformation::translation(-t) *
		oldViewMatrix *
		AffineTransformation::translation(orbitCenter() - ORIGIN) *
		AffineTransformation::rotationZ(deltaX) *
		AffineTransformation::translation(-(orbitCenter() - ORIGIN)));
}

/******************************************************************************
* Changes the way the center of rotation is chosen.
******************************************************************************/
void OrbitMode::setCenterMode(OrbitMode::CenterMode mode)
{
	if(_centerMode == mode) return;
	_centerMode = mode;
	VIEWPORT_MANAGER.updateViewports();
}

/******************************************************************************
* Sets the world space point around which the camera orbits.
******************************************************************************/
void OrbitMode::setOrbitCenter(const Point3& center)
{
	if(_orbitCenter == center) return;
	_orbitCenter = center;
	VIEWPORT_MANAGER.updateViewports();
}


/******************************************************************************
* Lets the input mode render its overlay content in a viewport.
******************************************************************************/
void OrbitMode::renderOverlay(Viewport* vp, bool isActive)
{
	NavigationMode::renderOverlay(vp, isActive);

	// Render center of rotation when in user mode.
	if(_centerMode == ORBIT_USER_DEFINED) {
		vp->setDepthTest(true);
		vp->setBackfaceCulling(true);
		vp->setLightingEnabled(false);
		vp->setWorldMatrix(AffineTransformation::translation(orbitCenter() - ORIGIN));

		FloatType symbolSize = 1.0 * vp->nonScalingSize(orbitCenter());

		Box3 bbox(Point3(-symbolSize), Point3(+symbolSize));
		Point3 verts[2];
		verts[0] = Point3(-symbolSize,0,0);
		verts[1] = Point3(+symbolSize,0,0);
		vp->setRenderingColor(Color(1,0,0));
		vp->renderLines(2, bbox, verts);
		verts[0] = Point3(0,-symbolSize,0);
		verts[1] = Point3(0,+symbolSize,0);
		vp->setRenderingColor(Color(0,1,0));
		vp->renderLines(2, bbox, verts);
		verts[0] = Point3(0,0,-symbolSize);
		verts[1] = Point3(0,0,+symbolSize);
		vp->setRenderingColor(Color(0,0,1));
		vp->renderLines(2, bbox, verts);
	}
}

/////////////////////////////////////////////////////////////////////////////////////
///////////////////////////// Pick Orbit Center Mode ////////////////////////////////

/******************************************************************************
* Handles the mouse down events for a Viewport.
******************************************************************************/
void PickOrbitCenterMode::onMousePressed(QMouseEvent* event)
{
	SimpleInputHandler::onMousePressed(event);

	Point3 p;
	if(findIntersection(viewport(), event->pos(), p)) {
		OrbitMode::instance()->setCenterMode(OrbitMode::ORBIT_USER_DEFINED);
		OrbitMode::instance()->setOrbitCenter(p);
	}
	else {
		OrbitMode::instance()->setCenterMode(OrbitMode::ORBIT_SELECTION_CENTER);
		OrbitMode::instance()->setOrbitCenter(ORIGIN);
	}

	onFinish();
}

/******************************************************************************
* Is called when the user moves the mouse while the operation is not active.
******************************************************************************/
void PickOrbitCenterMode::onMouseFreeMove(Viewport& vp, QMouseEvent* event)
{
	SimpleInputHandler::onMouseFreeMove(vp, event);

	Point3 p;
	bool isOverObject = findIntersection(&vp, event->pos(), p);

	if(!isOverObject && showCursor) {
		showCursor = false;
		updateCursor();
	}
	else if(isOverObject && !showCursor) {
		showCursor = true;
		updateCursor();
	}
}

/******************************************************************************
* Finds the closest intersection point between a ray originating from the
* current mouse cursor position and the whole scene.
******************************************************************************/
bool PickOrbitCenterMode::findIntersection(Viewport* vp, const Point2I& mousePos, Point3& intersectionPoint)
{
	OVITO_ASSERT(vp != NULL);

	TimeTicks time = ANIM_MANAGER.time();
	Ray3 ray = vp->screenRay(mousePos);

	// Iterate over all object nodes in the scene.
	SceneRoot* rootNode = DATASET_MANAGER.currentSet()->sceneRoot();
	if(rootNode == NULL) return false;

	FloatType closestDistance = FLOATTYPE_MAX;

	for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
		ObjectNode* objNode = dynamic_object_cast<ObjectNode>(iter.current());
		if(!objNode) continue;

		const PipelineFlowState& flowState = objNode->evalPipeline(time);
		if(flowState.result() == NULL) continue;

		TimeInterval iv;
		const AffineTransformation& nodeTM = objNode->getWorldTransform(time, iv);

		Ray3 transformedRay = nodeTM.inverse() * ray;
		Vector3 normal;
		FloatType t;

		if(flowState.result()->intersectRay(transformedRay, time, objNode, t, normal)) {
			if(t < closestDistance) {
				closestDistance = t;
				intersectionPoint = nodeTM * transformedRay.point(t);
			}
		}
	}

	return closestDistance != FLOATTYPE_MAX;
}

};
