/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isInteger = require( './../../../../base/assert/is-integer' );
var isnan = require( './../../../../base/assert/is-nan' );
var betaln = require( './../../../../base/special/betaln' );
var abs = require( './../../../../base/special/abs' );
var ln = require( './../../../../base/special/ln' );
var NINF = require( '@stdlib/constants/float64/ninf' );


// MAIN //

/**
* Computes the natural logarithm of the binomial coefficient of two integers.
*
* @param {integer} n - input value
* @param {integer} k - second input value
* @returns {number} function value
*
* @example
* var v = binomcoefln( 8, 2 );
* // returns ~3.332
*
* @example
* var v = binomcoefln( 0, 0 );
* // returns 0.0
*
* @example
* var v = binomcoefln( -4, 2 );
* // returns ~2.303
*
* @example
* var v = binomcoefln( 88, 3 );
* // returns ~11.606
*
* @example
* var v = binomcoefln( NaN, 3 );
* // returns NaN
*
* @example
* var v = binomcoefln( 5, NaN );
* // returns NaN
*
* @example
* var v = binomcoefln( NaN, NaN );
* // returns NaN
*/
function binomcoefln( n, k ) {
	if ( isnan( n ) || isnan( k ) ) {
		return NaN;
	}
	if ( !isInteger( n ) || !isInteger( k ) ) {
		return NaN;
	}
	if ( n < 0.0 ) {
		return binomcoefln( -n + k - 1.0, k );
	}
	if ( k < 0 ) {
		return NINF;
	}
	if ( k === 0 ) {
		return 0.0;
	}
	if ( k === 1 ) {
		return ln( abs( n ) );
	}
	if ( n < k ) {
		return NINF;
	}
	if ( n - k < 2 ) {
		return binomcoefln( n, n - k );
	}
	// Case: n - k >= 2
	return -ln( n + 1 ) - betaln( n - k + 1, k + 1 );
}


// EXPORTS //

module.exports = binomcoefln;
