/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/logging.h"

// GLib based libraries
#include <glib.h>

// C Standard Library
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

// Boost C++
#include <boost/algorithm/string.hpp>
#include <boost/assert.hpp>
#include <boost/locale/format.hpp>

// GCC Extensions to the C++ Standard Library
#include <ext/stdio_sync_filebuf.h>

// C++ Standard Library
#include <map>
#include <string>

// Media Scanner Library
#include "mediascanner/locale.h"

// Boost C++
using boost::algorithm::starts_with;
using boost::algorithm::to_lower_copy;
using boost::locale::format;

namespace mediascanner {
namespace logging {

const Domain::SettingsMap &Domain::ReadSettings() {
    const char *const current_env = ::getenv("MEDIASCANNER_DEBUG");
    static const char *parsed_env = nullptr;
    static SettingsMap settings;

    if (current_env != parsed_env) {
        std::istringstream env(safe_string(current_env));
        settings.clear();

        for (std::string name; std::getline(env, name, ':'); ) {
            if (name.empty())
                continue;

            Flags flags = Explicit | Enabled;

            if (name[0] == '-' || name[0] == '!') {
                name = name.substr(1);
                flags = Explicit | Disabled;
            } else if (name[0] == '+') {
                name = name.substr(1);
            }

            SettingsMap::iterator it = settings.find(name);

            if (it == settings.end())
                it = settings.insert(std::make_pair(name, flags)).first;
            else
                it->second = flags;
        }

        parsed_env = current_env;
    }

    return settings;
}

Domain::Flags Domain::LookupFlags(const std::string &name, Flags preset) {
    std::string best_match;
    Flags flags = preset;

    for (const auto &entry: ReadSettings()) {
        if (entry.first == name) {
            flags = entry.second;
            break;
        }

        if ((preset & (Explicit | Enabled | Disabled)) == (Explicit | Disabled))
            continue;

        if (starts_with(name, entry.first)
                && name[entry.first.length()] == '/'
                && entry.first.length() > best_match.length()) {
            best_match = entry.first;
            flags = entry.second;
        }
    }

    return flags;
}

// Leave parent as const to permit static const declarations of logging
// domains in application code. Of course once could work arround by using
// the parent()
Domain::Domain(const std::string &name, Flags flags,
               const Domain *parent, MessageSinkPtr sink)
    : message_sink_(sink)
    , parent_(const_cast<Domain *>(parent))
    , name_(to_lower_copy(name))
    , initial_flags_(LookupFlags(name_, flags))
    , flags_(initial_flags_)
    , default_message_sink_(message_sink()) {
    VerifyGraph();
    VerifyPrefix();
}

Domain::Domain(const std::string &name,
               const Domain *parent, MessageSinkPtr sink)
    : message_sink_(sink)
    , parent_(const_cast<Domain *>(parent))
    , name_(to_lower_copy(name))
    , initial_flags_(LookupFlags(name_, Enabled))
    , flags_(initial_flags_)
    , default_message_sink_(message_sink()) {
    VerifyGraph();
    VerifyPrefix();
}

void Domain::VerifyGraph() const {
    for (const Domain *other = parent_; other; other = other->parent_)
        BOOST_ASSERT_MSG(this != other, "Domain graphs must be cycle-free");
}

void Domain::VerifyPrefix() const {
    const std::string::size_type n = name().rfind('/');

    if (n != std::string::npos) {
        BOOST_ASSERT_MSG(parent_, "Parent needed for prefixed domain names");

        const std::string prefix = name().substr(0, n);

        if (parent_->name() != prefix) {
            const std::string parent_name = parent_->name();
            const std::string domain_name = name();
            const std::string error_message =
                    (format("The parent domain's name (\"{1}\") must match "
                            "the prefix of this domain's name (\"{2}\").")
                     % parent_name % domain_name).str();

            BOOST_ASSERT_MSG(parent_name == prefix,
                             error_message.c_str());
        }
    }
}

static MessageSinkPtr new_sink(const std::string &prefix,
                               DefaultMessageSink::Color color) {
    return MessageSinkPtr(new DefaultMessageSink(&std::clog, prefix, color));
}

static Domain* unknown() {
    static Domain domain("unknown", nullptr,
                         new_sink("UNKNOWN", DefaultMessageSink::Red));
    return &domain;
}

Domain* error() {
    static Domain domain("error", unknown(),
                         new_sink("ERROR", DefaultMessageSink::Red));
    return &domain;
}

Domain* critical() {
    static Domain domain("critical", error(),
                         new_sink("CRITICAL", DefaultMessageSink::Brown));
    return &domain;
}

Domain* warning() {
    static Domain domain("warning", critical(),
                         new_sink("WARNING", DefaultMessageSink::Brown));
    return &domain;
}

Domain* message() {
    static Domain domain("message", warning(),
                         new_sink("WARNING", DefaultMessageSink::Default));
    return &domain;
}

Domain* info() {
    static Domain domain("info", message(),
                         new_sink("INFO", DefaultMessageSink::Blue));
    return &domain;
}

Domain* debug() {
    static Domain domain("debug", Domain::Disabled, info(),
                         new_sink("DEBUG", DefaultMessageSink::Green));
    return &domain;
}

Domain* trace() {
    static Domain domain("trace", Domain::Disabled, debug(),
                         new_sink("TRACE", DefaultMessageSink::Cyan));
    return &domain;
}

static const MessageSinkPtr initial_default_sink(new DefaultMessageSink);
MessageSinkPtr MessageSink::default_instance_ = initial_default_sink;

MessageSink::~MessageSink() {
}

void MessageSink::Report(const std::string &domain_name,
                         const std::wstring &message) {
    Report(domain_name, FromUnicode(message));
}

void MessageSink::set_default_instance(MessageSinkPtr instance) {
    if (instance == initial_default_sink) {
        for (Domain *domain = trace(); domain; domain->parent()) {
            if (domain->message_sink() == default_instance_)
                domain->set_message_sink(domain->default_message_sink());
        }
    } else {
        for (Domain *domain = trace(); domain; domain->parent()) {
            if (domain->message_sink() == domain->default_message_sink())
                domain->set_message_sink(instance);
        }
    }

    default_instance_ = instance;
}

static bool supports_color_codes(std::ostream *stream) {
    using __gnu_cxx::stdio_sync_filebuf;

    if (stdio_sync_filebuf<char> *const stdio =
            dynamic_cast<stdio_sync_filebuf<char> *>(stream->rdbuf())) // NOLINT:runtime/rtti
        return ::isatty(::fileno(stdio->file()));

    return false;
}

static std::string ansi_color_code(DefaultMessageSink::Color color) {
    switch (color) {
    case DefaultMessageSink::Default:
        return "\033[0m";
    case DefaultMessageSink::Black:
        return "\033[1;30m";
    case DefaultMessageSink::Red:
        return "\033[1;31m";
    case DefaultMessageSink::Green:
        return "\033[1;32m";
    case DefaultMessageSink::Brown:
        return "\033[0;33m";
    case DefaultMessageSink::Blue:
        return "\033[1;34m";
    case DefaultMessageSink::Magenta:
        return "\033[1;35m";
    case DefaultMessageSink::Cyan:
        return "\033[1;36m";
    case DefaultMessageSink::White:
        return "\033[1;37m";
    case DefaultMessageSink::Bold:
        return "\033[1m";
    }

    return std::string();
}

DefaultMessageSink::DefaultMessageSink(std::ostream *stream,
                                       const std::string &prefix,
                                       Color color)
    : stream_(stream)
    , prefix_(prefix) {
    if (not prefix_.empty()) {
        if (color != Default && supports_color_codes(stream_))
            prefix_ = ansi_color_code(color) + prefix_
                    + ansi_color_code(Default);

        prefix_ += " ";
    }
}

void DefaultMessageSink::Report(const std::string &domain_name,
                                const std::string &message) {
    std::ostringstream buffer;

    buffer << program_invocation_short_name << "[" << getpid() << "]: "
           << prefix_ << domain_name << ": " << message << std::endl;

    // Write formatted message in one go
    // to avoid overlapping of messages from different threads.
    const std::string result = buffer ? buffer.str() : message + "/n";
    stream_->write(result.data(), result.size());
}

static void on_glib_message(const char *domain_name,
                            GLogLevelFlags log_level,
                            const char *text, void *) {
    Domain *domain;

    switch (log_level & G_LOG_LEVEL_MASK) {
    case G_LOG_LEVEL_ERROR:
        domain = error();
        break;

    case G_LOG_LEVEL_CRITICAL:
        domain = critical();
        break;

    case G_LOG_LEVEL_WARNING:
        domain = warning();
        break;

    case G_LOG_LEVEL_MESSAGE:
        domain = message();

    case G_LOG_LEVEL_INFO:
        domain = info();
        break;

    case G_LOG_LEVEL_DEBUG:
        domain = debug();
        break;

    default:
        domain = unknown();
    }

    if (domain_name) {
        Domain(domain->name() + "/" + domain_name, domain).print(text);
    } else {
        domain->print(text);
    }
}

static bool capturing_glib_messages = false;

void capture_glib_messages() {
    g_log_set_default_handler(on_glib_message, 0);
    capturing_glib_messages = true;
}

bool is_capturing_glib_messages() {
    return capturing_glib_messages;
}

} // namespace logging
} // namespace mediascanner
