//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Instrument/PolarizationAnalysisEditor.cpp
//! @brief     Implements class PolarizationAnalysisEditor
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Instrument/PolarizationAnalysisEditor.h"
#include "GUI/Model/Device/InstrumentItems.h"
#include "GUI/View/Numeric/DoubleSpinBox.h"
#include "GUI/View/Numeric/NumWidgetUtil.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"
#include <QLabel>

PolarizationAnalysisEditor::PolarizationAnalysisEditor(QWidget* parent, InstrumentItem* instrument)
    : QGroupBox("Polarization analysis", parent)
    , m_instrument(instrument)
{
    ASSERT(instrument);

    auto* layout = new QVBoxLayout(this);

    // polarizer
    {
        auto* polarizerCheckbox = new QCheckBox("Enable beam polarization");
        polarizerCheckbox->setChecked(m_instrument->withPolarizer());
        layout->addWidget(polarizerCheckbox);

        auto* polarizerWidget = new QWidget(this);
        auto* polarizerLayout = new QFormLayout(polarizerWidget);
        polarizerLayout->setFieldGrowthPolicy(QFormLayout::FieldsStayAtSizeHint);

        addBlochRow(polarizerLayout, m_instrument->polarizerBlochVector());

        layout->addWidget(polarizerWidget);
        polarizerWidget->setVisible(m_instrument->withPolarizer());

        connect(polarizerCheckbox, &QCheckBox::toggled, [this, polarizerWidget](bool b) {
            polarizerWidget->setVisible(b);
            m_instrument->setWithPolarizer(b);
            emit dataChanged();
        });
    }

    // analyzer
    {
        auto* analyzerCheckbox = new QCheckBox("Enable analyzer");
        analyzerCheckbox->setChecked(m_instrument->withAnalyzer());
        layout->addWidget(analyzerCheckbox);

        auto* analyzerWidget = new QWidget(this);
        auto* analyzerLayout = new QFormLayout(analyzerWidget);
        analyzerLayout->setFieldGrowthPolicy(QFormLayout::FieldsStayAtSizeHint);

        addBlochRow(analyzerLayout, m_instrument->analyzerBlochVector());

        layout->addWidget(analyzerWidget);
        analyzerWidget->setVisible(m_instrument->withAnalyzer());

        connect(analyzerCheckbox, &QCheckBox::toggled, [this, analyzerWidget](bool b) {
            analyzerWidget->setVisible(b);
            m_instrument->setWithAnalyzer(b);
            emit dataChanged();
        });
    }

    // collapser

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(this);
    collapser->setExpanded(instrument->isExpandPolarizerAlanyzer());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [instrument](bool b) { instrument->setExpandPolarizerAlanyzer(b); });
}

DoubleSpinBox* PolarizationAnalysisEditor::createSpinBox(DoubleProperty& d)
{
    auto* sb = new DoubleSpinBox(d);
    connect(sb, &DoubleSpinBox::baseValueChanged, [this, &d](double v) {
        if (d.value() != v) {
            d.setValue(v);
            emit dataChanged();
        }
    });
    return sb;
}

void PolarizationAnalysisEditor::addBlochRow(QFormLayout* parentLayout, VectorProperty& d)
{
    auto* layout = new QHBoxLayout;

    const auto add = [this, layout](DoubleProperty& d) {
        layout->addWidget(new QLabel(GUI::Util::labelWithUnit(d) + ":"));
        layout->addWidget(createSpinBox(d));
    };

    add(d.x());
    add(d.y());
    add(d.z());

    layout->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding));

    parentLayout->addRow(d.label() + ":", layout);
}
