/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <lame/lame.h>
#include "config.h"
#ifdef PACKAGED
#include <ripoff/RipOffPluginRawInterface.h>
#else
#include "lib/RipOffPluginRawInterface.h"
#endif

#define UI_NAME "Lame Mp3 Encoder Plugin v1.1"
struct LamePlugin_
{
	gchar *name;
	gchar *label;
	gdouble qualityrating;
	gdouble bitrate;
	lame_global_flags *gf;
	gboolean using_bitrate;
	gboolean using_mono;
	unsigned char mp3buffer[LAME_MAXMP3BUFFER];
};
typedef struct LamePlugin_ * LamePlugin;

struct helper_struct
{
	GtkWidget *quality_rating_hscale;
	GtkWidget *bitrate_hscale;
	GtkWidget *window;
	gboolean *using_bitrate;
};
	

void value_changed(GtkAdjustment *adjustment, gpointer data);
void quality_rating_toggled(	GtkToggleButton *button,
				gpointer data);
void bitrate_toggled(	GtkToggleButton *button,
			gpointer data);
void close_button_clicked(GtkButton *close_button,
			  gpointer data);
gboolean    destroy_handler(	GtkWidget *widget,
                                GdkEvent *event,
                                gpointer user_data);
void        checked(	GtkToggleButton *togglebutton,
                        gpointer user_data);

RipOffPluginRaw ripoff_plugin_raw_new(xmlDocPtr ptr)
{
	LamePlugin raw = g_new(struct LamePlugin_, 1);
	ripoff_config_file_set_config_file_pointer(ptr);
	raw->name = UI_NAME;
	raw->label = "ripoff_default_mp3_plugin";
	raw->qualityrating = ripoff_config_file_retrieve_double(raw->label, "qualityrating", 5.0);
	raw->bitrate = ripoff_config_file_retrieve_double(raw->label, "bitrate", 128.0);
	raw->using_bitrate = ripoff_config_file_retrieve_boolean(raw->label, "using_bitrate", TRUE);
	raw->using_mono = ripoff_config_file_retrieve_boolean(raw->label, "using_mono", FALSE);
	
	/* this function should take care of the stored preferences loading */
	return  (RipOffPluginRaw) raw;
}

gchar *ripoff_plugin_raw_get_name(RipOffPluginRaw raw)
{
	return raw->name;
}

gchar *ripoff_plugin_raw_get_label(RipOffPluginRaw raw)
{
	return raw->label;
}

const gchar *ripoff_plugin_raw_get_extension(RipOffPluginRaw raw)
{
	return "mp3";
}

gboolean ripoff_plugin_raw_has_prefs(RipOffPluginRaw raw)
{
	return TRUE;
}

GtkWidget *ripoff_plugin_raw_prefs(RipOffPluginRaw raw/*, GtkWidget *pref_window*/)
{
	GtkWidget *close_button;
	GtkWidget *quality_radio_button;
	GtkWidget *bitrate_radio_button;
	GtkWidget *window;
	GtkObject *quality_rating_adjustment;
	GtkObject *bitrate_adjustment;
	GtkWidget *quality_rating_hscale;
	GtkWidget *bitrate_hscale;
	GtkWidget *table;
	GtkWidget *mono_check_button;
	GtkTooltips *mono_tips;
	GtkTooltips *quality_tips;
	GtkTooltips *bitrate_tips;
	LamePlugin plugin = (LamePlugin) raw;
	struct helper_struct *helper;
	helper = g_new(struct helper_struct, 1);

	table = gtk_table_new(2, 5, FALSE);

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (GTK_WIDGET (window), 400, 170);
	gtk_window_set_title (GTK_WINDOW (window), UI_NAME" Preferences");

	mono_check_button = gtk_check_button_new_with_label("Mono");

	if(plugin->using_mono)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mono_check_button), TRUE);
	}
	else
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mono_check_button), FALSE);
	}

	mono_tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(mono_tips, mono_check_button, "Encode the MP3 to be monoural. You probably don't want to do this unless your source was monoural originally.", NULL);
				

	/* create quality rating adjustments and scales */
	quality_rating_adjustment = gtk_adjustment_new(	plugin->qualityrating,
							0.0,
							7.0,
							4.0,
							1.0,
							0.0); 

	quality_rating_hscale = gtk_hscale_new(GTK_ADJUSTMENT(quality_rating_adjustment));
	gtk_scale_set_digits(GTK_SCALE(quality_rating_hscale), 0);

	/* create bitrate adjustments and scales */
	bitrate_adjustment = gtk_adjustment_new(		plugin->bitrate,
							48.0,
							320.0,
							128.0,
							8.0,
							0.0); 

	bitrate_hscale = gtk_hscale_new(GTK_ADJUSTMENT(bitrate_adjustment));
	gtk_scale_set_digits(GTK_SCALE(bitrate_hscale), 0);

	/* helper struct setup */
	helper->using_bitrate = &(plugin->using_bitrate);
	helper->quality_rating_hscale = quality_rating_hscale;
	helper->bitrate_hscale = bitrate_hscale;
	helper->window = window;

	/* Creates quality setting radio button and scale bar*/
	quality_radio_button = gtk_radio_button_new_with_label(NULL, "LAME Quality Rating:");

	quality_tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(quality_tips, quality_radio_button, "The quality rating allows LAME to vary the bitrate (instead of having a static bitrate as is specified with the bitrate setting) in order to achieve an ideal compression-to-quality ratio. 0 is the highest quality and 7 is the lowest. Recommended encoding option.", NULL);


	g_signal_connect(	quality_rating_adjustment,
				"value-changed",
				G_CALLBACK (value_changed),
				&(plugin->qualityrating));

	g_signal_connect(	quality_radio_button,
				"toggled",
				G_CALLBACK (quality_rating_toggled),
				helper);

	/* complete bitrate setup */
	bitrate_radio_button = gtk_radio_button_new_with_label(	gtk_radio_button_get_group(
								GTK_RADIO_BUTTON(quality_radio_button)), "Bitrate:");

	bitrate_tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(bitrate_tips, bitrate_radio_button, "Specifies a static bitrate for the MP3 even when varying the bitrate would achieve a better compression-to-quality ration.", NULL);


	g_signal_connect(	bitrate_adjustment,
				"value-changed",
				G_CALLBACK (value_changed),
				&(plugin->bitrate));

	g_signal_connect(	bitrate_radio_button,
				"toggled",
				G_CALLBACK (bitrate_toggled),
				helper);


	/* close window button setup */
	close_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);


	g_signal_connect (		G_OBJECT (close_button), 
					"clicked",
                             		G_CALLBACK (close_button_clicked), 
                              		helper);

	g_signal_connect	 (	G_OBJECT (window), 
				"delete_event",
                             	G_CALLBACK (destroy_handler), 
                                helper);

	g_signal_connect(	mono_check_button,
			 	"toggled",
				G_CALLBACK (checked),
				&(plugin->using_mono));

	if(plugin->using_bitrate)
	{
		gtk_widget_set_sensitive(quality_rating_hscale, FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bitrate_radio_button), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(quality_radio_button), FALSE);
	}
	else
	{
		gtk_widget_set_sensitive(bitrate_hscale, FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(quality_radio_button), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bitrate_radio_button), FALSE);
	}

	/* table attachments */

	gtk_table_attach	 (	GTK_TABLE(table),
				mono_check_button, 1, 2, 0, 1,
				GTK_SHRINK,
				GTK_FILL | GTK_EXPAND,
				0, 0);

	gtk_table_attach (	GTK_TABLE(table),
				quality_radio_button, 0, 1, 1, 2,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				0, 5);

	gtk_table_attach (	GTK_TABLE(table),
				quality_rating_hscale, 1, 2, 1, 2,
				GTK_FILL | GTK_EXPAND,
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				bitrate_radio_button, 0, 1, 2, 3,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				0, 5);

	gtk_table_attach (	GTK_TABLE(table),
				bitrate_hscale, 1, 2, 2, 3,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				close_button, 1, 2, 3, 4,
				GTK_SHRINK,
				GTK_SHRINK,
				0, 5);


	gtk_container_add(GTK_CONTAINER(window), table);

	return window;
}

gboolean ripoff_plugin_raw_has_about(RipOffPluginRaw raw)
{
	return TRUE;
}

GtkWidget *ripoff_plugin_raw_about(RipOffPluginRaw raw)
{
	return about_window_draw(	        "Mp3 Encoder", 
						"Mp3 Encoder 1.1", 
						"Lame Mp3 Encoding Plugin for "NAME,
						"Copyright © "YEAR" Bobby Ryan Newberry",
						"http://ripoffc.sourceforge.net");
}

gboolean ripoff_plugin_raw_perform_setup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{	
	LamePlugin plugin = (LamePlugin) raw;
	plugin->gf = lame_init();
	

	if(plugin->using_bitrate)
	{
		lame_set_VBR(plugin->gf, vbr_off);
		lame_set_brate(plugin->gf, plugin->bitrate);
		lame_set_VBR_min_bitrate_kbps(plugin->gf,lame_get_brate(plugin->gf));
	}
	else
	{
		lame_set_VBR(plugin->gf, vbr_mtrh);
		lame_set_quality(plugin->gf, (int) plugin->qualityrating);

		/*if(((int) plugin->qualityrating) > 7)
		{*/
			 lame_set_exp_nspsytune(plugin->gf, 2);
		/*}*/
	}	

	if(plugin->using_mono)
	{
		lame_set_mode(plugin->gf, MONO);
	}

	/* id3 tag initialization and setting */
	id3tag_init(plugin->gf);
	
	id3tag_set_artist(plugin->gf, ripoff_track_get_artist(track));
	id3tag_set_genre(plugin->gf, ripoff_track_get_genre(track));
	id3tag_set_album(plugin->gf, ripoff_track_get_album_title(track));
	id3tag_set_title(plugin->gf, ripoff_track_get_track_title(track));
	id3tag_set_year(plugin->gf, ripoff_track_get_year(track));
	id3tag_set_comment(plugin->gf, "Ripped by "NAME" "WEBSITE);

	lame_init_params(plugin->gf);

	return TRUE;
}

gboolean ripoff_plugin_raw_encode_buffer(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						int16_t *audio_data,
						FILE *output_descriptor,
						RipOffTrack track)
{
	long i;
	int bytes_written;
	LamePlugin plugin = (LamePlugin) raw;

	bytes_written = lame_encode_buffer_interleaved(		plugin->gf,
								audio_data,
								CDIO_CD_FRAMESIZE_RAW / 4,
								plugin->mp3buffer, 
								sizeof(plugin->mp3buffer));
	fwrite(plugin->mp3buffer,  bytes_written, 1, output_descriptor);

	return TRUE;
}

gboolean ripoff_plugin_raw_perform_cleanup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{
	int bytes_written;
	LamePlugin plugin = (LamePlugin) raw;
	bytes_written = lame_encode_flush(	plugin->gf,
						plugin->mp3buffer,
						sizeof(plugin->mp3buffer));

	fwrite(plugin->mp3buffer, bytes_written, 1, output_descriptor);
	lame_close(plugin->gf);
	return TRUE;
}

void ripoff_plugin_raw_close(RipOffPluginRaw raw)
{
	LamePlugin plugin = (LamePlugin) raw;
	ripoff_config_file_save_double(plugin->label, "qualityrating", plugin->qualityrating);
	ripoff_config_file_save_double(plugin->label, "bitrate", plugin->bitrate);
	ripoff_config_file_save_boolean(plugin->label, "using_bitrate", plugin->using_bitrate);
	ripoff_config_file_save_boolean(plugin->label, "using_mono", plugin->using_mono);
}

void value_changed(GtkAdjustment *adjustment, gpointer data)
{
	*((gdouble *)data) = gtk_adjustment_get_value(adjustment);
}

void quality_rating_toggled(	GtkToggleButton *button,
				gpointer data)
{
	struct helper_struct *helper = data;
	
	if(gtk_toggle_button_get_active(button))
	{
		*(helper->using_bitrate)	 = FALSE;
		gtk_widget_set_sensitive(helper->quality_rating_hscale, TRUE);
		gtk_widget_set_sensitive(helper->bitrate_hscale, FALSE);

	}
}

void bitrate_toggled(	GtkToggleButton *button,
			gpointer data)
{
	struct helper_struct *helper = data;
	
	if(gtk_toggle_button_get_active(button))
	{
		*(helper->using_bitrate)	 = TRUE;
		gtk_widget_set_sensitive(helper->bitrate_hscale, TRUE);
		gtk_widget_set_sensitive(helper->quality_rating_hscale, FALSE);
	}
}

gboolean    destroy_handler(	GtkWidget *widget,
                                GdkEvent *event,
                                gpointer data)
{
	gtk_widget_destroy(((struct helper_struct *)data)->window);
	g_free(data);
	return FALSE;
}

void close_button_clicked(GtkButton *close_button,
			  gpointer data)
{
	gtk_widget_destroy(((struct helper_struct *)data)->window);
	g_free(data);
}

void        checked(	GtkToggleButton *toggle_button,
                        gpointer user_data)
{
	gboolean *use_mono = user_data;

	if(gtk_toggle_button_get_active(toggle_button))
	{
		*use_mono = TRUE;		
	}
	else
	{
		*use_mono = FALSE;
	}
}
