#! /usr/bin/env python

import openturns as ot

ot.TESTPREAMBLE()
ot.PlatformInfo.SetNumericalPrecision(3)

size = 10000

distribution = ot.Gumbel(1.5, -0.5)
print("distribution=", distribution)
sample = distribution.getSample(size)
factory = ot.MethodOfMomentsFactory(ot.Gumbel())
inf_distribution = factory.build(sample)
print("estimated distribution=", inf_distribution)

# set (a,b) out of (r, t, a, b)
distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
print("distribution=", distribution)
sample = distribution.getSample(size)
factory = ot.MethodOfMomentsFactory(ot.Beta())
factory.setKnownParameter([-1.0, 1.0], [2, 3])
inf_distribution = factory.build(sample)
print("estimated distribution=", inf_distribution)

# from moments
distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
factory = ot.MethodOfMomentsFactory(ot.Beta())
cm = [distribution.getCentralMoment(i + 2)[0] for i in range(3)]
moments = [distribution.getMean()[0]] + cm
inf_distribution = factory.buildFromMoments(moments)
print("estimated distribution (moments)=", inf_distribution)

# with bounds
data = [
    0.6852,
    0.9349,
    0.5884,
    1.727,
    1.581,
    0.3193,
    -0.5701,
    1.623,
    2.210,
    -0.3440,
    -0.1646,
]
sample = ot.Sample([[x] for x in data])
size = sample.getSize()
xMin = sample.getMin()[0]
xMax = sample.getMax()[0]
delta = xMax - xMin
a = xMin - delta / (size + 2)
b = xMax + delta / (size + 2)
distribution = ot.TruncatedNormal()
factory = ot.MethodOfMomentsFactory(distribution)
factory.setKnownParameter([a, b], [2, 3])
solver = factory.getOptimizationAlgorithm()
sampleMean = sample.computeMean()[0]
sampleSigma = sample.computeStandardDeviation()[0]
startingPoint = [sampleMean, sampleSigma]
solver.setStartingPoint(startingPoint)
factory.setOptimizationAlgorithm(solver)
lowerBound = [-1.0, 0]
upperBound = [-1.0, 1.5]
finiteLowerBound = [False, True]
finiteUpperBound = [False, True]
bounds = ot.Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound)
factory = ot.MethodOfMomentsFactory(distribution, bounds)
factory.setKnownParameter([a, b], [2, 3])
factory.setOptimizationBounds(bounds)
inf_distribution = factory.build(sample)
print("estimated distribution=", inf_distribution)

# setKnownParameter+buildEstimator
sample = ot.Normal(2.0, 1.0).getSample(10)
factory = ot.MethodOfMomentsFactory(ot.WeibullMin())
factory.setBootstrapSize(4)
factory.setKnownParameter([1.0], [1])  # set the sigma parameter to 1.0
result = factory.buildEstimator(sample)
print("ok")
