/* -------------------------------------------------------------------------- *
 *                                   OpenMMAmoeba                             *
 * -------------------------------------------------------------------------- *
 * This is part of the OpenMM molecular simulation toolkit originating from   *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org.               *
 *                                                                            *
 * Portions copyright (c) 2008-2012 Stanford University and the Authors.      *
 * Authors: Mark Friedrichs                                                   *
 * Contributors:                                                              *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,  *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,  *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS, CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,   *
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR      *
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE  *
 * USE OR OTHER DEALINGS IN THE SOFTWARE.                                     *
 * -------------------------------------------------------------------------- */

/**
 * This tests AmoebaGeneralizedKirkwoodForce.
 */

#include "openmm/internal/AssertionUtilities.h"
#include "openmm/Context.h"
#include "OpenMMAmoeba.h"
#include "openmm/System.h"
#include "openmm/AmoebaMultipoleForce.h"
#include "openmm/LangevinIntegrator.h"
#include <iostream>
#include <vector>
#include <stdlib.h>
#include <stdio.h>

#define ASSERT_EQUAL_TOL_MOD(expected, found, tol, testname) {double _scale_ = std::abs(expected) > 1.0 ? std::abs(expected) : 1.0; if (!(std::abs((expected)-(found))/_scale_ <= (tol))) {std::stringstream details; details << testname << " Expected "<<(expected)<<", found "<<(found); throwException(__FILE__, __LINE__, details.str());}};

#define ASSERT_EQUAL_VEC_MOD(expected, found, tol,testname) {ASSERT_EQUAL_TOL_MOD((expected)[0], (found)[0], (tol),(testname)); ASSERT_EQUAL_TOL_MOD((expected)[1], (found)[1], (tol),(testname)); ASSERT_EQUAL_TOL_MOD((expected)[2], (found)[2], (tol),(testname));};


using namespace OpenMM;
using namespace std;

const double TOL = 1e-4;

static void checkFiniteDifferences(vector<Vec3> forces, Context &context, vector<Vec3> positions)
{
    // Take a small step in the direction of the energy gradient and see whether the potential energy changes by the expected amount.

    double norm = 0.0;
    for (auto& f : forces)
        norm += f.dot(f);
    norm = std::sqrt(norm);
    const double stepSize = 1e-3;
    double step = 0.5*stepSize/norm;
    vector<Vec3> positions2(forces.size()), positions3(forces.size());
    for (int i = 0; i < (int) positions.size(); ++i) {
        Vec3 p = positions[i];
        Vec3 f = forces[i];
        positions2[i] = Vec3(p[0]-f[0]*step, p[1]-f[1]*step, p[2]-f[2]*step);
        positions3[i] = Vec3(p[0]+f[0]*step, p[1]+f[1]*step, p[2]+f[2]*step);
    }
    context.setPositions(positions2);
    State state2 = context.getState(State::Energy);
    context.setPositions(positions3);
    State state3 = context.getState(State::Energy);
    ASSERT_EQUAL_TOL(state3.getPotentialEnergy()+norm*stepSize, state2.getPotentialEnergy(), 1e-4);
}

// setup for 2 ammonia molecules

static void setupMultipoleAmmonia(System& system, AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce,
                                  AmoebaMultipoleForce::PolarizationType polarizationType, int includeCavityTerm) {

    // beginning of Multipole setup

    AmoebaMultipoleForce* amoebaMultipoleForce        = new AmoebaMultipoleForce();;
    int numberOfParticles                             = 8;

    amoebaMultipoleForce->setNonbondedMethod(AmoebaMultipoleForce::NoCutoff);
    amoebaMultipoleForce->setPolarizationType(polarizationType);
    amoebaMultipoleForce->setMutualInducedTargetEpsilon(1.0e-06);
    amoebaMultipoleForce->setMutualInducedMaxIterations(500);

    std::vector<double> nitrogenMolecularDipole(3);
    std::vector<double> nitrogenMolecularQuadrupole(9);

    nitrogenMolecularDipole[0]     =   8.3832254e-03;
    nitrogenMolecularDipole[1]     =   0.0000000e+00;
    nitrogenMolecularDipole[2]     =   3.4232474e-03;

    nitrogenMolecularQuadrupole[0] =  -4.0406249e-04;
    nitrogenMolecularQuadrupole[1] =   0.0000000e+00;
    nitrogenMolecularQuadrupole[2] =  -2.6883671e-04;
    nitrogenMolecularQuadrupole[3] =   0.0000000e+00;
    nitrogenMolecularQuadrupole[4] =   2.5463927e-04;
    nitrogenMolecularQuadrupole[5] =   0.0000000e+00;
    nitrogenMolecularQuadrupole[6] =  -2.6883671e-04;
    nitrogenMolecularQuadrupole[7] =   0.0000000e+00;
    nitrogenMolecularQuadrupole[8] =   1.4942322e-04;

    // first N

    system.addParticle(1.4007000e+01);
    amoebaMultipoleForce->addMultipole(-5.7960000e-01, nitrogenMolecularDipole, nitrogenMolecularQuadrupole, 2, 1, 2, 3,  3.9000000e-01,  3.1996314e-01,  1.0730000e-03);

    // 3 H attached to first N

    std::vector<double> hydrogenMolecularDipole(3);
    std::vector<double> hydrogenMolecularQuadrupole(9);
    hydrogenMolecularDipole[0]     =  -1.7388763e-03;
    hydrogenMolecularDipole[1]     =   0.0000000e+00;
    hydrogenMolecularDipole[2]     =  -4.6837475e-03;

    hydrogenMolecularQuadrupole[0] =  -4.4253841e-05;
    hydrogenMolecularQuadrupole[1] =   0.0000000e+00;
    hydrogenMolecularQuadrupole[2] =   1.5429571e-05;
    hydrogenMolecularQuadrupole[3] =   0.0000000e+00;
    hydrogenMolecularQuadrupole[4] =   4.1798924e-05;
    hydrogenMolecularQuadrupole[5] =   0.0000000e+00;
    hydrogenMolecularQuadrupole[6] =   1.5429571e-05;
    hydrogenMolecularQuadrupole[7] =   0.0000000e+00;
    hydrogenMolecularQuadrupole[8] =   2.4549167e-06;

    system.addParticle(1.0080000e+00);
    system.addParticle(1.0080000e+00);
    system.addParticle(1.0080000e+00);
    amoebaMultipoleForce->addMultipole(1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 0, 2, 3, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);
    amoebaMultipoleForce->addMultipole(1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 0, 1, 3, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);
    amoebaMultipoleForce->addMultipole(1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 0, 1, 2, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);

    // second N

    system.addParticle(  1.4007000e+01);
    amoebaMultipoleForce->addMultipole( -5.7960000e-01, nitrogenMolecularDipole, nitrogenMolecularQuadrupole, 2, 5, 6, 7,  3.9000000e-01,  3.1996314e-01,  1.0730000e-03);

    // 3 H attached to second N

    system.addParticle(  1.0080000e+00);
    system.addParticle(  1.0080000e+00);
    system.addParticle(  1.0080000e+00);
    amoebaMultipoleForce->addMultipole(  1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 4, 6, 7, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);
    amoebaMultipoleForce->addMultipole(  1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 4, 5, 7, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);
    amoebaMultipoleForce->addMultipole(  1.9320000e-01, hydrogenMolecularDipole, hydrogenMolecularQuadrupole, 2, 4, 5, 6, 3.9000000e-01,  2.8135002e-01,  4.9600000e-04);

    // covalent maps

    std::vector< int > covalentMap;
    covalentMap.resize(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(0, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(0, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    amoebaMultipoleForce->setCovalentMap(1, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(1, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(1, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    amoebaMultipoleForce->setCovalentMap(2, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(1);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(2, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(2, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    amoebaMultipoleForce->setCovalentMap(3, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    amoebaMultipoleForce->setCovalentMap(3, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(0);
    covalentMap.push_back(1);
    covalentMap.push_back(2);
    covalentMap.push_back(3);
    amoebaMultipoleForce->setCovalentMap(3, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(4, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(4, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    amoebaMultipoleForce->setCovalentMap(5, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(5, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(5, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    amoebaMultipoleForce->setCovalentMap(6, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(5);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(6, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(6, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    amoebaMultipoleForce->setCovalentMap(7, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(0), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    amoebaMultipoleForce->setCovalentMap(7, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(1), covalentMap);

    covalentMap.resize(0);
    covalentMap.push_back(4);
    covalentMap.push_back(5);
    covalentMap.push_back(6);
    covalentMap.push_back(7);
    amoebaMultipoleForce->setCovalentMap(7, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(4), covalentMap);
    system.addForce(amoebaMultipoleForce);

    // GK force

    amoebaGeneralizedKirkwoodForce->setSolventDielectric(  7.8300000e+01);
    amoebaGeneralizedKirkwoodForce->setSoluteDielectric(   1.0000000e+00);
    amoebaGeneralizedKirkwoodForce->setIncludeCavityTerm(includeCavityTerm);

    // addParticle: charge, radius, scalingFactor

    for (unsigned int ii = 0; ii < 2; ii++) {
        amoebaGeneralizedKirkwoodForce->addParticle( -5.7960000e-01,   1.5965000e-01,   6.9000000e-01);
        amoebaGeneralizedKirkwoodForce->addParticle(  1.9320000e-01,   1.2360000e-01,   6.9000000e-01);
        amoebaGeneralizedKirkwoodForce->addParticle(  1.9320000e-01,   1.2360000e-01,   6.9000000e-01);
        amoebaGeneralizedKirkwoodForce->addParticle(  1.9320000e-01,   1.2360000e-01,   6.9000000e-01);
    }
    system.addForce(amoebaGeneralizedKirkwoodForce);
}

static void getForcesEnergyMultipoleAmmonia(Context& context, std::vector<Vec3>& forces, double& energy) {
    std::vector<Vec3> positions(context.getSystem().getNumParticles());

    positions[0]              = Vec3(  1.5927280e-01,  1.7000000e-06,   1.6491000e-03);
    positions[1]              = Vec3(  2.0805540e-01, -8.1258800e-02,   3.7282500e-02);
    positions[2]              = Vec3(  2.0843610e-01,  8.0953200e-02,   3.7462200e-02);
    positions[3]              = Vec3(  1.7280780e-01,  2.0730000e-04,  -9.8741700e-02);
    positions[4]              = Vec3( -1.6743680e-01,  1.5900000e-05,  -6.6149000e-03);
    positions[5]              = Vec3( -2.0428260e-01,  8.1071500e-02,   4.1343900e-02);
    positions[6]              = Vec3( -6.7308300e-02,  1.2800000e-05,   1.0623300e-02);
    positions[7]              = Vec3( -2.0426290e-01, -8.1231400e-02,   4.1033500e-02);

    context.setPositions(positions);
    State state                      = context.getState(State::Forces | State::Energy);
    forces                           = state.getForces();
    energy                           = state.getPotentialEnergy();
    
    // Check that the forces and energy are consistent.
    
    checkFiniteDifferences(forces, context, positions);
}

// setup for villin

static void setupAndGetForcesEnergyMultipoleVillin(AmoebaMultipoleForce::PolarizationType polarizationType,
                                                   int includeCavityTerm, std::vector<Vec3>& forces, double& energy) {

    // beginning of Multipole setup

    System system;

    AmoebaMultipoleForce* amoebaMultipoleForce        = new AmoebaMultipoleForce();;
    int numberOfParticles                             = 596;

    amoebaMultipoleForce->setNonbondedMethod(AmoebaMultipoleForce::NoCutoff);
    amoebaMultipoleForce->setPolarizationType(polarizationType);
    amoebaMultipoleForce->setMutualInducedTargetEpsilon(1.0e-06);
    amoebaMultipoleForce->setMutualInducedMaxIterations(500);

    for (unsigned int ii = 0; ii < numberOfParticles; ii++) {
        system.addParticle(1.0);
    }

    static const double multipoleData[] = {
        0,   1.1164000e-01,   0.0000000e+00,   0.0000000e+00,   7.3465672e-03,  -1.5868283e-04,   0.0000000e+00,  -1.3245349e-05,   0.0000000e+00,  -3.2179944e-04,   0.0000000e+00,  -1.3245349e-05,   0.0000000e+00,   4.8048227e-04, 0, 1, 4, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
        1,   1.8220000e-02,   8.7176654e-03,   6.6850957e-03,   1.4157078e-02,  -1.9774681e-04,  -1.5194348e-04,  -1.9436780e-04,  -1.5194348e-04,  -2.6774460e-04,  -5.9291372e-05,  -1.9436780e-04,  -5.9291372e-05,   4.6549141e-04, 0, 0, 2, 7,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
        2,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 3, 1, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
        3,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 2, 1, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
        4,   2.1240000e-01,   0.0000000e+00,   0.0000000e+00,  -6.6094234e-03,   3.3808777e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -1.3413366e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -2.0395411e-05, 0, 0, 1, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
        5,   2.1240000e-01,   0.0000000e+00,   0.0000000e+00,  -6.6094234e-03,   3.3808777e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -1.3413366e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -2.0395411e-05, 0, 0, 1, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
        6,   2.1240000e-01,   0.0000000e+00,   0.0000000e+00,  -6.6094234e-03,   3.3808777e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -1.3413366e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -2.0395411e-05, 0, 0, 1, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
        7,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 1, 8, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
        8,  -2.9490000e-02,   1.3651714e-02,   0.0000000e+00,   1.0618999e-02,   1.6947326e-04,   0.0000000e+00,  -2.1876761e-04,   0.0000000e+00,  -5.0077500e-04,   0.0000000e+00,  -2.1876761e-04,   0.0000000e+00,   3.3130174e-04, 0, 1, 9, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
        9,  -7.2850000e-02,   5.4473502e-03,   0.0000000e+00,  -7.8106556e-03,   7.1033901e-05,   0.0000000e+00,  -3.8243495e-04,   0.0000000e+00,   8.8675698e-06,   0.0000000e+00,  -3.8243495e-04,   0.0000000e+00,  -7.9901471e-05, 0, 10, 8, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       10,   6.9690000e-02,   2.4070684e-02,   0.0000000e+00,   2.1202543e-02,   1.1563684e-03,   0.0000000e+00,  -9.1397575e-04,   0.0000000e+00,  -2.6583948e-03,   0.0000000e+00,  -9.1397575e-04,   0.0000000e+00,   1.5020263e-03, 0, 9, 11, -1,   3.9000000e-01,   3.8585043e-01,   3.3000000e-03,
       11,  -1.5553000e-01,   4.0413264e-03,   0.0000000e+00,   2.8771365e-03,  -2.3124755e-04,   0.0000000e+00,   1.6577688e-04,   0.0000000e+00,   2.3636274e-04,   0.0000000e+00,   1.6577688e-04,   0.0000000e+00,  -5.1151876e-06, 0, 10, 9, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       12,   6.3730000e-02,   1.1636607e-03,   0.0000000e+00,  -4.7192024e-03,  -1.1387827e-06,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,   1.7669800e-05,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,  -1.6531017e-05, 0, 8, 9, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       13,   6.3730000e-02,   1.1636607e-03,   0.0000000e+00,  -4.7192024e-03,  -1.1387827e-06,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,   1.7669800e-05,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,  -1.6531017e-05, 0, 8, 9, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       14,   2.5100000e-03,   4.8684303e-04,   0.0000000e+00,  -3.6158679e-03,   6.3165100e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -4.6354054e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -1.6811046e-05, 0, 9, 10, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       15,   2.5100000e-03,   4.8684303e-04,   0.0000000e+00,  -3.6158679e-03,   6.3165100e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -4.6354054e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -1.6811046e-05, 0, 9, 10, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       16,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 11, 10, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       17,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 11, 10, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       18,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 11, 10, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       19,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 20, 23, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       20,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 19, 21, 24,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       21,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 22, 20, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       22,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 21, 20, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       23,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 19, 20, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       24,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 20, 25, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       25,  -5.8800000e-02,   1.4047538e-02,   0.0000000e+00,   1.1230728e-02,   1.9550658e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,  -5.4669968e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,   3.5119310e-04, 0, 20, 26, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       26,   2.4250000e-02,   7.3121707e-03,   0.0000000e+00,  -5.7045303e-03,   3.1945654e-04,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,   3.2520646e-05,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,  -3.5197718e-04, 0, 31, 25, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       27,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 26, 25, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       28,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 26, 25, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       29,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 25, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       30,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 25, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       31,   8.4100000e-03,  -3.5560709e-04,   0.0000000e+00,  -3.4571147e-03,  -2.1002139e-05,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,  -2.0535425e-07,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,   2.1207493e-05, 0, 26, 25, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       32,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 27, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       33,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 27, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       34,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 27, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       35,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 28, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       36,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 28, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       37,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 28, 26, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       38,  -7.7000000e-02,   3.7852046e-03,   0.0000000e+00,   2.5059716e-02,   3.9082647e-05,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,  -1.0511804e-03,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,   1.0120977e-03, 0, 39, 42, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       39,  -3.1424000e-01,   5.8093074e-03,   9.8781510e-03,   3.1025660e-03,  -1.2709561e-04,  -2.9052959e-04,   2.3671744e-05,  -2.9052959e-04,  -3.1528411e-04,  -2.5555403e-04,   2.3671744e-05,  -2.5555403e-04,   4.4237972e-04, 0, 38, 40, 43,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       40,   8.6830000e-01,  -4.3974626e-04,   0.0000000e+00,   1.5173098e-02,   2.8948415e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,  -3.8031607e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,   9.0831918e-05, 0, 41, 39, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       41,  -7.8568000e-01,  -2.5135918e-04,   0.0000000e+00,  -1.1452453e-02,  -5.9135490e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   3.1936319e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   2.7199170e-04, 0, 40, 39, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       42,   1.3014000e-01,  -7.7418626e-04,   0.0000000e+00,  -7.6873573e-03,   4.0501458e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -2.4371816e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -1.6129643e-05, 0, 38, 39, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       43,   9.0300000e-02,   2.9633924e-04,   0.0000000e+00,   3.0385355e-03,  -1.5868283e-06,   0.0000000e+00,   9.8009982e-06,   0.0000000e+00,   1.6241654e-06,   0.0000000e+00,   9.8009982e-06,   0.0000000e+00,  -3.7337136e-08, 0, 39, 44, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       44,   1.8740000e-01,   1.7278694e-02,   0.0000000e+00,   8.9452116e-03,   3.6295430e-04,   0.0000000e+00,  -3.7583561e-04,   0.0000000e+00,  -6.0389084e-04,   0.0000000e+00,  -3.7583561e-04,   0.0000000e+00,   2.4093654e-04, 0, 39, 45, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       45,  -4.1295000e-01,   1.7699919e-02,   0.0000000e+00,  -7.5413044e-03,   3.9622169e-04,   0.0000000e+00,  -3.5747508e-04,   0.0000000e+00,  -3.8488987e-04,   0.0000000e+00,  -3.5747508e-04,   0.0000000e+00,  -1.1331821e-05, 0, 48, 44, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
       46,   1.9820000e-02,  -6.5565056e-04,   0.0000000e+00,  -6.2162447e-03,   1.8397874e-05,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,   3.5563622e-06,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,  -2.1954236e-05, 0, 44, 39, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       47,   1.9820000e-02,  -6.5565056e-04,   0.0000000e+00,  -6.2162447e-03,   1.8397874e-05,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,   3.5563622e-06,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,  -2.1954236e-05, 0, 44, 39, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       48,   2.7409000e-01,  -1.2118158e-04,   0.0000000e+00,  -7.4899742e-03,   3.3407403e-05,   0.0000000e+00,   7.1743307e-05,   0.0000000e+00,  -1.7828482e-06,   0.0000000e+00,   7.1743307e-05,   0.0000000e+00,  -3.1624554e-05, 0, 45, 44, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       49,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 50, 53, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       50,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 49, 51, 54,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       51,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 52, 50, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       52,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 51, 50, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       53,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 49, 50, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       54,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 50, 55, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       55,  -3.3040000e-01,  -5.9029718e-03,   0.0000000e+00,   1.7571329e-02,  -5.7842691e-04,   0.0000000e+00,  -1.3539379e-04,   0.0000000e+00,  -2.4709717e-04,   0.0000000e+00,  -1.3539379e-04,   0.0000000e+00,   8.2552408e-04, 0, 50, 56, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       56,   1.0164400e+00,   0.0000000e+00,   0.0000000e+00,   4.4387384e-03,   3.6350502e-04,   0.0000000e+00,  -4.5513035e-04,   0.0000000e+00,  -3.7872924e-04,   0.0000000e+00,  -4.5513035e-04,   0.0000000e+00,   1.5224217e-05, 1, 57, 58, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       57,  -8.5689000e-01,  -6.5268717e-03,   0.0000000e+00,  -9.4722721e-03,  -2.8521838e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4084501e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4437337e-04, 0, 56, 55, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       58,  -8.5689000e-01,  -6.5268717e-03,   0.0000000e+00,  -9.4722721e-03,  -2.8521838e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4084501e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4437337e-04, 0, 56, 55, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       59,   4.8930000e-02,   2.7215584e-03,   0.0000000e+00,  -3.5179701e-03,   2.6610177e-04,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,   9.5116354e-06,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,  -2.7561340e-04, 0, 55, 56, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       60,   4.8930000e-02,   2.7215584e-03,   0.0000000e+00,  -3.5179701e-03,   2.6610177e-04,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,   9.5116354e-06,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,  -2.7561340e-04, 0, 55, 56, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       61,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 62, 65, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       62,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 61, 63, 66,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       63,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 64, 62, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       64,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 63, 62, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       65,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 61, 62, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       66,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 62, 67, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       67,   4.1000000e-03,   1.0333243e-02,   0.0000000e+00,   9.9988034e-03,   2.0035107e-04,   0.0000000e+00,  -5.4810916e-04,   0.0000000e+00,  -8.1410825e-04,   0.0000000e+00,  -5.4810916e-04,   0.0000000e+00,   6.1375718e-04, 0, 62, 68, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       68,  -3.6014000e-01,  -1.1218028e-02,   0.0000000e+00,   1.3508836e-02,  -8.6359862e-04,   0.0000000e+00,  -1.8817917e-05,   0.0000000e+00,   1.0361989e-04,   0.0000000e+00,  -1.8817917e-05,   0.0000000e+00,   7.5997874e-04, 0, 67, 69, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       69,   1.1459600e+00,   0.0000000e+00,   0.0000000e+00,  -4.2916272e-03,   4.4307979e-04,   0.0000000e+00,  -1.9475984e-04,   0.0000000e+00,  -5.0209114e-04,   0.0000000e+00,  -1.9475984e-04,   0.0000000e+00,   5.9011344e-05, 1, 70, 71, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       70,  -8.9716000e-01,   1.2218173e-02,   0.0000000e+00,  -1.4797912e-02,  -4.5542905e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   2.6634446e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   1.8908459e-04, 0, 69, 71, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       71,  -8.9716000e-01,   1.2218173e-02,   0.0000000e+00,  -1.4797912e-02,  -4.5542905e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   2.6634446e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   1.8908459e-04, 0, 69, 70, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       72,   4.0820000e-02,   2.9596881e-03,   0.0000000e+00,  -7.2015727e-03,   1.8051572e-04,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,   4.5364620e-06,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,  -1.8505218e-04, 0, 67, 68, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       73,   4.0820000e-02,   2.9596881e-03,   0.0000000e+00,  -7.2015727e-03,   1.8051572e-04,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,   4.5364620e-06,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,  -1.8505218e-04, 0, 67, 68, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       74,  -3.5600000e-03,   7.5936930e-04,   0.0000000e+00,  -1.9754185e-03,   2.8601180e-04,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -9.5480391e-05,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -1.9053141e-04, 0, 68, 69, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       75,  -3.5600000e-03,   7.5936930e-04,   0.0000000e+00,  -1.9754185e-03,   2.8601180e-04,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -9.5480391e-05,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -1.9053141e-04, 0, 68, 69, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       76,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 77, 80, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       77,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 76, 78, 81,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       78,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 79, 77, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       79,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 78, 77, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       80,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 76, 77, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       81,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 77, 82, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       82,  -3.3040000e-01,  -5.9029718e-03,   0.0000000e+00,   1.7571329e-02,  -5.7842691e-04,   0.0000000e+00,  -1.3539379e-04,   0.0000000e+00,  -2.4709717e-04,   0.0000000e+00,  -1.3539379e-04,   0.0000000e+00,   8.2552408e-04, 0, 77, 83, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       83,   1.0164400e+00,   0.0000000e+00,   0.0000000e+00,   4.4387384e-03,   3.6350502e-04,   0.0000000e+00,  -4.5513035e-04,   0.0000000e+00,  -3.7872924e-04,   0.0000000e+00,  -4.5513035e-04,   0.0000000e+00,   1.5224217e-05, 1, 84, 85, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       84,  -8.5689000e-01,  -6.5268717e-03,   0.0000000e+00,  -9.4722721e-03,  -2.8521838e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4084501e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4437337e-04, 0, 83, 82, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       85,  -8.5689000e-01,  -6.5268717e-03,   0.0000000e+00,  -9.4722721e-03,  -2.8521838e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4084501e-04,   0.0000000e+00,  -2.3889233e-04,   0.0000000e+00,   1.4437337e-04, 0, 83, 82, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       86,   4.8930000e-02,   2.7215584e-03,   0.0000000e+00,  -3.5179701e-03,   2.6610177e-04,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,   9.5116354e-06,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,  -2.7561340e-04, 0, 82, 83, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       87,   4.8930000e-02,   2.7215584e-03,   0.0000000e+00,  -3.5179701e-03,   2.6610177e-04,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,   9.5116354e-06,   0.0000000e+00,   9.5489726e-06,   0.0000000e+00,  -2.7561340e-04, 0, 82, 83, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       88,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 89, 92, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
       89,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 88, 90, 93,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       90,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 91, 89, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       91,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 90, 89, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
       92,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 88, 89, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       93,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 89, 94, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
       94,  -1.8950000e-02,   1.9556273e-02,   0.0000000e+00,   5.8950341e-04,   3.8441382e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,  -6.1989914e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,   2.3548532e-04, 0, 95, 89, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
       95,  -6.2750000e-02,  -8.0170347e-04,   0.0000000e+00,   6.2003694e-03,   9.7860634e-05,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,  -2.5117625e-04,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,   1.5331562e-04, 0, 94, 96, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
       96,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 95, 98, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
       97,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 95, 99, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
       98,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 96, 100, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
       99,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 97, 100, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      100,  -5.8260000e-02,  -5.1859367e-05,   0.0000000e+00,   6.5882563e-04,  -3.0261749e-05,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,  -1.6282725e-04,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,   1.9308900e-04, 0, 98, 99, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      101,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 94, 95, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      102,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 94, 95, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      103,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 96, 95, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      104,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 97, 95, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      105,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 98, 96, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      106,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 99, 97, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      107,   3.5880000e-02,   2.0743747e-04,   0.0000000e+00,  -7.8995574e-03,   4.4888572e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,   1.2059895e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,  -5.6948467e-05, 0, 100, 98, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      108,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 109, 112, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      109,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 108, 110, 113,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      110,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 111, 109, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      111,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 110, 109, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      112,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 108, 109, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      113,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 109, 114, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      114,   1.4170000e-02,   7.5783468e-03,   0.0000000e+00,   1.0956085e-02,   2.8334219e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,  -7.3121047e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,   4.4786828e-04, 0, 109, 115, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      115,  -1.4658000e-01,   3.9196156e-03,   0.0000000e+00,   1.4685197e-02,  -6.8233616e-06,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,  -4.2669813e-04,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,   4.3352149e-04, 0, 116, 114, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      116,  -1.2517000e-01,   5.4621672e-03,   0.0000000e+00,   1.6352105e-02,  -1.0743761e-05,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,  -5.1248953e-04,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,   5.2323329e-04, 0, 117, 115, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      117,  -6.3700000e-03,   7.8831529e-03,   0.0000000e+00,   2.4261717e-02,  -1.5507046e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,  -7.0800544e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,   8.6307590e-04, 0, 118, 116, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      118,   1.0679000e-01,   0.0000000e+00,   0.0000000e+00,   6.7078503e-03,  -1.5848681e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,  -1.2231646e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,   2.8080327e-04, 0, 117, 127, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      119,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 114, 115, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      120,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 114, 115, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      121,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 115, 122, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      122,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 115, 121, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      123,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 116, 124, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      124,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 116, 123, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      125,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 117, 126, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      126,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 117, 125, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      127,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 118, 117, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      128,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 118, 117, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      129,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 118, 117, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      130,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 131, 134, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      131,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 130, 132, 135,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      132,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 133, 131, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      133,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 132, 131, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      134,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 130, 131, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      135,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 131, 136, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      136,  -1.5440000e-01,  -5.2917721e-06,   0.0000000e+00,   1.9202253e-02,  -2.7512802e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,  -2.7337318e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,   5.4850120e-04, 0, 131, 135, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      137,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 136, 131, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      138,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 136, 131, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      139,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 136, 131, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      140,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 141, 144, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      141,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 140, 142, 145,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      142,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 143, 141, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      143,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 142, 141, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      144,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 140, 141, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      145,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 141, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      146,   1.4820000e-02,   4.2440012e-03,   0.0000000e+00,  -1.4286197e-02,   3.1767369e-04,   0.0000000e+00,  -2.3279704e-05,   0.0000000e+00,   1.3306022e-04,   0.0000000e+00,  -2.3279704e-05,   0.0000000e+00,  -4.5073391e-04, 0, 149, 141, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      147,  -1.7773000e-01,   8.6732145e-04,   0.0000000e+00,   1.4602645e-02,  -2.4876800e-04,   0.0000000e+00,  -2.8189538e-06,   0.0000000e+00,  -2.2548830e-04,   0.0000000e+00,  -2.8189538e-06,   0.0000000e+00,   4.7425630e-04, 0, 146, 141, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      148,  -1.7773000e-01,   8.6732145e-04,   0.0000000e+00,   1.4602645e-02,  -2.4876800e-04,   0.0000000e+00,  -2.8189538e-06,   0.0000000e+00,  -2.2548830e-04,   0.0000000e+00,  -2.8189538e-06,   0.0000000e+00,   4.7425630e-04, 0, 146, 141, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      149,   6.6180000e-02,  -8.2022468e-04,   0.0000000e+00,  -2.0103442e-03,  -1.9527322e-05,   0.0000000e+00,   1.9601996e-06,   0.0000000e+00,  -8.0741557e-06,   0.0000000e+00,   1.9601996e-06,   0.0000000e+00,   2.7601478e-05, 0, 146, 141, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      150,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 147, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      151,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 147, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      152,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 147, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      153,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 148, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      154,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 148, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      155,   5.7430000e-02,   1.3758607e-04,   0.0000000e+00,  -5.1896409e-03,  -9.2782783e-06,   1.1014455e-06,   0.0000000e+00,   1.1014455e-06,  -1.8257860e-05,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   2.7536138e-05, 0, 148, 146, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      156,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 157, 160, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      157,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 156, 158, 161,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      158,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 159, 157, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      159,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 158, 157, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      160,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 156, 157, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      161,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 157, 162, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      162,  -1.8950000e-02,   1.9556273e-02,   0.0000000e+00,   5.8950341e-04,   3.8441382e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,  -6.1989914e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,   2.3548532e-04, 0, 163, 157, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      163,  -6.2750000e-02,  -8.0170347e-04,   0.0000000e+00,   6.2003694e-03,   9.7860634e-05,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,  -2.5117625e-04,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,   1.5331562e-04, 0, 162, 164, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      164,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 163, 166, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      165,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 163, 167, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      166,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 164, 168, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      167,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 165, 168, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      168,  -5.8260000e-02,  -5.1859367e-05,   0.0000000e+00,   6.5882563e-04,  -3.0261749e-05,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,  -1.6282725e-04,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,   1.9308900e-04, 0, 166, 167, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      169,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 162, 163, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      170,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 162, 163, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      171,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 164, 163, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      172,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 165, 163, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      173,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 166, 164, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      174,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 167, 165, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      175,   3.5880000e-02,   2.0743747e-04,   0.0000000e+00,  -7.8995574e-03,   4.4888572e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,   1.2059895e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,  -5.6948467e-05, 0, 168, 166, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      176,  -2.2620000e-01,   4.3466616e-03,   0.0000000e+00,   1.8459289e-02,   1.0991119e-04,   0.0000000e+00,  -1.6386336e-04,   0.0000000e+00,  -9.5382381e-04,   0.0000000e+00,  -1.6386336e-04,   0.0000000e+00,   8.4391262e-04, 0, 177, 180, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      177,  -1.5245000e-01,   1.0327952e-02,   0.0000000e+00,   1.0417912e-02,  -1.9300499e-04,   0.0000000e+00,  -1.5606923e-05,   0.0000000e+00,  -4.4882971e-04,   0.0000000e+00,  -1.5606923e-05,   0.0000000e+00,   6.4183470e-04, 0, 176, 178, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      178,   8.4374000e-01,  -5.4669298e-03,   0.0000000e+00,   1.1569931e-02,   1.2128035e-04,   0.0000000e+00,   2.7722824e-06,   0.0000000e+00,  -2.9704492e-04,   0.0000000e+00,   2.7722824e-06,   0.0000000e+00,   1.7576457e-04, 0, 179, 177, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      179,  -7.3597000e-01,  -1.0901051e-04,   0.0000000e+00,  -1.0256513e-02,  -4.9353160e-04,   0.0000000e+00,   1.6036300e-05,   0.0000000e+00,   2.9642886e-04,   0.0000000e+00,   1.6036300e-05,   0.0000000e+00,   1.9710274e-04, 0, 178, 177, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      180,   1.2752000e-01,   3.4237766e-04,   0.0000000e+00,  -6.7644723e-03,   2.9897712e-05,   0.0000000e+00,  -6.2353017e-06,   0.0000000e+00,  -2.2635639e-05,   0.0000000e+00,  -6.2353017e-06,   0.0000000e+00,  -7.2620730e-06, 0, 176, 177, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      181,   7.1680000e-02,  -3.5454873e-04,   0.0000000e+00,  -1.8806958e-03,  -1.0687755e-05,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   3.9577364e-06,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   6.7300188e-06, 0, 177, 182, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      182,   7.1680000e-02,  -3.5454873e-04,   0.0000000e+00,  -1.8806958e-03,  -1.0687755e-05,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   3.9577364e-06,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   6.7300188e-06, 0, 177, 181, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      183,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 184, 187, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      184,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 183, 185, 188,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      185,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 186, 184, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      186,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 185, 184, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      187,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 183, 184, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      188,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 184, 189, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      189,  -2.9490000e-02,   1.3651714e-02,   0.0000000e+00,   1.0618999e-02,   1.6947326e-04,   0.0000000e+00,  -2.1876761e-04,   0.0000000e+00,  -5.0077500e-04,   0.0000000e+00,  -2.1876761e-04,   0.0000000e+00,   3.3130174e-04, 0, 184, 190, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      190,  -7.2850000e-02,   5.4473502e-03,   0.0000000e+00,  -7.8106556e-03,   7.1033901e-05,   0.0000000e+00,  -3.8243495e-04,   0.0000000e+00,   8.8675698e-06,   0.0000000e+00,  -3.8243495e-04,   0.0000000e+00,  -7.9901471e-05, 0, 191, 189, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      191,   6.9690000e-02,   2.4070684e-02,   0.0000000e+00,   2.1202543e-02,   1.1563684e-03,   0.0000000e+00,  -9.1397575e-04,   0.0000000e+00,  -2.6583948e-03,   0.0000000e+00,  -9.1397575e-04,   0.0000000e+00,   1.5020263e-03, 0, 190, 192, -1,   3.9000000e-01,   3.8585043e-01,   3.3000000e-03,
      192,  -1.5553000e-01,   4.0413264e-03,   0.0000000e+00,   2.8771365e-03,  -2.3124755e-04,   0.0000000e+00,   1.6577688e-04,   0.0000000e+00,   2.3636274e-04,   0.0000000e+00,   1.6577688e-04,   0.0000000e+00,  -5.1151876e-06, 0, 191, 190, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      193,   6.3730000e-02,   1.1636607e-03,   0.0000000e+00,  -4.7192024e-03,  -1.1387827e-06,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,   1.7669800e-05,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,  -1.6531017e-05, 0, 189, 190, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      194,   6.3730000e-02,   1.1636607e-03,   0.0000000e+00,  -4.7192024e-03,  -1.1387827e-06,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,   1.7669800e-05,   0.0000000e+00,   6.9503079e-05,   0.0000000e+00,  -1.6531017e-05, 0, 189, 190, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      195,   2.5100000e-03,   4.8684303e-04,   0.0000000e+00,  -3.6158679e-03,   6.3165100e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -4.6354054e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -1.6811046e-05, 0, 190, 191, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      196,   2.5100000e-03,   4.8684303e-04,   0.0000000e+00,  -3.6158679e-03,   6.3165100e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -4.6354054e-05,   0.0000000e+00,   4.7782200e-05,   0.0000000e+00,  -1.6811046e-05, 0, 190, 191, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      197,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 192, 191, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      198,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 192, 191, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      199,   4.1940000e-02,   5.3341063e-04,   0.0000000e+00,  -5.4626963e-03,   6.7430868e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -2.1515525e-05,   0.0000000e+00,  -2.9197640e-05,   0.0000000e+00,  -4.5915343e-05, 0, 192, 191, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      200,  -7.7000000e-02,   3.7852046e-03,   0.0000000e+00,   2.5059716e-02,   3.9082647e-05,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,  -1.0511804e-03,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,   1.0120977e-03, 0, 201, 204, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      201,  -3.1424000e-01,   5.8093074e-03,   9.8781510e-03,   3.1025660e-03,  -1.2709561e-04,  -2.9052959e-04,   2.3671744e-05,  -2.9052959e-04,  -3.1528411e-04,  -2.5555403e-04,   2.3671744e-05,  -2.5555403e-04,   4.4237972e-04, 0, 200, 202, 205,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      202,   8.6830000e-01,  -4.3974626e-04,   0.0000000e+00,   1.5173098e-02,   2.8948415e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,  -3.8031607e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,   9.0831918e-05, 0, 203, 201, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      203,  -7.8568000e-01,  -2.5135918e-04,   0.0000000e+00,  -1.1452453e-02,  -5.9135490e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   3.1936319e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   2.7199170e-04, 0, 202, 201, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      204,   1.3014000e-01,  -7.7418626e-04,   0.0000000e+00,  -7.6873573e-03,   4.0501458e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -2.4371816e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -1.6129643e-05, 0, 200, 201, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      205,   8.9720000e-02,  -4.6990936e-04,   0.0000000e+00,   4.1889668e-03,  -1.6503014e-05,   0.0000000e+00,  -1.9975368e-06,   0.0000000e+00,  -1.7903157e-05,   0.0000000e+00,  -1.9975368e-06,   0.0000000e+00,   3.4406171e-05, 0, 201, 206, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      206,   1.7854000e-01,   1.2636752e-02,   0.0000000e+00,   1.6965421e-02,   9.8318014e-05,   0.0000000e+00,  -2.7276645e-04,   0.0000000e+00,  -6.0115589e-04,   0.0000000e+00,  -2.7276645e-04,   0.0000000e+00,   5.0283788e-04, 0, 207, 201, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      207,  -3.9938000e-01,   1.6875990e-02,   0.0000000e+00,   3.2607900e-03,   1.6615959e-04,   0.0000000e+00,  -3.2564517e-04,   0.0000000e+00,  -6.1119025e-04,   0.0000000e+00,  -3.2564517e-04,   0.0000000e+00,   4.4503066e-04, 0, 210, 206, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      208,  -1.5307000e-01,  -1.3229430e-05,   0.0000000e+00,   2.0133076e-02,  -2.8164335e-04,   0.0000000e+00,  -5.9254035e-05,   0.0000000e+00,  -2.1424982e-04,   0.0000000e+00,  -5.9254035e-05,   0.0000000e+00,   4.9589317e-04, 0, 206, 211, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      209,   2.1340000e-02,  -5.0271835e-04,   0.0000000e+00,  -5.2536714e-03,   6.4033188e-06,   0.0000000e+00,   5.3541453e-05,   0.0000000e+00,   5.7312504e-06,   0.0000000e+00,   5.3541453e-05,   0.0000000e+00,  -1.2134569e-05, 0, 206, 207, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      210,   2.2557000e-01,  -2.6760492e-03,   0.0000000e+00,  -1.3922652e-03,  -3.3967460e-05,   0.0000000e+00,  -3.4312828e-05,   0.0000000e+00,  -7.4338238e-05,   0.0000000e+00,  -3.4312828e-05,   0.0000000e+00,   1.0830570e-04, 0, 207, 206, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      211,   7.1920000e-02,  -3.5931133e-04,   0.0000000e+00,  -5.3198185e-03,   6.2082323e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,   2.0806119e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,  -8.2888442e-05, 0, 208, 206, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      212,   7.1920000e-02,  -3.5931133e-04,   0.0000000e+00,  -5.3198185e-03,   6.2082323e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,   2.0806119e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,  -8.2888442e-05, 0, 208, 206, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      213,   7.1920000e-02,  -3.5931133e-04,   0.0000000e+00,  -5.3198185e-03,   6.2082323e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,   2.0806119e-05,   0.0000000e+00,  -2.6229338e-06,   0.0000000e+00,  -8.2888442e-05, 0, 208, 206, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      214,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 215, 218, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      215,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 214, 216, 219,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      216,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 217, 215, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      217,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 216, 215, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      218,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 214, 215, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      219,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 215, 220, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      220,  -5.8100000e-02,   1.0820086e-02,   0.0000000e+00,   1.1999093e-02,   1.4391599e-04,   0.0000000e+00,  -1.9024204e-04,   0.0000000e+00,  -5.7209827e-04,   0.0000000e+00,  -1.9024204e-04,   0.0000000e+00,   4.2818228e-04, 0, 215, 221, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      221,  -1.2460000e-01,   1.8130670e-02,   0.0000000e+00,   5.7637982e-03,   2.7841369e-04,   0.0000000e+00,  -3.8348039e-04,   0.0000000e+00,  -5.7686809e-04,   0.0000000e+00,  -3.8348039e-04,   0.0000000e+00,   2.9845440e-04, 0, 220, 222, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      222,  -5.0400000e-03,   1.6519325e-02,   0.0000000e+00,   6.4469660e-03,   5.9600337e-04,   0.0000000e+00,  -3.9900330e-04,   0.0000000e+00,  -5.6310002e-04,   0.0000000e+00,  -3.9900330e-04,   0.0000000e+00,  -3.2903351e-05, 0, 221, 223, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      223,  -2.6789000e-01,  -9.0129463e-03,   0.0000000e+00,   2.0332576e-02,   1.2257782e-04,   0.0000000e+00,   1.7695002e-04,   0.0000000e+00,  -8.0230038e-04,   0.0000000e+00,   1.7695002e-04,   0.0000000e+00,   6.7972256e-04, 0, 222, 224, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      224,   9.0680000e-01,   2.1802101e-04,   0.0000000e+00,   9.6204417e-04,  -1.0379724e-04,   0.0000000e+00,  -1.9686005e-05,   0.0000000e+00,   6.5433331e-05,   0.0000000e+00,  -1.9686005e-05,   0.0000000e+00,   3.8363907e-05, 1, 225, 226, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      225,  -3.0169000e-01,  -8.8901771e-05,   0.0000000e+00,  -1.3081790e-02,   4.0242899e-04,   0.0000000e+00,  -6.8877682e-05,   0.0000000e+00,  -6.4565243e-04,   0.0000000e+00,  -6.8877682e-05,   0.0000000e+00,   2.4322344e-04, 0, 224, 226, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      226,  -3.0169000e-01,  -8.8901771e-05,   0.0000000e+00,  -1.3081790e-02,   4.0242899e-04,   0.0000000e+00,  -6.8877682e-05,   0.0000000e+00,  -6.4565243e-04,   0.0000000e+00,  -6.8877682e-05,   0.0000000e+00,   2.4322344e-04, 0, 224, 225, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      227,   7.3320000e-02,   1.5160927e-03,   0.0000000e+00,  -3.7391662e-03,  -1.6120309e-05,   0.0000000e+00,  -1.9116614e-05,   0.0000000e+00,   4.7940883e-05,   0.0000000e+00,  -1.9116614e-05,   0.0000000e+00,  -3.1820574e-05, 0, 220, 221, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      228,   7.3320000e-02,   1.5160927e-03,   0.0000000e+00,  -3.7391662e-03,  -1.6120309e-05,   0.0000000e+00,  -1.9116614e-05,   0.0000000e+00,   4.7940883e-05,   0.0000000e+00,  -1.9116614e-05,   0.0000000e+00,  -3.1820574e-05, 0, 220, 221, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      229,   1.0013000e-01,  -1.2732004e-03,   0.0000000e+00,  -2.4204566e-03,   4.0660141e-05,   0.0000000e+00,   3.3640760e-05,   0.0000000e+00,   3.9932067e-05,   0.0000000e+00,   3.3640760e-05,   0.0000000e+00,  -8.0592208e-05, 0, 221, 230, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      230,   1.0013000e-01,  -1.2732004e-03,   0.0000000e+00,  -2.4204566e-03,   4.0660141e-05,   0.0000000e+00,   3.3640760e-05,   0.0000000e+00,   3.9932067e-05,   0.0000000e+00,   3.3640760e-05,   0.0000000e+00,  -8.0592208e-05, 0, 221, 229, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      231,   6.3860000e-02,   2.7570133e-04,   0.0000000e+00,  -6.6708079e-03,   6.3519803e-05,   0.0000000e+00,   4.1518895e-05,   0.0000000e+00,   1.1230077e-04,   0.0000000e+00,   4.1518895e-05,   0.0000000e+00,  -1.7582057e-04, 0, 222, 232, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      232,   6.3860000e-02,   2.7570133e-04,   0.0000000e+00,  -6.6708079e-03,   6.3519803e-05,   0.0000000e+00,   4.1518895e-05,   0.0000000e+00,   1.1230077e-04,   0.0000000e+00,   4.1518895e-05,   0.0000000e+00,  -1.7582057e-04, 0, 222, 231, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      233,   1.4895000e-01,  -1.0864008e-03,   0.0000000e+00,  -1.0259688e-02,   6.9671096e-05,   0.0000000e+00,   8.0209503e-05,   0.0000000e+00,   3.4424839e-05,   0.0000000e+00,   8.0209503e-05,   0.0000000e+00,  -1.0409594e-04, 0, 223, 224, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      234,   1.4969000e-01,   8.6890898e-04,   0.0000000e+00,  -1.1642957e-02,   1.6250988e-04,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -3.0877812e-05,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -1.3163207e-04, 0, 225, 224, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      235,   1.4969000e-01,   8.6890898e-04,   0.0000000e+00,  -1.1642957e-02,   1.6250988e-04,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -3.0877812e-05,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -1.3163207e-04, 0, 225, 224, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      236,   1.4969000e-01,   8.6890898e-04,   0.0000000e+00,  -1.1642957e-02,   1.6250988e-04,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -3.0877812e-05,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -1.3163207e-04, 0, 226, 224, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      237,   1.4969000e-01,   8.6890898e-04,   0.0000000e+00,  -1.1642957e-02,   1.6250988e-04,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -3.0877812e-05,   0.0000000e+00,  -2.5827964e-05,   0.0000000e+00,  -1.3163207e-04, 0, 226, 224, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      238,  -7.7000000e-02,   3.7852046e-03,   0.0000000e+00,   2.5059716e-02,   3.9082647e-05,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,  -1.0511804e-03,   0.0000000e+00,  -1.2421132e-04,   0.0000000e+00,   1.0120977e-03, 0, 239, 242, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      239,  -3.1424000e-01,   5.8093074e-03,   9.8781510e-03,   3.1025660e-03,  -1.2709561e-04,  -2.9052959e-04,   2.3671744e-05,  -2.9052959e-04,  -3.1528411e-04,  -2.5555403e-04,   2.3671744e-05,  -2.5555403e-04,   4.4237972e-04, 0, 238, 240, 243,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      240,   8.6830000e-01,  -4.3974626e-04,   0.0000000e+00,   1.5173098e-02,   2.8948415e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,  -3.8031607e-04,   0.0000000e+00,  -1.8733908e-05,   0.0000000e+00,   9.0831918e-05, 0, 241, 239, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      241,  -7.8568000e-01,  -2.5135918e-04,   0.0000000e+00,  -1.1452453e-02,  -5.9135490e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   3.1936319e-04,   0.0000000e+00,   2.0834122e-05,   0.0000000e+00,   2.7199170e-04, 0, 240, 239, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      242,   1.3014000e-01,  -7.7418626e-04,   0.0000000e+00,  -7.6873573e-03,   4.0501458e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -2.4371816e-05,   0.0000000e+00,  -3.1829909e-06,   0.0000000e+00,  -1.6129643e-05, 0, 238, 239, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      243,   9.0300000e-02,   2.9633924e-04,   0.0000000e+00,   3.0385355e-03,  -1.5868283e-06,   0.0000000e+00,   9.8009982e-06,   0.0000000e+00,   1.6241654e-06,   0.0000000e+00,   9.8009982e-06,   0.0000000e+00,  -3.7337136e-08, 0, 239, 244, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      244,   1.8740000e-01,   1.7278694e-02,   0.0000000e+00,   8.9452116e-03,   3.6295430e-04,   0.0000000e+00,  -3.7583561e-04,   0.0000000e+00,  -6.0389084e-04,   0.0000000e+00,  -3.7583561e-04,   0.0000000e+00,   2.4093654e-04, 0, 239, 245, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      245,  -4.1295000e-01,   1.7699919e-02,   0.0000000e+00,  -7.5413044e-03,   3.9622169e-04,   0.0000000e+00,  -3.5747508e-04,   0.0000000e+00,  -3.8488987e-04,   0.0000000e+00,  -3.5747508e-04,   0.0000000e+00,  -1.1331821e-05, 0, 248, 244, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      246,   1.9820000e-02,  -6.5565056e-04,   0.0000000e+00,  -6.2162447e-03,   1.8397874e-05,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,   3.5563622e-06,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,  -2.1954236e-05, 0, 244, 239, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      247,   1.9820000e-02,  -6.5565056e-04,   0.0000000e+00,  -6.2162447e-03,   1.8397874e-05,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,   3.5563622e-06,   0.0000000e+00,  -9.4556297e-06,   0.0000000e+00,  -2.1954236e-05, 0, 244, 239, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      248,   2.7409000e-01,  -1.2118158e-04,   0.0000000e+00,  -7.4899742e-03,   3.3407403e-05,   0.0000000e+00,   7.1743307e-05,   0.0000000e+00,  -1.7828482e-06,   0.0000000e+00,   7.1743307e-05,   0.0000000e+00,  -3.1624554e-05, 0, 245, 244, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      249,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 250, 253, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      250,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 249, 251, 254,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      251,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 252, 250, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      252,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 251, 250, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      253,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 249, 250, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      254,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 250, 255, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      255,  -1.5440000e-01,  -5.2917721e-06,   0.0000000e+00,   1.9202253e-02,  -2.7512802e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,  -2.7337318e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,   5.4850120e-04, 0, 250, 254, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      256,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 255, 250, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      257,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 255, 250, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      258,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 255, 250, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      259,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 260, 263, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      260,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 259, 261, 264,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      261,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 262, 260, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      262,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 261, 260, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      263,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 259, 260, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      264,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 260, 265, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      265,  -1.8950000e-02,   1.9556273e-02,   0.0000000e+00,   5.8950341e-04,   3.8441382e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,  -6.1989914e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,   2.3548532e-04, 0, 266, 260, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      266,  -6.2750000e-02,  -8.0170347e-04,   0.0000000e+00,   6.2003694e-03,   9.7860634e-05,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,  -2.5117625e-04,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,   1.5331562e-04, 0, 265, 267, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      267,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 266, 269, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      268,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 266, 270, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      269,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 267, 271, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      270,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 268, 271, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      271,  -5.8260000e-02,  -5.1859367e-05,   0.0000000e+00,   6.5882563e-04,  -3.0261749e-05,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,  -1.6282725e-04,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,   1.9308900e-04, 0, 269, 270, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      272,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 265, 266, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      273,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 265, 266, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      274,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 267, 266, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      275,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 268, 266, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      276,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 269, 267, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      277,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 270, 268, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      278,   3.5880000e-02,   2.0743747e-04,   0.0000000e+00,  -7.8995574e-03,   4.4888572e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,   1.2059895e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,  -5.6948467e-05, 0, 271, 269, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      279,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 280, 283, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      280,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 279, 281, 284,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      281,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 282, 280, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      282,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 281, 280, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      283,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 279, 280, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      284,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 280, 285, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      285,  -1.5440000e-01,  -5.2917721e-06,   0.0000000e+00,   1.9202253e-02,  -2.7512802e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,  -2.7337318e-04,   0.0000000e+00,   7.1127244e-06,   0.0000000e+00,   5.4850120e-04, 0, 280, 284, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      286,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 285, 280, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      287,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 285, 280, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      288,   7.4840000e-02,  -5.8738670e-05,   0.0000000e+00,  -5.2970639e-03,  -2.8376223e-06,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,  -1.4356129e-05,   0.0000000e+00,   1.1574512e-05,   0.0000000e+00,   1.7193751e-05, 0, 285, 280, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      289,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 290, 293, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      290,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 289, 291, 294,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      291,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 292, 290, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      292,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 291, 290, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      293,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 289, 290, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      294,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 290, 295, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      295,  -1.5072000e-01,   1.1297933e-02,   0.0000000e+00,   1.3891960e-02,   5.0134439e-05,   0.0000000e+00,  -1.3707396e-04,   0.0000000e+00,  -4.7686057e-04,   0.0000000e+00,  -1.3707396e-04,   0.0000000e+00,   4.2672613e-04, 0, 290, 296, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      296,   7.6960000e-01,   1.6207110e-02,   0.0000000e+00,  -4.9541570e-03,   1.9747611e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,  -3.7147650e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,   1.7400039e-04, 0, 298, 297, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      297,  -7.2950000e-01,  -1.9516056e-03,   0.0000000e+00,  -8.4678937e-03,  -2.5694484e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   2.5177364e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   5.1711933e-06, 0, 296, 298, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      298,  -3.8020000e-01,   4.9493945e-03,   0.0000000e+00,   1.1738209e-02,  -1.4391599e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,  -4.3525766e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,   5.7917365e-04, 0, 296, 297, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      299,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 295, 296, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      300,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 295, 296, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      301,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 298, 296, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      302,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 298, 296, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      303,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 304, 307, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      304,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 303, 305, 308,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      305,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 306, 304, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      306,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 305, 304, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      307,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 303, 304, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      308,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 304, 309, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      309,  -5.8800000e-02,   1.4047538e-02,   0.0000000e+00,   1.1230728e-02,   1.9550658e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,  -5.4669968e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,   3.5119310e-04, 0, 304, 310, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      310,   2.4250000e-02,   7.3121707e-03,   0.0000000e+00,  -5.7045303e-03,   3.1945654e-04,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,   3.2520646e-05,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,  -3.5197718e-04, 0, 315, 309, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      311,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 310, 309, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      312,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 310, 309, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      313,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 309, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      314,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 309, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      315,   8.4100000e-03,  -3.5560709e-04,   0.0000000e+00,  -3.4571147e-03,  -2.1002139e-05,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,  -2.0535425e-07,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,   2.1207493e-05, 0, 310, 309, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      316,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 311, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      317,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 311, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      318,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 311, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      319,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 312, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      320,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 312, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      321,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 312, 310, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      322,  -1.1893000e-01,   1.8015309e-02,   0.0000000e+00,   1.8269843e-02,   1.2558346e-04,   0.0000000e+00,  -5.1772606e-04,   0.0000000e+00,  -1.1838766e-03,   0.0000000e+00,  -5.1772606e-04,   0.0000000e+00,   1.0582931e-03, 0, 323, 329, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      323,  -3.8385000e-01,   4.9229356e-03,   1.6080637e-02,   1.1009003e-02,  -2.6614844e-04,  -1.0846438e-04,  -6.3099760e-05,  -1.0846438e-04,  -1.7866753e-04,  -5.1814611e-05,  -6.3099760e-05,  -5.1814611e-05,   4.4481597e-04, 0, 322, 324, 326,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      324,   9.8874000e-01,   1.4769336e-03,   0.0000000e+00,   1.6227748e-02,   2.8630116e-04,   0.0000000e+00,   8.1973682e-05,   0.0000000e+00,  -4.5175134e-04,   0.0000000e+00,   8.1973682e-05,   0.0000000e+00,   1.6545018e-04, 0, 325, 323, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      325,  -8.2816000e-01,  -3.5719462e-03,   0.0000000e+00,  -1.3246364e-02,  -6.6432099e-04,   0.0000000e+00,  -5.0965191e-05,   0.0000000e+00,   3.7206456e-04,   0.0000000e+00,  -5.0965191e-05,   0.0000000e+00,   2.9225643e-04, 0, 324, 323, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      326,   1.0715000e-01,   3.2226892e-04,   0.0000000e+00,   3.9788834e-03,   1.5588254e-05,   0.0000000e+00,   1.0547741e-05,   0.0000000e+00,  -1.5382900e-05,   0.0000000e+00,   1.0547741e-05,   0.0000000e+00,  -2.0535425e-07, 0, 323, 327, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      327,  -9.4950000e-02,   1.6468524e-02,   0.0000000e+00,   2.1487241e-02,   2.1391379e-04,   0.0000000e+00,  -7.0931224e-05,   0.0000000e+00,  -6.2650781e-04,   0.0000000e+00,  -7.0931224e-05,   0.0000000e+00,   4.1259402e-04, 0, 323, 328, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      328,  -1.1624000e-01,   2.0756976e-02,   0.0000000e+00,   1.2811380e-02,   3.1367861e-04,   0.0000000e+00,  -3.1111169e-05,   0.0000000e+00,  -6.0971543e-04,   0.0000000e+00,  -3.1111169e-05,   0.0000000e+00,   2.9603682e-04, 0, 327, 329, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      329,  -2.0160000e-02,   8.4372015e-03,   0.0000000e+00,   2.0552185e-02,  -1.4990860e-05,   0.0000000e+00,  -5.4549556e-05,   0.0000000e+00,  -5.1335762e-04,   0.0000000e+00,  -5.4549556e-05,   0.0000000e+00,   5.2834848e-04, 0, 322, 328, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      330,   9.2750000e-02,  -6.6887999e-04,   0.0000000e+00,  -4.0069298e-03,   1.1947884e-06,   0.0000000e+00,   9.2502755e-06,   0.0000000e+00,  -2.0498088e-05,   0.0000000e+00,   9.2502755e-06,   0.0000000e+00,   1.9303299e-05, 0, 327, 323, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      331,   9.2750000e-02,  -6.6887999e-04,   0.0000000e+00,  -4.0069298e-03,   1.1947884e-06,   0.0000000e+00,   9.2502755e-06,   0.0000000e+00,  -2.0498088e-05,   0.0000000e+00,   9.2502755e-06,   0.0000000e+00,   1.9303299e-05, 0, 327, 323, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      332,   9.2170000e-02,  -8.0434936e-05,   0.0000000e+00,  -3.7857338e-03,  -7.5607701e-06,   0.0000000e+00,   2.1188825e-06,   0.0000000e+00,  -6.5339988e-06,   0.0000000e+00,   2.1188825e-06,   0.0000000e+00,   1.4094769e-05, 0, 328, 327, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      333,   9.2170000e-02,  -8.0434936e-05,   0.0000000e+00,  -3.7857338e-03,  -7.5607701e-06,   0.0000000e+00,   2.1188825e-06,   0.0000000e+00,  -6.5339988e-06,   0.0000000e+00,   2.1188825e-06,   0.0000000e+00,   1.4094769e-05, 0, 328, 327, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      334,   4.8280000e-02,  -1.5012757e-03,   0.0000000e+00,  -5.5505398e-03,  -1.1201141e-05,   0.0000000e+00,  -1.9938031e-05,   0.0000000e+00,   8.7182213e-06,   0.0000000e+00,  -1.9938031e-05,   0.0000000e+00,   2.4829195e-06, 0, 329, 322, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      335,   4.8280000e-02,  -1.5012757e-03,   0.0000000e+00,  -5.5505398e-03,  -1.1201141e-05,   0.0000000e+00,  -1.9938031e-05,   0.0000000e+00,   8.7182213e-06,   0.0000000e+00,  -1.9938031e-05,   0.0000000e+00,   2.4829195e-06, 0, 329, 322, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      336,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 337, 340, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      337,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 336, 338, 341,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      338,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 339, 337, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      339,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 338, 337, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      340,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 336, 337, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      341,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 337, 342, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      342,  -5.8800000e-02,   1.4047538e-02,   0.0000000e+00,   1.1230728e-02,   1.9550658e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,  -5.4669968e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,   3.5119310e-04, 0, 337, 343, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      343,   2.4250000e-02,   7.3121707e-03,   0.0000000e+00,  -5.7045303e-03,   3.1945654e-04,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,   3.2520646e-05,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,  -3.5197718e-04, 0, 348, 342, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      344,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 343, 342, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      345,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 343, 342, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      346,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 342, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      347,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 342, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      348,   8.4100000e-03,  -3.5560709e-04,   0.0000000e+00,  -3.4571147e-03,  -2.1002139e-05,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,  -2.0535425e-07,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,   2.1207493e-05, 0, 343, 342, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      349,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 344, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      350,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 344, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      351,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 344, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      352,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 345, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      353,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 345, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      354,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 345, 343, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      355,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 356, 359, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      356,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 355, 357, 360,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      357,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 358, 356, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      358,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 357, 356, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      359,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 355, 356, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      360,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 356, 361, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      361,  -1.6392000e-01,  -7.1862265e-04,   0.0000000e+00,   2.1433794e-02,  -2.6248940e-04,   0.0000000e+00,   1.7917158e-04,   0.0000000e+00,  -3.7835587e-04,   0.0000000e+00,   1.7917158e-04,   0.0000000e+00,   6.4084527e-04, 0, 356, 362, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      362,  -1.0518000e-01,  -1.7869785e-02,   0.0000000e+00,  -7.1999851e-03,   7.8305309e-05,   0.0000000e+00,   6.0131458e-04,   0.0000000e+00,  -3.0527776e-04,   0.0000000e+00,   6.0131458e-04,   0.0000000e+00,   2.2697245e-04, 0, 363, 364, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      363,   6.3630000e-02,   1.6669082e-04,   0.0000000e+00,   2.4118310e-02,  -1.2754366e-04,   0.0000000e+00,   4.6830103e-04,   0.0000000e+00,  -4.2212433e-04,   0.0000000e+00,   4.6830103e-04,   0.0000000e+00,   5.4966798e-04, 0, 365, 362, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      364,   6.7000000e-03,   1.3768662e-02,   0.0000000e+00,  -2.7215584e-03,   2.3727750e-04,   0.0000000e+00,  -1.0206106e-04,   0.0000000e+00,  -2.6173332e-05,   0.0000000e+00,  -1.0206106e-04,   0.0000000e+00,  -2.1110417e-04, 0, 362, 367, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      365,  -4.0140000e-02,   6.4781874e-03,   0.0000000e+00,   1.7598847e-02,   4.4041019e-04,   0.0000000e+00,   9.3559396e-04,   0.0000000e+00,  -8.8743838e-04,   0.0000000e+00,   9.3559396e-04,   0.0000000e+00,   4.4702820e-04, 0, 363, 366, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      366,   2.7213000e-01,  -5.3976076e-04,   0.0000000e+00,  -2.1643348e-03,  -5.2785376e-04,   0.0000000e+00,   1.6428340e-05,   0.0000000e+00,  -7.8627341e-04,   0.0000000e+00,   1.6428340e-05,   0.0000000e+00,   1.3141272e-03, 0, 365, 368, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      367,  -2.1794000e-01,  -1.8678897e-02,   0.0000000e+00,   5.3494524e-03,  -3.7789849e-04,   0.0000000e+00,   3.5132378e-04,   0.0000000e+00,   2.7103027e-04,   0.0000000e+00,   3.5132378e-04,   0.0000000e+00,   1.0686822e-04, 0, 369, 364, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      368,  -1.1761000e-01,  -7.4661613e-03,   0.0000000e+00,  -2.1119462e-03,  -9.8430025e-05,   0.0000000e+00,   8.2515071e-06,   0.0000000e+00,   7.9248071e-06,   0.0000000e+00,   8.2515071e-06,   0.0000000e+00,   9.0505218e-05, 0, 370, 366, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      369,   1.0201000e-01,   2.0841115e-02,   0.0000000e+00,   8.9118734e-03,   7.5159655e-04,   0.0000000e+00,   5.9733817e-04,   0.0000000e+00,  -6.9133441e-04,   0.0000000e+00,   5.9733817e-04,   0.0000000e+00,  -6.0262138e-05, 0, 367, 370, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      370,  -5.7640000e-02,  -2.6024935e-03,   0.0000000e+00,   3.5984050e-05,   3.3090037e-05,   0.0000000e+00,  -1.5303559e-04,   0.0000000e+00,  -1.6316328e-04,   0.0000000e+00,  -1.5303559e-04,   0.0000000e+00,   1.3007325e-04, 0, 369, 368, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      371,   9.3070000e-02,   1.4171366e-03,   0.0000000e+00,  -1.6632040e-03,   1.0239710e-05,   0.0000000e+00,   2.8189538e-06,   0.0000000e+00,  -8.9049070e-06,   0.0000000e+00,   2.8189538e-06,   0.0000000e+00,  -1.3348026e-06, 0, 361, 362, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      372,   9.3070000e-02,   1.4171366e-03,   0.0000000e+00,  -1.6632040e-03,   1.0239710e-05,   0.0000000e+00,   2.8189538e-06,   0.0000000e+00,  -8.9049070e-06,   0.0000000e+00,   2.8189538e-06,   0.0000000e+00,  -1.3348026e-06, 0, 361, 362, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      373,   2.8580000e-02,  -1.8891626e-04,   0.0000000e+00,  -1.0509989e-02,   4.5187269e-05,   0.0000000e+00,  -9.3342840e-06,   0.0000000e+00,   4.6578077e-05,   0.0000000e+00,  -9.3342840e-06,   0.0000000e+00,  -9.1765346e-05, 0, 363, 365, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      374,   8.9430000e-02,   1.3599854e-04,   0.0000000e+00,  -1.1892199e-02,   4.4879238e-05,   0.0000000e+00,  -2.1282168e-06,   0.0000000e+00,   7.1332598e-05,   0.0000000e+00,  -2.1282168e-06,   0.0000000e+00,  -1.1621184e-04, 0, 365, 363, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      375,   8.0100000e-03,  -1.1853570e-04,   0.0000000e+00,  -9.5262482e-03,   5.6752447e-05,   0.0000000e+00,  -7.8034614e-06,   0.0000000e+00,   5.2271991e-06,   0.0000000e+00,  -7.8034614e-06,   0.0000000e+00,  -6.1979646e-05, 0, 367, 369, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      376,   4.4000000e-03,   2.5559259e-04,   0.0000000e+00,  -9.7955994e-03,   4.2134958e-05,   0.0000000e+00,   1.8948597e-06,   0.0000000e+00,   3.5507616e-05,   0.0000000e+00,   1.8948597e-06,   0.0000000e+00,  -7.7642574e-05, 0, 368, 370, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      377,   7.7500000e-03,   2.5400506e-04,   0.0000000e+00,  -9.4034790e-03,   4.8454268e-05,   0.0000000e+00,   4.0324107e-06,   0.0000000e+00,   2.4166461e-05,   0.0000000e+00,   4.0324107e-06,   0.0000000e+00,  -7.2620730e-05, 0, 369, 367, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      378,   3.7700000e-03,  -2.9633924e-05,   0.0000000e+00,  -9.6421380e-03,   3.1223180e-05,   0.0000000e+00,  -1.2545278e-05,   0.0000000e+00,   5.6939133e-07,   0.0000000e+00,  -1.2545278e-05,   0.0000000e+00,  -3.1792571e-05, 0, 370, 369, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      379,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 380, 383, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      380,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 379, 381, 384,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      381,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 382, 380, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      382,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 381, 380, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      383,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 379, 380, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      384,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 380, 385, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      385,   1.4170000e-02,   7.5783468e-03,   0.0000000e+00,   1.0956085e-02,   2.8334219e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,  -7.3121047e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,   4.4786828e-04, 0, 380, 386, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      386,  -1.4658000e-01,   3.9196156e-03,   0.0000000e+00,   1.4685197e-02,  -6.8233616e-06,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,  -4.2669813e-04,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,   4.3352149e-04, 0, 387, 385, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      387,  -1.2517000e-01,   5.4621672e-03,   0.0000000e+00,   1.6352105e-02,  -1.0743761e-05,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,  -5.1248953e-04,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,   5.2323329e-04, 0, 388, 386, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      388,  -6.3700000e-03,   7.8831529e-03,   0.0000000e+00,   2.4261717e-02,  -1.5507046e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,  -7.0800544e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,   8.6307590e-04, 0, 389, 387, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      389,   1.0679000e-01,   0.0000000e+00,   0.0000000e+00,   6.7078503e-03,  -1.5848681e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,  -1.2231646e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,   2.8080327e-04, 0, 388, 398, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      390,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 385, 386, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      391,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 385, 386, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      392,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 386, 393, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      393,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 386, 392, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      394,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 387, 395, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      395,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 387, 394, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      396,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 388, 397, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      397,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 388, 396, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      398,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 389, 388, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      399,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 389, 388, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      400,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 389, 388, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      401,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 402, 405, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      402,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 401, 403, 406,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      403,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 404, 402, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      404,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 403, 402, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      405,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 401, 402, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      406,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 402, 407, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      407,  -1.2108000e-01,   1.1022232e-02,   0.0000000e+00,   1.3354316e-02,   1.8811383e-04,   0.0000000e+00,  -2.9191106e-04,   0.0000000e+00,  -5.7985506e-04,   0.0000000e+00,  -2.9191106e-04,   0.0000000e+00,   3.9174123e-04, 0, 402, 408, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      408,  -1.4858000e-01,   1.9061492e-02,   0.0000000e+00,   1.5940934e-02,   2.1888896e-05,   0.0000000e+00,   2.2635639e-05,   0.0000000e+00,  -5.2966461e-04,   0.0000000e+00,   2.2635639e-05,   0.0000000e+00,   5.0777572e-04, 0, 407, 409, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      409,   7.6960000e-01,   1.6207110e-02,   0.0000000e+00,  -4.9541570e-03,   1.9747611e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,  -3.7147650e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,   1.7400039e-04, 0, 411, 410, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      410,  -7.2950000e-01,  -1.9516056e-03,   0.0000000e+00,  -8.4678937e-03,  -2.5694484e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   2.5177364e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   5.1711933e-06, 0, 409, 411, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      411,  -3.8020000e-01,   4.9493945e-03,   0.0000000e+00,   1.1738209e-02,  -1.4391599e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,  -4.3525766e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,   5.7917365e-04, 0, 409, 410, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      412,   5.9470000e-02,  -2.7675968e-04,   0.0000000e+00,  -4.9097062e-03,  -1.4281455e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,   9.3622869e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,  -7.9341414e-06, 0, 407, 402, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      413,   5.9470000e-02,  -2.7675968e-04,   0.0000000e+00,  -4.9097062e-03,  -1.4281455e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,   9.3622869e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,  -7.9341414e-06, 0, 407, 402, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      414,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 408, 409, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      415,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 408, 409, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      416,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 411, 409, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      417,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 411, 409, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      418,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 419, 422, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      419,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 418, 420, 423,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      420,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 421, 419, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      421,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 420, 419, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      422,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 418, 419, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      423,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 419, 424, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      424,  -1.2108000e-01,   1.1022232e-02,   0.0000000e+00,   1.3354316e-02,   1.8811383e-04,   0.0000000e+00,  -2.9191106e-04,   0.0000000e+00,  -5.7985506e-04,   0.0000000e+00,  -2.9191106e-04,   0.0000000e+00,   3.9174123e-04, 0, 419, 425, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      425,  -1.4858000e-01,   1.9061492e-02,   0.0000000e+00,   1.5940934e-02,   2.1888896e-05,   0.0000000e+00,   2.2635639e-05,   0.0000000e+00,  -5.2966461e-04,   0.0000000e+00,   2.2635639e-05,   0.0000000e+00,   5.0777572e-04, 0, 424, 426, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      426,   7.6960000e-01,   1.6207110e-02,   0.0000000e+00,  -4.9541570e-03,   1.9747611e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,  -3.7147650e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,   1.7400039e-04, 0, 428, 427, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      427,  -7.2950000e-01,  -1.9516056e-03,   0.0000000e+00,  -8.4678937e-03,  -2.5694484e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   2.5177364e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   5.1711933e-06, 0, 426, 428, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      428,  -3.8020000e-01,   4.9493945e-03,   0.0000000e+00,   1.1738209e-02,  -1.4391599e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,  -4.3525766e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,   5.7917365e-04, 0, 426, 427, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      429,   5.9470000e-02,  -2.7675968e-04,   0.0000000e+00,  -4.9097062e-03,  -1.4281455e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,   9.3622869e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,  -7.9341414e-06, 0, 424, 419, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      430,   5.9470000e-02,  -2.7675968e-04,   0.0000000e+00,  -4.9097062e-03,  -1.4281455e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,   9.3622869e-06,   0.0000000e+00,   1.0862306e-04,   0.0000000e+00,  -7.9341414e-06, 0, 424, 419, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      431,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 425, 426, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      432,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 425, 426, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      433,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 428, 426, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      434,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 428, 426, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      435,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 436, 439, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      436,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 435, 437, 440,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      437,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 438, 436, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      438,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 437, 436, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      439,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 435, 436, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      440,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 436, 441, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      441,  -1.5072000e-01,   1.1297933e-02,   0.0000000e+00,   1.3891960e-02,   5.0134439e-05,   0.0000000e+00,  -1.3707396e-04,   0.0000000e+00,  -4.7686057e-04,   0.0000000e+00,  -1.3707396e-04,   0.0000000e+00,   4.2672613e-04, 0, 436, 442, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      442,   7.6960000e-01,   1.6207110e-02,   0.0000000e+00,  -4.9541570e-03,   1.9747611e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,  -3.7147650e-04,   0.0000000e+00,   3.1223180e-05,   0.0000000e+00,   1.7400039e-04, 0, 444, 443, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      443,  -7.2950000e-01,  -1.9516056e-03,   0.0000000e+00,  -8.4678937e-03,  -2.5694484e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   2.5177364e-04,   0.0000000e+00,   9.9596810e-06,   0.0000000e+00,   5.1711933e-06, 0, 442, 444, -1,   3.9000000e-01,   3.0680399e-01,   8.3400000e-04,
      444,  -3.8020000e-01,   4.9493945e-03,   0.0000000e+00,   1.1738209e-02,  -1.4391599e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,  -4.3525766e-04,   0.0000000e+00,   2.1060012e-04,   0.0000000e+00,   5.7917365e-04, 0, 442, 443, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      445,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 441, 442, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      446,   9.6790000e-02,   1.7616309e-03,   0.0000000e+00,  -5.1176728e-03,   3.6536255e-04,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -6.4219874e-05,   0.0000000e+00,   8.1581642e-06,   0.0000000e+00,  -3.0114267e-04, 0, 441, 442, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      447,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 444, 442, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      448,   1.8368000e-01,  -2.1050669e-03,   0.0000000e+00,  -1.6793439e-02,   9.9904842e-05,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,   1.8174784e-04,   0.0000000e+00,  -1.0130498e-04,   0.0000000e+00,  -2.8165269e-04, 0, 444, 442, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      449,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 450, 453, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      450,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 449, 451, 454,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      451,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 452, 450, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      452,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 451, 450, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      453,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 449, 450, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      454,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 450, 455, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      455,  -5.8800000e-02,   1.4047538e-02,   0.0000000e+00,   1.1230728e-02,   1.9550658e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,  -5.4669968e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,   3.5119310e-04, 0, 450, 456, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      456,   2.4250000e-02,   7.3121707e-03,   0.0000000e+00,  -5.7045303e-03,   3.1945654e-04,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,   3.2520646e-05,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,  -3.5197718e-04, 0, 461, 455, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      457,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 456, 455, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      458,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 456, 455, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      459,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 455, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      460,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 455, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      461,   8.4100000e-03,  -3.5560709e-04,   0.0000000e+00,  -3.4571147e-03,  -2.1002139e-05,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,  -2.0535425e-07,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,   2.1207493e-05, 0, 456, 455, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      462,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 457, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      463,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 457, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      464,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 457, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      465,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 458, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      466,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 458, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      467,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 458, 456, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      468,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 469, 472, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      469,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 468, 470, 473,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      470,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 471, 469, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      471,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 470, 469, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      472,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 468, 469, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      473,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 469, 474, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      474,   1.4170000e-02,   7.5783468e-03,   0.0000000e+00,   1.0956085e-02,   2.8334219e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,  -7.3121047e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,   4.4786828e-04, 0, 469, 475, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      475,  -1.4658000e-01,   3.9196156e-03,   0.0000000e+00,   1.4685197e-02,  -6.8233616e-06,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,  -4.2669813e-04,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,   4.3352149e-04, 0, 476, 474, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      476,  -1.2517000e-01,   5.4621672e-03,   0.0000000e+00,   1.6352105e-02,  -1.0743761e-05,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,  -5.1248953e-04,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,   5.2323329e-04, 0, 477, 475, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      477,  -6.3700000e-03,   7.8831529e-03,   0.0000000e+00,   2.4261717e-02,  -1.5507046e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,  -7.0800544e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,   8.6307590e-04, 0, 478, 476, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      478,   1.0679000e-01,   0.0000000e+00,   0.0000000e+00,   6.7078503e-03,  -1.5848681e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,  -1.2231646e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,   2.8080327e-04, 0, 477, 487, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      479,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 474, 475, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      480,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 474, 475, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      481,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 475, 482, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      482,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 475, 481, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      483,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 476, 484, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      484,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 476, 483, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      485,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 477, 486, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      486,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 477, 485, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      487,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 478, 477, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      488,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 478, 477, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      489,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 478, 477, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      490,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 491, 494, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      491,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 490, 492, 495,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      492,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 493, 491, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      493,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 492, 491, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      494,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 490, 491, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      495,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 491, 496, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      496,   1.4170000e-02,   7.5783468e-03,   0.0000000e+00,   1.0956085e-02,   2.8334219e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,  -7.3121047e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,   4.4786828e-04, 0, 491, 497, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      497,  -1.4658000e-01,   3.9196156e-03,   0.0000000e+00,   1.4685197e-02,  -6.8233616e-06,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,  -4.2669813e-04,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,   4.3352149e-04, 0, 498, 496, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      498,  -1.2517000e-01,   5.4621672e-03,   0.0000000e+00,   1.6352105e-02,  -1.0743761e-05,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,  -5.1248953e-04,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,   5.2323329e-04, 0, 499, 497, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      499,  -6.3700000e-03,   7.8831529e-03,   0.0000000e+00,   2.4261717e-02,  -1.5507046e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,  -7.0800544e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,   8.6307590e-04, 0, 500, 498, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      500,   1.0679000e-01,   0.0000000e+00,   0.0000000e+00,   6.7078503e-03,  -1.5848681e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,  -1.2231646e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,   2.8080327e-04, 0, 499, 509, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      501,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 496, 497, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      502,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 496, 497, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      503,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 497, 504, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      504,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 497, 503, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      505,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 498, 506, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      506,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 498, 505, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      507,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 499, 508, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      508,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 499, 507, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      509,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 500, 499, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      510,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 500, 499, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      511,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 500, 499, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      512,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 513, 516, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      513,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 512, 514, 517,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      514,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 515, 513, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      515,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 514, 513, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      516,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 512, 513, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      517,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 513, 518, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      518,   4.1000000e-03,   1.0333243e-02,   0.0000000e+00,   9.9988034e-03,   2.0035107e-04,   0.0000000e+00,  -5.4810916e-04,   0.0000000e+00,  -8.1410825e-04,   0.0000000e+00,  -5.4810916e-04,   0.0000000e+00,   6.1375718e-04, 0, 513, 519, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      519,  -3.6014000e-01,  -1.1218028e-02,   0.0000000e+00,   1.3508836e-02,  -8.6359862e-04,   0.0000000e+00,  -1.8817917e-05,   0.0000000e+00,   1.0361989e-04,   0.0000000e+00,  -1.8817917e-05,   0.0000000e+00,   7.5997874e-04, 0, 518, 520, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      520,   1.1459600e+00,   0.0000000e+00,   0.0000000e+00,  -4.2916272e-03,   4.4307979e-04,   0.0000000e+00,  -1.9475984e-04,   0.0000000e+00,  -5.0209114e-04,   0.0000000e+00,  -1.9475984e-04,   0.0000000e+00,   5.9011344e-05, 1, 521, 522, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      521,  -8.9716000e-01,   1.2218173e-02,   0.0000000e+00,  -1.4797912e-02,  -4.5542905e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   2.6634446e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   1.8908459e-04, 0, 520, 522, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      522,  -8.9716000e-01,   1.2218173e-02,   0.0000000e+00,  -1.4797912e-02,  -4.5542905e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   2.6634446e-04,   0.0000000e+00,   2.3940572e-04,   0.0000000e+00,   1.8908459e-04, 0, 520, 521, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      523,   4.0820000e-02,   2.9596881e-03,   0.0000000e+00,  -7.2015727e-03,   1.8051572e-04,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,   4.5364620e-06,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,  -1.8505218e-04, 0, 518, 519, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      524,   4.0820000e-02,   2.9596881e-03,   0.0000000e+00,  -7.2015727e-03,   1.8051572e-04,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,   4.5364620e-06,   0.0000000e+00,   1.2739431e-04,   0.0000000e+00,  -1.8505218e-04, 0, 518, 519, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      525,  -3.5600000e-03,   7.5936930e-04,   0.0000000e+00,  -1.9754185e-03,   2.8601180e-04,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -9.5480391e-05,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -1.9053141e-04, 0, 519, 520, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      526,  -3.5600000e-03,   7.5936930e-04,   0.0000000e+00,  -1.9754185e-03,   2.8601180e-04,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -9.5480391e-05,   0.0000000e+00,  -1.6633694e-04,   0.0000000e+00,  -1.9053141e-04, 0, 519, 520, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      527,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 528, 531, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      528,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 527, 529, 532,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      529,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 530, 528, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      530,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 529, 528, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      531,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 527, 528, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      532,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 528, 533, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      533,   1.4170000e-02,   7.5783468e-03,   0.0000000e+00,   1.0956085e-02,   2.8334219e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,  -7.3121047e-04,   0.0000000e+00,  -4.6563142e-04,   0.0000000e+00,   4.4786828e-04, 0, 528, 534, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      534,  -1.4658000e-01,   3.9196156e-03,   0.0000000e+00,   1.4685197e-02,  -6.8233616e-06,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,  -4.2669813e-04,   0.0000000e+00,  -3.1456537e-05,   0.0000000e+00,   4.3352149e-04, 0, 535, 533, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      535,  -1.2517000e-01,   5.4621672e-03,   0.0000000e+00,   1.6352105e-02,  -1.0743761e-05,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,  -5.1248953e-04,   0.0000000e+00,  -1.5651727e-04,   0.0000000e+00,   5.2323329e-04, 0, 536, 534, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      536,  -6.3700000e-03,   7.8831529e-03,   0.0000000e+00,   2.4261717e-02,  -1.5507046e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,  -7.0800544e-04,   0.0000000e+00,  -1.2499540e-04,   0.0000000e+00,   8.6307590e-04, 0, 537, 535, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      537,   1.0679000e-01,   0.0000000e+00,   0.0000000e+00,   6.7078503e-03,  -1.5848681e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,  -1.2231646e-04,   0.0000000e+00,   3.0233746e-05,   0.0000000e+00,   2.8080327e-04, 0, 536, 546, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      538,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 533, 534, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      539,   6.1680000e-02,   5.4716924e-04,   0.0000000e+00,  -3.6650814e-03,  -4.4897906e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,  -9.7169897e-06,   0.0000000e+00,   1.1201141e-06,   0.0000000e+00,   1.4206780e-05, 0, 533, 534, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      540,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 534, 541, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      541,   7.9140000e-02,  -1.1715983e-03,   0.0000000e+00,  -3.0189560e-03,  -1.3096000e-05,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,  -8.4288585e-06,   0.0000000e+00,  -9.4182926e-06,   0.0000000e+00,   2.1524859e-05, 0, 534, 540, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      542,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 535, 543, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      543,   7.9470000e-02,  -9.5463569e-04,   0.0000000e+00,  -2.9723884e-03,  -3.8830622e-06,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,  -2.1898230e-05,   0.0000000e+00,  -9.1475983e-06,   0.0000000e+00,   2.5781292e-05, 0, 535, 542, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      544,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 536, 545, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      545,   1.0424000e-01,  -1.1467270e-03,   0.0000000e+00,  -3.6433851e-03,   5.3858819e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,  -7.9061386e-06,   0.0000000e+00,  -2.8936280e-06,   0.0000000e+00,   2.5202567e-06, 0, 536, 544, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      546,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 537, 536, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      547,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 537, 536, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      548,   1.9274000e-01,   2.0050525e-03,   0.0000000e+00,  -7.4502860e-03,   1.1649186e-05,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -3.1549880e-06,   0.0000000e+00,   1.4104103e-05,   0.0000000e+00,  -8.4941985e-06, 0, 537, 536, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      549,  -2.2620000e-01,   4.3466616e-03,   0.0000000e+00,   1.8459289e-02,   1.0991119e-04,   0.0000000e+00,  -1.6386336e-04,   0.0000000e+00,  -9.5382381e-04,   0.0000000e+00,  -1.6386336e-04,   0.0000000e+00,   8.4391262e-04, 0, 550, 553, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      550,  -1.5245000e-01,   1.0327952e-02,   0.0000000e+00,   1.0417912e-02,  -1.9300499e-04,   0.0000000e+00,  -1.5606923e-05,   0.0000000e+00,  -4.4882971e-04,   0.0000000e+00,  -1.5606923e-05,   0.0000000e+00,   6.4183470e-04, 0, 549, 551, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      551,   8.4374000e-01,  -5.4669298e-03,   0.0000000e+00,   1.1569931e-02,   1.2128035e-04,   0.0000000e+00,   2.7722824e-06,   0.0000000e+00,  -2.9704492e-04,   0.0000000e+00,   2.7722824e-06,   0.0000000e+00,   1.7576457e-04, 0, 552, 550, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      552,  -7.3597000e-01,  -1.0901051e-04,   0.0000000e+00,  -1.0256513e-02,  -4.9353160e-04,   0.0000000e+00,   1.6036300e-05,   0.0000000e+00,   2.9642886e-04,   0.0000000e+00,   1.6036300e-05,   0.0000000e+00,   1.9710274e-04, 0, 551, 550, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      553,   1.2752000e-01,   3.4237766e-04,   0.0000000e+00,  -6.7644723e-03,   2.9897712e-05,   0.0000000e+00,  -6.2353017e-06,   0.0000000e+00,  -2.2635639e-05,   0.0000000e+00,  -6.2353017e-06,   0.0000000e+00,  -7.2620730e-06, 0, 549, 550, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      554,   7.1680000e-02,  -3.5454873e-04,   0.0000000e+00,  -1.8806958e-03,  -1.0687755e-05,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   3.9577364e-06,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   6.7300188e-06, 0, 550, 555, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      555,   7.1680000e-02,  -3.5454873e-04,   0.0000000e+00,  -1.8806958e-03,  -1.0687755e-05,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   3.9577364e-06,   0.0000000e+00,   5.6005704e-08,   0.0000000e+00,   6.7300188e-06, 0, 550, 554, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      556,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 557, 560, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      557,  -2.1238000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 556, 558, 561,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      558,   8.5068000e-01,  -8.9166360e-04,   0.0000000e+00,   1.4682022e-02,   2.7501601e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,  -3.8828755e-04,   0.0000000e+00,  -6.9353730e-06,   0.0000000e+00,   1.1327154e-04, 0, 559, 557, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      559,  -7.7770000e-01,  -1.0038492e-03,   0.0000000e+00,  -1.1528655e-02,  -5.7445051e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   3.1676826e-04,   0.0000000e+00,   8.0834900e-06,   0.0000000e+00,   2.5768224e-04, 0, 558, 557, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      560,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 556, 557, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      561,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 557, 562, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      562,  -5.8800000e-02,   1.4047538e-02,   0.0000000e+00,   1.1230728e-02,   1.9550658e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,  -5.4669968e-04,   0.0000000e+00,  -1.0493602e-04,   0.0000000e+00,   3.5119310e-04, 0, 557, 563, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      563,   2.4250000e-02,   7.3121707e-03,   0.0000000e+00,  -5.7045303e-03,   3.1945654e-04,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,   3.2520646e-05,   0.0000000e+00,  -6.6618785e-05,   0.0000000e+00,  -3.5197718e-04, 0, 568, 562, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      564,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 563, 562, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      565,  -1.1850000e-01,   1.0805799e-03,   0.0000000e+00,   1.4733352e-02,  -2.0342205e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,  -1.9862423e-04,   0.0000000e+00,  -4.9854411e-05,   0.0000000e+00,   4.0204628e-04, 0, 563, 562, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      566,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 562, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      567,   4.6960000e-02,   3.2121057e-04,   0.0000000e+00,  -3.5370205e-03,  -6.4985285e-05,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,  -3.6217022e-06,   0.0000000e+00,   7.0875219e-05,   0.0000000e+00,   6.8606988e-05, 0, 562, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      568,   8.4100000e-03,  -3.5560709e-04,   0.0000000e+00,  -3.4571147e-03,  -2.1002139e-05,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,  -2.0535425e-07,   0.0000000e+00,   2.2019576e-05,   0.0000000e+00,   2.1207493e-05, 0, 563, 562, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      569,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 564, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      570,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 564, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      571,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 564, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      572,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 565, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      573,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 565, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      574,   3.9890000e-02,  -1.4816962e-04,   0.0000000e+00,  -5.3124100e-03,   3.8914630e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,   4.1033513e-05,   0.0000000e+00,  -4.5672652e-05,   0.0000000e+00,  -7.9948143e-05, 0, 565, 563, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      575,  -1.4985000e-01,   1.8785791e-03,   0.0000000e+00,   2.3443080e-02,   8.5595384e-05,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,  -1.0406887e-03,   0.0000000e+00,  -2.2727115e-04,   0.0000000e+00,   9.5509328e-04, 0, 576, 579, -1,   3.9000000e-01,   3.1996314e-01,   1.0730000e-03,
      576,  -3.5660000e-01,   8.2959111e-03,   5.5277851e-03,   7.6968825e-03,  -1.6264990e-04,  -2.1019874e-04,  -4.1070850e-07,  -2.1019874e-04,  -3.7849588e-04,  -2.0935866e-04,  -4.1070850e-07,  -2.0935866e-04,   5.4114578e-04, 0, 575, 577, 580,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      577,   1.0266900e+00,   0.0000000e+00,   0.0000000e+00,  -3.6444435e-03,   3.3967460e-04,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,  -4.2628742e-04,   0.0000000e+00,   0.0000000e+00,   0.0000000e+00,   8.6612821e-05, 1, 578, 595, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      578,  -9.0443000e-01,   1.0117339e-02,   0.0000000e+00,  -1.0191424e-02,  -4.2070552e-04,   0.0000000e+00,   1.5798276e-04,   0.0000000e+00,   2.2781254e-04,   0.0000000e+00,   1.5798276e-04,   0.0000000e+00,   1.9289298e-04, 0, 577, 595, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04,
      579,   1.2992000e-01,  -4.6726348e-04,   0.0000000e+00,  -7.4979119e-03,   3.8494587e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -2.4297141e-05,   0.0000000e+00,  -4.3497764e-06,   0.0000000e+00,  -1.4197446e-05, 0, 575, 576, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      580,   8.9210000e-02,   7.4349398e-04,   0.0000000e+00,   2.2146066e-03,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,  -9.2409412e-07,   0.0000000e+00,   7.4767615e-06,   0.0000000e+00,   1.8481882e-06, 0, 576, 581, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      581,  -1.8950000e-02,   1.9556273e-02,   0.0000000e+00,   5.8950341e-04,   3.8441382e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,  -6.1989914e-04,   0.0000000e+00,  -3.0664990e-04,   0.0000000e+00,   2.3548532e-04, 0, 582, 576, -1,   3.9000000e-01,   3.3178695e-01,   1.3340000e-03,
      582,  -6.2750000e-02,  -8.0170347e-04,   0.0000000e+00,   6.2003694e-03,   9.7860634e-05,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,  -2.5117625e-04,   0.0000000e+00,   2.3242367e-06,   0.0000000e+00,   1.5331562e-04, 0, 581, 583, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      583,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 582, 585, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      584,   8.9000000e-03,   9.8141206e-03,   0.0000000e+00,   7.9752297e-03,   4.4312647e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -3.9988073e-04,   0.0000000e+00,   2.6799663e-04,   0.0000000e+00,  -4.3245738e-05, 0, 582, 586, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      585,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 583, 587, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      586,  -5.4020000e-02,  -5.0007246e-04,   0.0000000e+00,   1.5462558e-03,  -2.5762624e-06,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,  -1.2537810e-04,   0.0000000e+00,  -1.2951319e-04,   0.0000000e+00,   1.2795437e-04, 0, 584, 587, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      587,  -5.8260000e-02,  -5.1859367e-05,   0.0000000e+00,   6.5882563e-04,  -3.0261749e-05,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,  -1.6282725e-04,   0.0000000e+00,  -1.4155442e-04,   0.0000000e+00,   1.9308900e-04, 0, 585, 586, -1,   3.9000000e-01,   3.4714134e-01,   1.7500000e-03,
      588,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 581, 582, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      589,   6.8390000e-02,   1.4748169e-03,   0.0000000e+00,  -7.3465672e-03,  -8.1114928e-06,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,   7.8557334e-05,   0.0000000e+00,   8.7490244e-05,   0.0000000e+00,  -7.0445841e-05, 0, 581, 582, -1,   3.9000000e-01,   2.8135002e-01,   4.9600000e-04,
      590,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 583, 582, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      591,   2.8180000e-02,  -7.0115980e-04,   0.0000000e+00,  -7.8630442e-03,   7.0949893e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,   9.0477215e-05,   0.0000000e+00,   1.0463732e-05,   0.0000000e+00,  -1.6142711e-04, 0, 584, 582, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      592,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 585, 583, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      593,   3.5650000e-02,   2.5453424e-04,   0.0000000e+00,  -8.0186223e-03,   5.0386465e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,   3.5600959e-05,   0.0000000e+00,   3.2109937e-06,   0.0000000e+00,  -8.5987424e-05, 0, 586, 584, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      594,   3.5880000e-02,   2.0743747e-04,   0.0000000e+00,  -7.8995574e-03,   4.4888572e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,   1.2059895e-05,   0.0000000e+00,   1.3581383e-05,   0.0000000e+00,  -5.6948467e-05, 0, 587, 585, -1,   3.9000000e-01,   2.9769271e-01,   6.9600000e-04,
      595,  -9.0443000e-01,   1.0117339e-02,   0.0000000e+00,  -1.0191424e-02,  -4.2070552e-04,   0.0000000e+00,   1.5798276e-04,   0.0000000e+00,   2.2781254e-04,   0.0000000e+00,   1.5798276e-04,   0.0000000e+00,   1.9289298e-04, 0, 577, 578, -1,   3.9000000e-01,   3.0698765e-01,   8.3700000e-04
    };

    unsigned int index                      = 0;
    unsigned int particleIndex              = index++;
    unsigned int chargeIndex                = index++;
    unsigned int dipoleIndex                = index; index += 3;
    unsigned int quadrupoleIndex            = index; index += 9;
    unsigned int axisTypeIndex              = index++;
    unsigned int multipoleAtomZIndex        = index++;
    unsigned int multipoleAtomXIndex        = index++;
    unsigned int multipoleAtomYIndex        = index++;
    unsigned int tholeIndex                 = index++;
    unsigned int dampingFactorIndex         = index++;
    unsigned int polarityIndex              = index++;
    std::vector<double> dipole(3);
    std::vector<double> quadrupole(9);
    unsigned int entriesPerParticle = 21;
    const double* data    = multipoleData;
    for (unsigned int ii = 0; ii < numberOfParticles; ii++) { 

        dipole[0]       = data[dipoleIndex + 0];
        dipole[1]       = data[dipoleIndex + 1];
        dipole[2]       = data[dipoleIndex + 2];

        quadrupole[0]   = data[quadrupoleIndex + 0];
        quadrupole[1]   = data[quadrupoleIndex + 1];
        quadrupole[2]   = data[quadrupoleIndex + 2];
        quadrupole[3]   = data[quadrupoleIndex + 3];
        quadrupole[4]   = data[quadrupoleIndex + 4];
        quadrupole[5]   = data[quadrupoleIndex + 5];
        quadrupole[6]   = data[quadrupoleIndex + 6];
        quadrupole[7]   = data[quadrupoleIndex + 7];
        quadrupole[8]   = data[quadrupoleIndex + 8];

        amoebaMultipoleForce->addMultipole(data[chargeIndex], dipole, quadrupole, static_cast<int>(data[axisTypeIndex]),
                                           static_cast<int>(data[multipoleAtomZIndex]), static_cast<int>(data[multipoleAtomXIndex]), static_cast<int>(data[multipoleAtomYIndex]),
                                           data[tholeIndex], data[dampingFactorIndex], data[polarityIndex]);
        data           += entriesPerParticle;
    }

    // covalent maps for multipole force
    // structure: particle index, map type, number of entries, entries ... 

    static const int covalentMapData[] = {
        0, 0, 4,  1, 4, 5, 6,
        0, 1, 3,  2, 7, 8,
        0, 2, 5,  3, 9, 12, 13, 19,
        0, 3, 5,  10, 14, 15, 20, 23,
        0, 4, 4,  0, 4, 5, 6,
        0, 5, 2,  1, 7,
        0, 6, 7,  2, 3, 8, 12, 13, 19, 23,
        0, 7, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        1, 0, 4,  0, 2, 7, 8,
        1, 1, 8,  3, 4, 5, 6, 9, 12, 13, 19,
        1, 2, 5,  10, 14, 15, 20, 23,
        1, 3, 4,  11, 21, 24, 25,
        1, 4, 2,  1, 7,
        1, 5, 11,  0, 2, 3, 4, 5, 6, 8, 12, 13, 19, 23,
        1, 6, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        1, 7, 7,  21, 22, 25, 29, 30, 38, 42,
        2, 0, 3,  1, 3, 19,
        2, 1, 5,  0, 7, 8, 20, 23,
        2, 2, 9,  4, 5, 6, 9, 12, 13, 21, 24, 25,
        2, 3, 8,  10, 14, 15, 22, 26, 29, 30, 38,
        2, 4, 4,  2, 3, 19, 23,
        2, 5, 4,  1, 7, 20, 24,
        2, 6, 14,  0, 4, 5, 6, 8, 12, 13, 21, 22, 25, 29, 30, 38, 42,
        2, 7, 12,  9, 10, 11, 14, 15, 16, 17, 18, 26, 31, 39, 43,
        3, 0, 1,  2,
        3, 1, 2,  1, 19,
        3, 2, 5,  0, 7, 8, 20, 23,
        3, 3, 9,  4, 5, 6, 9, 12, 13, 21, 24, 25,
        3, 4, 4,  2, 3, 19, 23,
        3, 5, 4,  1, 7, 20, 24,
        3, 6, 14,  0, 4, 5, 6, 8, 12, 13, 21, 22, 25, 29, 30, 38, 42,
        3, 7, 12,  9, 10, 11, 14, 15, 16, 17, 18, 26, 31, 39, 43,
        4, 0, 1,  0,
        4, 1, 3,  1, 5, 6,
        4, 2, 3,  2, 7, 8,
        4, 3, 5,  3, 9, 12, 13, 19,
        4, 4, 4,  0, 4, 5, 6,
        4, 5, 2,  1, 7,
        4, 6, 7,  2, 3, 8, 12, 13, 19, 23,
        4, 7, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        5, 0, 1,  0,
        5, 1, 3,  1, 4, 6,
        5, 2, 3,  2, 7, 8,
        5, 3, 5,  3, 9, 12, 13, 19,
        5, 4, 4,  0, 4, 5, 6,
        5, 5, 2,  1, 7,
        5, 6, 7,  2, 3, 8, 12, 13, 19, 23,
        5, 7, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        6, 0, 1,  0,
        6, 1, 3,  1, 4, 5,
        6, 2, 3,  2, 7, 8,
        6, 3, 5,  3, 9, 12, 13, 19,
        6, 4, 4,  0, 4, 5, 6,
        6, 5, 2,  1, 7,
        6, 6, 7,  2, 3, 8, 12, 13, 19, 23,
        6, 7, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        7, 0, 1,  1,
        7, 1, 3,  0, 2, 8,
        7, 2, 8,  3, 4, 5, 6, 9, 12, 13, 19,
        7, 3, 5,  10, 14, 15, 20, 23,
        7, 4, 2,  1, 7,
        7, 5, 11,  0, 2, 3, 4, 5, 6, 8, 12, 13, 19, 23,
        7, 6, 10,  9, 10, 11, 14, 15, 16, 17, 18, 20, 24,
        7, 7, 7,  21, 22, 25, 29, 30, 38, 42,
        8, 0, 4,  1, 9, 12, 13,
        8, 1, 6,  0, 2, 7, 10, 14, 15,
        8, 2, 6,  3, 4, 5, 6, 11, 19,
        8, 3, 5,  16, 17, 18, 20, 23,
        8, 4, 3,  8, 12, 13,
        8, 5, 10,  1, 7, 9, 10, 11, 14, 15, 16, 17, 18,
        8, 6, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        8, 7, 2,  20, 24,
        9, 0, 4,  8, 10, 14, 15,
        9, 1, 4,  1, 11, 12, 13,
        9, 2, 6,  0, 2, 7, 16, 17, 18,
        9, 3, 5,  3, 4, 5, 6, 19,
        9, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        9, 5, 3,  8, 12, 13,
        9, 6, 2,  1, 7,
        9, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        10, 0, 2,  9, 11,
        10, 1, 6,  8, 14, 15, 16, 17, 18,
        10, 2, 3,  1, 12, 13,
        10, 3, 3,  0, 2, 7,
        10, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        10, 5, 3,  8, 12, 13,
        10, 6, 2,  1, 7,
        10, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        11, 0, 4,  10, 16, 17, 18,
        11, 1, 1,  9,
        11, 2, 3,  8, 14, 15,
        11, 3, 3,  1, 12, 13,
        11, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        11, 5, 3,  8, 12, 13,
        11, 6, 2,  1, 7,
        11, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        12, 0, 1,  8,
        12, 1, 3,  1, 9, 13,
        12, 2, 6,  0, 2, 7, 10, 14, 15,
        12, 3, 6,  3, 4, 5, 6, 11, 19,
        12, 4, 3,  8, 12, 13,
        12, 5, 10,  1, 7, 9, 10, 11, 14, 15, 16, 17, 18,
        12, 6, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        12, 7, 2,  20, 24,
        13, 0, 1,  8,
        13, 1, 3,  1, 9, 12,
        13, 2, 6,  0, 2, 7, 10, 14, 15,
        13, 3, 6,  3, 4, 5, 6, 11, 19,
        13, 4, 3,  8, 12, 13,
        13, 5, 10,  1, 7, 9, 10, 11, 14, 15, 16, 17, 18,
        13, 6, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        13, 7, 2,  20, 24,
        14, 0, 1,  9,
        14, 1, 3,  8, 10, 15,
        14, 2, 4,  1, 11, 12, 13,
        14, 3, 6,  0, 2, 7, 16, 17, 18,
        14, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        14, 5, 3,  8, 12, 13,
        14, 6, 2,  1, 7,
        14, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        15, 0, 1,  9,
        15, 1, 3,  8, 10, 14,
        15, 2, 4,  1, 11, 12, 13,
        15, 3, 6,  0, 2, 7, 16, 17, 18,
        15, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        15, 5, 3,  8, 12, 13,
        15, 6, 2,  1, 7,
        15, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        16, 0, 1,  11,
        16, 1, 3,  10, 17, 18,
        16, 2, 1,  9,
        16, 3, 3,  8, 14, 15,
        16, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        16, 5, 3,  8, 12, 13,
        16, 6, 2,  1, 7,
        16, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        17, 0, 1,  11,
        17, 1, 3,  10, 16, 18,
        17, 2, 1,  9,
        17, 3, 3,  8, 14, 15,
        17, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        17, 5, 3,  8, 12, 13,
        17, 6, 2,  1, 7,
        17, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        18, 0, 1,  11,
        18, 1, 3,  10, 16, 17,
        18, 2, 1,  9,
        18, 3, 3,  8, 14, 15,
        18, 4, 8,  9, 10, 11, 14, 15, 16, 17, 18,
        18, 5, 3,  8, 12, 13,
        18, 6, 2,  1, 7,
        18, 7, 8,  0, 2, 3, 4, 5, 6, 19, 23,
        19, 0, 3,  2, 20, 23,
        19, 1, 5,  1, 3, 21, 24, 25,
        19, 2, 8,  0, 7, 8, 22, 26, 29, 30, 38,
        19, 3, 11,  4, 5, 6, 9, 12, 13, 27, 28, 31, 39, 42,
        19, 4, 4,  2, 3, 19, 23,
        19, 5, 4,  1, 7, 20, 24,
        19, 6, 14,  0, 4, 5, 6, 8, 12, 13, 21, 22, 25, 29, 30, 38, 42,
        19, 7, 12,  9, 10, 11, 14, 15, 16, 17, 18, 26, 31, 39, 43,
        20, 0, 4,  19, 21, 24, 25,
        20, 1, 7,  2, 22, 23, 26, 29, 30, 38,
        20, 2, 7,  1, 3, 27, 28, 31, 39, 42,
        20, 3, 12,  0, 7, 8, 32, 33, 34, 35, 36, 37, 40, 43, 44,
        20, 4, 2,  20, 24,
        20, 5, 11,  2, 3, 19, 21, 22, 23, 25, 29, 30, 38, 42,
        20, 6, 6,  1, 7, 26, 31, 39, 43,
        20, 7, 24,  0, 4, 5, 6, 8, 12, 13, 27, 28, 32, 33, 34, 35, 36, 37, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        21, 0, 3,  20, 22, 38,
        21, 1, 5,  19, 24, 25, 39, 42,
        21, 2, 8,  2, 23, 26, 29, 30, 40, 43, 44,
        21, 3, 10,  1, 3, 27, 28, 31, 41, 45, 46, 47, 49,
        21, 4, 4,  21, 22, 38, 42,
        21, 5, 4,  20, 24, 39, 43,
        21, 6, 16,  2, 3, 19, 23, 25, 29, 30, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        21, 7, 6,  1, 7, 26, 31, 50, 54,
        22, 0, 1,  21,
        22, 1, 2,  20, 38,
        22, 2, 5,  19, 24, 25, 39, 42,
        22, 3, 8,  2, 23, 26, 29, 30, 40, 43, 44,
        22, 4, 4,  21, 22, 38, 42,
        22, 5, 4,  20, 24, 39, 43,
        22, 6, 16,  2, 3, 19, 23, 25, 29, 30, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        22, 7, 6,  1, 7, 26, 31, 50, 54,
        23, 0, 1,  19,
        23, 1, 2,  2, 20,
        23, 2, 5,  1, 3, 21, 24, 25,
        23, 3, 8,  0, 7, 8, 22, 26, 29, 30, 38,
        23, 4, 4,  2, 3, 19, 23,
        23, 5, 4,  1, 7, 20, 24,
        23, 6, 14,  0, 4, 5, 6, 8, 12, 13, 21, 22, 25, 29, 30, 38, 42,
        23, 7, 12,  9, 10, 11, 14, 15, 16, 17, 18, 26, 31, 39, 43,
        24, 0, 1,  20,
        24, 1, 3,  19, 21, 25,
        24, 2, 7,  2, 22, 23, 26, 29, 30, 38,
        24, 3, 7,  1, 3, 27, 28, 31, 39, 42,
        24, 4, 2,  20, 24,
        24, 5, 11,  2, 3, 19, 21, 22, 23, 25, 29, 30, 38, 42,
        24, 6, 6,  1, 7, 26, 31, 39, 43,
        24, 7, 24,  0, 4, 5, 6, 8, 12, 13, 27, 28, 32, 33, 34, 35, 36, 37, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        25, 0, 4,  20, 26, 29, 30,
        25, 1, 6,  19, 21, 24, 27, 28, 31,
        25, 2, 10,  2, 22, 23, 32, 33, 34, 35, 36, 37, 38,
        25, 3, 4,  1, 3, 39, 42,
        25, 4, 3,  25, 29, 30,
        25, 5, 4,  20, 24, 26, 31,
        25, 6, 16,  2, 3, 19, 21, 22, 23, 27, 28, 32, 33, 34, 35, 36, 37, 38, 42,
        25, 7, 4,  1, 7, 39, 43,
        26, 0, 4,  25, 27, 28, 31,
        26, 1, 9,  20, 29, 30, 32, 33, 34, 35, 36, 37,
        26, 2, 3,  19, 21, 24,
        26, 3, 4,  2, 22, 23, 38,
        26, 4, 2,  26, 31,
        26, 5, 11,  25, 27, 28, 29, 30, 32, 33, 34, 35, 36, 37,
        26, 6, 2,  20, 24,
        26, 7, 8,  2, 3, 19, 21, 22, 23, 38, 42,
        27, 0, 4,  26, 32, 33, 34,
        27, 1, 3,  25, 28, 31,
        27, 2, 6,  20, 29, 30, 35, 36, 37,
        27, 3, 3,  19, 21, 24,
        27, 4, 4,  27, 32, 33, 34,
        27, 5, 2,  26, 31,
        27, 6, 7,  25, 28, 29, 30, 35, 36, 37,
        27, 7, 2,  20, 24,
        28, 0, 4,  26, 35, 36, 37,
        28, 1, 3,  25, 27, 31,
        28, 2, 6,  20, 29, 30, 32, 33, 34,
        28, 3, 3,  19, 21, 24,
        28, 4, 4,  28, 35, 36, 37,
        28, 5, 2,  26, 31,
        28, 6, 7,  25, 27, 29, 30, 32, 33, 34,
        28, 7, 2,  20, 24,
        29, 0, 1,  25,
        29, 1, 3,  20, 26, 30,
        29, 2, 6,  19, 21, 24, 27, 28, 31,
        29, 3, 10,  2, 22, 23, 32, 33, 34, 35, 36, 37, 38,
        29, 4, 3,  25, 29, 30,
        29, 5, 4,  20, 24, 26, 31,
        29, 6, 16,  2, 3, 19, 21, 22, 23, 27, 28, 32, 33, 34, 35, 36, 37, 38, 42,
        29, 7, 4,  1, 7, 39, 43,
        30, 0, 1,  25,
        30, 1, 3,  20, 26, 29,
        30, 2, 6,  19, 21, 24, 27, 28, 31,
        30, 3, 10,  2, 22, 23, 32, 33, 34, 35, 36, 37, 38,
        30, 4, 3,  25, 29, 30,
        30, 5, 4,  20, 24, 26, 31,
        30, 6, 16,  2, 3, 19, 21, 22, 23, 27, 28, 32, 33, 34, 35, 36, 37, 38, 42,
        30, 7, 4,  1, 7, 39, 43,
        31, 0, 1,  26,
        31, 1, 3,  25, 27, 28,
        31, 2, 9,  20, 29, 30, 32, 33, 34, 35, 36, 37,
        31, 3, 3,  19, 21, 24,
        31, 4, 2,  26, 31,
        31, 5, 11,  25, 27, 28, 29, 30, 32, 33, 34, 35, 36, 37,
        31, 6, 2,  20, 24,
        31, 7, 8,  2, 3, 19, 21, 22, 23, 38, 42,
        32, 0, 1,  27,
        32, 1, 3,  26, 33, 34,
        32, 2, 3,  25, 28, 31,
        32, 3, 6,  20, 29, 30, 35, 36, 37,
        32, 4, 4,  27, 32, 33, 34,
        32, 5, 2,  26, 31,
        32, 6, 7,  25, 28, 29, 30, 35, 36, 37,
        32, 7, 2,  20, 24,
        33, 0, 1,  27,
        33, 1, 3,  26, 32, 34,
        33, 2, 3,  25, 28, 31,
        33, 3, 6,  20, 29, 30, 35, 36, 37,
        33, 4, 4,  27, 32, 33, 34,
        33, 5, 2,  26, 31,
        33, 6, 7,  25, 28, 29, 30, 35, 36, 37,
        33, 7, 2,  20, 24,
        34, 0, 1,  27,
        34, 1, 3,  26, 32, 33,
        34, 2, 3,  25, 28, 31,
        34, 3, 6,  20, 29, 30, 35, 36, 37,
        34, 4, 4,  27, 32, 33, 34,
        34, 5, 2,  26, 31,
        34, 6, 7,  25, 28, 29, 30, 35, 36, 37,
        34, 7, 2,  20, 24,
        35, 0, 1,  28,
        35, 1, 3,  26, 36, 37,
        35, 2, 3,  25, 27, 31,
        35, 3, 6,  20, 29, 30, 32, 33, 34,
        35, 4, 4,  28, 35, 36, 37,
        35, 5, 2,  26, 31,
        35, 6, 7,  25, 27, 29, 30, 32, 33, 34,
        35, 7, 2,  20, 24,
        36, 0, 1,  28,
        36, 1, 3,  26, 35, 37,
        36, 2, 3,  25, 27, 31,
        36, 3, 6,  20, 29, 30, 32, 33, 34,
        36, 4, 4,  28, 35, 36, 37,
        36, 5, 2,  26, 31,
        36, 6, 7,  25, 27, 29, 30, 32, 33, 34,
        36, 7, 2,  20, 24,
        37, 0, 1,  28,
        37, 1, 3,  26, 35, 36,
        37, 2, 3,  25, 27, 31,
        37, 3, 6,  20, 29, 30, 32, 33, 34,
        37, 4, 4,  28, 35, 36, 37,
        37, 5, 2,  26, 31,
        37, 6, 7,  25, 27, 29, 30, 32, 33, 34,
        37, 7, 2,  20, 24,
        38, 0, 3,  21, 39, 42,
        38, 1, 5,  20, 22, 40, 43, 44,
        38, 2, 8,  19, 24, 25, 41, 45, 46, 47, 49,
        38, 3, 8,  2, 23, 26, 29, 30, 48, 50, 53,
        38, 4, 4,  21, 22, 38, 42,
        38, 5, 4,  20, 24, 39, 43,
        38, 6, 16,  2, 3, 19, 23, 25, 29, 30, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        38, 7, 6,  1, 7, 26, 31, 50, 54,
        39, 0, 4,  38, 40, 43, 44,
        39, 1, 7,  21, 41, 42, 45, 46, 47, 49,
        39, 2, 5,  20, 22, 48, 50, 53,
        39, 3, 6,  19, 24, 25, 51, 54, 55,
        39, 4, 2,  39, 43,
        39, 5, 13,  21, 22, 38, 40, 41, 42, 44, 45, 46, 47, 48, 49, 53,
        39, 6, 4,  20, 24, 50, 54,
        39, 7, 14,  2, 3, 19, 23, 25, 29, 30, 51, 52, 55, 59, 60, 61, 65,
        40, 0, 3,  39, 41, 49,
        40, 1, 5,  38, 43, 44, 50, 53,
        40, 2, 8,  21, 42, 45, 46, 47, 51, 54, 55,
        40, 3, 8,  20, 22, 48, 52, 56, 59, 60, 61,
        40, 4, 4,  40, 41, 49, 53,
        40, 5, 4,  39, 43, 50, 54,
        40, 6, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 51, 52, 55, 59, 60, 61, 65,
        40, 7, 7,  20, 24, 56, 57, 58, 62, 66,
        41, 0, 1,  40,
        41, 1, 2,  39, 49,
        41, 2, 5,  38, 43, 44, 50, 53,
        41, 3, 8,  21, 42, 45, 46, 47, 51, 54, 55,
        41, 4, 4,  40, 41, 49, 53,
        41, 5, 4,  39, 43, 50, 54,
        41, 6, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 51, 52, 55, 59, 60, 61, 65,
        41, 7, 7,  20, 24, 56, 57, 58, 62, 66,
        42, 0, 1,  38,
        42, 1, 2,  21, 39,
        42, 2, 5,  20, 22, 40, 43, 44,
        42, 3, 8,  19, 24, 25, 41, 45, 46, 47, 49,
        42, 4, 4,  21, 22, 38, 42,
        42, 5, 4,  20, 24, 39, 43,
        42, 6, 16,  2, 3, 19, 23, 25, 29, 30, 40, 41, 44, 45, 46, 47, 48, 49, 53,
        42, 7, 6,  1, 7, 26, 31, 50, 54,
        43, 0, 1,  39,
        43, 1, 3,  38, 40, 44,
        43, 2, 7,  21, 41, 42, 45, 46, 47, 49,
        43, 3, 5,  20, 22, 48, 50, 53,
        43, 4, 2,  39, 43,
        43, 5, 13,  21, 22, 38, 40, 41, 42, 44, 45, 46, 47, 48, 49, 53,
        43, 6, 4,  20, 24, 50, 54,
        43, 7, 14,  2, 3, 19, 23, 25, 29, 30, 51, 52, 55, 59, 60, 61, 65,
        44, 0, 4,  39, 45, 46, 47,
        44, 1, 4,  38, 40, 43, 48,
        44, 2, 4,  21, 41, 42, 49,
        44, 3, 4,  20, 22, 50, 53,
        44, 4, 5,  44, 45, 46, 47, 48,
        44, 5, 2,  39, 43,
        44, 6, 8,  21, 22, 38, 40, 41, 42, 49, 53,
        44, 7, 4,  20, 24, 50, 54,
        45, 0, 2,  44, 48,
        45, 1, 3,  39, 46, 47,
        45, 2, 3,  38, 40, 43,
        45, 3, 4,  21, 41, 42, 49,
        45, 4, 5,  44, 45, 46, 47, 48,
        45, 5, 2,  39, 43,
        45, 6, 8,  21, 22, 38, 40, 41, 42, 49, 53,
        45, 7, 4,  20, 24, 50, 54,
        46, 0, 1,  44,
        46, 1, 3,  39, 45, 47,
        46, 2, 4,  38, 40, 43, 48,
        46, 3, 4,  21, 41, 42, 49,
        46, 4, 5,  44, 45, 46, 47, 48,
        46, 5, 2,  39, 43,
        46, 6, 8,  21, 22, 38, 40, 41, 42, 49, 53,
        46, 7, 4,  20, 24, 50, 54,
        47, 0, 1,  44,
        47, 1, 3,  39, 45, 46,
        47, 2, 4,  38, 40, 43, 48,
        47, 3, 4,  21, 41, 42, 49,
        47, 4, 5,  44, 45, 46, 47, 48,
        47, 5, 2,  39, 43,
        47, 6, 8,  21, 22, 38, 40, 41, 42, 49, 53,
        47, 7, 4,  20, 24, 50, 54,
        48, 0, 1,  45,
        48, 1, 1,  44,
        48, 2, 3,  39, 46, 47,
        48, 3, 3,  38, 40, 43,
        48, 4, 5,  44, 45, 46, 47, 48,
        48, 5, 2,  39, 43,
        48, 6, 8,  21, 22, 38, 40, 41, 42, 49, 53,
        48, 7, 4,  20, 24, 50, 54,
        49, 0, 3,  40, 50, 53,
        49, 1, 5,  39, 41, 51, 54, 55,
        49, 2, 8,  38, 43, 44, 52, 56, 59, 60, 61,
        49, 3, 9,  21, 42, 45, 46, 47, 57, 58, 62, 65,
        49, 4, 4,  40, 41, 49, 53,
        49, 5, 4,  39, 43, 50, 54,
        49, 6, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 51, 52, 55, 59, 60, 61, 65,
        49, 7, 7,  20, 24, 56, 57, 58, 62, 66,
        50, 0, 4,  49, 51, 54, 55,
        50, 1, 7,  40, 52, 53, 56, 59, 60, 61,
        50, 2, 6,  39, 41, 57, 58, 62, 65,
        50, 3, 6,  38, 43, 44, 63, 66, 67,
        50, 4, 2,  50, 54,
        50, 5, 11,  40, 41, 49, 51, 52, 53, 55, 59, 60, 61, 65,
        50, 6, 7,  39, 43, 56, 57, 58, 62, 66,
        50, 7, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 63, 64, 67, 72, 73, 76, 80,
        51, 0, 3,  50, 52, 61,
        51, 1, 5,  49, 54, 55, 62, 65,
        51, 2, 8,  40, 53, 56, 59, 60, 63, 66, 67,
        51, 3, 9,  39, 41, 57, 58, 64, 68, 72, 73, 76,
        51, 4, 4,  51, 52, 61, 65,
        51, 5, 4,  50, 54, 62, 66,
        51, 6, 14,  40, 41, 49, 53, 55, 59, 60, 63, 64, 67, 72, 73, 76, 80,
        51, 7, 10,  39, 43, 56, 57, 58, 68, 74, 75, 77, 81,
        52, 0, 1,  51,
        52, 1, 2,  50, 61,
        52, 2, 5,  49, 54, 55, 62, 65,
        52, 3, 8,  40, 53, 56, 59, 60, 63, 66, 67,
        52, 4, 4,  51, 52, 61, 65,
        52, 5, 4,  50, 54, 62, 66,
        52, 6, 14,  40, 41, 49, 53, 55, 59, 60, 63, 64, 67, 72, 73, 76, 80,
        52, 7, 10,  39, 43, 56, 57, 58, 68, 74, 75, 77, 81,
        53, 0, 1,  49,
        53, 1, 2,  40, 50,
        53, 2, 5,  39, 41, 51, 54, 55,
        53, 3, 8,  38, 43, 44, 52, 56, 59, 60, 61,
        53, 4, 4,  40, 41, 49, 53,
        53, 5, 4,  39, 43, 50, 54,
        53, 6, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 51, 52, 55, 59, 60, 61, 65,
        53, 7, 7,  20, 24, 56, 57, 58, 62, 66,
        54, 0, 1,  50,
        54, 1, 3,  49, 51, 55,
        54, 2, 7,  40, 52, 53, 56, 59, 60, 61,
        54, 3, 6,  39, 41, 57, 58, 62, 65,
        54, 4, 2,  50, 54,
        54, 5, 11,  40, 41, 49, 51, 52, 53, 55, 59, 60, 61, 65,
        54, 6, 7,  39, 43, 56, 57, 58, 62, 66,
        54, 7, 16,  21, 22, 38, 42, 44, 45, 46, 47, 48, 63, 64, 67, 72, 73, 76, 80,
        55, 0, 4,  50, 56, 59, 60,
        55, 1, 5,  49, 51, 54, 57, 58,
        55, 2, 4,  40, 52, 53, 61,
        55, 3, 4,  39, 41, 62, 65,
        55, 4, 3,  55, 59, 60,
        55, 5, 5,  50, 54, 56, 57, 58,
        55, 6, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        55, 7, 4,  39, 43, 62, 66,
        56, 0, 3,  55, 57, 58,
        56, 1, 3,  50, 59, 60,
        56, 2, 3,  49, 51, 54,
        56, 3, 4,  40, 52, 53, 61,
        56, 4, 3,  56, 57, 58,
        56, 5, 3,  55, 59, 60,
        56, 6, 2,  50, 54,
        56, 7, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        57, 0, 1,  56,
        57, 1, 2,  55, 58,
        57, 2, 3,  50, 59, 60,
        57, 3, 3,  49, 51, 54,
        57, 4, 3,  56, 57, 58,
        57, 5, 3,  55, 59, 60,
        57, 6, 2,  50, 54,
        57, 7, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        58, 0, 1,  56,
        58, 1, 2,  55, 57,
        58, 2, 3,  50, 59, 60,
        58, 3, 3,  49, 51, 54,
        58, 4, 3,  56, 57, 58,
        58, 5, 3,  55, 59, 60,
        58, 6, 2,  50, 54,
        58, 7, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        59, 0, 1,  55,
        59, 1, 3,  50, 56, 60,
        59, 2, 5,  49, 51, 54, 57, 58,
        59, 3, 4,  40, 52, 53, 61,
        59, 4, 3,  55, 59, 60,
        59, 5, 5,  50, 54, 56, 57, 58,
        59, 6, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        59, 7, 4,  39, 43, 62, 66,
        60, 0, 1,  55,
        60, 1, 3,  50, 56, 59,
        60, 2, 5,  49, 51, 54, 57, 58,
        60, 3, 4,  40, 52, 53, 61,
        60, 4, 3,  55, 59, 60,
        60, 5, 5,  50, 54, 56, 57, 58,
        60, 6, 8,  40, 41, 49, 51, 52, 53, 61, 65,
        60, 7, 4,  39, 43, 62, 66,
        61, 0, 3,  51, 62, 65,
        61, 1, 5,  50, 52, 63, 66, 67,
        61, 2, 8,  49, 54, 55, 64, 68, 72, 73, 76,
        61, 3, 10,  40, 53, 56, 59, 60, 69, 74, 75, 77, 80,
        61, 4, 4,  51, 52, 61, 65,
        61, 5, 4,  50, 54, 62, 66,
        61, 6, 14,  40, 41, 49, 53, 55, 59, 60, 63, 64, 67, 72, 73, 76, 80,
        61, 7, 10,  39, 43, 56, 57, 58, 68, 74, 75, 77, 81,
        62, 0, 4,  61, 63, 66, 67,
        62, 1, 7,  51, 64, 65, 68, 72, 73, 76,
        62, 2, 7,  50, 52, 69, 74, 75, 77, 80,
        62, 3, 8,  49, 54, 55, 70, 71, 78, 81, 82,
        62, 4, 2,  62, 66,
        62, 5, 11,  51, 52, 61, 63, 64, 65, 67, 72, 73, 76, 80,
        62, 6, 7,  50, 54, 68, 74, 75, 77, 81,
        62, 7, 17,  40, 41, 49, 53, 55, 59, 60, 69, 70, 71, 78, 79, 82, 86, 87, 88, 92,
        63, 0, 3,  62, 64, 76,
        63, 1, 5,  61, 66, 67, 77, 80,
        63, 2, 8,  51, 65, 68, 72, 73, 78, 81, 82,
        63, 3, 10,  50, 52, 69, 74, 75, 79, 83, 86, 87, 88,
        63, 4, 4,  63, 64, 76, 80,
        63, 5, 4,  62, 66, 77, 81,
        63, 6, 14,  51, 52, 61, 65, 67, 72, 73, 78, 79, 82, 86, 87, 88, 92,
        63, 7, 10,  50, 54, 68, 74, 75, 83, 84, 85, 89, 93,
        64, 0, 1,  63,
        64, 1, 2,  62, 76,
        64, 2, 5,  61, 66, 67, 77, 80,
        64, 3, 8,  51, 65, 68, 72, 73, 78, 81, 82,
        64, 4, 4,  63, 64, 76, 80,
        64, 5, 4,  62, 66, 77, 81,
        64, 6, 14,  51, 52, 61, 65, 67, 72, 73, 78, 79, 82, 86, 87, 88, 92,
        64, 7, 10,  50, 54, 68, 74, 75, 83, 84, 85, 89, 93,
        65, 0, 1,  61,
        65, 1, 2,  51, 62,
        65, 2, 5,  50, 52, 63, 66, 67,
        65, 3, 8,  49, 54, 55, 64, 68, 72, 73, 76,
        65, 4, 4,  51, 52, 61, 65,
        65, 5, 4,  50, 54, 62, 66,
        65, 6, 14,  40, 41, 49, 53, 55, 59, 60, 63, 64, 67, 72, 73, 76, 80,
        65, 7, 10,  39, 43, 56, 57, 58, 68, 74, 75, 77, 81,
        66, 0, 1,  62,
        66, 1, 3,  61, 63, 67,
        66, 2, 7,  51, 64, 65, 68, 72, 73, 76,
        66, 3, 7,  50, 52, 69, 74, 75, 77, 80,
        66, 4, 2,  62, 66,
        66, 5, 11,  51, 52, 61, 63, 64, 65, 67, 72, 73, 76, 80,
        66, 6, 7,  50, 54, 68, 74, 75, 77, 81,
        66, 7, 17,  40, 41, 49, 53, 55, 59, 60, 69, 70, 71, 78, 79, 82, 86, 87, 88, 92,
        67, 0, 4,  62, 68, 72, 73,
        67, 1, 6,  61, 63, 66, 69, 74, 75,
        67, 2, 6,  51, 64, 65, 70, 71, 76,
        67, 3, 4,  50, 52, 77, 80,
        67, 4, 3,  67, 72, 73,
        67, 5, 5,  62, 66, 68, 74, 75,
        67, 6, 11,  51, 52, 61, 63, 64, 65, 69, 70, 71, 76, 80,
        67, 7, 4,  50, 54, 77, 81,
        68, 0, 4,  67, 69, 74, 75,
        68, 1, 5,  62, 70, 71, 72, 73,
        68, 2, 3,  61, 63, 66,
        68, 3, 4,  51, 64, 65, 76,
        68, 4, 3,  68, 74, 75,
        68, 5, 6,  67, 69, 70, 71, 72, 73,
        68, 6, 2,  62, 66,
        68, 7, 8,  51, 52, 61, 63, 64, 65, 76, 80,
        69, 0, 3,  68, 70, 71,
        69, 1, 3,  67, 74, 75,
        69, 2, 3,  62, 72, 73,
        69, 3, 3,  61, 63, 66,
        69, 4, 3,  69, 70, 71,
        69, 5, 3,  68, 74, 75,
        69, 6, 3,  67, 72, 73,
        69, 7, 2,  62, 66,
        70, 0, 1,  69,
        70, 1, 2,  68, 71,
        70, 2, 3,  67, 74, 75,
        70, 3, 3,  62, 72, 73,
        70, 4, 3,  69, 70, 71,
        70, 5, 3,  68, 74, 75,
        70, 6, 3,  67, 72, 73,
        70, 7, 2,  62, 66,
        71, 0, 1,  69,
        71, 1, 2,  68, 70,
        71, 2, 3,  67, 74, 75,
        71, 3, 3,  62, 72, 73,
        71, 4, 3,  69, 70, 71,
        71, 5, 3,  68, 74, 75,
        71, 6, 3,  67, 72, 73,
        71, 7, 2,  62, 66,
        72, 0, 1,  67,
        72, 1, 3,  62, 68, 73,
        72, 2, 6,  61, 63, 66, 69, 74, 75,
        72, 3, 6,  51, 64, 65, 70, 71, 76,
        72, 4, 3,  67, 72, 73,
        72, 5, 5,  62, 66, 68, 74, 75,
        72, 6, 11,  51, 52, 61, 63, 64, 65, 69, 70, 71, 76, 80,
        72, 7, 4,  50, 54, 77, 81,
        73, 0, 1,  67,
        73, 1, 3,  62, 68, 72,
        73, 2, 6,  61, 63, 66, 69, 74, 75,
        73, 3, 6,  51, 64, 65, 70, 71, 76,
        73, 4, 3,  67, 72, 73,
        73, 5, 5,  62, 66, 68, 74, 75,
        73, 6, 11,  51, 52, 61, 63, 64, 65, 69, 70, 71, 76, 80,
        73, 7, 4,  50, 54, 77, 81,
        74, 0, 1,  68,
        74, 1, 3,  67, 69, 75,
        74, 2, 5,  62, 70, 71, 72, 73,
        74, 3, 3,  61, 63, 66,
        74, 4, 3,  68, 74, 75,
        74, 5, 6,  67, 69, 70, 71, 72, 73,
        74, 6, 2,  62, 66,
        74, 7, 8,  51, 52, 61, 63, 64, 65, 76, 80,
        75, 0, 1,  68,
        75, 1, 3,  67, 69, 74,
        75, 2, 5,  62, 70, 71, 72, 73,
        75, 3, 3,  61, 63, 66,
        75, 4, 3,  68, 74, 75,
        75, 5, 6,  67, 69, 70, 71, 72, 73,
        75, 6, 2,  62, 66,
        75, 7, 8,  51, 52, 61, 63, 64, 65, 76, 80,
        76, 0, 3,  63, 77, 80,
        76, 1, 5,  62, 64, 78, 81, 82,
        76, 2, 8,  61, 66, 67, 79, 83, 86, 87, 88,
        76, 3, 9,  51, 65, 68, 72, 73, 84, 85, 89, 92,
        76, 4, 4,  63, 64, 76, 80,
        76, 5, 4,  62, 66, 77, 81,
        76, 6, 14,  51, 52, 61, 65, 67, 72, 73, 78, 79, 82, 86, 87, 88, 92,
        76, 7, 10,  50, 54, 68, 74, 75, 83, 84, 85, 89, 93,
        77, 0, 4,  76, 78, 81, 82,
        77, 1, 7,  63, 79, 80, 83, 86, 87, 88,
        77, 2, 6,  62, 64, 84, 85, 89, 92,
        77, 3, 6,  61, 66, 67, 90, 93, 94,
        77, 4, 2,  77, 81,
        77, 5, 11,  63, 64, 76, 78, 79, 80, 82, 86, 87, 88, 92,
        77, 6, 7,  62, 66, 83, 84, 85, 89, 93,
        77, 7, 14,  51, 52, 61, 65, 67, 72, 73, 90, 91, 94, 101, 102, 108, 112,
        78, 0, 3,  77, 79, 88,
        78, 1, 5,  76, 81, 82, 89, 92,
        78, 2, 8,  63, 80, 83, 86, 87, 90, 93, 94,
        78, 3, 9,  62, 64, 84, 85, 91, 95, 101, 102, 108,
        78, 4, 4,  78, 79, 88, 92,
        78, 5, 4,  77, 81, 89, 93,
        78, 6, 14,  63, 64, 76, 80, 82, 86, 87, 90, 91, 94, 101, 102, 108, 112,
        78, 7, 18,  62, 66, 83, 84, 85, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        79, 0, 1,  78,
        79, 1, 2,  77, 88,
        79, 2, 5,  76, 81, 82, 89, 92,
        79, 3, 8,  63, 80, 83, 86, 87, 90, 93, 94,
        79, 4, 4,  78, 79, 88, 92,
        79, 5, 4,  77, 81, 89, 93,
        79, 6, 14,  63, 64, 76, 80, 82, 86, 87, 90, 91, 94, 101, 102, 108, 112,
        79, 7, 18,  62, 66, 83, 84, 85, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        80, 0, 1,  76,
        80, 1, 2,  63, 77,
        80, 2, 5,  62, 64, 78, 81, 82,
        80, 3, 8,  61, 66, 67, 79, 83, 86, 87, 88,
        80, 4, 4,  63, 64, 76, 80,
        80, 5, 4,  62, 66, 77, 81,
        80, 6, 14,  51, 52, 61, 65, 67, 72, 73, 78, 79, 82, 86, 87, 88, 92,
        80, 7, 10,  50, 54, 68, 74, 75, 83, 84, 85, 89, 93,
        81, 0, 1,  77,
        81, 1, 3,  76, 78, 82,
        81, 2, 7,  63, 79, 80, 83, 86, 87, 88,
        81, 3, 6,  62, 64, 84, 85, 89, 92,
        81, 4, 2,  77, 81,
        81, 5, 11,  63, 64, 76, 78, 79, 80, 82, 86, 87, 88, 92,
        81, 6, 7,  62, 66, 83, 84, 85, 89, 93,
        81, 7, 14,  51, 52, 61, 65, 67, 72, 73, 90, 91, 94, 101, 102, 108, 112,
        82, 0, 4,  77, 83, 86, 87,
        82, 1, 5,  76, 78, 81, 84, 85,
        82, 2, 4,  63, 79, 80, 88,
        82, 3, 4,  62, 64, 89, 92,
        82, 4, 3,  82, 86, 87,
        82, 5, 5,  77, 81, 83, 84, 85,
        82, 6, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        82, 7, 4,  62, 66, 89, 93,
        83, 0, 3,  82, 84, 85,
        83, 1, 3,  77, 86, 87,
        83, 2, 3,  76, 78, 81,
        83, 3, 4,  63, 79, 80, 88,
        83, 4, 3,  83, 84, 85,
        83, 5, 3,  82, 86, 87,
        83, 6, 2,  77, 81,
        83, 7, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        84, 0, 1,  83,
        84, 1, 2,  82, 85,
        84, 2, 3,  77, 86, 87,
        84, 3, 3,  76, 78, 81,
        84, 4, 3,  83, 84, 85,
        84, 5, 3,  82, 86, 87,
        84, 6, 2,  77, 81,
        84, 7, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        85, 0, 1,  83,
        85, 1, 2,  82, 84,
        85, 2, 3,  77, 86, 87,
        85, 3, 3,  76, 78, 81,
        85, 4, 3,  83, 84, 85,
        85, 5, 3,  82, 86, 87,
        85, 6, 2,  77, 81,
        85, 7, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        86, 0, 1,  82,
        86, 1, 3,  77, 83, 87,
        86, 2, 5,  76, 78, 81, 84, 85,
        86, 3, 4,  63, 79, 80, 88,
        86, 4, 3,  82, 86, 87,
        86, 5, 5,  77, 81, 83, 84, 85,
        86, 6, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        86, 7, 4,  62, 66, 89, 93,
        87, 0, 1,  82,
        87, 1, 3,  77, 83, 86,
        87, 2, 5,  76, 78, 81, 84, 85,
        87, 3, 4,  63, 79, 80, 88,
        87, 4, 3,  82, 86, 87,
        87, 5, 5,  77, 81, 83, 84, 85,
        87, 6, 8,  63, 64, 76, 78, 79, 80, 88, 92,
        87, 7, 4,  62, 66, 89, 93,
        88, 0, 3,  78, 89, 92,
        88, 1, 5,  77, 79, 90, 93, 94,
        88, 2, 8,  76, 81, 82, 91, 95, 101, 102, 108,
        88, 3, 9,  63, 80, 83, 86, 87, 96, 97, 109, 112,
        88, 4, 4,  78, 79, 88, 92,
        88, 5, 4,  77, 81, 89, 93,
        88, 6, 14,  63, 64, 76, 80, 82, 86, 87, 90, 91, 94, 101, 102, 108, 112,
        88, 7, 18,  62, 66, 83, 84, 85, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        89, 0, 4,  88, 90, 93, 94,
        89, 1, 7,  78, 91, 92, 95, 101, 102, 108,
        89, 2, 6,  77, 79, 96, 97, 109, 112,
        89, 3, 10,  76, 81, 82, 98, 99, 103, 104, 110, 113, 114,
        89, 4, 2,  89, 93,
        89, 5, 11,  78, 79, 88, 90, 91, 92, 94, 101, 102, 108, 112,
        89, 6, 15,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        89, 7, 14,  63, 64, 76, 80, 82, 86, 87, 110, 111, 114, 119, 120, 130, 134,
        90, 0, 3,  89, 91, 108,
        90, 1, 5,  88, 93, 94, 109, 112,
        90, 2, 8,  78, 92, 95, 101, 102, 110, 113, 114,
        90, 3, 9,  77, 79, 96, 97, 111, 115, 119, 120, 130,
        90, 4, 4,  90, 91, 108, 112,
        90, 5, 4,  89, 93, 109, 113,
        90, 6, 14,  78, 79, 88, 92, 94, 101, 102, 110, 111, 114, 119, 120, 130, 134,
        90, 7, 18,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 115, 121, 122, 131, 135,
        91, 0, 1,  90,
        91, 1, 2,  89, 108,
        91, 2, 5,  88, 93, 94, 109, 112,
        91, 3, 8,  78, 92, 95, 101, 102, 110, 113, 114,
        91, 4, 4,  90, 91, 108, 112,
        91, 5, 4,  89, 93, 109, 113,
        91, 6, 14,  78, 79, 88, 92, 94, 101, 102, 110, 111, 114, 119, 120, 130, 134,
        91, 7, 18,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 115, 121, 122, 131, 135,
        92, 0, 1,  88,
        92, 1, 2,  78, 89,
        92, 2, 5,  77, 79, 90, 93, 94,
        92, 3, 8,  76, 81, 82, 91, 95, 101, 102, 108,
        92, 4, 4,  78, 79, 88, 92,
        92, 5, 4,  77, 81, 89, 93,
        92, 6, 14,  63, 64, 76, 80, 82, 86, 87, 90, 91, 94, 101, 102, 108, 112,
        92, 7, 18,  62, 66, 83, 84, 85, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        93, 0, 1,  89,
        93, 1, 3,  88, 90, 94,
        93, 2, 7,  78, 91, 92, 95, 101, 102, 108,
        93, 3, 6,  77, 79, 96, 97, 109, 112,
        93, 4, 2,  89, 93,
        93, 5, 11,  78, 79, 88, 90, 91, 92, 94, 101, 102, 108, 112,
        93, 6, 15,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 109, 113,
        93, 7, 14,  63, 64, 76, 80, 82, 86, 87, 110, 111, 114, 119, 120, 130, 134,
        94, 0, 4,  89, 95, 101, 102,
        94, 1, 5,  88, 90, 93, 96, 97,
        94, 2, 8,  78, 91, 92, 98, 99, 103, 104, 108,
        94, 3, 8,  77, 79, 100, 100, 105, 106, 109, 112,
        94, 4, 3,  94, 101, 102,
        94, 5, 13,  89, 93, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        94, 6, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        94, 7, 4,  77, 81, 109, 113,
        95, 0, 3,  94, 96, 97,
        95, 1, 7,  89, 98, 99, 101, 102, 103, 104,
        95, 2, 7,  88, 90, 93, 100, 100, 105, 106,
        95, 3, 6,  78, 91, 92, 107, 107, 108,
        95, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        95, 5, 3,  94, 101, 102,
        95, 6, 2,  89, 93,
        95, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        96, 0, 3,  95, 98, 103,
        96, 1, 4,  94, 97, 100, 105,
        96, 2, 7,  89, 99, 99, 101, 102, 104, 107,
        96, 3, 5,  88, 90, 93, 106, 106,
        96, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        96, 5, 3,  94, 101, 102,
        96, 6, 2,  89, 93,
        96, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        97, 0, 3,  95, 99, 104,
        97, 1, 4,  94, 96, 100, 106,
        97, 2, 7,  89, 98, 98, 101, 102, 103, 107,
        97, 3, 5,  88, 90, 93, 105, 105,
        97, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        97, 5, 3,  94, 101, 102,
        97, 6, 2,  89, 93,
        97, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        98, 0, 3,  96, 100, 105,
        98, 1, 4,  95, 99, 103, 107,
        98, 2, 4,  94, 97, 97, 106,
        98, 3, 5,  89, 101, 102, 104, 104,
        98, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        98, 5, 3,  94, 101, 102,
        98, 6, 2,  89, 93,
        98, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        99, 0, 3,  97, 100, 106,
        99, 1, 4,  95, 98, 104, 107,
        99, 2, 4,  94, 96, 96, 105,
        99, 3, 5,  89, 101, 102, 103, 103,
        99, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        99, 5, 3,  94, 101, 102,
        99, 6, 2,  89, 93,
        99, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        100, 0, 3,  98, 99, 107,
        100, 1, 4,  96, 97, 105, 106,
        100, 2, 4,  95, 95, 103, 104,
        100, 3, 2,  94, 94,
        100, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        100, 5, 3,  94, 101, 102,
        100, 6, 2,  89, 93,
        100, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        101, 0, 1,  94,
        101, 1, 3,  89, 95, 102,
        101, 2, 5,  88, 90, 93, 96, 97,
        101, 3, 8,  78, 91, 92, 98, 99, 103, 104, 108,
        101, 4, 3,  94, 101, 102,
        101, 5, 13,  89, 93, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        101, 6, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        101, 7, 4,  77, 81, 109, 113,
        102, 0, 1,  94,
        102, 1, 3,  89, 95, 101,
        102, 2, 5,  88, 90, 93, 96, 97,
        102, 3, 8,  78, 91, 92, 98, 99, 103, 104, 108,
        102, 4, 3,  94, 101, 102,
        102, 5, 13,  89, 93, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        102, 6, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        102, 7, 4,  77, 81, 109, 113,
        103, 0, 1,  96,
        103, 1, 2,  95, 98,
        103, 2, 4,  94, 97, 100, 105,
        103, 3, 7,  89, 99, 99, 101, 102, 104, 107,
        103, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        103, 5, 3,  94, 101, 102,
        103, 6, 2,  89, 93,
        103, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        104, 0, 1,  97,
        104, 1, 2,  95, 99,
        104, 2, 4,  94, 96, 100, 106,
        104, 3, 7,  89, 98, 98, 101, 102, 103, 107,
        104, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        104, 5, 3,  94, 101, 102,
        104, 6, 2,  89, 93,
        104, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        105, 0, 1,  98,
        105, 1, 2,  96, 100,
        105, 2, 4,  95, 99, 103, 107,
        105, 3, 4,  94, 97, 97, 106,
        105, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        105, 5, 3,  94, 101, 102,
        105, 6, 2,  89, 93,
        105, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        106, 0, 1,  99,
        106, 1, 2,  97, 100,
        106, 2, 4,  95, 98, 104, 107,
        106, 3, 4,  94, 96, 96, 105,
        106, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        106, 5, 3,  94, 101, 102,
        106, 6, 2,  89, 93,
        106, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        107, 0, 1,  100,
        107, 1, 2,  98, 99,
        107, 2, 4,  96, 97, 105, 106,
        107, 3, 4,  95, 95, 103, 104,
        107, 4, 11,  95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107,
        107, 5, 3,  94, 101, 102,
        107, 6, 2,  89, 93,
        107, 7, 8,  78, 79, 88, 90, 91, 92, 108, 112,
        108, 0, 3,  90, 109, 112,
        108, 1, 5,  89, 91, 110, 113, 114,
        108, 2, 8,  88, 93, 94, 111, 115, 119, 120, 130,
        108, 3, 10,  78, 92, 95, 101, 102, 116, 121, 122, 131, 134,
        108, 4, 4,  90, 91, 108, 112,
        108, 5, 4,  89, 93, 109, 113,
        108, 6, 14,  78, 79, 88, 92, 94, 101, 102, 110, 111, 114, 119, 120, 130, 134,
        108, 7, 18,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 115, 121, 122, 131, 135,
        109, 0, 4,  108, 110, 113, 114,
        109, 1, 7,  90, 111, 112, 115, 119, 120, 130,
        109, 2, 7,  89, 91, 116, 121, 122, 131, 134,
        109, 3, 9,  88, 93, 94, 117, 123, 124, 132, 135, 136,
        109, 4, 2,  109, 113,
        109, 5, 11,  90, 91, 108, 110, 111, 112, 114, 119, 120, 130, 134,
        109, 6, 7,  89, 93, 115, 121, 122, 131, 135,
        109, 7, 18,  78, 79, 88, 92, 94, 101, 102, 116, 123, 124, 132, 133, 136, 137, 138, 139, 140, 144,
        110, 0, 3,  109, 111, 130,
        110, 1, 5,  108, 113, 114, 131, 134,
        110, 2, 8,  90, 112, 115, 119, 120, 132, 135, 136,
        110, 3, 10,  89, 91, 116, 121, 122, 133, 137, 138, 139, 140,
        110, 4, 4,  110, 111, 130, 134,
        110, 5, 4,  109, 113, 131, 135,
        110, 6, 15,  90, 91, 108, 112, 114, 119, 120, 132, 133, 136, 137, 138, 139, 140, 144,
        110, 7, 7,  89, 93, 115, 121, 122, 141, 145,
        111, 0, 1,  110,
        111, 1, 2,  109, 130,
        111, 2, 5,  108, 113, 114, 131, 134,
        111, 3, 8,  90, 112, 115, 119, 120, 132, 135, 136,
        111, 4, 4,  110, 111, 130, 134,
        111, 5, 4,  109, 113, 131, 135,
        111, 6, 15,  90, 91, 108, 112, 114, 119, 120, 132, 133, 136, 137, 138, 139, 140, 144,
        111, 7, 7,  89, 93, 115, 121, 122, 141, 145,
        112, 0, 1,  108,
        112, 1, 2,  90, 109,
        112, 2, 5,  89, 91, 110, 113, 114,
        112, 3, 8,  88, 93, 94, 111, 115, 119, 120, 130,
        112, 4, 4,  90, 91, 108, 112,
        112, 5, 4,  89, 93, 109, 113,
        112, 6, 14,  78, 79, 88, 92, 94, 101, 102, 110, 111, 114, 119, 120, 130, 134,
        112, 7, 18,  77, 81, 95, 96, 97, 98, 99, 100, 103, 104, 105, 106, 107, 115, 121, 122, 131, 135,
        113, 0, 1,  109,
        113, 1, 3,  108, 110, 114,
        113, 2, 7,  90, 111, 112, 115, 119, 120, 130,
        113, 3, 7,  89, 91, 116, 121, 122, 131, 134,
        113, 4, 2,  109, 113,
        113, 5, 11,  90, 91, 108, 110, 111, 112, 114, 119, 120, 130, 134,
        113, 6, 7,  89, 93, 115, 121, 122, 131, 135,
        113, 7, 18,  78, 79, 88, 92, 94, 101, 102, 116, 123, 124, 132, 133, 136, 137, 138, 139, 140, 144,
        114, 0, 4,  109, 115, 119, 120,
        114, 1, 6,  108, 110, 113, 116, 121, 122,
        114, 2, 7,  90, 111, 112, 117, 123, 124, 130,
        114, 3, 7,  89, 91, 118, 125, 126, 131, 134,
        114, 4, 3,  114, 119, 120,
        114, 5, 5,  109, 113, 115, 121, 122,
        114, 6, 11,  90, 91, 108, 110, 111, 112, 116, 123, 124, 130, 134,
        114, 7, 7,  89, 93, 117, 125, 126, 131, 135,
        115, 0, 4,  114, 116, 121, 122,
        115, 1, 6,  109, 117, 119, 120, 123, 124,
        115, 2, 6,  108, 110, 113, 118, 125, 126,
        115, 3, 7,  90, 111, 112, 127, 128, 129, 130,
        115, 4, 3,  115, 121, 122,
        115, 5, 6,  114, 116, 119, 120, 123, 124,
        115, 6, 5,  109, 113, 117, 125, 126,
        115, 7, 12,  90, 91, 108, 110, 111, 112, 118, 127, 128, 129, 130, 134,
        116, 0, 4,  115, 117, 123, 124,
        116, 1, 6,  114, 118, 121, 122, 125, 126,
        116, 2, 6,  109, 119, 120, 127, 128, 129,
        116, 3, 3,  108, 110, 113,
        116, 4, 3,  116, 123, 124,
        116, 5, 6,  115, 117, 121, 122, 125, 126,
        116, 6, 7,  114, 118, 119, 120, 127, 128, 129,
        116, 7, 2,  109, 113,
        117, 0, 4,  116, 118, 125, 126,
        117, 1, 6,  115, 123, 124, 127, 128, 129,
        117, 2, 3,  114, 121, 122,
        117, 3, 3,  109, 119, 120,
        117, 4, 3,  117, 125, 126,
        117, 5, 7,  116, 118, 123, 124, 127, 128, 129,
        117, 6, 3,  115, 121, 122,
        117, 7, 3,  114, 119, 120,
        118, 0, 4,  117, 127, 128, 129,
        118, 1, 3,  116, 125, 126,
        118, 2, 3,  115, 123, 124,
        118, 3, 3,  114, 121, 122,
        118, 4, 4,  118, 127, 128, 129,
        118, 5, 3,  117, 125, 126,
        118, 6, 3,  116, 123, 124,
        118, 7, 3,  115, 121, 122,
        119, 0, 1,  114,
        119, 1, 3,  109, 115, 120,
        119, 2, 6,  108, 110, 113, 116, 121, 122,
        119, 3, 7,  90, 111, 112, 117, 123, 124, 130,
        119, 4, 3,  114, 119, 120,
        119, 5, 5,  109, 113, 115, 121, 122,
        119, 6, 11,  90, 91, 108, 110, 111, 112, 116, 123, 124, 130, 134,
        119, 7, 7,  89, 93, 117, 125, 126, 131, 135,
        120, 0, 1,  114,
        120, 1, 3,  109, 115, 119,
        120, 2, 6,  108, 110, 113, 116, 121, 122,
        120, 3, 7,  90, 111, 112, 117, 123, 124, 130,
        120, 4, 3,  114, 119, 120,
        120, 5, 5,  109, 113, 115, 121, 122,
        120, 6, 11,  90, 91, 108, 110, 111, 112, 116, 123, 124, 130, 134,
        120, 7, 7,  89, 93, 117, 125, 126, 131, 135,
        121, 0, 1,  115,
        121, 1, 3,  114, 116, 122,
        121, 2, 6,  109, 117, 119, 120, 123, 124,
        121, 3, 6,  108, 110, 113, 118, 125, 126,
        121, 4, 3,  115, 121, 122,
        121, 5, 6,  114, 116, 119, 120, 123, 124,
        121, 6, 5,  109, 113, 117, 125, 126,
        121, 7, 12,  90, 91, 108, 110, 111, 112, 118, 127, 128, 129, 130, 134,
        122, 0, 1,  115,
        122, 1, 3,  114, 116, 121,
        122, 2, 6,  109, 117, 119, 120, 123, 124,
        122, 3, 6,  108, 110, 113, 118, 125, 126,
        122, 4, 3,  115, 121, 122,
        122, 5, 6,  114, 116, 119, 120, 123, 124,
        122, 6, 5,  109, 113, 117, 125, 126,
        122, 7, 12,  90, 91, 108, 110, 111, 112, 118, 127, 128, 129, 130, 134,
        123, 0, 1,  116,
        123, 1, 3,  115, 117, 124,
        123, 2, 6,  114, 118, 121, 122, 125, 126,
        123, 3, 6,  109, 119, 120, 127, 128, 129,
        123, 4, 3,  116, 123, 124,
        123, 5, 6,  115, 117, 121, 122, 125, 126,
        123, 6, 7,  114, 118, 119, 120, 127, 128, 129,
        123, 7, 2,  109, 113,
        124, 0, 1,  116,
        124, 1, 3,  115, 117, 123,
        124, 2, 6,  114, 118, 121, 122, 125, 126,
        124, 3, 6,  109, 119, 120, 127, 128, 129,
        124, 4, 3,  116, 123, 124,
        124, 5, 6,  115, 117, 121, 122, 125, 126,
        124, 6, 7,  114, 118, 119, 120, 127, 128, 129,
        124, 7, 2,  109, 113,
        125, 0, 1,  117,
        125, 1, 3,  116, 118, 126,
        125, 2, 6,  115, 123, 124, 127, 128, 129,
        125, 3, 3,  114, 121, 122,
        125, 4, 3,  117, 125, 126,
        125, 5, 7,  116, 118, 123, 124, 127, 128, 129,
        125, 6, 3,  115, 121, 122,
        125, 7, 3,  114, 119, 120,
        126, 0, 1,  117,
        126, 1, 3,  116, 118, 125,
        126, 2, 6,  115, 123, 124, 127, 128, 129,
        126, 3, 3,  114, 121, 122,
        126, 4, 3,  117, 125, 126,
        126, 5, 7,  116, 118, 123, 124, 127, 128, 129,
        126, 6, 3,  115, 121, 122,
        126, 7, 3,  114, 119, 120,
        127, 0, 1,  118,
        127, 1, 3,  117, 128, 129,
        127, 2, 3,  116, 125, 126,
        127, 3, 3,  115, 123, 124,
        127, 4, 4,  118, 127, 128, 129,
        127, 5, 3,  117, 125, 126,
        127, 6, 3,  116, 123, 124,
        127, 7, 3,  115, 121, 122,
        128, 0, 1,  118,
        128, 1, 3,  117, 127, 129,
        128, 2, 3,  116, 125, 126,
        128, 3, 3,  115, 123, 124,
        128, 4, 4,  118, 127, 128, 129,
        128, 5, 3,  117, 125, 126,
        128, 6, 3,  116, 123, 124,
        128, 7, 3,  115, 121, 122,
        129, 0, 1,  118,
        129, 1, 3,  117, 127, 128,
        129, 2, 3,  116, 125, 126,
        129, 3, 3,  115, 123, 124,
        129, 4, 4,  118, 127, 128, 129,
        129, 5, 3,  117, 125, 126,
        129, 6, 3,  116, 123, 124,
        129, 7, 3,  115, 121, 122,
        130, 0, 3,  110, 131, 134,
        130, 1, 5,  109, 111, 132, 135, 136,
        130, 2, 8,  108, 113, 114, 133, 137, 138, 139, 140,
        130, 3, 7,  90, 112, 115, 119, 120, 141, 144,
        130, 4, 4,  110, 111, 130, 134,
        130, 5, 4,  109, 113, 131, 135,
        130, 6, 15,  90, 91, 108, 112, 114, 119, 120, 132, 133, 136, 137, 138, 139, 140, 144,
        130, 7, 7,  89, 93, 115, 121, 122, 141, 145,
        131, 0, 4,  130, 132, 135, 136,
        131, 1, 7,  110, 133, 134, 137, 138, 139, 140,
        131, 2, 4,  109, 111, 141, 144,
        131, 3, 6,  108, 113, 114, 142, 145, 146,
        131, 4, 2,  131, 135,
        131, 5, 12,  110, 111, 130, 132, 133, 134, 136, 137, 138, 139, 140, 144,
        131, 6, 4,  109, 113, 141, 145,
        131, 7, 13,  90, 91, 108, 112, 114, 119, 120, 142, 143, 146, 149, 156, 160,
        132, 0, 3,  131, 133, 140,
        132, 1, 5,  130, 135, 136, 141, 144,
        132, 2, 8,  110, 134, 137, 138, 139, 142, 145, 146,
        132, 3, 7,  109, 111, 143, 147, 148, 149, 156,
        132, 4, 4,  132, 133, 140, 144,
        132, 5, 4,  131, 135, 141, 145,
        132, 6, 14,  110, 111, 130, 134, 136, 137, 138, 139, 142, 143, 146, 149, 156, 160,
        132, 7, 12,  109, 113, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        133, 0, 1,  132,
        133, 1, 2,  131, 140,
        133, 2, 5,  130, 135, 136, 141, 144,
        133, 3, 8,  110, 134, 137, 138, 139, 142, 145, 146,
        133, 4, 4,  132, 133, 140, 144,
        133, 5, 4,  131, 135, 141, 145,
        133, 6, 14,  110, 111, 130, 134, 136, 137, 138, 139, 142, 143, 146, 149, 156, 160,
        133, 7, 12,  109, 113, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        134, 0, 1,  130,
        134, 1, 2,  110, 131,
        134, 2, 5,  109, 111, 132, 135, 136,
        134, 3, 8,  108, 113, 114, 133, 137, 138, 139, 140,
        134, 4, 4,  110, 111, 130, 134,
        134, 5, 4,  109, 113, 131, 135,
        134, 6, 15,  90, 91, 108, 112, 114, 119, 120, 132, 133, 136, 137, 138, 139, 140, 144,
        134, 7, 7,  89, 93, 115, 121, 122, 141, 145,
        135, 0, 1,  131,
        135, 1, 3,  130, 132, 136,
        135, 2, 7,  110, 133, 134, 137, 138, 139, 140,
        135, 3, 4,  109, 111, 141, 144,
        135, 4, 2,  131, 135,
        135, 5, 12,  110, 111, 130, 132, 133, 134, 136, 137, 138, 139, 140, 144,
        135, 6, 4,  109, 113, 141, 145,
        135, 7, 13,  90, 91, 108, 112, 114, 119, 120, 142, 143, 146, 149, 156, 160,
        136, 0, 4,  131, 137, 138, 139,
        136, 1, 3,  130, 132, 135,
        136, 2, 4,  110, 133, 134, 140,
        136, 3, 4,  109, 111, 141, 144,
        136, 4, 4,  136, 137, 138, 139,
        136, 5, 2,  131, 135,
        136, 6, 8,  110, 111, 130, 132, 133, 134, 140, 144,
        136, 7, 4,  109, 113, 141, 145,
        137, 0, 1,  136,
        137, 1, 3,  131, 138, 139,
        137, 2, 3,  130, 132, 135,
        137, 3, 4,  110, 133, 134, 140,
        137, 4, 4,  136, 137, 138, 139,
        137, 5, 2,  131, 135,
        137, 6, 8,  110, 111, 130, 132, 133, 134, 140, 144,
        137, 7, 4,  109, 113, 141, 145,
        138, 0, 1,  136,
        138, 1, 3,  131, 137, 139,
        138, 2, 3,  130, 132, 135,
        138, 3, 4,  110, 133, 134, 140,
        138, 4, 4,  136, 137, 138, 139,
        138, 5, 2,  131, 135,
        138, 6, 8,  110, 111, 130, 132, 133, 134, 140, 144,
        138, 7, 4,  109, 113, 141, 145,
        139, 0, 1,  136,
        139, 1, 3,  131, 137, 138,
        139, 2, 3,  130, 132, 135,
        139, 3, 4,  110, 133, 134, 140,
        139, 4, 4,  136, 137, 138, 139,
        139, 5, 2,  131, 135,
        139, 6, 8,  110, 111, 130, 132, 133, 134, 140, 144,
        139, 7, 4,  109, 113, 141, 145,
        140, 0, 3,  132, 141, 144,
        140, 1, 5,  131, 133, 142, 145, 146,
        140, 2, 8,  130, 135, 136, 143, 147, 148, 149, 156,
        140, 3, 13,  110, 134, 137, 138, 139, 150, 151, 152, 153, 154, 155, 157, 160,
        140, 4, 4,  132, 133, 140, 144,
        140, 5, 4,  131, 135, 141, 145,
        140, 6, 14,  110, 111, 130, 134, 136, 137, 138, 139, 142, 143, 146, 149, 156, 160,
        140, 7, 12,  109, 113, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        141, 0, 4,  140, 142, 145, 146,
        141, 1, 7,  132, 143, 144, 147, 148, 149, 156,
        141, 2, 10,  131, 133, 150, 151, 152, 153, 154, 155, 157, 160,
        141, 3, 6,  130, 135, 136, 158, 161, 162,
        141, 4, 2,  141, 145,
        141, 5, 10,  132, 133, 140, 142, 143, 144, 146, 149, 156, 160,
        141, 6, 12,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        141, 7, 15,  110, 111, 130, 134, 136, 137, 138, 139, 158, 159, 162, 169, 170, 176, 180,
        142, 0, 3,  141, 143, 156,
        142, 1, 5,  140, 145, 146, 157, 160,
        142, 2, 8,  132, 144, 147, 148, 149, 158, 161, 162,
        142, 3, 13,  131, 133, 150, 151, 152, 153, 154, 155, 159, 163, 169, 170, 176,
        142, 4, 4,  142, 143, 156, 160,
        142, 5, 4,  141, 145, 157, 161,
        142, 6, 13,  132, 133, 140, 144, 146, 149, 158, 159, 162, 169, 170, 176, 180,
        142, 7, 24,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        143, 0, 1,  142,
        143, 1, 2,  141, 156,
        143, 2, 5,  140, 145, 146, 157, 160,
        143, 3, 8,  132, 144, 147, 148, 149, 158, 161, 162,
        143, 4, 4,  142, 143, 156, 160,
        143, 5, 4,  141, 145, 157, 161,
        143, 6, 13,  132, 133, 140, 144, 146, 149, 158, 159, 162, 169, 170, 176, 180,
        143, 7, 24,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        144, 0, 1,  140,
        144, 1, 2,  132, 141,
        144, 2, 5,  131, 133, 142, 145, 146,
        144, 3, 8,  130, 135, 136, 143, 147, 148, 149, 156,
        144, 4, 4,  132, 133, 140, 144,
        144, 5, 4,  131, 135, 141, 145,
        144, 6, 14,  110, 111, 130, 134, 136, 137, 138, 139, 142, 143, 146, 149, 156, 160,
        144, 7, 12,  109, 113, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        145, 0, 1,  141,
        145, 1, 3,  140, 142, 146,
        145, 2, 7,  132, 143, 144, 147, 148, 149, 156,
        145, 3, 10,  131, 133, 150, 151, 152, 153, 154, 155, 157, 160,
        145, 4, 2,  141, 145,
        145, 5, 10,  132, 133, 140, 142, 143, 144, 146, 149, 156, 160,
        145, 6, 12,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 157, 161,
        145, 7, 15,  110, 111, 130, 134, 136, 137, 138, 139, 158, 159, 162, 169, 170, 176, 180,
        146, 0, 4,  141, 147, 148, 149,
        146, 1, 9,  140, 142, 145, 150, 151, 152, 153, 154, 155,
        146, 2, 4,  132, 143, 144, 156,
        146, 3, 4,  131, 133, 157, 160,
        146, 4, 2,  146, 149,
        146, 5, 10,  141, 145, 147, 148, 150, 151, 152, 153, 154, 155,
        146, 6, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        146, 7, 4,  131, 135, 157, 161,
        147, 0, 4,  146, 150, 151, 152,
        147, 1, 3,  141, 148, 149,
        147, 2, 6,  140, 142, 145, 153, 154, 155,
        147, 3, 4,  132, 143, 144, 156,
        147, 4, 4,  147, 150, 151, 152,
        147, 5, 2,  146, 149,
        147, 6, 6,  141, 145, 148, 153, 154, 155,
        147, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        148, 0, 4,  146, 153, 154, 155,
        148, 1, 3,  141, 147, 149,
        148, 2, 6,  140, 142, 145, 150, 151, 152,
        148, 3, 4,  132, 143, 144, 156,
        148, 4, 4,  148, 153, 154, 155,
        148, 5, 2,  146, 149,
        148, 6, 6,  141, 145, 147, 150, 151, 152,
        148, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        149, 0, 1,  146,
        149, 1, 3,  141, 147, 148,
        149, 2, 9,  140, 142, 145, 150, 151, 152, 153, 154, 155,
        149, 3, 4,  132, 143, 144, 156,
        149, 4, 2,  146, 149,
        149, 5, 10,  141, 145, 147, 148, 150, 151, 152, 153, 154, 155,
        149, 6, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        149, 7, 4,  131, 135, 157, 161,
        150, 0, 1,  147,
        150, 1, 3,  146, 151, 152,
        150, 2, 3,  141, 148, 149,
        150, 3, 6,  140, 142, 145, 153, 154, 155,
        150, 4, 4,  147, 150, 151, 152,
        150, 5, 2,  146, 149,
        150, 6, 6,  141, 145, 148, 153, 154, 155,
        150, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        151, 0, 1,  147,
        151, 1, 3,  146, 150, 152,
        151, 2, 3,  141, 148, 149,
        151, 3, 6,  140, 142, 145, 153, 154, 155,
        151, 4, 4,  147, 150, 151, 152,
        151, 5, 2,  146, 149,
        151, 6, 6,  141, 145, 148, 153, 154, 155,
        151, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        152, 0, 1,  147,
        152, 1, 3,  146, 150, 151,
        152, 2, 3,  141, 148, 149,
        152, 3, 6,  140, 142, 145, 153, 154, 155,
        152, 4, 4,  147, 150, 151, 152,
        152, 5, 2,  146, 149,
        152, 6, 6,  141, 145, 148, 153, 154, 155,
        152, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        153, 0, 1,  148,
        153, 1, 3,  146, 154, 155,
        153, 2, 3,  141, 147, 149,
        153, 3, 6,  140, 142, 145, 150, 151, 152,
        153, 4, 4,  148, 153, 154, 155,
        153, 5, 2,  146, 149,
        153, 6, 6,  141, 145, 147, 150, 151, 152,
        153, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        154, 0, 1,  148,
        154, 1, 3,  146, 153, 155,
        154, 2, 3,  141, 147, 149,
        154, 3, 6,  140, 142, 145, 150, 151, 152,
        154, 4, 4,  148, 153, 154, 155,
        154, 5, 2,  146, 149,
        154, 6, 6,  141, 145, 147, 150, 151, 152,
        154, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        155, 0, 1,  148,
        155, 1, 3,  146, 153, 154,
        155, 2, 3,  141, 147, 149,
        155, 3, 6,  140, 142, 145, 150, 151, 152,
        155, 4, 4,  148, 153, 154, 155,
        155, 5, 2,  146, 149,
        155, 6, 6,  141, 145, 147, 150, 151, 152,
        155, 7, 8,  132, 133, 140, 142, 143, 144, 156, 160,
        156, 0, 3,  142, 157, 160,
        156, 1, 5,  141, 143, 158, 161, 162,
        156, 2, 8,  140, 145, 146, 159, 163, 169, 170, 176,
        156, 3, 9,  132, 144, 147, 148, 149, 164, 165, 177, 180,
        156, 4, 4,  142, 143, 156, 160,
        156, 5, 4,  141, 145, 157, 161,
        156, 6, 13,  132, 133, 140, 144, 146, 149, 158, 159, 162, 169, 170, 176, 180,
        156, 7, 24,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        157, 0, 4,  156, 158, 161, 162,
        157, 1, 7,  142, 159, 160, 163, 169, 170, 176,
        157, 2, 6,  141, 143, 164, 165, 177, 180,
        157, 3, 10,  140, 145, 146, 166, 167, 171, 172, 178, 181, 182,
        157, 4, 2,  157, 161,
        157, 5, 11,  142, 143, 156, 158, 159, 160, 162, 169, 170, 176, 180,
        157, 6, 16,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        157, 7, 10,  132, 133, 140, 144, 146, 149, 178, 179, 183, 187,
        158, 0, 3,  157, 159, 176,
        158, 1, 5,  156, 161, 162, 177, 180,
        158, 2, 8,  142, 160, 163, 169, 170, 178, 181, 182,
        158, 3, 6,  141, 143, 164, 165, 179, 183,
        158, 4, 4,  158, 159, 176, 180,
        158, 5, 5,  157, 161, 177, 181, 182,
        158, 6, 11,  142, 143, 156, 160, 162, 169, 170, 178, 179, 183, 187,
        158, 7, 15,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 184, 188,
        159, 0, 1,  158,
        159, 1, 2,  157, 176,
        159, 2, 5,  156, 161, 162, 177, 180,
        159, 3, 8,  142, 160, 163, 169, 170, 178, 181, 182,
        159, 4, 4,  158, 159, 176, 180,
        159, 5, 5,  157, 161, 177, 181, 182,
        159, 6, 11,  142, 143, 156, 160, 162, 169, 170, 178, 179, 183, 187,
        159, 7, 15,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 184, 188,
        160, 0, 1,  156,
        160, 1, 2,  142, 157,
        160, 2, 5,  141, 143, 158, 161, 162,
        160, 3, 8,  140, 145, 146, 159, 163, 169, 170, 176,
        160, 4, 4,  142, 143, 156, 160,
        160, 5, 4,  141, 145, 157, 161,
        160, 6, 13,  132, 133, 140, 144, 146, 149, 158, 159, 162, 169, 170, 176, 180,
        160, 7, 24,  131, 135, 147, 148, 150, 151, 152, 153, 154, 155, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        161, 0, 1,  157,
        161, 1, 3,  156, 158, 162,
        161, 2, 7,  142, 159, 160, 163, 169, 170, 176,
        161, 3, 6,  141, 143, 164, 165, 177, 180,
        161, 4, 2,  157, 161,
        161, 5, 11,  142, 143, 156, 158, 159, 160, 162, 169, 170, 176, 180,
        161, 6, 16,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 177, 181, 182,
        161, 7, 10,  132, 133, 140, 144, 146, 149, 178, 179, 183, 187,
        162, 0, 4,  157, 163, 169, 170,
        162, 1, 5,  156, 158, 161, 164, 165,
        162, 2, 8,  142, 159, 160, 166, 167, 171, 172, 176,
        162, 3, 8,  141, 143, 168, 168, 173, 174, 177, 180,
        162, 4, 3,  162, 169, 170,
        162, 5, 13,  157, 161, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        162, 6, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        162, 7, 5,  141, 145, 177, 181, 182,
        163, 0, 3,  162, 164, 165,
        163, 1, 7,  157, 166, 167, 169, 170, 171, 172,
        163, 2, 7,  156, 158, 161, 168, 168, 173, 174,
        163, 3, 6,  142, 159, 160, 175, 175, 176,
        163, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        163, 5, 3,  162, 169, 170,
        163, 6, 2,  157, 161,
        163, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        164, 0, 3,  163, 166, 171,
        164, 1, 4,  162, 165, 168, 173,
        164, 2, 7,  157, 167, 167, 169, 170, 172, 175,
        164, 3, 5,  156, 158, 161, 174, 174,
        164, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        164, 5, 3,  162, 169, 170,
        164, 6, 2,  157, 161,
        164, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        165, 0, 3,  163, 167, 172,
        165, 1, 4,  162, 164, 168, 174,
        165, 2, 7,  157, 166, 166, 169, 170, 171, 175,
        165, 3, 5,  156, 158, 161, 173, 173,
        165, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        165, 5, 3,  162, 169, 170,
        165, 6, 2,  157, 161,
        165, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        166, 0, 3,  164, 168, 173,
        166, 1, 4,  163, 167, 171, 175,
        166, 2, 4,  162, 165, 165, 174,
        166, 3, 5,  157, 169, 170, 172, 172,
        166, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        166, 5, 3,  162, 169, 170,
        166, 6, 2,  157, 161,
        166, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        167, 0, 3,  165, 168, 174,
        167, 1, 4,  163, 166, 172, 175,
        167, 2, 4,  162, 164, 164, 173,
        167, 3, 5,  157, 169, 170, 171, 171,
        167, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        167, 5, 3,  162, 169, 170,
        167, 6, 2,  157, 161,
        167, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        168, 0, 3,  166, 167, 175,
        168, 1, 4,  164, 165, 173, 174,
        168, 2, 4,  163, 163, 171, 172,
        168, 3, 2,  162, 162,
        168, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        168, 5, 3,  162, 169, 170,
        168, 6, 2,  157, 161,
        168, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        169, 0, 1,  162,
        169, 1, 3,  157, 163, 170,
        169, 2, 5,  156, 158, 161, 164, 165,
        169, 3, 8,  142, 159, 160, 166, 167, 171, 172, 176,
        169, 4, 3,  162, 169, 170,
        169, 5, 13,  157, 161, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        169, 6, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        169, 7, 5,  141, 145, 177, 181, 182,
        170, 0, 1,  162,
        170, 1, 3,  157, 163, 169,
        170, 2, 5,  156, 158, 161, 164, 165,
        170, 3, 8,  142, 159, 160, 166, 167, 171, 172, 176,
        170, 4, 3,  162, 169, 170,
        170, 5, 13,  157, 161, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        170, 6, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        170, 7, 5,  141, 145, 177, 181, 182,
        171, 0, 1,  164,
        171, 1, 2,  163, 166,
        171, 2, 4,  162, 165, 168, 173,
        171, 3, 7,  157, 167, 167, 169, 170, 172, 175,
        171, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        171, 5, 3,  162, 169, 170,
        171, 6, 2,  157, 161,
        171, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        172, 0, 1,  165,
        172, 1, 2,  163, 167,
        172, 2, 4,  162, 164, 168, 174,
        172, 3, 7,  157, 166, 166, 169, 170, 171, 175,
        172, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        172, 5, 3,  162, 169, 170,
        172, 6, 2,  157, 161,
        172, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        173, 0, 1,  166,
        173, 1, 2,  164, 168,
        173, 2, 4,  163, 167, 171, 175,
        173, 3, 4,  162, 165, 165, 174,
        173, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        173, 5, 3,  162, 169, 170,
        173, 6, 2,  157, 161,
        173, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        174, 0, 1,  167,
        174, 1, 2,  165, 168,
        174, 2, 4,  163, 166, 172, 175,
        174, 3, 4,  162, 164, 164, 173,
        174, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        174, 5, 3,  162, 169, 170,
        174, 6, 2,  157, 161,
        174, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        175, 0, 1,  168,
        175, 1, 2,  166, 167,
        175, 2, 4,  164, 165, 173, 174,
        175, 3, 4,  163, 163, 171, 172,
        175, 4, 11,  163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175,
        175, 5, 3,  162, 169, 170,
        175, 6, 2,  157, 161,
        175, 7, 8,  142, 143, 156, 158, 159, 160, 176, 180,
        176, 0, 3,  158, 177, 180,
        176, 1, 5,  157, 159, 178, 181, 182,
        176, 2, 5,  156, 161, 162, 179, 183,
        176, 3, 7,  142, 160, 163, 169, 170, 184, 187,
        176, 4, 4,  158, 159, 176, 180,
        176, 5, 5,  157, 161, 177, 181, 182,
        176, 6, 11,  142, 143, 156, 160, 162, 169, 170, 178, 179, 183, 187,
        176, 7, 15,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 184, 188,
        177, 0, 4,  176, 178, 181, 182,
        177, 1, 4,  158, 179, 180, 183,
        177, 2, 4,  157, 159, 184, 187,
        177, 3, 6,  156, 161, 162, 185, 188, 189,
        177, 4, 3,  177, 181, 182,
        177, 5, 8,  158, 159, 176, 178, 179, 180, 183, 187,
        177, 6, 4,  157, 161, 184, 188,
        177, 7, 14,  142, 143, 156, 160, 162, 169, 170, 185, 186, 189, 193, 194, 200, 204,
        178, 0, 3,  177, 179, 183,
        178, 1, 5,  176, 181, 182, 184, 187,
        178, 2, 5,  158, 180, 185, 188, 189,
        178, 3, 7,  157, 159, 186, 190, 193, 194, 200,
        178, 4, 4,  178, 179, 183, 187,
        178, 5, 5,  177, 181, 182, 184, 188,
        178, 6, 11,  158, 159, 176, 180, 185, 186, 189, 193, 194, 200, 204,
        178, 7, 12,  157, 161, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        179, 0, 1,  178,
        179, 1, 2,  177, 183,
        179, 2, 5,  176, 181, 182, 184, 187,
        179, 3, 5,  158, 180, 185, 188, 189,
        179, 4, 4,  178, 179, 183, 187,
        179, 5, 5,  177, 181, 182, 184, 188,
        179, 6, 11,  158, 159, 176, 180, 185, 186, 189, 193, 194, 200, 204,
        179, 7, 12,  157, 161, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        180, 0, 1,  176,
        180, 1, 2,  158, 177,
        180, 2, 5,  157, 159, 178, 181, 182,
        180, 3, 5,  156, 161, 162, 179, 183,
        180, 4, 4,  158, 159, 176, 180,
        180, 5, 5,  157, 161, 177, 181, 182,
        180, 6, 11,  142, 143, 156, 160, 162, 169, 170, 178, 179, 183, 187,
        180, 7, 15,  141, 145, 163, 164, 165, 166, 167, 168, 171, 172, 173, 174, 175, 184, 188,
        181, 0, 1,  177,
        181, 1, 3,  176, 178, 182,
        181, 2, 4,  158, 179, 180, 183,
        181, 3, 4,  157, 159, 184, 187,
        181, 4, 3,  177, 181, 182,
        181, 5, 8,  158, 159, 176, 178, 179, 180, 183, 187,
        181, 6, 4,  157, 161, 184, 188,
        181, 7, 14,  142, 143, 156, 160, 162, 169, 170, 185, 186, 189, 193, 194, 200, 204,
        182, 0, 1,  177,
        182, 1, 3,  176, 178, 181,
        182, 2, 4,  158, 179, 180, 183,
        182, 3, 4,  157, 159, 184, 187,
        182, 4, 3,  177, 181, 182,
        182, 5, 8,  158, 159, 176, 178, 179, 180, 183, 187,
        182, 6, 4,  157, 161, 184, 188,
        182, 7, 14,  142, 143, 156, 160, 162, 169, 170, 185, 186, 189, 193, 194, 200, 204,
        183, 0, 3,  178, 184, 187,
        183, 1, 5,  177, 179, 185, 188, 189,
        183, 2, 8,  176, 181, 182, 186, 190, 193, 194, 200,
        183, 3, 7,  158, 180, 191, 195, 196, 201, 204,
        183, 4, 4,  178, 179, 183, 187,
        183, 5, 5,  177, 181, 182, 184, 188,
        183, 6, 11,  158, 159, 176, 180, 185, 186, 189, 193, 194, 200, 204,
        183, 7, 12,  157, 161, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        184, 0, 4,  183, 185, 188, 189,
        184, 1, 7,  178, 186, 187, 190, 193, 194, 200,
        184, 2, 7,  177, 179, 191, 195, 196, 201, 204,
        184, 3, 7,  176, 181, 182, 192, 202, 205, 206,
        184, 4, 2,  184, 188,
        184, 5, 11,  178, 179, 183, 185, 186, 187, 189, 193, 194, 200, 204,
        184, 6, 13,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        184, 7, 12,  158, 159, 176, 180, 202, 203, 206, 207, 209, 210, 214, 218,
        185, 0, 3,  184, 186, 200,
        185, 1, 5,  183, 188, 189, 201, 204,
        185, 2, 8,  178, 187, 190, 193, 194, 202, 205, 206,
        185, 3, 10,  177, 179, 191, 195, 196, 203, 207, 208, 209, 214,
        185, 4, 4,  185, 186, 200, 204,
        185, 5, 4,  184, 188, 201, 205,
        185, 6, 15,  178, 179, 183, 187, 189, 193, 194, 202, 203, 206, 207, 209, 210, 214, 218,
        185, 7, 17,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 208, 211, 212, 213, 215, 219,
        186, 0, 1,  185,
        186, 1, 2,  184, 200,
        186, 2, 5,  183, 188, 189, 201, 204,
        186, 3, 8,  178, 187, 190, 193, 194, 202, 205, 206,
        186, 4, 4,  185, 186, 200, 204,
        186, 5, 4,  184, 188, 201, 205,
        186, 6, 15,  178, 179, 183, 187, 189, 193, 194, 202, 203, 206, 207, 209, 210, 214, 218,
        186, 7, 17,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 208, 211, 212, 213, 215, 219,
        187, 0, 1,  183,
        187, 1, 2,  178, 184,
        187, 2, 5,  177, 179, 185, 188, 189,
        187, 3, 8,  176, 181, 182, 186, 190, 193, 194, 200,
        187, 4, 4,  178, 179, 183, 187,
        187, 5, 5,  177, 181, 182, 184, 188,
        187, 6, 11,  158, 159, 176, 180, 185, 186, 189, 193, 194, 200, 204,
        187, 7, 12,  157, 161, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        188, 0, 1,  184,
        188, 1, 3,  183, 185, 189,
        188, 2, 7,  178, 186, 187, 190, 193, 194, 200,
        188, 3, 7,  177, 179, 191, 195, 196, 201, 204,
        188, 4, 2,  184, 188,
        188, 5, 11,  178, 179, 183, 185, 186, 187, 189, 193, 194, 200, 204,
        188, 6, 13,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 201, 205,
        188, 7, 12,  158, 159, 176, 180, 202, 203, 206, 207, 209, 210, 214, 218,
        189, 0, 4,  184, 190, 193, 194,
        189, 1, 6,  183, 185, 188, 191, 195, 196,
        189, 2, 5,  178, 186, 187, 192, 200,
        189, 3, 7,  177, 179, 197, 198, 199, 201, 204,
        189, 4, 3,  189, 193, 194,
        189, 5, 10,  184, 188, 190, 191, 192, 195, 196, 197, 198, 199,
        189, 6, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        189, 7, 5,  177, 181, 182, 201, 205,
        190, 0, 4,  189, 191, 195, 196,
        190, 1, 4,  184, 192, 193, 194,
        190, 2, 6,  183, 185, 188, 197, 198, 199,
        190, 3, 4,  178, 186, 187, 200,
        190, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        190, 5, 3,  189, 193, 194,
        190, 6, 2,  184, 188,
        190, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        191, 0, 2,  190, 192,
        191, 1, 6,  189, 195, 196, 197, 198, 199,
        191, 2, 3,  184, 193, 194,
        191, 3, 3,  183, 185, 188,
        191, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        191, 5, 3,  189, 193, 194,
        191, 6, 2,  184, 188,
        191, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        192, 0, 4,  191, 197, 198, 199,
        192, 1, 1,  190,
        192, 2, 3,  189, 195, 196,
        192, 3, 3,  184, 193, 194,
        192, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        192, 5, 3,  189, 193, 194,
        192, 6, 2,  184, 188,
        192, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        193, 0, 1,  189,
        193, 1, 3,  184, 190, 194,
        193, 2, 6,  183, 185, 188, 191, 195, 196,
        193, 3, 5,  178, 186, 187, 192, 200,
        193, 4, 3,  189, 193, 194,
        193, 5, 10,  184, 188, 190, 191, 192, 195, 196, 197, 198, 199,
        193, 6, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        193, 7, 5,  177, 181, 182, 201, 205,
        194, 0, 1,  189,
        194, 1, 3,  184, 190, 193,
        194, 2, 6,  183, 185, 188, 191, 195, 196,
        194, 3, 5,  178, 186, 187, 192, 200,
        194, 4, 3,  189, 193, 194,
        194, 5, 10,  184, 188, 190, 191, 192, 195, 196, 197, 198, 199,
        194, 6, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        194, 7, 5,  177, 181, 182, 201, 205,
        195, 0, 1,  190,
        195, 1, 3,  189, 191, 196,
        195, 2, 4,  184, 192, 193, 194,
        195, 3, 6,  183, 185, 188, 197, 198, 199,
        195, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        195, 5, 3,  189, 193, 194,
        195, 6, 2,  184, 188,
        195, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        196, 0, 1,  190,
        196, 1, 3,  189, 191, 195,
        196, 2, 4,  184, 192, 193, 194,
        196, 3, 6,  183, 185, 188, 197, 198, 199,
        196, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        196, 5, 3,  189, 193, 194,
        196, 6, 2,  184, 188,
        196, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        197, 0, 1,  192,
        197, 1, 3,  191, 198, 199,
        197, 2, 1,  190,
        197, 3, 3,  189, 195, 196,
        197, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        197, 5, 3,  189, 193, 194,
        197, 6, 2,  184, 188,
        197, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        198, 0, 1,  192,
        198, 1, 3,  191, 197, 199,
        198, 2, 1,  190,
        198, 3, 3,  189, 195, 196,
        198, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        198, 5, 3,  189, 193, 194,
        198, 6, 2,  184, 188,
        198, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        199, 0, 1,  192,
        199, 1, 3,  191, 197, 198,
        199, 2, 1,  190,
        199, 3, 3,  189, 195, 196,
        199, 4, 8,  190, 191, 192, 195, 196, 197, 198, 199,
        199, 5, 3,  189, 193, 194,
        199, 6, 2,  184, 188,
        199, 7, 8,  178, 179, 183, 185, 186, 187, 200, 204,
        200, 0, 3,  185, 201, 204,
        200, 1, 5,  184, 186, 202, 205, 206,
        200, 2, 8,  183, 188, 189, 203, 207, 208, 209, 214,
        200, 3, 11,  178, 187, 190, 193, 194, 210, 211, 212, 213, 215, 218,
        200, 4, 4,  185, 186, 200, 204,
        200, 5, 4,  184, 188, 201, 205,
        200, 6, 15,  178, 179, 183, 187, 189, 193, 194, 202, 203, 206, 207, 209, 210, 214, 218,
        200, 7, 17,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 208, 211, 212, 213, 215, 219,
        201, 0, 4,  200, 202, 205, 206,
        201, 1, 7,  185, 203, 204, 207, 208, 209, 214,
        201, 2, 8,  184, 186, 210, 211, 212, 213, 215, 218,
        201, 3, 6,  183, 188, 189, 216, 219, 220,
        201, 4, 2,  201, 205,
        201, 5, 12,  185, 186, 200, 202, 203, 204, 206, 207, 209, 210, 214, 218,
        201, 6, 8,  184, 188, 208, 211, 212, 213, 215, 219,
        201, 7, 14,  178, 179, 183, 187, 189, 193, 194, 216, 217, 220, 227, 228, 238, 242,
        202, 0, 3,  201, 203, 214,
        202, 1, 5,  200, 205, 206, 215, 218,
        202, 2, 8,  185, 204, 207, 208, 209, 216, 219, 220,
        202, 3, 11,  184, 186, 210, 211, 212, 213, 217, 221, 227, 228, 238,
        202, 4, 4,  202, 203, 214, 218,
        202, 5, 4,  201, 205, 215, 219,
        202, 6, 15,  185, 186, 200, 204, 206, 207, 209, 210, 216, 217, 220, 227, 228, 238, 242,
        202, 7, 11,  184, 188, 208, 211, 212, 213, 221, 229, 230, 239, 243,
        203, 0, 1,  202,
        203, 1, 2,  201, 214,
        203, 2, 5,  200, 205, 206, 215, 218,
        203, 3, 8,  185, 204, 207, 208, 209, 216, 219, 220,
        203, 4, 4,  202, 203, 214, 218,
        203, 5, 4,  201, 205, 215, 219,
        203, 6, 15,  185, 186, 200, 204, 206, 207, 209, 210, 216, 217, 220, 227, 228, 238, 242,
        203, 7, 11,  184, 188, 208, 211, 212, 213, 221, 229, 230, 239, 243,
        204, 0, 1,  200,
        204, 1, 2,  185, 201,
        204, 2, 5,  184, 186, 202, 205, 206,
        204, 3, 8,  183, 188, 189, 203, 207, 208, 209, 214,
        204, 4, 4,  185, 186, 200, 204,
        204, 5, 4,  184, 188, 201, 205,
        204, 6, 15,  178, 179, 183, 187, 189, 193, 194, 202, 203, 206, 207, 209, 210, 214, 218,
        204, 7, 17,  177, 181, 182, 190, 191, 192, 195, 196, 197, 198, 199, 208, 211, 212, 213, 215, 219,
        205, 0, 1,  201,
        205, 1, 3,  200, 202, 206,
        205, 2, 7,  185, 203, 204, 207, 208, 209, 214,
        205, 3, 8,  184, 186, 210, 211, 212, 213, 215, 218,
        205, 4, 2,  201, 205,
        205, 5, 12,  185, 186, 200, 202, 203, 204, 206, 207, 209, 210, 214, 218,
        205, 6, 8,  184, 188, 208, 211, 212, 213, 215, 219,
        205, 7, 14,  178, 179, 183, 187, 189, 193, 194, 216, 217, 220, 227, 228, 238, 242,
        206, 0, 4,  201, 207, 208, 209,
        206, 1, 7,  200, 202, 205, 210, 211, 212, 213,
        206, 2, 4,  185, 203, 204, 214,
        206, 3, 4,  184, 186, 215, 218,
        206, 4, 4,  206, 207, 209, 210,
        206, 5, 6,  201, 205, 208, 211, 212, 213,
        206, 6, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        206, 7, 4,  184, 188, 215, 219,
        207, 0, 2,  206, 210,
        207, 1, 3,  201, 208, 209,
        207, 2, 6,  200, 202, 205, 211, 212, 213,
        207, 3, 4,  185, 203, 204, 214,
        207, 4, 4,  206, 207, 209, 210,
        207, 5, 6,  201, 205, 208, 211, 212, 213,
        207, 6, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        207, 7, 4,  184, 188, 215, 219,
        208, 0, 4,  206, 211, 212, 213,
        208, 1, 3,  201, 207, 209,
        208, 2, 4,  200, 202, 205, 210,
        208, 3, 4,  185, 203, 204, 214,
        208, 4, 4,  208, 211, 212, 213,
        208, 5, 4,  206, 207, 209, 210,
        208, 6, 2,  201, 205,
        208, 7, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        209, 0, 1,  206,
        209, 1, 3,  201, 207, 208,
        209, 2, 7,  200, 202, 205, 210, 211, 212, 213,
        209, 3, 4,  185, 203, 204, 214,
        209, 4, 4,  206, 207, 209, 210,
        209, 5, 6,  201, 205, 208, 211, 212, 213,
        209, 6, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        209, 7, 4,  184, 188, 215, 219,
        210, 0, 1,  207,
        210, 1, 1,  206,
        210, 2, 3,  201, 208, 209,
        210, 3, 6,  200, 202, 205, 211, 212, 213,
        210, 4, 4,  206, 207, 209, 210,
        210, 5, 6,  201, 205, 208, 211, 212, 213,
        210, 6, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        210, 7, 4,  184, 188, 215, 219,
        211, 0, 1,  208,
        211, 1, 3,  206, 212, 213,
        211, 2, 3,  201, 207, 209,
        211, 3, 4,  200, 202, 205, 210,
        211, 4, 4,  208, 211, 212, 213,
        211, 5, 4,  206, 207, 209, 210,
        211, 6, 2,  201, 205,
        211, 7, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        212, 0, 1,  208,
        212, 1, 3,  206, 211, 213,
        212, 2, 3,  201, 207, 209,
        212, 3, 4,  200, 202, 205, 210,
        212, 4, 4,  208, 211, 212, 213,
        212, 5, 4,  206, 207, 209, 210,
        212, 6, 2,  201, 205,
        212, 7, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        213, 0, 1,  208,
        213, 1, 3,  206, 211, 212,
        213, 2, 3,  201, 207, 209,
        213, 3, 4,  200, 202, 205, 210,
        213, 4, 4,  208, 211, 212, 213,
        213, 5, 4,  206, 207, 209, 210,
        213, 6, 2,  201, 205,
        213, 7, 8,  185, 186, 200, 202, 203, 204, 214, 218,
        214, 0, 3,  202, 215, 218,
        214, 1, 5,  201, 203, 216, 219, 220,
        214, 2, 8,  200, 205, 206, 217, 221, 227, 228, 238,
        214, 3, 10,  185, 204, 207, 208, 209, 222, 229, 230, 239, 242,
        214, 4, 4,  202, 203, 214, 218,
        214, 5, 4,  201, 205, 215, 219,
        214, 6, 15,  185, 186, 200, 204, 206, 207, 209, 210, 216, 217, 220, 227, 228, 238, 242,
        214, 7, 11,  184, 188, 208, 211, 212, 213, 221, 229, 230, 239, 243,
        215, 0, 4,  214, 216, 219, 220,
        215, 1, 7,  202, 217, 218, 221, 227, 228, 238,
        215, 2, 7,  201, 203, 222, 229, 230, 239, 242,
        215, 3, 9,  200, 205, 206, 223, 231, 232, 240, 243, 244,
        215, 4, 2,  215, 219,
        215, 5, 11,  202, 203, 214, 216, 217, 218, 220, 227, 228, 238, 242,
        215, 6, 7,  201, 205, 221, 229, 230, 239, 243,
        215, 7, 29,  185, 186, 200, 204, 206, 207, 209, 210, 222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        216, 0, 3,  215, 217, 238,
        216, 1, 5,  214, 219, 220, 239, 242,
        216, 2, 8,  202, 218, 221, 227, 228, 240, 243, 244,
        216, 3, 10,  201, 203, 222, 229, 230, 241, 245, 246, 247, 249,
        216, 4, 4,  216, 217, 238, 242,
        216, 5, 4,  215, 219, 239, 243,
        216, 6, 16,  202, 203, 214, 218, 220, 227, 228, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        216, 7, 7,  201, 205, 221, 229, 230, 250, 254,
        217, 0, 1,  216,
        217, 1, 2,  215, 238,
        217, 2, 5,  214, 219, 220, 239, 242,
        217, 3, 8,  202, 218, 221, 227, 228, 240, 243, 244,
        217, 4, 4,  216, 217, 238, 242,
        217, 5, 4,  215, 219, 239, 243,
        217, 6, 16,  202, 203, 214, 218, 220, 227, 228, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        217, 7, 7,  201, 205, 221, 229, 230, 250, 254,
        218, 0, 1,  214,
        218, 1, 2,  202, 215,
        218, 2, 5,  201, 203, 216, 219, 220,
        218, 3, 8,  200, 205, 206, 217, 221, 227, 228, 238,
        218, 4, 4,  202, 203, 214, 218,
        218, 5, 4,  201, 205, 215, 219,
        218, 6, 15,  185, 186, 200, 204, 206, 207, 209, 210, 216, 217, 220, 227, 228, 238, 242,
        218, 7, 11,  184, 188, 208, 211, 212, 213, 221, 229, 230, 239, 243,
        219, 0, 1,  215,
        219, 1, 3,  214, 216, 220,
        219, 2, 7,  202, 217, 218, 221, 227, 228, 238,
        219, 3, 7,  201, 203, 222, 229, 230, 239, 242,
        219, 4, 2,  215, 219,
        219, 5, 11,  202, 203, 214, 216, 217, 218, 220, 227, 228, 238, 242,
        219, 6, 7,  201, 205, 221, 229, 230, 239, 243,
        219, 7, 29,  185, 186, 200, 204, 206, 207, 209, 210, 222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        220, 0, 4,  215, 221, 227, 228,
        220, 1, 6,  214, 216, 219, 222, 229, 230,
        220, 2, 7,  202, 217, 218, 223, 231, 232, 238,
        220, 3, 6,  201, 203, 224, 233, 239, 242,
        220, 4, 3,  220, 227, 228,
        220, 5, 5,  215, 219, 221, 229, 230,
        220, 6, 20,  202, 203, 214, 216, 217, 218, 222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237, 238, 242,
        220, 7, 4,  201, 205, 239, 243,
        221, 0, 4,  220, 222, 229, 230,
        221, 1, 6,  215, 223, 227, 228, 231, 232,
        221, 2, 5,  214, 216, 219, 224, 233,
        221, 3, 6,  202, 217, 218, 225, 226, 238,
        221, 4, 3,  221, 229, 230,
        221, 5, 15,  220, 222, 223, 224, 225, 226, 227, 228, 231, 232, 233, 234, 235, 236, 237,
        221, 6, 2,  215, 219,
        221, 7, 8,  202, 203, 214, 216, 217, 218, 238, 242,
        222, 0, 4,  221, 223, 231, 232,
        222, 1, 5,  220, 224, 229, 230, 233,
        222, 2, 5,  215, 225, 226, 227, 228,
        222, 3, 7,  214, 216, 219, 234, 235, 236, 237,
        222, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        222, 5, 3,  221, 229, 230,
        222, 6, 3,  220, 227, 228,
        222, 7, 2,  215, 219,
        223, 0, 3,  222, 224, 233,
        223, 1, 5,  221, 225, 226, 231, 232,
        223, 2, 7,  220, 229, 230, 234, 235, 236, 237,
        223, 3, 3,  215, 227, 228,
        223, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        223, 5, 3,  221, 229, 230,
        223, 6, 3,  220, 227, 228,
        223, 7, 2,  215, 219,
        224, 0, 3,  223, 225, 226,
        224, 1, 6,  222, 233, 234, 235, 236, 237,
        224, 2, 3,  221, 231, 232,
        224, 3, 3,  220, 229, 230,
        224, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        224, 5, 3,  221, 229, 230,
        224, 6, 3,  220, 227, 228,
        224, 7, 2,  215, 219,
        225, 0, 3,  224, 234, 235,
        225, 1, 2,  223, 226,
        225, 2, 4,  222, 233, 236, 237,
        225, 3, 3,  221, 231, 232,
        225, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        225, 5, 3,  221, 229, 230,
        225, 6, 3,  220, 227, 228,
        225, 7, 2,  215, 219,
        226, 0, 3,  224, 236, 237,
        226, 1, 2,  223, 225,
        226, 2, 4,  222, 233, 234, 235,
        226, 3, 3,  221, 231, 232,
        226, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        226, 5, 3,  221, 229, 230,
        226, 6, 3,  220, 227, 228,
        226, 7, 2,  215, 219,
        227, 0, 1,  220,
        227, 1, 3,  215, 221, 228,
        227, 2, 6,  214, 216, 219, 222, 229, 230,
        227, 3, 7,  202, 217, 218, 223, 231, 232, 238,
        227, 4, 3,  220, 227, 228,
        227, 5, 5,  215, 219, 221, 229, 230,
        227, 6, 20,  202, 203, 214, 216, 217, 218, 222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237, 238, 242,
        227, 7, 4,  201, 205, 239, 243,
        228, 0, 1,  220,
        228, 1, 3,  215, 221, 227,
        228, 2, 6,  214, 216, 219, 222, 229, 230,
        228, 3, 7,  202, 217, 218, 223, 231, 232, 238,
        228, 4, 3,  220, 227, 228,
        228, 5, 5,  215, 219, 221, 229, 230,
        228, 6, 20,  202, 203, 214, 216, 217, 218, 222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237, 238, 242,
        228, 7, 4,  201, 205, 239, 243,
        229, 0, 1,  221,
        229, 1, 3,  220, 222, 230,
        229, 2, 6,  215, 223, 227, 228, 231, 232,
        229, 3, 5,  214, 216, 219, 224, 233,
        229, 4, 3,  221, 229, 230,
        229, 5, 15,  220, 222, 223, 224, 225, 226, 227, 228, 231, 232, 233, 234, 235, 236, 237,
        229, 6, 2,  215, 219,
        229, 7, 8,  202, 203, 214, 216, 217, 218, 238, 242,
        230, 0, 1,  221,
        230, 1, 3,  220, 222, 229,
        230, 2, 6,  215, 223, 227, 228, 231, 232,
        230, 3, 5,  214, 216, 219, 224, 233,
        230, 4, 3,  221, 229, 230,
        230, 5, 15,  220, 222, 223, 224, 225, 226, 227, 228, 231, 232, 233, 234, 235, 236, 237,
        230, 6, 2,  215, 219,
        230, 7, 8,  202, 203, 214, 216, 217, 218, 238, 242,
        231, 0, 1,  222,
        231, 1, 3,  221, 223, 232,
        231, 2, 5,  220, 224, 229, 230, 233,
        231, 3, 5,  215, 225, 226, 227, 228,
        231, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        231, 5, 3,  221, 229, 230,
        231, 6, 3,  220, 227, 228,
        231, 7, 2,  215, 219,
        232, 0, 1,  222,
        232, 1, 3,  221, 223, 231,
        232, 2, 5,  220, 224, 229, 230, 233,
        232, 3, 5,  215, 225, 226, 227, 228,
        232, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        232, 5, 3,  221, 229, 230,
        232, 6, 3,  220, 227, 228,
        232, 7, 2,  215, 219,
        233, 0, 1,  223,
        233, 1, 2,  222, 224,
        233, 2, 5,  221, 225, 226, 231, 232,
        233, 3, 7,  220, 229, 230, 234, 235, 236, 237,
        233, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        233, 5, 3,  221, 229, 230,
        233, 6, 3,  220, 227, 228,
        233, 7, 2,  215, 219,
        234, 0, 1,  225,
        234, 1, 2,  224, 235,
        234, 2, 2,  223, 226,
        234, 3, 4,  222, 233, 236, 237,
        234, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        234, 5, 3,  221, 229, 230,
        234, 6, 3,  220, 227, 228,
        234, 7, 2,  215, 219,
        235, 0, 1,  225,
        235, 1, 2,  224, 234,
        235, 2, 2,  223, 226,
        235, 3, 4,  222, 233, 236, 237,
        235, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        235, 5, 3,  221, 229, 230,
        235, 6, 3,  220, 227, 228,
        235, 7, 2,  215, 219,
        236, 0, 1,  226,
        236, 1, 2,  224, 237,
        236, 2, 2,  223, 225,
        236, 3, 4,  222, 233, 234, 235,
        236, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        236, 5, 3,  221, 229, 230,
        236, 6, 3,  220, 227, 228,
        236, 7, 2,  215, 219,
        237, 0, 1,  226,
        237, 1, 2,  224, 236,
        237, 2, 2,  223, 225,
        237, 3, 4,  222, 233, 234, 235,
        237, 4, 12,  222, 223, 224, 225, 226, 231, 232, 233, 234, 235, 236, 237,
        237, 5, 3,  221, 229, 230,
        237, 6, 3,  220, 227, 228,
        237, 7, 2,  215, 219,
        238, 0, 3,  216, 239, 242,
        238, 1, 5,  215, 217, 240, 243, 244,
        238, 2, 8,  214, 219, 220, 241, 245, 246, 247, 249,
        238, 3, 8,  202, 218, 221, 227, 228, 248, 250, 253,
        238, 4, 4,  216, 217, 238, 242,
        238, 5, 4,  215, 219, 239, 243,
        238, 6, 16,  202, 203, 214, 218, 220, 227, 228, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        238, 7, 7,  201, 205, 221, 229, 230, 250, 254,
        239, 0, 4,  238, 240, 243, 244,
        239, 1, 7,  216, 241, 242, 245, 246, 247, 249,
        239, 2, 5,  215, 217, 248, 250, 253,
        239, 3, 6,  214, 219, 220, 251, 254, 255,
        239, 4, 2,  239, 243,
        239, 5, 13,  216, 217, 238, 240, 241, 242, 244, 245, 246, 247, 248, 249, 253,
        239, 6, 4,  215, 219, 250, 254,
        239, 7, 15,  202, 203, 214, 218, 220, 227, 228, 251, 252, 255, 256, 257, 258, 259, 263,
        240, 0, 3,  239, 241, 249,
        240, 1, 5,  238, 243, 244, 250, 253,
        240, 2, 8,  216, 242, 245, 246, 247, 251, 254, 255,
        240, 3, 8,  215, 217, 248, 252, 256, 257, 258, 259,
        240, 4, 4,  240, 241, 249, 253,
        240, 5, 4,  239, 243, 250, 254,
        240, 6, 17,  216, 217, 238, 242, 244, 245, 246, 247, 248, 251, 252, 255, 256, 257, 258, 259, 263,
        240, 7, 4,  215, 219, 260, 264,
        241, 0, 1,  240,
        241, 1, 2,  239, 249,
        241, 2, 5,  238, 243, 244, 250, 253,
        241, 3, 8,  216, 242, 245, 246, 247, 251, 254, 255,
        241, 4, 4,  240, 241, 249, 253,
        241, 5, 4,  239, 243, 250, 254,
        241, 6, 17,  216, 217, 238, 242, 244, 245, 246, 247, 248, 251, 252, 255, 256, 257, 258, 259, 263,
        241, 7, 4,  215, 219, 260, 264,
        242, 0, 1,  238,
        242, 1, 2,  216, 239,
        242, 2, 5,  215, 217, 240, 243, 244,
        242, 3, 8,  214, 219, 220, 241, 245, 246, 247, 249,
        242, 4, 4,  216, 217, 238, 242,
        242, 5, 4,  215, 219, 239, 243,
        242, 6, 16,  202, 203, 214, 218, 220, 227, 228, 240, 241, 244, 245, 246, 247, 248, 249, 253,
        242, 7, 7,  201, 205, 221, 229, 230, 250, 254,
        243, 0, 1,  239,
        243, 1, 3,  238, 240, 244,
        243, 2, 7,  216, 241, 242, 245, 246, 247, 249,
        243, 3, 5,  215, 217, 248, 250, 253,
        243, 4, 2,  239, 243,
        243, 5, 13,  216, 217, 238, 240, 241, 242, 244, 245, 246, 247, 248, 249, 253,
        243, 6, 4,  215, 219, 250, 254,
        243, 7, 15,  202, 203, 214, 218, 220, 227, 228, 251, 252, 255, 256, 257, 258, 259, 263,
        244, 0, 4,  239, 245, 246, 247,
        244, 1, 4,  238, 240, 243, 248,
        244, 2, 4,  216, 241, 242, 249,
        244, 3, 4,  215, 217, 250, 253,
        244, 4, 5,  244, 245, 246, 247, 248,
        244, 5, 2,  239, 243,
        244, 6, 8,  216, 217, 238, 240, 241, 242, 249, 253,
        244, 7, 4,  215, 219, 250, 254,
        245, 0, 2,  244, 248,
        245, 1, 3,  239, 246, 247,
        245, 2, 3,  238, 240, 243,
        245, 3, 4,  216, 241, 242, 249,
        245, 4, 5,  244, 245, 246, 247, 248,
        245, 5, 2,  239, 243,
        245, 6, 8,  216, 217, 238, 240, 241, 242, 249, 253,
        245, 7, 4,  215, 219, 250, 254,
        246, 0, 1,  244,
        246, 1, 3,  239, 245, 247,
        246, 2, 4,  238, 240, 243, 248,
        246, 3, 4,  216, 241, 242, 249,
        246, 4, 5,  244, 245, 246, 247, 248,
        246, 5, 2,  239, 243,
        246, 6, 8,  216, 217, 238, 240, 241, 242, 249, 253,
        246, 7, 4,  215, 219, 250, 254,
        247, 0, 1,  244,
        247, 1, 3,  239, 245, 246,
        247, 2, 4,  238, 240, 243, 248,
        247, 3, 4,  216, 241, 242, 249,
        247, 4, 5,  244, 245, 246, 247, 248,
        247, 5, 2,  239, 243,
        247, 6, 8,  216, 217, 238, 240, 241, 242, 249, 253,
        247, 7, 4,  215, 219, 250, 254,
        248, 0, 1,  245,
        248, 1, 1,  244,
        248, 2, 3,  239, 246, 247,
        248, 3, 3,  238, 240, 243,
        248, 4, 5,  244, 245, 246, 247, 248,
        248, 5, 2,  239, 243,
        248, 6, 8,  216, 217, 238, 240, 241, 242, 249, 253,
        248, 7, 4,  215, 219, 250, 254,
        249, 0, 3,  240, 250, 253,
        249, 1, 5,  239, 241, 251, 254, 255,
        249, 2, 8,  238, 243, 244, 252, 256, 257, 258, 259,
        249, 3, 7,  216, 242, 245, 246, 247, 260, 263,
        249, 4, 4,  240, 241, 249, 253,
        249, 5, 4,  239, 243, 250, 254,
        249, 6, 17,  216, 217, 238, 242, 244, 245, 246, 247, 248, 251, 252, 255, 256, 257, 258, 259, 263,
        249, 7, 4,  215, 219, 260, 264,
        250, 0, 4,  249, 251, 254, 255,
        250, 1, 7,  240, 252, 253, 256, 257, 258, 259,
        250, 2, 4,  239, 241, 260, 263,
        250, 3, 6,  238, 243, 244, 261, 264, 265,
        250, 4, 2,  250, 254,
        250, 5, 12,  240, 241, 249, 251, 252, 253, 255, 256, 257, 258, 259, 263,
        250, 6, 4,  239, 243, 260, 264,
        250, 7, 16,  216, 217, 238, 242, 244, 245, 246, 247, 248, 261, 262, 265, 272, 273, 279, 283,
        251, 0, 3,  250, 252, 259,
        251, 1, 5,  249, 254, 255, 260, 263,
        251, 2, 8,  240, 253, 256, 257, 258, 261, 264, 265,
        251, 3, 7,  239, 241, 262, 266, 272, 273, 279,
        251, 4, 4,  251, 252, 259, 263,
        251, 5, 4,  250, 254, 260, 264,
        251, 6, 15,  240, 241, 249, 253, 255, 256, 257, 258, 261, 262, 265, 272, 273, 279, 283,
        251, 7, 15,  239, 243, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        252, 0, 1,  251,
        252, 1, 2,  250, 259,
        252, 2, 5,  249, 254, 255, 260, 263,
        252, 3, 8,  240, 253, 256, 257, 258, 261, 264, 265,
        252, 4, 4,  251, 252, 259, 263,
        252, 5, 4,  250, 254, 260, 264,
        252, 6, 15,  240, 241, 249, 253, 255, 256, 257, 258, 261, 262, 265, 272, 273, 279, 283,
        252, 7, 15,  239, 243, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        253, 0, 1,  249,
        253, 1, 2,  240, 250,
        253, 2, 5,  239, 241, 251, 254, 255,
        253, 3, 8,  238, 243, 244, 252, 256, 257, 258, 259,
        253, 4, 4,  240, 241, 249, 253,
        253, 5, 4,  239, 243, 250, 254,
        253, 6, 17,  216, 217, 238, 242, 244, 245, 246, 247, 248, 251, 252, 255, 256, 257, 258, 259, 263,
        253, 7, 4,  215, 219, 260, 264,
        254, 0, 1,  250,
        254, 1, 3,  249, 251, 255,
        254, 2, 7,  240, 252, 253, 256, 257, 258, 259,
        254, 3, 4,  239, 241, 260, 263,
        254, 4, 2,  250, 254,
        254, 5, 12,  240, 241, 249, 251, 252, 253, 255, 256, 257, 258, 259, 263,
        254, 6, 4,  239, 243, 260, 264,
        254, 7, 16,  216, 217, 238, 242, 244, 245, 246, 247, 248, 261, 262, 265, 272, 273, 279, 283,
        255, 0, 4,  250, 256, 257, 258,
        255, 1, 3,  249, 251, 254,
        255, 2, 4,  240, 252, 253, 259,
        255, 3, 4,  239, 241, 260, 263,
        255, 4, 4,  255, 256, 257, 258,
        255, 5, 2,  250, 254,
        255, 6, 8,  240, 241, 249, 251, 252, 253, 259, 263,
        255, 7, 4,  239, 243, 260, 264,
        256, 0, 1,  255,
        256, 1, 3,  250, 257, 258,
        256, 2, 3,  249, 251, 254,
        256, 3, 4,  240, 252, 253, 259,
        256, 4, 4,  255, 256, 257, 258,
        256, 5, 2,  250, 254,
        256, 6, 8,  240, 241, 249, 251, 252, 253, 259, 263,
        256, 7, 4,  239, 243, 260, 264,
        257, 0, 1,  255,
        257, 1, 3,  250, 256, 258,
        257, 2, 3,  249, 251, 254,
        257, 3, 4,  240, 252, 253, 259,
        257, 4, 4,  255, 256, 257, 258,
        257, 5, 2,  250, 254,
        257, 6, 8,  240, 241, 249, 251, 252, 253, 259, 263,
        257, 7, 4,  239, 243, 260, 264,
        258, 0, 1,  255,
        258, 1, 3,  250, 256, 257,
        258, 2, 3,  249, 251, 254,
        258, 3, 4,  240, 252, 253, 259,
        258, 4, 4,  255, 256, 257, 258,
        258, 5, 2,  250, 254,
        258, 6, 8,  240, 241, 249, 251, 252, 253, 259, 263,
        258, 7, 4,  239, 243, 260, 264,
        259, 0, 3,  251, 260, 263,
        259, 1, 5,  250, 252, 261, 264, 265,
        259, 2, 8,  249, 254, 255, 262, 266, 272, 273, 279,
        259, 3, 9,  240, 253, 256, 257, 258, 267, 268, 280, 283,
        259, 4, 4,  251, 252, 259, 263,
        259, 5, 4,  250, 254, 260, 264,
        259, 6, 15,  240, 241, 249, 253, 255, 256, 257, 258, 261, 262, 265, 272, 273, 279, 283,
        259, 7, 15,  239, 243, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        260, 0, 4,  259, 261, 264, 265,
        260, 1, 7,  251, 262, 263, 266, 272, 273, 279,
        260, 2, 6,  250, 252, 267, 268, 280, 283,
        260, 3, 10,  249, 254, 255, 269, 270, 274, 275, 281, 284, 285,
        260, 4, 2,  260, 264,
        260, 5, 11,  251, 252, 259, 261, 262, 263, 265, 272, 273, 279, 283,
        260, 6, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        260, 7, 16,  240, 241, 249, 253, 255, 256, 257, 258, 281, 282, 285, 286, 287, 288, 289, 293,
        261, 0, 3,  260, 262, 279,
        261, 1, 5,  259, 264, 265, 280, 283,
        261, 2, 8,  251, 263, 266, 272, 273, 281, 284, 285,
        261, 3, 9,  250, 252, 267, 268, 282, 286, 287, 288, 289,
        261, 4, 4,  261, 262, 279, 283,
        261, 5, 4,  260, 264, 280, 284,
        261, 6, 15,  251, 252, 259, 263, 265, 272, 273, 281, 282, 285, 286, 287, 288, 289, 293,
        261, 7, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 290, 294,
        262, 0, 1,  261,
        262, 1, 2,  260, 279,
        262, 2, 5,  259, 264, 265, 280, 283,
        262, 3, 8,  251, 263, 266, 272, 273, 281, 284, 285,
        262, 4, 4,  261, 262, 279, 283,
        262, 5, 4,  260, 264, 280, 284,
        262, 6, 15,  251, 252, 259, 263, 265, 272, 273, 281, 282, 285, 286, 287, 288, 289, 293,
        262, 7, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 290, 294,
        263, 0, 1,  259,
        263, 1, 2,  251, 260,
        263, 2, 5,  250, 252, 261, 264, 265,
        263, 3, 8,  249, 254, 255, 262, 266, 272, 273, 279,
        263, 4, 4,  251, 252, 259, 263,
        263, 5, 4,  250, 254, 260, 264,
        263, 6, 15,  240, 241, 249, 253, 255, 256, 257, 258, 261, 262, 265, 272, 273, 279, 283,
        263, 7, 15,  239, 243, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        264, 0, 1,  260,
        264, 1, 3,  259, 261, 265,
        264, 2, 7,  251, 262, 263, 266, 272, 273, 279,
        264, 3, 6,  250, 252, 267, 268, 280, 283,
        264, 4, 2,  260, 264,
        264, 5, 11,  251, 252, 259, 261, 262, 263, 265, 272, 273, 279, 283,
        264, 6, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 280, 284,
        264, 7, 16,  240, 241, 249, 253, 255, 256, 257, 258, 281, 282, 285, 286, 287, 288, 289, 293,
        265, 0, 4,  260, 266, 272, 273,
        265, 1, 5,  259, 261, 264, 267, 268,
        265, 2, 8,  251, 262, 263, 269, 270, 274, 275, 279,
        265, 3, 8,  250, 252, 271, 271, 276, 277, 280, 283,
        265, 4, 3,  265, 272, 273,
        265, 5, 13,  260, 264, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        265, 6, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        265, 7, 4,  250, 254, 280, 284,
        266, 0, 3,  265, 267, 268,
        266, 1, 7,  260, 269, 270, 272, 273, 274, 275,
        266, 2, 7,  259, 261, 264, 271, 271, 276, 277,
        266, 3, 6,  251, 262, 263, 278, 278, 279,
        266, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        266, 5, 3,  265, 272, 273,
        266, 6, 2,  260, 264,
        266, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        267, 0, 3,  266, 269, 274,
        267, 1, 4,  265, 268, 271, 276,
        267, 2, 7,  260, 270, 270, 272, 273, 275, 278,
        267, 3, 5,  259, 261, 264, 277, 277,
        267, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        267, 5, 3,  265, 272, 273,
        267, 6, 2,  260, 264,
        267, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        268, 0, 3,  266, 270, 275,
        268, 1, 4,  265, 267, 271, 277,
        268, 2, 7,  260, 269, 269, 272, 273, 274, 278,
        268, 3, 5,  259, 261, 264, 276, 276,
        268, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        268, 5, 3,  265, 272, 273,
        268, 6, 2,  260, 264,
        268, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        269, 0, 3,  267, 271, 276,
        269, 1, 4,  266, 270, 274, 278,
        269, 2, 4,  265, 268, 268, 277,
        269, 3, 5,  260, 272, 273, 275, 275,
        269, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        269, 5, 3,  265, 272, 273,
        269, 6, 2,  260, 264,
        269, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        270, 0, 3,  268, 271, 277,
        270, 1, 4,  266, 269, 275, 278,
        270, 2, 4,  265, 267, 267, 276,
        270, 3, 5,  260, 272, 273, 274, 274,
        270, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        270, 5, 3,  265, 272, 273,
        270, 6, 2,  260, 264,
        270, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        271, 0, 3,  269, 270, 278,
        271, 1, 4,  267, 268, 276, 277,
        271, 2, 4,  266, 266, 274, 275,
        271, 3, 2,  265, 265,
        271, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        271, 5, 3,  265, 272, 273,
        271, 6, 2,  260, 264,
        271, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        272, 0, 1,  265,
        272, 1, 3,  260, 266, 273,
        272, 2, 5,  259, 261, 264, 267, 268,
        272, 3, 8,  251, 262, 263, 269, 270, 274, 275, 279,
        272, 4, 3,  265, 272, 273,
        272, 5, 13,  260, 264, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        272, 6, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        272, 7, 4,  250, 254, 280, 284,
        273, 0, 1,  265,
        273, 1, 3,  260, 266, 272,
        273, 2, 5,  259, 261, 264, 267, 268,
        273, 3, 8,  251, 262, 263, 269, 270, 274, 275, 279,
        273, 4, 3,  265, 272, 273,
        273, 5, 13,  260, 264, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        273, 6, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        273, 7, 4,  250, 254, 280, 284,
        274, 0, 1,  267,
        274, 1, 2,  266, 269,
        274, 2, 4,  265, 268, 271, 276,
        274, 3, 7,  260, 270, 270, 272, 273, 275, 278,
        274, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        274, 5, 3,  265, 272, 273,
        274, 6, 2,  260, 264,
        274, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        275, 0, 1,  268,
        275, 1, 2,  266, 270,
        275, 2, 4,  265, 267, 271, 277,
        275, 3, 7,  260, 269, 269, 272, 273, 274, 278,
        275, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        275, 5, 3,  265, 272, 273,
        275, 6, 2,  260, 264,
        275, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        276, 0, 1,  269,
        276, 1, 2,  267, 271,
        276, 2, 4,  266, 270, 274, 278,
        276, 3, 4,  265, 268, 268, 277,
        276, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        276, 5, 3,  265, 272, 273,
        276, 6, 2,  260, 264,
        276, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        277, 0, 1,  270,
        277, 1, 2,  268, 271,
        277, 2, 4,  266, 269, 275, 278,
        277, 3, 4,  265, 267, 267, 276,
        277, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        277, 5, 3,  265, 272, 273,
        277, 6, 2,  260, 264,
        277, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        278, 0, 1,  271,
        278, 1, 2,  269, 270,
        278, 2, 4,  267, 268, 276, 277,
        278, 3, 4,  266, 266, 274, 275,
        278, 4, 11,  266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278,
        278, 5, 3,  265, 272, 273,
        278, 6, 2,  260, 264,
        278, 7, 8,  251, 252, 259, 261, 262, 263, 279, 283,
        279, 0, 3,  261, 280, 283,
        279, 1, 5,  260, 262, 281, 284, 285,
        279, 2, 8,  259, 264, 265, 282, 286, 287, 288, 289,
        279, 3, 7,  251, 263, 266, 272, 273, 290, 293,
        279, 4, 4,  261, 262, 279, 283,
        279, 5, 4,  260, 264, 280, 284,
        279, 6, 15,  251, 252, 259, 263, 265, 272, 273, 281, 282, 285, 286, 287, 288, 289, 293,
        279, 7, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 290, 294,
        280, 0, 4,  279, 281, 284, 285,
        280, 1, 7,  261, 282, 283, 286, 287, 288, 289,
        280, 2, 4,  260, 262, 290, 293,
        280, 3, 6,  259, 264, 265, 291, 294, 295,
        280, 4, 2,  280, 284,
        280, 5, 12,  261, 262, 279, 281, 282, 283, 285, 286, 287, 288, 289, 293,
        280, 6, 4,  260, 264, 290, 294,
        280, 7, 14,  251, 252, 259, 263, 265, 272, 273, 291, 292, 295, 299, 300, 303, 307,
        281, 0, 3,  280, 282, 289,
        281, 1, 5,  279, 284, 285, 290, 293,
        281, 2, 8,  261, 283, 286, 287, 288, 291, 294, 295,
        281, 3, 7,  260, 262, 292, 296, 299, 300, 303,
        281, 4, 4,  281, 282, 289, 293,
        281, 5, 4,  280, 284, 290, 294,
        281, 6, 15,  261, 262, 279, 283, 285, 286, 287, 288, 291, 292, 295, 299, 300, 303, 307,
        281, 7, 9,  260, 264, 296, 297, 298, 301, 302, 304, 308,
        282, 0, 1,  281,
        282, 1, 2,  280, 289,
        282, 2, 5,  279, 284, 285, 290, 293,
        282, 3, 8,  261, 283, 286, 287, 288, 291, 294, 295,
        282, 4, 4,  281, 282, 289, 293,
        282, 5, 4,  280, 284, 290, 294,
        282, 6, 15,  261, 262, 279, 283, 285, 286, 287, 288, 291, 292, 295, 299, 300, 303, 307,
        282, 7, 9,  260, 264, 296, 297, 298, 301, 302, 304, 308,
        283, 0, 1,  279,
        283, 1, 2,  261, 280,
        283, 2, 5,  260, 262, 281, 284, 285,
        283, 3, 8,  259, 264, 265, 282, 286, 287, 288, 289,
        283, 4, 4,  261, 262, 279, 283,
        283, 5, 4,  260, 264, 280, 284,
        283, 6, 15,  251, 252, 259, 263, 265, 272, 273, 281, 282, 285, 286, 287, 288, 289, 293,
        283, 7, 15,  250, 254, 266, 267, 268, 269, 270, 271, 274, 275, 276, 277, 278, 290, 294,
        284, 0, 1,  280,
        284, 1, 3,  279, 281, 285,
        284, 2, 7,  261, 282, 283, 286, 287, 288, 289,
        284, 3, 4,  260, 262, 290, 293,
        284, 4, 2,  280, 284,
        284, 5, 12,  261, 262, 279, 281, 282, 283, 285, 286, 287, 288, 289, 293,
        284, 6, 4,  260, 264, 290, 294,
        284, 7, 14,  251, 252, 259, 263, 265, 272, 273, 291, 292, 295, 299, 300, 303, 307,
        285, 0, 4,  280, 286, 287, 288,
        285, 1, 3,  279, 281, 284,
        285, 2, 4,  261, 282, 283, 289,
        285, 3, 4,  260, 262, 290, 293,
        285, 4, 4,  285, 286, 287, 288,
        285, 5, 2,  280, 284,
        285, 6, 8,  261, 262, 279, 281, 282, 283, 289, 293,
        285, 7, 4,  260, 264, 290, 294,
        286, 0, 1,  285,
        286, 1, 3,  280, 287, 288,
        286, 2, 3,  279, 281, 284,
        286, 3, 4,  261, 282, 283, 289,
        286, 4, 4,  285, 286, 287, 288,
        286, 5, 2,  280, 284,
        286, 6, 8,  261, 262, 279, 281, 282, 283, 289, 293,
        286, 7, 4,  260, 264, 290, 294,
        287, 0, 1,  285,
        287, 1, 3,  280, 286, 288,
        287, 2, 3,  279, 281, 284,
        287, 3, 4,  261, 282, 283, 289,
        287, 4, 4,  285, 286, 287, 288,
        287, 5, 2,  280, 284,
        287, 6, 8,  261, 262, 279, 281, 282, 283, 289, 293,
        287, 7, 4,  260, 264, 290, 294,
        288, 0, 1,  285,
        288, 1, 3,  280, 286, 287,
        288, 2, 3,  279, 281, 284,
        288, 3, 4,  261, 282, 283, 289,
        288, 4, 4,  285, 286, 287, 288,
        288, 5, 2,  280, 284,
        288, 6, 8,  261, 262, 279, 281, 282, 283, 289, 293,
        288, 7, 4,  260, 264, 290, 294,
        289, 0, 3,  281, 290, 293,
        289, 1, 5,  280, 282, 291, 294, 295,
        289, 2, 8,  279, 284, 285, 292, 296, 299, 300, 303,
        289, 3, 9,  261, 283, 286, 287, 288, 297, 298, 304, 307,
        289, 4, 4,  281, 282, 289, 293,
        289, 5, 4,  280, 284, 290, 294,
        289, 6, 15,  261, 262, 279, 283, 285, 286, 287, 288, 291, 292, 295, 299, 300, 303, 307,
        289, 7, 9,  260, 264, 296, 297, 298, 301, 302, 304, 308,
        290, 0, 4,  289, 291, 294, 295,
        290, 1, 7,  281, 292, 293, 296, 299, 300, 303,
        290, 2, 6,  280, 282, 297, 298, 304, 307,
        290, 3, 8,  279, 284, 285, 301, 302, 305, 308, 309,
        290, 4, 2,  290, 294,
        290, 5, 11,  281, 282, 289, 291, 292, 293, 295, 299, 300, 303, 307,
        290, 6, 9,  280, 284, 296, 297, 298, 301, 302, 304, 308,
        290, 7, 17,  261, 262, 279, 283, 285, 286, 287, 288, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        291, 0, 3,  290, 292, 303,
        291, 1, 5,  289, 294, 295, 304, 307,
        291, 2, 8,  281, 293, 296, 299, 300, 305, 308, 309,
        291, 3, 9,  280, 282, 297, 298, 306, 310, 313, 314, 322,
        291, 4, 4,  291, 292, 303, 307,
        291, 5, 4,  290, 294, 304, 308,
        291, 6, 16,  281, 282, 289, 293, 295, 299, 300, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        291, 7, 14,  280, 284, 296, 297, 298, 301, 302, 310, 315, 323, 326, 328, 332, 333,
        292, 0, 1,  291,
        292, 1, 2,  290, 303,
        292, 2, 5,  289, 294, 295, 304, 307,
        292, 3, 8,  281, 293, 296, 299, 300, 305, 308, 309,
        292, 4, 4,  291, 292, 303, 307,
        292, 5, 4,  290, 294, 304, 308,
        292, 6, 16,  281, 282, 289, 293, 295, 299, 300, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        292, 7, 14,  280, 284, 296, 297, 298, 301, 302, 310, 315, 323, 326, 328, 332, 333,
        293, 0, 1,  289,
        293, 1, 2,  281, 290,
        293, 2, 5,  280, 282, 291, 294, 295,
        293, 3, 8,  279, 284, 285, 292, 296, 299, 300, 303,
        293, 4, 4,  281, 282, 289, 293,
        293, 5, 4,  280, 284, 290, 294,
        293, 6, 15,  261, 262, 279, 283, 285, 286, 287, 288, 291, 292, 295, 299, 300, 303, 307,
        293, 7, 9,  260, 264, 296, 297, 298, 301, 302, 304, 308,
        294, 0, 1,  290,
        294, 1, 3,  289, 291, 295,
        294, 2, 7,  281, 292, 293, 296, 299, 300, 303,
        294, 3, 6,  280, 282, 297, 298, 304, 307,
        294, 4, 2,  290, 294,
        294, 5, 11,  281, 282, 289, 291, 292, 293, 295, 299, 300, 303, 307,
        294, 6, 9,  280, 284, 296, 297, 298, 301, 302, 304, 308,
        294, 7, 17,  261, 262, 279, 283, 285, 286, 287, 288, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        295, 0, 4,  290, 296, 299, 300,
        295, 1, 5,  289, 291, 294, 297, 298,
        295, 2, 6,  281, 292, 293, 301, 302, 303,
        295, 3, 4,  280, 282, 304, 307,
        295, 4, 3,  295, 299, 300,
        295, 5, 7,  290, 294, 296, 297, 298, 301, 302,
        295, 6, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        295, 7, 4,  280, 284, 304, 308,
        296, 0, 3,  295, 297, 298,
        296, 1, 5,  290, 299, 300, 301, 302,
        296, 2, 3,  289, 291, 294,
        296, 3, 4,  281, 292, 293, 303,
        296, 4, 5,  296, 297, 298, 301, 302,
        296, 5, 3,  295, 299, 300,
        296, 6, 2,  290, 294,
        296, 7, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        297, 0, 1,  296,
        297, 1, 2,  295, 298,
        297, 2, 5,  290, 299, 300, 301, 302,
        297, 3, 3,  289, 291, 294,
        297, 4, 5,  296, 297, 298, 301, 302,
        297, 5, 3,  295, 299, 300,
        297, 6, 2,  290, 294,
        297, 7, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        298, 0, 3,  296, 301, 302,
        298, 1, 2,  295, 297,
        298, 2, 3,  290, 299, 300,
        298, 3, 3,  289, 291, 294,
        298, 4, 5,  296, 297, 298, 301, 302,
        298, 5, 3,  295, 299, 300,
        298, 6, 2,  290, 294,
        298, 7, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        299, 0, 1,  295,
        299, 1, 3,  290, 296, 300,
        299, 2, 5,  289, 291, 294, 297, 298,
        299, 3, 6,  281, 292, 293, 301, 302, 303,
        299, 4, 3,  295, 299, 300,
        299, 5, 7,  290, 294, 296, 297, 298, 301, 302,
        299, 6, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        299, 7, 4,  280, 284, 304, 308,
        300, 0, 1,  295,
        300, 1, 3,  290, 296, 299,
        300, 2, 5,  289, 291, 294, 297, 298,
        300, 3, 6,  281, 292, 293, 301, 302, 303,
        300, 4, 3,  295, 299, 300,
        300, 5, 7,  290, 294, 296, 297, 298, 301, 302,
        300, 6, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        300, 7, 4,  280, 284, 304, 308,
        301, 0, 1,  298,
        301, 1, 2,  296, 302,
        301, 2, 2,  295, 297,
        301, 3, 3,  290, 299, 300,
        301, 4, 5,  296, 297, 298, 301, 302,
        301, 5, 3,  295, 299, 300,
        301, 6, 2,  290, 294,
        301, 7, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        302, 0, 1,  298,
        302, 1, 2,  296, 301,
        302, 2, 2,  295, 297,
        302, 3, 3,  290, 299, 300,
        302, 4, 5,  296, 297, 298, 301, 302,
        302, 5, 3,  295, 299, 300,
        302, 6, 2,  290, 294,
        302, 7, 8,  281, 282, 289, 291, 292, 293, 303, 307,
        303, 0, 3,  291, 304, 307,
        303, 1, 5,  290, 292, 305, 308, 309,
        303, 2, 8,  289, 294, 295, 306, 310, 313, 314, 322,
        303, 3, 10,  281, 293, 296, 299, 300, 311, 312, 315, 323, 329,
        303, 4, 4,  291, 292, 303, 307,
        303, 5, 4,  290, 294, 304, 308,
        303, 6, 16,  281, 282, 289, 293, 295, 299, 300, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        303, 7, 14,  280, 284, 296, 297, 298, 301, 302, 310, 315, 323, 326, 328, 332, 333,
        304, 0, 4,  303, 305, 308, 309,
        304, 1, 7,  291, 306, 307, 310, 313, 314, 322,
        304, 2, 7,  290, 292, 311, 312, 315, 323, 329,
        304, 3, 15,  289, 294, 295, 316, 317, 318, 319, 320, 321, 324, 326, 327, 328, 334, 335,
        304, 4, 2,  304, 308,
        304, 5, 13,  291, 292, 303, 305, 306, 307, 309, 313, 314, 322, 329, 334, 335,
        304, 6, 9,  290, 294, 310, 315, 323, 326, 328, 332, 333,
        304, 7, 22,  281, 282, 289, 293, 295, 299, 300, 311, 312, 316, 317, 318, 319, 320, 321, 324, 325, 327, 330, 331, 336, 340,
        305, 0, 3,  304, 306, 322,
        305, 1, 5,  303, 308, 309, 323, 329,
        305, 2, 11,  291, 307, 310, 313, 314, 324, 326, 327, 328, 334, 335,
        305, 3, 11,  290, 292, 311, 312, 315, 325, 330, 331, 332, 333, 336,
        305, 4, 6,  305, 306, 322, 329, 334, 335,
        305, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        305, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        305, 7, 6,  290, 294, 310, 315, 337, 341,
        306, 0, 1,  305,
        306, 1, 2,  304, 322,
        306, 2, 5,  303, 308, 309, 323, 329,
        306, 3, 11,  291, 307, 310, 313, 314, 324, 326, 327, 328, 334, 335,
        306, 4, 6,  305, 306, 322, 329, 334, 335,
        306, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        306, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        306, 7, 6,  290, 294, 310, 315, 337, 341,
        307, 0, 1,  303,
        307, 1, 2,  291, 304,
        307, 2, 5,  290, 292, 305, 308, 309,
        307, 3, 8,  289, 294, 295, 306, 310, 313, 314, 322,
        307, 4, 4,  291, 292, 303, 307,
        307, 5, 4,  290, 294, 304, 308,
        307, 6, 16,  281, 282, 289, 293, 295, 299, 300, 305, 306, 309, 313, 314, 322, 329, 334, 335,
        307, 7, 14,  280, 284, 296, 297, 298, 301, 302, 310, 315, 323, 326, 328, 332, 333,
        308, 0, 1,  304,
        308, 1, 3,  303, 305, 309,
        308, 2, 7,  291, 306, 307, 310, 313, 314, 322,
        308, 3, 7,  290, 292, 311, 312, 315, 323, 329,
        308, 4, 2,  304, 308,
        308, 5, 13,  291, 292, 303, 305, 306, 307, 309, 313, 314, 322, 329, 334, 335,
        308, 6, 9,  290, 294, 310, 315, 323, 326, 328, 332, 333,
        308, 7, 22,  281, 282, 289, 293, 295, 299, 300, 311, 312, 316, 317, 318, 319, 320, 321, 324, 325, 327, 330, 331, 336, 340,
        309, 0, 4,  304, 310, 313, 314,
        309, 1, 6,  303, 305, 308, 311, 312, 315,
        309, 2, 10,  291, 306, 307, 316, 317, 318, 319, 320, 321, 322,
        309, 3, 4,  290, 292, 323, 329,
        309, 4, 3,  309, 313, 314,
        309, 5, 4,  304, 308, 310, 315,
        309, 6, 18,  291, 292, 303, 305, 306, 307, 311, 312, 316, 317, 318, 319, 320, 321, 322, 329, 334, 335,
        309, 7, 7,  290, 294, 323, 326, 328, 332, 333,
        310, 0, 4,  309, 311, 312, 315,
        310, 1, 9,  304, 313, 314, 316, 317, 318, 319, 320, 321,
        310, 2, 3,  303, 305, 308,
        310, 3, 4,  291, 306, 307, 322,
        310, 4, 2,  310, 315,
        310, 5, 11,  309, 311, 312, 313, 314, 316, 317, 318, 319, 320, 321,
        310, 6, 2,  304, 308,
        310, 7, 10,  291, 292, 303, 305, 306, 307, 322, 329, 334, 335,
        311, 0, 4,  310, 316, 317, 318,
        311, 1, 3,  309, 312, 315,
        311, 2, 6,  304, 313, 314, 319, 320, 321,
        311, 3, 3,  303, 305, 308,
        311, 4, 4,  311, 316, 317, 318,
        311, 5, 2,  310, 315,
        311, 6, 7,  309, 312, 313, 314, 319, 320, 321,
        311, 7, 2,  304, 308,
        312, 0, 4,  310, 319, 320, 321,
        312, 1, 3,  309, 311, 315,
        312, 2, 6,  304, 313, 314, 316, 317, 318,
        312, 3, 3,  303, 305, 308,
        312, 4, 4,  312, 319, 320, 321,
        312, 5, 2,  310, 315,
        312, 6, 7,  309, 311, 313, 314, 316, 317, 318,
        312, 7, 2,  304, 308,
        313, 0, 1,  309,
        313, 1, 3,  304, 310, 314,
        313, 2, 6,  303, 305, 308, 311, 312, 315,
        313, 3, 10,  291, 306, 307, 316, 317, 318, 319, 320, 321, 322,
        313, 4, 3,  309, 313, 314,
        313, 5, 4,  304, 308, 310, 315,
        313, 6, 18,  291, 292, 303, 305, 306, 307, 311, 312, 316, 317, 318, 319, 320, 321, 322, 329, 334, 335,
        313, 7, 7,  290, 294, 323, 326, 328, 332, 333,
        314, 0, 1,  309,
        314, 1, 3,  304, 310, 313,
        314, 2, 6,  303, 305, 308, 311, 312, 315,
        314, 3, 10,  291, 306, 307, 316, 317, 318, 319, 320, 321, 322,
        314, 4, 3,  309, 313, 314,
        314, 5, 4,  304, 308, 310, 315,
        314, 6, 18,  291, 292, 303, 305, 306, 307, 311, 312, 316, 317, 318, 319, 320, 321, 322, 329, 334, 335,
        314, 7, 7,  290, 294, 323, 326, 328, 332, 333,
        315, 0, 1,  310,
        315, 1, 3,  309, 311, 312,
        315, 2, 9,  304, 313, 314, 316, 317, 318, 319, 320, 321,
        315, 3, 3,  303, 305, 308,
        315, 4, 2,  310, 315,
        315, 5, 11,  309, 311, 312, 313, 314, 316, 317, 318, 319, 320, 321,
        315, 6, 2,  304, 308,
        315, 7, 10,  291, 292, 303, 305, 306, 307, 322, 329, 334, 335,
        316, 0, 1,  311,
        316, 1, 3,  310, 317, 318,
        316, 2, 3,  309, 312, 315,
        316, 3, 6,  304, 313, 314, 319, 320, 321,
        316, 4, 4,  311, 316, 317, 318,
        316, 5, 2,  310, 315,
        316, 6, 7,  309, 312, 313, 314, 319, 320, 321,
        316, 7, 2,  304, 308,
        317, 0, 1,  311,
        317, 1, 3,  310, 316, 318,
        317, 2, 3,  309, 312, 315,
        317, 3, 6,  304, 313, 314, 319, 320, 321,
        317, 4, 4,  311, 316, 317, 318,
        317, 5, 2,  310, 315,
        317, 6, 7,  309, 312, 313, 314, 319, 320, 321,
        317, 7, 2,  304, 308,
        318, 0, 1,  311,
        318, 1, 3,  310, 316, 317,
        318, 2, 3,  309, 312, 315,
        318, 3, 6,  304, 313, 314, 319, 320, 321,
        318, 4, 4,  311, 316, 317, 318,
        318, 5, 2,  310, 315,
        318, 6, 7,  309, 312, 313, 314, 319, 320, 321,
        318, 7, 2,  304, 308,
        319, 0, 1,  312,
        319, 1, 3,  310, 320, 321,
        319, 2, 3,  309, 311, 315,
        319, 3, 6,  304, 313, 314, 316, 317, 318,
        319, 4, 4,  312, 319, 320, 321,
        319, 5, 2,  310, 315,
        319, 6, 7,  309, 311, 313, 314, 316, 317, 318,
        319, 7, 2,  304, 308,
        320, 0, 1,  312,
        320, 1, 3,  310, 319, 321,
        320, 2, 3,  309, 311, 315,
        320, 3, 6,  304, 313, 314, 316, 317, 318,
        320, 4, 4,  312, 319, 320, 321,
        320, 5, 2,  310, 315,
        320, 6, 7,  309, 311, 313, 314, 316, 317, 318,
        320, 7, 2,  304, 308,
        321, 0, 1,  312,
        321, 1, 3,  310, 319, 320,
        321, 2, 3,  309, 311, 315,
        321, 3, 6,  304, 313, 314, 316, 317, 318,
        321, 4, 4,  312, 319, 320, 321,
        321, 5, 2,  310, 315,
        321, 6, 7,  309, 311, 313, 314, 316, 317, 318,
        321, 7, 2,  304, 308,
        322, 0, 3,  305, 323, 329,
        322, 1, 8,  304, 306, 324, 326, 327, 328, 334, 335,
        322, 2, 9,  303, 308, 309, 325, 330, 331, 332, 333, 336,
        322, 3, 7,  291, 307, 310, 313, 314, 337, 340,
        322, 4, 6,  305, 306, 322, 329, 334, 335,
        322, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        322, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        322, 7, 6,  290, 294, 310, 315, 337, 341,
        323, 0, 4,  322, 324, 326, 327,
        323, 1, 7,  305, 325, 328, 329, 330, 331, 336,
        323, 2, 8,  304, 306, 332, 333, 334, 335, 337, 340,
        323, 3, 6,  303, 308, 309, 338, 341, 342,
        323, 4, 2,  323, 326,
        323, 5, 13,  305, 306, 322, 324, 325, 327, 329, 330, 331, 334, 335, 336, 340,
        323, 6, 7,  304, 308, 328, 332, 333, 337, 341,
        323, 7, 14,  291, 292, 303, 307, 309, 313, 314, 338, 339, 342, 346, 347, 355, 359,
        324, 0, 3,  323, 325, 336,
        324, 1, 5,  322, 326, 327, 337, 340,
        324, 2, 8,  305, 328, 329, 330, 331, 338, 341, 342,
        324, 3, 11,  304, 306, 332, 333, 334, 335, 339, 343, 346, 347, 355,
        324, 4, 4,  324, 325, 336, 340,
        324, 5, 4,  323, 326, 337, 341,
        324, 6, 16,  305, 306, 322, 327, 329, 330, 331, 334, 335, 338, 339, 342, 346, 347, 355, 359,
        324, 7, 9,  304, 308, 328, 332, 333, 343, 348, 356, 360,
        325, 0, 1,  324,
        325, 1, 2,  323, 336,
        325, 2, 5,  322, 326, 327, 337, 340,
        325, 3, 8,  305, 328, 329, 330, 331, 338, 341, 342,
        325, 4, 4,  324, 325, 336, 340,
        325, 5, 4,  323, 326, 337, 341,
        325, 6, 16,  305, 306, 322, 327, 329, 330, 331, 334, 335, 338, 339, 342, 346, 347, 355, 359,
        325, 7, 9,  304, 308, 328, 332, 333, 343, 348, 356, 360,
        326, 0, 1,  323,
        326, 1, 3,  322, 324, 327,
        326, 2, 7,  305, 325, 328, 329, 330, 331, 336,
        326, 3, 8,  304, 306, 332, 333, 334, 335, 337, 340,
        326, 4, 2,  323, 326,
        326, 5, 13,  305, 306, 322, 324, 325, 327, 329, 330, 331, 334, 335, 336, 340,
        326, 6, 7,  304, 308, 328, 332, 333, 337, 341,
        326, 7, 14,  291, 292, 303, 307, 309, 313, 314, 338, 339, 342, 346, 347, 355, 359,
        327, 0, 4,  323, 328, 330, 331,
        327, 1, 6,  322, 324, 326, 329, 332, 333,
        327, 2, 5,  305, 325, 334, 335, 336,
        327, 3, 4,  304, 306, 337, 340,
        327, 4, 3,  327, 330, 331,
        327, 5, 5,  323, 326, 328, 332, 333,
        327, 6, 10,  305, 306, 322, 324, 325, 329, 334, 335, 336, 340,
        327, 7, 4,  304, 308, 337, 341,
        328, 0, 4,  327, 329, 332, 333,
        328, 1, 6,  322, 323, 330, 331, 334, 335,
        328, 2, 3,  305, 324, 326,
        328, 3, 4,  304, 306, 325, 336,
        328, 4, 3,  328, 332, 333,
        328, 5, 9,  305, 306, 322, 327, 329, 330, 331, 334, 335,
        328, 6, 4,  304, 308, 323, 326,
        328, 7, 11,  291, 292, 303, 307, 309, 313, 314, 324, 325, 336, 340,
        329, 0, 4,  322, 328, 334, 335,
        329, 1, 5,  305, 323, 327, 332, 333,
        329, 2, 6,  304, 306, 324, 326, 330, 331,
        329, 3, 5,  303, 308, 309, 325, 336,
        329, 4, 6,  305, 306, 322, 329, 334, 335,
        329, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        329, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        329, 7, 6,  290, 294, 310, 315, 337, 341,
        330, 0, 1,  327,
        330, 1, 3,  323, 328, 331,
        330, 2, 6,  322, 324, 326, 329, 332, 333,
        330, 3, 5,  305, 325, 334, 335, 336,
        330, 4, 3,  327, 330, 331,
        330, 5, 5,  323, 326, 328, 332, 333,
        330, 6, 10,  305, 306, 322, 324, 325, 329, 334, 335, 336, 340,
        330, 7, 4,  304, 308, 337, 341,
        331, 0, 1,  327,
        331, 1, 3,  323, 328, 330,
        331, 2, 6,  322, 324, 326, 329, 332, 333,
        331, 3, 5,  305, 325, 334, 335, 336,
        331, 4, 3,  327, 330, 331,
        331, 5, 5,  323, 326, 328, 332, 333,
        331, 6, 10,  305, 306, 322, 324, 325, 329, 334, 335, 336, 340,
        331, 7, 4,  304, 308, 337, 341,
        332, 0, 1,  328,
        332, 1, 3,  327, 329, 333,
        332, 2, 6,  322, 323, 330, 331, 334, 335,
        332, 3, 3,  305, 324, 326,
        332, 4, 3,  328, 332, 333,
        332, 5, 9,  305, 306, 322, 327, 329, 330, 331, 334, 335,
        332, 6, 4,  304, 308, 323, 326,
        332, 7, 11,  291, 292, 303, 307, 309, 313, 314, 324, 325, 336, 340,
        333, 0, 1,  328,
        333, 1, 3,  327, 329, 332,
        333, 2, 6,  322, 323, 330, 331, 334, 335,
        333, 3, 3,  305, 324, 326,
        333, 4, 3,  328, 332, 333,
        333, 5, 9,  305, 306, 322, 327, 329, 330, 331, 334, 335,
        333, 6, 4,  304, 308, 323, 326,
        333, 7, 11,  291, 292, 303, 307, 309, 313, 314, 324, 325, 336, 340,
        334, 0, 1,  329,
        334, 1, 3,  322, 328, 335,
        334, 2, 5,  305, 323, 327, 332, 333,
        334, 3, 6,  304, 306, 324, 326, 330, 331,
        334, 4, 6,  305, 306, 322, 329, 334, 335,
        334, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        334, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        334, 7, 6,  290, 294, 310, 315, 337, 341,
        335, 0, 1,  329,
        335, 1, 3,  322, 328, 334,
        335, 2, 5,  305, 323, 327, 332, 333,
        335, 3, 6,  304, 306, 324, 326, 330, 331,
        335, 4, 6,  305, 306, 322, 329, 334, 335,
        335, 5, 7,  304, 308, 323, 326, 328, 332, 333,
        335, 6, 14,  291, 292, 303, 307, 309, 313, 314, 324, 325, 327, 330, 331, 336, 340,
        335, 7, 6,  290, 294, 310, 315, 337, 341,
        336, 0, 3,  324, 337, 340,
        336, 1, 5,  323, 325, 338, 341, 342,
        336, 2, 8,  322, 326, 327, 339, 343, 346, 347, 355,
        336, 3, 10,  305, 328, 329, 330, 331, 344, 345, 348, 356, 359,
        336, 4, 4,  324, 325, 336, 340,
        336, 5, 4,  323, 326, 337, 341,
        336, 6, 16,  305, 306, 322, 327, 329, 330, 331, 334, 335, 338, 339, 342, 346, 347, 355, 359,
        336, 7, 9,  304, 308, 328, 332, 333, 343, 348, 356, 360,
        337, 0, 4,  336, 338, 341, 342,
        337, 1, 7,  324, 339, 340, 343, 346, 347, 355,
        337, 2, 7,  323, 325, 344, 345, 348, 356, 359,
        337, 3, 12,  322, 326, 327, 349, 350, 351, 352, 353, 354, 357, 360, 361,
        337, 4, 2,  337, 341,
        337, 5, 11,  324, 325, 336, 338, 339, 340, 342, 346, 347, 355, 359,
        337, 6, 6,  323, 326, 343, 348, 356, 360,
        337, 7, 24,  305, 306, 322, 327, 329, 330, 331, 334, 335, 344, 345, 349, 350, 351, 352, 353, 354, 357, 358, 361, 371, 372, 379, 383,
        338, 0, 3,  337, 339, 355,
        338, 1, 5,  336, 341, 342, 356, 359,
        338, 2, 8,  324, 340, 343, 346, 347, 357, 360, 361,
        338, 3, 10,  323, 325, 344, 345, 348, 358, 362, 371, 372, 379,
        338, 4, 4,  338, 339, 355, 359,
        338, 5, 4,  337, 341, 356, 360,
        338, 6, 14,  324, 325, 336, 340, 342, 346, 347, 357, 358, 361, 371, 372, 379, 383,
        338, 7, 21,  323, 326, 343, 348, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        339, 0, 1,  338,
        339, 1, 2,  337, 355,
        339, 2, 5,  336, 341, 342, 356, 359,
        339, 3, 8,  324, 340, 343, 346, 347, 357, 360, 361,
        339, 4, 4,  338, 339, 355, 359,
        339, 5, 4,  337, 341, 356, 360,
        339, 6, 14,  324, 325, 336, 340, 342, 346, 347, 357, 358, 361, 371, 372, 379, 383,
        339, 7, 21,  323, 326, 343, 348, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        340, 0, 1,  336,
        340, 1, 2,  324, 337,
        340, 2, 5,  323, 325, 338, 341, 342,
        340, 3, 8,  322, 326, 327, 339, 343, 346, 347, 355,
        340, 4, 4,  324, 325, 336, 340,
        340, 5, 4,  323, 326, 337, 341,
        340, 6, 16,  305, 306, 322, 327, 329, 330, 331, 334, 335, 338, 339, 342, 346, 347, 355, 359,
        340, 7, 9,  304, 308, 328, 332, 333, 343, 348, 356, 360,
        341, 0, 1,  337,
        341, 1, 3,  336, 338, 342,
        341, 2, 7,  324, 339, 340, 343, 346, 347, 355,
        341, 3, 7,  323, 325, 344, 345, 348, 356, 359,
        341, 4, 2,  337, 341,
        341, 5, 11,  324, 325, 336, 338, 339, 340, 342, 346, 347, 355, 359,
        341, 6, 6,  323, 326, 343, 348, 356, 360,
        341, 7, 24,  305, 306, 322, 327, 329, 330, 331, 334, 335, 344, 345, 349, 350, 351, 352, 353, 354, 357, 358, 361, 371, 372, 379, 383,
        342, 0, 4,  337, 343, 346, 347,
        342, 1, 6,  336, 338, 341, 344, 345, 348,
        342, 2, 10,  324, 339, 340, 349, 350, 351, 352, 353, 354, 355,
        342, 3, 4,  323, 325, 356, 359,
        342, 4, 3,  342, 346, 347,
        342, 5, 4,  337, 341, 343, 348,
        342, 6, 16,  324, 325, 336, 338, 339, 340, 344, 345, 349, 350, 351, 352, 353, 354, 355, 359,
        342, 7, 4,  323, 326, 356, 360,
        343, 0, 4,  342, 344, 345, 348,
        343, 1, 9,  337, 346, 347, 349, 350, 351, 352, 353, 354,
        343, 2, 3,  336, 338, 341,
        343, 3, 4,  324, 339, 340, 355,
        343, 4, 2,  343, 348,
        343, 5, 11,  342, 344, 345, 346, 347, 349, 350, 351, 352, 353, 354,
        343, 6, 2,  337, 341,
        343, 7, 8,  324, 325, 336, 338, 339, 340, 355, 359,
        344, 0, 4,  343, 349, 350, 351,
        344, 1, 3,  342, 345, 348,
        344, 2, 6,  337, 346, 347, 352, 353, 354,
        344, 3, 3,  336, 338, 341,
        344, 4, 4,  344, 349, 350, 351,
        344, 5, 2,  343, 348,
        344, 6, 7,  342, 345, 346, 347, 352, 353, 354,
        344, 7, 2,  337, 341,
        345, 0, 4,  343, 352, 353, 354,
        345, 1, 3,  342, 344, 348,
        345, 2, 6,  337, 346, 347, 349, 350, 351,
        345, 3, 3,  336, 338, 341,
        345, 4, 4,  345, 352, 353, 354,
        345, 5, 2,  343, 348,
        345, 6, 7,  342, 344, 346, 347, 349, 350, 351,
        345, 7, 2,  337, 341,
        346, 0, 1,  342,
        346, 1, 3,  337, 343, 347,
        346, 2, 6,  336, 338, 341, 344, 345, 348,
        346, 3, 10,  324, 339, 340, 349, 350, 351, 352, 353, 354, 355,
        346, 4, 3,  342, 346, 347,
        346, 5, 4,  337, 341, 343, 348,
        346, 6, 16,  324, 325, 336, 338, 339, 340, 344, 345, 349, 350, 351, 352, 353, 354, 355, 359,
        346, 7, 4,  323, 326, 356, 360,
        347, 0, 1,  342,
        347, 1, 3,  337, 343, 346,
        347, 2, 6,  336, 338, 341, 344, 345, 348,
        347, 3, 10,  324, 339, 340, 349, 350, 351, 352, 353, 354, 355,
        347, 4, 3,  342, 346, 347,
        347, 5, 4,  337, 341, 343, 348,
        347, 6, 16,  324, 325, 336, 338, 339, 340, 344, 345, 349, 350, 351, 352, 353, 354, 355, 359,
        347, 7, 4,  323, 326, 356, 360,
        348, 0, 1,  343,
        348, 1, 3,  342, 344, 345,
        348, 2, 9,  337, 346, 347, 349, 350, 351, 352, 353, 354,
        348, 3, 3,  336, 338, 341,
        348, 4, 2,  343, 348,
        348, 5, 11,  342, 344, 345, 346, 347, 349, 350, 351, 352, 353, 354,
        348, 6, 2,  337, 341,
        348, 7, 8,  324, 325, 336, 338, 339, 340, 355, 359,
        349, 0, 1,  344,
        349, 1, 3,  343, 350, 351,
        349, 2, 3,  342, 345, 348,
        349, 3, 6,  337, 346, 347, 352, 353, 354,
        349, 4, 4,  344, 349, 350, 351,
        349, 5, 2,  343, 348,
        349, 6, 7,  342, 345, 346, 347, 352, 353, 354,
        349, 7, 2,  337, 341,
        350, 0, 1,  344,
        350, 1, 3,  343, 349, 351,
        350, 2, 3,  342, 345, 348,
        350, 3, 6,  337, 346, 347, 352, 353, 354,
        350, 4, 4,  344, 349, 350, 351,
        350, 5, 2,  343, 348,
        350, 6, 7,  342, 345, 346, 347, 352, 353, 354,
        350, 7, 2,  337, 341,
        351, 0, 1,  344,
        351, 1, 3,  343, 349, 350,
        351, 2, 3,  342, 345, 348,
        351, 3, 6,  337, 346, 347, 352, 353, 354,
        351, 4, 4,  344, 349, 350, 351,
        351, 5, 2,  343, 348,
        351, 6, 7,  342, 345, 346, 347, 352, 353, 354,
        351, 7, 2,  337, 341,
        352, 0, 1,  345,
        352, 1, 3,  343, 353, 354,
        352, 2, 3,  342, 344, 348,
        352, 3, 6,  337, 346, 347, 349, 350, 351,
        352, 4, 4,  345, 352, 353, 354,
        352, 5, 2,  343, 348,
        352, 6, 7,  342, 344, 346, 347, 349, 350, 351,
        352, 7, 2,  337, 341,
        353, 0, 1,  345,
        353, 1, 3,  343, 352, 354,
        353, 2, 3,  342, 344, 348,
        353, 3, 6,  337, 346, 347, 349, 350, 351,
        353, 4, 4,  345, 352, 353, 354,
        353, 5, 2,  343, 348,
        353, 6, 7,  342, 344, 346, 347, 349, 350, 351,
        353, 7, 2,  337, 341,
        354, 0, 1,  345,
        354, 1, 3,  343, 352, 353,
        354, 2, 3,  342, 344, 348,
        354, 3, 6,  337, 346, 347, 349, 350, 351,
        354, 4, 4,  345, 352, 353, 354,
        354, 5, 2,  343, 348,
        354, 6, 7,  342, 344, 346, 347, 349, 350, 351,
        354, 7, 2,  337, 341,
        355, 0, 3,  338, 356, 359,
        355, 1, 5,  337, 339, 357, 360, 361,
        355, 2, 8,  336, 341, 342, 358, 362, 371, 372, 379,
        355, 3, 9,  324, 340, 343, 346, 347, 363, 364, 380, 383,
        355, 4, 4,  338, 339, 355, 359,
        355, 5, 4,  337, 341, 356, 360,
        355, 6, 14,  324, 325, 336, 340, 342, 346, 347, 357, 358, 361, 371, 372, 379, 383,
        355, 7, 21,  323, 326, 343, 348, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        356, 0, 4,  355, 357, 360, 361,
        356, 1, 7,  338, 358, 359, 362, 371, 372, 379,
        356, 2, 6,  337, 339, 363, 364, 380, 383,
        356, 3, 10,  336, 341, 342, 365, 366, 367, 373, 381, 384, 385,
        356, 4, 2,  356, 360,
        356, 5, 11,  338, 339, 355, 357, 358, 359, 361, 371, 372, 379, 383,
        356, 6, 19,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        356, 7, 14,  324, 325, 336, 340, 342, 346, 347, 381, 382, 385, 390, 391, 401, 405,
        357, 0, 3,  356, 358, 379,
        357, 1, 5,  355, 360, 361, 380, 383,
        357, 2, 8,  338, 359, 362, 371, 372, 381, 384, 385,
        357, 3, 9,  337, 339, 363, 364, 382, 386, 390, 391, 401,
        357, 4, 4,  357, 358, 379, 383,
        357, 5, 4,  356, 360, 380, 384,
        357, 6, 14,  338, 339, 355, 359, 361, 371, 372, 381, 382, 385, 390, 391, 401, 405,
        357, 7, 22,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 386, 392, 393, 402, 406,
        358, 0, 1,  357,
        358, 1, 2,  356, 379,
        358, 2, 5,  355, 360, 361, 380, 383,
        358, 3, 8,  338, 359, 362, 371, 372, 381, 384, 385,
        358, 4, 4,  357, 358, 379, 383,
        358, 5, 4,  356, 360, 380, 384,
        358, 6, 14,  338, 339, 355, 359, 361, 371, 372, 381, 382, 385, 390, 391, 401, 405,
        358, 7, 22,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 386, 392, 393, 402, 406,
        359, 0, 1,  355,
        359, 1, 2,  338, 356,
        359, 2, 5,  337, 339, 357, 360, 361,
        359, 3, 8,  336, 341, 342, 358, 362, 371, 372, 379,
        359, 4, 4,  338, 339, 355, 359,
        359, 5, 4,  337, 341, 356, 360,
        359, 6, 14,  324, 325, 336, 340, 342, 346, 347, 357, 358, 361, 371, 372, 379, 383,
        359, 7, 21,  323, 326, 343, 348, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        360, 0, 1,  356,
        360, 1, 3,  355, 357, 361,
        360, 2, 7,  338, 358, 359, 362, 371, 372, 379,
        360, 3, 6,  337, 339, 363, 364, 380, 383,
        360, 4, 2,  356, 360,
        360, 5, 11,  338, 339, 355, 357, 358, 359, 361, 371, 372, 379, 383,
        360, 6, 19,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 380, 384,
        360, 7, 14,  324, 325, 336, 340, 342, 346, 347, 381, 382, 385, 390, 391, 401, 405,
        361, 0, 4,  356, 362, 371, 372,
        361, 1, 5,  355, 357, 360, 363, 364,
        361, 2, 8,  338, 358, 359, 365, 366, 367, 373, 379,
        361, 3, 8,  337, 339, 368, 369, 374, 375, 380, 383,
        361, 4, 3,  361, 371, 372,
        361, 5, 17,  356, 360, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        361, 6, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        361, 7, 4,  337, 341, 380, 384,
        362, 0, 3,  361, 363, 364,
        362, 1, 7,  356, 365, 366, 367, 371, 372, 373,
        362, 2, 7,  355, 357, 360, 368, 369, 374, 375,
        362, 3, 8,  338, 358, 359, 370, 370, 376, 377, 379,
        362, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        362, 5, 3,  361, 371, 372,
        362, 6, 2,  356, 360,
        362, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        363, 0, 3,  362, 365, 373,
        363, 1, 4,  361, 364, 366, 374,
        363, 2, 5,  356, 367, 368, 371, 372,
        363, 3, 7,  355, 357, 360, 369, 370, 375, 376,
        363, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        363, 5, 3,  361, 371, 372,
        363, 6, 2,  356, 360,
        363, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        364, 0, 3,  362, 366, 367,
        364, 1, 6,  361, 363, 365, 368, 369, 375,
        364, 2, 9,  356, 370, 370, 371, 372, 373, 374, 376, 377,
        364, 3, 5,  355, 357, 360, 378, 378,
        364, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        364, 5, 3,  361, 371, 372,
        364, 6, 2,  356, 360,
        364, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        365, 0, 3,  363, 366, 374,
        365, 1, 4,  362, 364, 368, 373,
        365, 2, 4,  361, 367, 370, 376,
        365, 3, 7,  356, 369, 369, 371, 372, 375, 378,
        365, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        365, 5, 3,  361, 371, 372,
        365, 6, 2,  356, 360,
        365, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        366, 0, 3,  364, 365, 368,
        366, 1, 6,  362, 363, 367, 370, 374, 376,
        366, 2, 6,  361, 369, 369, 373, 375, 378,
        366, 3, 5,  356, 371, 372, 377, 377,
        366, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        366, 5, 3,  361, 371, 372,
        366, 6, 2,  356, 360,
        366, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        367, 0, 3,  364, 369, 375,
        367, 1, 4,  362, 366, 370, 377,
        367, 2, 6,  361, 363, 365, 368, 368, 378,
        367, 3, 7,  356, 371, 372, 373, 374, 376, 376,
        367, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        367, 5, 3,  361, 371, 372,
        367, 6, 2,  356, 360,
        367, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        368, 0, 3,  366, 370, 376,
        368, 1, 4,  364, 365, 369, 378,
        368, 2, 6,  362, 363, 367, 367, 374, 377,
        368, 3, 4,  361, 373, 375, 375,
        368, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        368, 5, 3,  361, 371, 372,
        368, 6, 2,  356, 360,
        368, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        369, 0, 3,  367, 370, 377,
        369, 1, 4,  364, 368, 375, 378,
        369, 2, 4,  362, 366, 366, 376,
        369, 3, 4,  361, 363, 365, 365,
        369, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        369, 5, 3,  361, 371, 372,
        369, 6, 2,  356, 360,
        369, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        370, 0, 3,  368, 369, 378,
        370, 1, 4,  366, 367, 376, 377,
        370, 2, 4,  364, 364, 365, 375,
        370, 3, 4,  362, 362, 363, 374,
        370, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        370, 5, 3,  361, 371, 372,
        370, 6, 2,  356, 360,
        370, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        371, 0, 1,  361,
        371, 1, 3,  356, 362, 372,
        371, 2, 5,  355, 357, 360, 363, 364,
        371, 3, 8,  338, 358, 359, 365, 366, 367, 373, 379,
        371, 4, 3,  361, 371, 372,
        371, 5, 17,  356, 360, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        371, 6, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        371, 7, 4,  337, 341, 380, 384,
        372, 0, 1,  361,
        372, 1, 3,  356, 362, 371,
        372, 2, 5,  355, 357, 360, 363, 364,
        372, 3, 8,  338, 358, 359, 365, 366, 367, 373, 379,
        372, 4, 3,  361, 371, 372,
        372, 5, 17,  356, 360, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        372, 6, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        372, 7, 4,  337, 341, 380, 384,
        373, 0, 1,  363,
        373, 1, 2,  362, 365,
        373, 2, 4,  361, 364, 366, 374,
        373, 3, 5,  356, 367, 368, 371, 372,
        373, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        373, 5, 3,  361, 371, 372,
        373, 6, 2,  356, 360,
        373, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        374, 0, 1,  365,
        374, 1, 2,  363, 366,
        374, 2, 4,  362, 364, 368, 373,
        374, 3, 4,  361, 367, 370, 376,
        374, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        374, 5, 3,  361, 371, 372,
        374, 6, 2,  356, 360,
        374, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        375, 0, 1,  367,
        375, 1, 2,  364, 369,
        375, 2, 4,  362, 366, 370, 377,
        375, 3, 6,  361, 363, 365, 368, 368, 378,
        375, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        375, 5, 3,  361, 371, 372,
        375, 6, 2,  356, 360,
        375, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        376, 0, 1,  368,
        376, 1, 2,  366, 370,
        376, 2, 4,  364, 365, 369, 378,
        376, 3, 6,  362, 363, 367, 367, 374, 377,
        376, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        376, 5, 3,  361, 371, 372,
        376, 6, 2,  356, 360,
        376, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        377, 0, 1,  369,
        377, 1, 2,  367, 370,
        377, 2, 4,  364, 368, 375, 378,
        377, 3, 4,  362, 366, 366, 376,
        377, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        377, 5, 3,  361, 371, 372,
        377, 6, 2,  356, 360,
        377, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        378, 0, 1,  370,
        378, 1, 2,  368, 369,
        378, 2, 4,  366, 367, 376, 377,
        378, 3, 4,  364, 364, 365, 375,
        378, 4, 15,  362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378,
        378, 5, 3,  361, 371, 372,
        378, 6, 2,  356, 360,
        378, 7, 8,  338, 339, 355, 357, 358, 359, 379, 383,
        379, 0, 3,  357, 380, 383,
        379, 1, 5,  356, 358, 381, 384, 385,
        379, 2, 8,  355, 360, 361, 382, 386, 390, 391, 401,
        379, 3, 10,  338, 359, 362, 371, 372, 387, 392, 393, 402, 405,
        379, 4, 4,  357, 358, 379, 383,
        379, 5, 4,  356, 360, 380, 384,
        379, 6, 14,  338, 339, 355, 359, 361, 371, 372, 381, 382, 385, 390, 391, 401, 405,
        379, 7, 22,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 386, 392, 393, 402, 406,
        380, 0, 4,  379, 381, 384, 385,
        380, 1, 7,  357, 382, 383, 386, 390, 391, 401,
        380, 2, 7,  356, 358, 387, 392, 393, 402, 405,
        380, 3, 9,  355, 360, 361, 388, 394, 395, 403, 406, 407,
        380, 4, 2,  380, 384,
        380, 5, 11,  357, 358, 379, 381, 382, 383, 385, 390, 391, 401, 405,
        380, 6, 7,  356, 360, 386, 392, 393, 402, 406,
        380, 7, 17,  338, 339, 355, 359, 361, 371, 372, 387, 394, 395, 403, 404, 407, 412, 413, 418, 422,
        381, 0, 3,  380, 382, 401,
        381, 1, 5,  379, 384, 385, 402, 405,
        381, 2, 8,  357, 383, 386, 390, 391, 403, 406, 407,
        381, 3, 10,  356, 358, 387, 392, 393, 404, 408, 412, 413, 418,
        381, 4, 4,  381, 382, 401, 405,
        381, 5, 4,  380, 384, 402, 406,
        381, 6, 14,  357, 358, 379, 383, 385, 390, 391, 403, 404, 407, 412, 413, 418, 422,
        381, 7, 10,  356, 360, 386, 392, 393, 408, 414, 415, 419, 423,
        382, 0, 1,  381,
        382, 1, 2,  380, 401,
        382, 2, 5,  379, 384, 385, 402, 405,
        382, 3, 8,  357, 383, 386, 390, 391, 403, 406, 407,
        382, 4, 4,  381, 382, 401, 405,
        382, 5, 4,  380, 384, 402, 406,
        382, 6, 14,  357, 358, 379, 383, 385, 390, 391, 403, 404, 407, 412, 413, 418, 422,
        382, 7, 10,  356, 360, 386, 392, 393, 408, 414, 415, 419, 423,
        383, 0, 1,  379,
        383, 1, 2,  357, 380,
        383, 2, 5,  356, 358, 381, 384, 385,
        383, 3, 8,  355, 360, 361, 382, 386, 390, 391, 401,
        383, 4, 4,  357, 358, 379, 383,
        383, 5, 4,  356, 360, 380, 384,
        383, 6, 14,  338, 339, 355, 359, 361, 371, 372, 381, 382, 385, 390, 391, 401, 405,
        383, 7, 22,  337, 341, 362, 363, 364, 365, 366, 367, 368, 369, 370, 373, 374, 375, 376, 377, 378, 386, 392, 393, 402, 406,
        384, 0, 1,  380,
        384, 1, 3,  379, 381, 385,
        384, 2, 7,  357, 382, 383, 386, 390, 391, 401,
        384, 3, 7,  356, 358, 387, 392, 393, 402, 405,
        384, 4, 2,  380, 384,
        384, 5, 11,  357, 358, 379, 381, 382, 383, 385, 390, 391, 401, 405,
        384, 6, 7,  356, 360, 386, 392, 393, 402, 406,
        384, 7, 17,  338, 339, 355, 359, 361, 371, 372, 387, 394, 395, 403, 404, 407, 412, 413, 418, 422,
        385, 0, 4,  380, 386, 390, 391,
        385, 1, 6,  379, 381, 384, 387, 392, 393,
        385, 2, 7,  357, 382, 383, 388, 394, 395, 401,
        385, 3, 7,  356, 358, 389, 396, 397, 402, 405,
        385, 4, 3,  385, 390, 391,
        385, 5, 5,  380, 384, 386, 392, 393,
        385, 6, 11,  357, 358, 379, 381, 382, 383, 387, 394, 395, 401, 405,
        385, 7, 7,  356, 360, 388, 396, 397, 402, 406,
        386, 0, 4,  385, 387, 392, 393,
        386, 1, 6,  380, 388, 390, 391, 394, 395,
        386, 2, 6,  379, 381, 384, 389, 396, 397,
        386, 3, 7,  357, 382, 383, 398, 399, 400, 401,
        386, 4, 3,  386, 392, 393,
        386, 5, 6,  385, 387, 390, 391, 394, 395,
        386, 6, 5,  380, 384, 388, 396, 397,
        386, 7, 12,  357, 358, 379, 381, 382, 383, 389, 398, 399, 400, 401, 405,
        387, 0, 4,  386, 388, 394, 395,
        387, 1, 6,  385, 389, 392, 393, 396, 397,
        387, 2, 6,  380, 390, 391, 398, 399, 400,
        387, 3, 3,  379, 381, 384,
        387, 4, 3,  387, 394, 395,
        387, 5, 6,  386, 388, 392, 393, 396, 397,
        387, 6, 7,  385, 389, 390, 391, 398, 399, 400,
        387, 7, 2,  380, 384,
        388, 0, 4,  387, 389, 396, 397,
        388, 1, 6,  386, 394, 395, 398, 399, 400,
        388, 2, 3,  385, 392, 393,
        388, 3, 3,  380, 390, 391,
        388, 4, 3,  388, 396, 397,
        388, 5, 7,  387, 389, 394, 395, 398, 399, 400,
        388, 6, 3,  386, 392, 393,
        388, 7, 3,  385, 390, 391,
        389, 0, 4,  388, 398, 399, 400,
        389, 1, 3,  387, 396, 397,
        389, 2, 3,  386, 394, 395,
        389, 3, 3,  385, 392, 393,
        389, 4, 4,  389, 398, 399, 400,
        389, 5, 3,  388, 396, 397,
        389, 6, 3,  387, 394, 395,
        389, 7, 3,  386, 392, 393,
        390, 0, 1,  385,
        390, 1, 3,  380, 386, 391,
        390, 2, 6,  379, 381, 384, 387, 392, 393,
        390, 3, 7,  357, 382, 383, 388, 394, 395, 401,
        390, 4, 3,  385, 390, 391,
        390, 5, 5,  380, 384, 386, 392, 393,
        390, 6, 11,  357, 358, 379, 381, 382, 383, 387, 394, 395, 401, 405,
        390, 7, 7,  356, 360, 388, 396, 397, 402, 406,
        391, 0, 1,  385,
        391, 1, 3,  380, 386, 390,
        391, 2, 6,  379, 381, 384, 387, 392, 393,
        391, 3, 7,  357, 382, 383, 388, 394, 395, 401,
        391, 4, 3,  385, 390, 391,
        391, 5, 5,  380, 384, 386, 392, 393,
        391, 6, 11,  357, 358, 379, 381, 382, 383, 387, 394, 395, 401, 405,
        391, 7, 7,  356, 360, 388, 396, 397, 402, 406,
        392, 0, 1,  386,
        392, 1, 3,  385, 387, 393,
        392, 2, 6,  380, 388, 390, 391, 394, 395,
        392, 3, 6,  379, 381, 384, 389, 396, 397,
        392, 4, 3,  386, 392, 393,
        392, 5, 6,  385, 387, 390, 391, 394, 395,
        392, 6, 5,  380, 384, 388, 396, 397,
        392, 7, 12,  357, 358, 379, 381, 382, 383, 389, 398, 399, 400, 401, 405,
        393, 0, 1,  386,
        393, 1, 3,  385, 387, 392,
        393, 2, 6,  380, 388, 390, 391, 394, 395,
        393, 3, 6,  379, 381, 384, 389, 396, 397,
        393, 4, 3,  386, 392, 393,
        393, 5, 6,  385, 387, 390, 391, 394, 395,
        393, 6, 5,  380, 384, 388, 396, 397,
        393, 7, 12,  357, 358, 379, 381, 382, 383, 389, 398, 399, 400, 401, 405,
        394, 0, 1,  387,
        394, 1, 3,  386, 388, 395,
        394, 2, 6,  385, 389, 392, 393, 396, 397,
        394, 3, 6,  380, 390, 391, 398, 399, 400,
        394, 4, 3,  387, 394, 395,
        394, 5, 6,  386, 388, 392, 393, 396, 397,
        394, 6, 7,  385, 389, 390, 391, 398, 399, 400,
        394, 7, 2,  380, 384,
        395, 0, 1,  387,
        395, 1, 3,  386, 388, 394,
        395, 2, 6,  385, 389, 392, 393, 396, 397,
        395, 3, 6,  380, 390, 391, 398, 399, 400,
        395, 4, 3,  387, 394, 395,
        395, 5, 6,  386, 388, 392, 393, 396, 397,
        395, 6, 7,  385, 389, 390, 391, 398, 399, 400,
        395, 7, 2,  380, 384,
        396, 0, 1,  388,
        396, 1, 3,  387, 389, 397,
        396, 2, 6,  386, 394, 395, 398, 399, 400,
        396, 3, 3,  385, 392, 393,
        396, 4, 3,  388, 396, 397,
        396, 5, 7,  387, 389, 394, 395, 398, 399, 400,
        396, 6, 3,  386, 392, 393,
        396, 7, 3,  385, 390, 391,
        397, 0, 1,  388,
        397, 1, 3,  387, 389, 396,
        397, 2, 6,  386, 394, 395, 398, 399, 400,
        397, 3, 3,  385, 392, 393,
        397, 4, 3,  388, 396, 397,
        397, 5, 7,  387, 389, 394, 395, 398, 399, 400,
        397, 6, 3,  386, 392, 393,
        397, 7, 3,  385, 390, 391,
        398, 0, 1,  389,
        398, 1, 3,  388, 399, 400,
        398, 2, 3,  387, 396, 397,
        398, 3, 3,  386, 394, 395,
        398, 4, 4,  389, 398, 399, 400,
        398, 5, 3,  388, 396, 397,
        398, 6, 3,  387, 394, 395,
        398, 7, 3,  386, 392, 393,
        399, 0, 1,  389,
        399, 1, 3,  388, 398, 400,
        399, 2, 3,  387, 396, 397,
        399, 3, 3,  386, 394, 395,
        399, 4, 4,  389, 398, 399, 400,
        399, 5, 3,  388, 396, 397,
        399, 6, 3,  387, 394, 395,
        399, 7, 3,  386, 392, 393,
        400, 0, 1,  389,
        400, 1, 3,  388, 398, 399,
        400, 2, 3,  387, 396, 397,
        400, 3, 3,  386, 394, 395,
        400, 4, 4,  389, 398, 399, 400,
        400, 5, 3,  388, 396, 397,
        400, 6, 3,  387, 394, 395,
        400, 7, 3,  386, 392, 393,
        401, 0, 3,  381, 402, 405,
        401, 1, 5,  380, 382, 403, 406, 407,
        401, 2, 8,  379, 384, 385, 404, 408, 412, 413, 418,
        401, 3, 10,  357, 383, 386, 390, 391, 409, 414, 415, 419, 422,
        401, 4, 4,  381, 382, 401, 405,
        401, 5, 4,  380, 384, 402, 406,
        401, 6, 14,  357, 358, 379, 383, 385, 390, 391, 403, 404, 407, 412, 413, 418, 422,
        401, 7, 10,  356, 360, 386, 392, 393, 408, 414, 415, 419, 423,
        402, 0, 4,  401, 403, 406, 407,
        402, 1, 7,  381, 404, 405, 408, 412, 413, 418,
        402, 2, 7,  380, 382, 409, 414, 415, 419, 422,
        402, 3, 8,  379, 384, 385, 410, 411, 420, 423, 424,
        402, 4, 2,  402, 406,
        402, 5, 11,  381, 382, 401, 403, 404, 405, 407, 412, 413, 418, 422,
        402, 6, 7,  380, 384, 408, 414, 415, 419, 423,
        402, 7, 19,  357, 358, 379, 383, 385, 390, 391, 409, 410, 411, 416, 417, 420, 421, 424, 429, 430, 435, 439,
        403, 0, 3,  402, 404, 418,
        403, 1, 5,  401, 406, 407, 419, 422,
        403, 2, 8,  381, 405, 408, 412, 413, 420, 423, 424,
        403, 3, 10,  380, 382, 409, 414, 415, 421, 425, 429, 430, 435,
        403, 4, 4,  403, 404, 418, 422,
        403, 5, 4,  402, 406, 419, 423,
        403, 6, 14,  381, 382, 401, 405, 407, 412, 413, 420, 421, 424, 429, 430, 435, 439,
        403, 7, 10,  380, 384, 408, 414, 415, 425, 431, 432, 436, 440,
        404, 0, 1,  403,
        404, 1, 2,  402, 418,
        404, 2, 5,  401, 406, 407, 419, 422,
        404, 3, 8,  381, 405, 408, 412, 413, 420, 423, 424,
        404, 4, 4,  403, 404, 418, 422,
        404, 5, 4,  402, 406, 419, 423,
        404, 6, 14,  381, 382, 401, 405, 407, 412, 413, 420, 421, 424, 429, 430, 435, 439,
        404, 7, 10,  380, 384, 408, 414, 415, 425, 431, 432, 436, 440,
        405, 0, 1,  401,
        405, 1, 2,  381, 402,
        405, 2, 5,  380, 382, 403, 406, 407,
        405, 3, 8,  379, 384, 385, 404, 408, 412, 413, 418,
        405, 4, 4,  381, 382, 401, 405,
        405, 5, 4,  380, 384, 402, 406,
        405, 6, 14,  357, 358, 379, 383, 385, 390, 391, 403, 404, 407, 412, 413, 418, 422,
        405, 7, 10,  356, 360, 386, 392, 393, 408, 414, 415, 419, 423,
        406, 0, 1,  402,
        406, 1, 3,  401, 403, 407,
        406, 2, 7,  381, 404, 405, 408, 412, 413, 418,
        406, 3, 7,  380, 382, 409, 414, 415, 419, 422,
        406, 4, 2,  402, 406,
        406, 5, 11,  381, 382, 401, 403, 404, 405, 407, 412, 413, 418, 422,
        406, 6, 7,  380, 384, 408, 414, 415, 419, 423,
        406, 7, 19,  357, 358, 379, 383, 385, 390, 391, 409, 410, 411, 416, 417, 420, 421, 424, 429, 430, 435, 439,
        407, 0, 4,  402, 408, 412, 413,
        407, 1, 6,  401, 403, 406, 409, 414, 415,
        407, 2, 6,  381, 404, 405, 410, 411, 418,
        407, 3, 6,  380, 382, 416, 417, 419, 422,
        407, 4, 3,  407, 412, 413,
        407, 5, 5,  402, 406, 408, 414, 415,
        407, 6, 13,  381, 382, 401, 403, 404, 405, 409, 410, 411, 416, 417, 418, 422,
        407, 7, 4,  380, 384, 419, 423,
        408, 0, 4,  407, 409, 414, 415,
        408, 1, 5,  402, 410, 411, 412, 413,
        408, 2, 5,  401, 403, 406, 416, 417,
        408, 3, 4,  381, 404, 405, 418,
        408, 4, 3,  408, 414, 415,
        408, 5, 8,  407, 409, 410, 411, 412, 413, 416, 417,
        408, 6, 2,  402, 406,
        408, 7, 8,  381, 382, 401, 403, 404, 405, 418, 422,
        409, 0, 3,  408, 410, 411,
        409, 1, 5,  407, 414, 415, 416, 417,
        409, 2, 3,  402, 412, 413,
        409, 3, 3,  401, 403, 406,
        409, 4, 5,  409, 410, 411, 416, 417,
        409, 5, 3,  408, 414, 415,
        409, 6, 3,  407, 412, 413,
        409, 7, 2,  402, 406,
        410, 0, 1,  409,
        410, 1, 2,  408, 411,
        410, 2, 5,  407, 414, 415, 416, 417,
        410, 3, 3,  402, 412, 413,
        410, 4, 5,  409, 410, 411, 416, 417,
        410, 5, 3,  408, 414, 415,
        410, 6, 3,  407, 412, 413,
        410, 7, 2,  402, 406,
        411, 0, 3,  409, 416, 417,
        411, 1, 2,  408, 410,
        411, 2, 3,  407, 414, 415,
        411, 3, 3,  402, 412, 413,
        411, 4, 5,  409, 410, 411, 416, 417,
        411, 5, 3,  408, 414, 415,
        411, 6, 3,  407, 412, 413,
        411, 7, 2,  402, 406,
        412, 0, 1,  407,
        412, 1, 3,  402, 408, 413,
        412, 2, 6,  401, 403, 406, 409, 414, 415,
        412, 3, 6,  381, 404, 405, 410, 411, 418,
        412, 4, 3,  407, 412, 413,
        412, 5, 5,  402, 406, 408, 414, 415,
        412, 6, 13,  381, 382, 401, 403, 404, 405, 409, 410, 411, 416, 417, 418, 422,
        412, 7, 4,  380, 384, 419, 423,
        413, 0, 1,  407,
        413, 1, 3,  402, 408, 412,
        413, 2, 6,  401, 403, 406, 409, 414, 415,
        413, 3, 6,  381, 404, 405, 410, 411, 418,
        413, 4, 3,  407, 412, 413,
        413, 5, 5,  402, 406, 408, 414, 415,
        413, 6, 13,  381, 382, 401, 403, 404, 405, 409, 410, 411, 416, 417, 418, 422,
        413, 7, 4,  380, 384, 419, 423,
        414, 0, 1,  408,
        414, 1, 3,  407, 409, 415,
        414, 2, 5,  402, 410, 411, 412, 413,
        414, 3, 5,  401, 403, 406, 416, 417,
        414, 4, 3,  408, 414, 415,
        414, 5, 8,  407, 409, 410, 411, 412, 413, 416, 417,
        414, 6, 2,  402, 406,
        414, 7, 8,  381, 382, 401, 403, 404, 405, 418, 422,
        415, 0, 1,  408,
        415, 1, 3,  407, 409, 414,
        415, 2, 5,  402, 410, 411, 412, 413,
        415, 3, 5,  401, 403, 406, 416, 417,
        415, 4, 3,  408, 414, 415,
        415, 5, 8,  407, 409, 410, 411, 412, 413, 416, 417,
        415, 6, 2,  402, 406,
        415, 7, 8,  381, 382, 401, 403, 404, 405, 418, 422,
        416, 0, 1,  411,
        416, 1, 2,  409, 417,
        416, 2, 2,  408, 410,
        416, 3, 3,  407, 414, 415,
        416, 4, 5,  409, 410, 411, 416, 417,
        416, 5, 3,  408, 414, 415,
        416, 6, 3,  407, 412, 413,
        416, 7, 2,  402, 406,
        417, 0, 1,  411,
        417, 1, 2,  409, 416,
        417, 2, 2,  408, 410,
        417, 3, 3,  407, 414, 415,
        417, 4, 5,  409, 410, 411, 416, 417,
        417, 5, 3,  408, 414, 415,
        417, 6, 3,  407, 412, 413,
        417, 7, 2,  402, 406,
        418, 0, 3,  403, 419, 422,
        418, 1, 5,  402, 404, 420, 423, 424,
        418, 2, 8,  401, 406, 407, 421, 425, 429, 430, 435,
        418, 3, 10,  381, 405, 408, 412, 413, 426, 431, 432, 436, 439,
        418, 4, 4,  403, 404, 418, 422,
        418, 5, 4,  402, 406, 419, 423,
        418, 6, 14,  381, 382, 401, 405, 407, 412, 413, 420, 421, 424, 429, 430, 435, 439,
        418, 7, 10,  380, 384, 408, 414, 415, 425, 431, 432, 436, 440,
        419, 0, 4,  418, 420, 423, 424,
        419, 1, 7,  403, 421, 422, 425, 429, 430, 435,
        419, 2, 7,  402, 404, 426, 431, 432, 436, 439,
        419, 3, 8,  401, 406, 407, 427, 428, 437, 440, 441,
        419, 4, 2,  419, 423,
        419, 5, 11,  403, 404, 418, 420, 421, 422, 424, 429, 430, 435, 439,
        419, 6, 7,  402, 406, 425, 431, 432, 436, 440,
        419, 7, 19,  381, 382, 401, 405, 407, 412, 413, 426, 427, 428, 433, 434, 437, 438, 441, 445, 446, 449, 453,
        420, 0, 3,  419, 421, 435,
        420, 1, 5,  418, 423, 424, 436, 439,
        420, 2, 8,  403, 422, 425, 429, 430, 437, 440, 441,
        420, 3, 10,  402, 404, 426, 431, 432, 438, 442, 445, 446, 449,
        420, 4, 4,  420, 421, 435, 439,
        420, 5, 4,  419, 423, 436, 440,
        420, 6, 14,  403, 404, 418, 422, 424, 429, 430, 437, 438, 441, 445, 446, 449, 453,
        420, 7, 12,  402, 406, 425, 431, 432, 442, 443, 444, 447, 448, 450, 454,
        421, 0, 1,  420,
        421, 1, 2,  419, 435,
        421, 2, 5,  418, 423, 424, 436, 439,
        421, 3, 8,  403, 422, 425, 429, 430, 437, 440, 441,
        421, 4, 4,  420, 421, 435, 439,
        421, 5, 4,  419, 423, 436, 440,
        421, 6, 14,  403, 404, 418, 422, 424, 429, 430, 437, 438, 441, 445, 446, 449, 453,
        421, 7, 12,  402, 406, 425, 431, 432, 442, 443, 444, 447, 448, 450, 454,
        422, 0, 1,  418,
        422, 1, 2,  403, 419,
        422, 2, 5,  402, 404, 420, 423, 424,
        422, 3, 8,  401, 406, 407, 421, 425, 429, 430, 435,
        422, 4, 4,  403, 404, 418, 422,
        422, 5, 4,  402, 406, 419, 423,
        422, 6, 14,  381, 382, 401, 405, 407, 412, 413, 420, 421, 424, 429, 430, 435, 439,
        422, 7, 10,  380, 384, 408, 414, 415, 425, 431, 432, 436, 440,
        423, 0, 1,  419,
        423, 1, 3,  418, 420, 424,
        423, 2, 7,  403, 421, 422, 425, 429, 430, 435,
        423, 3, 7,  402, 404, 426, 431, 432, 436, 439,
        423, 4, 2,  419, 423,
        423, 5, 11,  403, 404, 418, 420, 421, 422, 424, 429, 430, 435, 439,
        423, 6, 7,  402, 406, 425, 431, 432, 436, 440,
        423, 7, 19,  381, 382, 401, 405, 407, 412, 413, 426, 427, 428, 433, 434, 437, 438, 441, 445, 446, 449, 453,
        424, 0, 4,  419, 425, 429, 430,
        424, 1, 6,  418, 420, 423, 426, 431, 432,
        424, 2, 6,  403, 421, 422, 427, 428, 435,
        424, 3, 6,  402, 404, 433, 434, 436, 439,
        424, 4, 3,  424, 429, 430,
        424, 5, 5,  419, 423, 425, 431, 432,
        424, 6, 13,  403, 404, 418, 420, 421, 422, 426, 427, 428, 433, 434, 435, 439,
        424, 7, 4,  402, 406, 436, 440,
        425, 0, 4,  424, 426, 431, 432,
        425, 1, 5,  419, 427, 428, 429, 430,
        425, 2, 5,  418, 420, 423, 433, 434,
        425, 3, 4,  403, 421, 422, 435,
        425, 4, 3,  425, 431, 432,
        425, 5, 8,  424, 426, 427, 428, 429, 430, 433, 434,
        425, 6, 2,  419, 423,
        425, 7, 8,  403, 404, 418, 420, 421, 422, 435, 439,
        426, 0, 3,  425, 427, 428,
        426, 1, 5,  424, 431, 432, 433, 434,
        426, 2, 3,  419, 429, 430,
        426, 3, 3,  418, 420, 423,
        426, 4, 5,  426, 427, 428, 433, 434,
        426, 5, 3,  425, 431, 432,
        426, 6, 3,  424, 429, 430,
        426, 7, 2,  419, 423,
        427, 0, 1,  426,
        427, 1, 2,  425, 428,
        427, 2, 5,  424, 431, 432, 433, 434,
        427, 3, 3,  419, 429, 430,
        427, 4, 5,  426, 427, 428, 433, 434,
        427, 5, 3,  425, 431, 432,
        427, 6, 3,  424, 429, 430,
        427, 7, 2,  419, 423,
        428, 0, 3,  426, 433, 434,
        428, 1, 2,  425, 427,
        428, 2, 3,  424, 431, 432,
        428, 3, 3,  419, 429, 430,
        428, 4, 5,  426, 427, 428, 433, 434,
        428, 5, 3,  425, 431, 432,
        428, 6, 3,  424, 429, 430,
        428, 7, 2,  419, 423,
        429, 0, 1,  424,
        429, 1, 3,  419, 425, 430,
        429, 2, 6,  418, 420, 423, 426, 431, 432,
        429, 3, 6,  403, 421, 422, 427, 428, 435,
        429, 4, 3,  424, 429, 430,
        429, 5, 5,  419, 423, 425, 431, 432,
        429, 6, 13,  403, 404, 418, 420, 421, 422, 426, 427, 428, 433, 434, 435, 439,
        429, 7, 4,  402, 406, 436, 440,
        430, 0, 1,  424,
        430, 1, 3,  419, 425, 429,
        430, 2, 6,  418, 420, 423, 426, 431, 432,
        430, 3, 6,  403, 421, 422, 427, 428, 435,
        430, 4, 3,  424, 429, 430,
        430, 5, 5,  419, 423, 425, 431, 432,
        430, 6, 13,  403, 404, 418, 420, 421, 422, 426, 427, 428, 433, 434, 435, 439,
        430, 7, 4,  402, 406, 436, 440,
        431, 0, 1,  425,
        431, 1, 3,  424, 426, 432,
        431, 2, 5,  419, 427, 428, 429, 430,
        431, 3, 5,  418, 420, 423, 433, 434,
        431, 4, 3,  425, 431, 432,
        431, 5, 8,  424, 426, 427, 428, 429, 430, 433, 434,
        431, 6, 2,  419, 423,
        431, 7, 8,  403, 404, 418, 420, 421, 422, 435, 439,
        432, 0, 1,  425,
        432, 1, 3,  424, 426, 431,
        432, 2, 5,  419, 427, 428, 429, 430,
        432, 3, 5,  418, 420, 423, 433, 434,
        432, 4, 3,  425, 431, 432,
        432, 5, 8,  424, 426, 427, 428, 429, 430, 433, 434,
        432, 6, 2,  419, 423,
        432, 7, 8,  403, 404, 418, 420, 421, 422, 435, 439,
        433, 0, 1,  428,
        433, 1, 2,  426, 434,
        433, 2, 2,  425, 427,
        433, 3, 3,  424, 431, 432,
        433, 4, 5,  426, 427, 428, 433, 434,
        433, 5, 3,  425, 431, 432,
        433, 6, 3,  424, 429, 430,
        433, 7, 2,  419, 423,
        434, 0, 1,  428,
        434, 1, 2,  426, 433,
        434, 2, 2,  425, 427,
        434, 3, 3,  424, 431, 432,
        434, 4, 5,  426, 427, 428, 433, 434,
        434, 5, 3,  425, 431, 432,
        434, 6, 3,  424, 429, 430,
        434, 7, 2,  419, 423,
        435, 0, 3,  420, 436, 439,
        435, 1, 5,  419, 421, 437, 440, 441,
        435, 2, 8,  418, 423, 424, 438, 442, 445, 446, 449,
        435, 3, 9,  403, 422, 425, 429, 430, 443, 444, 450, 453,
        435, 4, 4,  420, 421, 435, 439,
        435, 5, 4,  419, 423, 436, 440,
        435, 6, 14,  403, 404, 418, 422, 424, 429, 430, 437, 438, 441, 445, 446, 449, 453,
        435, 7, 12,  402, 406, 425, 431, 432, 442, 443, 444, 447, 448, 450, 454,
        436, 0, 4,  435, 437, 440, 441,
        436, 1, 7,  420, 438, 439, 442, 445, 446, 449,
        436, 2, 6,  419, 421, 443, 444, 450, 453,
        436, 3, 8,  418, 423, 424, 447, 448, 451, 454, 455,
        436, 4, 2,  436, 440,
        436, 5, 11,  420, 421, 435, 437, 438, 439, 441, 445, 446, 449, 453,
        436, 6, 9,  419, 423, 442, 443, 444, 447, 448, 450, 454,
        436, 7, 14,  403, 404, 418, 422, 424, 429, 430, 451, 452, 455, 459, 460, 468, 472,
        437, 0, 3,  436, 438, 449,
        437, 1, 5,  435, 440, 441, 450, 453,
        437, 2, 8,  420, 439, 442, 445, 446, 451, 454, 455,
        437, 3, 9,  419, 421, 443, 444, 452, 456, 459, 460, 468,
        437, 4, 4,  437, 438, 449, 453,
        437, 5, 4,  436, 440, 450, 454,
        437, 6, 14,  420, 421, 435, 439, 441, 445, 446, 451, 452, 455, 459, 460, 468, 472,
        437, 7, 11,  419, 423, 442, 443, 444, 447, 448, 456, 461, 469, 473,
        438, 0, 1,  437,
        438, 1, 2,  436, 449,
        438, 2, 5,  435, 440, 441, 450, 453,
        438, 3, 8,  420, 439, 442, 445, 446, 451, 454, 455,
        438, 4, 4,  437, 438, 449, 453,
        438, 5, 4,  436, 440, 450, 454,
        438, 6, 14,  420, 421, 435, 439, 441, 445, 446, 451, 452, 455, 459, 460, 468, 472,
        438, 7, 11,  419, 423, 442, 443, 444, 447, 448, 456, 461, 469, 473,
        439, 0, 1,  435,
        439, 1, 2,  420, 436,
        439, 2, 5,  419, 421, 437, 440, 441,
        439, 3, 8,  418, 423, 424, 438, 442, 445, 446, 449,
        439, 4, 4,  420, 421, 435, 439,
        439, 5, 4,  419, 423, 436, 440,
        439, 6, 14,  403, 404, 418, 422, 424, 429, 430, 437, 438, 441, 445, 446, 449, 453,
        439, 7, 12,  402, 406, 425, 431, 432, 442, 443, 444, 447, 448, 450, 454,
        440, 0, 1,  436,
        440, 1, 3,  435, 437, 441,
        440, 2, 7,  420, 438, 439, 442, 445, 446, 449,
        440, 3, 6,  419, 421, 443, 444, 450, 453,
        440, 4, 2,  436, 440,
        440, 5, 11,  420, 421, 435, 437, 438, 439, 441, 445, 446, 449, 453,
        440, 6, 9,  419, 423, 442, 443, 444, 447, 448, 450, 454,
        440, 7, 14,  403, 404, 418, 422, 424, 429, 430, 451, 452, 455, 459, 460, 468, 472,
        441, 0, 4,  436, 442, 445, 446,
        441, 1, 5,  435, 437, 440, 443, 444,
        441, 2, 6,  420, 438, 439, 447, 448, 449,
        441, 3, 4,  419, 421, 450, 453,
        441, 4, 3,  441, 445, 446,
        441, 5, 7,  436, 440, 442, 443, 444, 447, 448,
        441, 6, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        441, 7, 4,  419, 423, 450, 454,
        442, 0, 3,  441, 443, 444,
        442, 1, 5,  436, 445, 446, 447, 448,
        442, 2, 3,  435, 437, 440,
        442, 3, 4,  420, 438, 439, 449,
        442, 4, 5,  442, 443, 444, 447, 448,
        442, 5, 3,  441, 445, 446,
        442, 6, 2,  436, 440,
        442, 7, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        443, 0, 1,  442,
        443, 1, 2,  441, 444,
        443, 2, 5,  436, 445, 446, 447, 448,
        443, 3, 3,  435, 437, 440,
        443, 4, 5,  442, 443, 444, 447, 448,
        443, 5, 3,  441, 445, 446,
        443, 6, 2,  436, 440,
        443, 7, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        444, 0, 3,  442, 447, 448,
        444, 1, 2,  441, 443,
        444, 2, 3,  436, 445, 446,
        444, 3, 3,  435, 437, 440,
        444, 4, 5,  442, 443, 444, 447, 448,
        444, 5, 3,  441, 445, 446,
        444, 6, 2,  436, 440,
        444, 7, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        445, 0, 1,  441,
        445, 1, 3,  436, 442, 446,
        445, 2, 5,  435, 437, 440, 443, 444,
        445, 3, 6,  420, 438, 439, 447, 448, 449,
        445, 4, 3,  441, 445, 446,
        445, 5, 7,  436, 440, 442, 443, 444, 447, 448,
        445, 6, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        445, 7, 4,  419, 423, 450, 454,
        446, 0, 1,  441,
        446, 1, 3,  436, 442, 445,
        446, 2, 5,  435, 437, 440, 443, 444,
        446, 3, 6,  420, 438, 439, 447, 448, 449,
        446, 4, 3,  441, 445, 446,
        446, 5, 7,  436, 440, 442, 443, 444, 447, 448,
        446, 6, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        446, 7, 4,  419, 423, 450, 454,
        447, 0, 1,  444,
        447, 1, 2,  442, 448,
        447, 2, 2,  441, 443,
        447, 3, 3,  436, 445, 446,
        447, 4, 5,  442, 443, 444, 447, 448,
        447, 5, 3,  441, 445, 446,
        447, 6, 2,  436, 440,
        447, 7, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        448, 0, 1,  444,
        448, 1, 2,  442, 447,
        448, 2, 2,  441, 443,
        448, 3, 3,  436, 445, 446,
        448, 4, 5,  442, 443, 444, 447, 448,
        448, 5, 3,  441, 445, 446,
        448, 6, 2,  436, 440,
        448, 7, 8,  420, 421, 435, 437, 438, 439, 449, 453,
        449, 0, 3,  437, 450, 453,
        449, 1, 5,  436, 438, 451, 454, 455,
        449, 2, 8,  435, 440, 441, 452, 456, 459, 460, 468,
        449, 3, 10,  420, 439, 442, 445, 446, 457, 458, 461, 469, 472,
        449, 4, 4,  437, 438, 449, 453,
        449, 5, 4,  436, 440, 450, 454,
        449, 6, 14,  420, 421, 435, 439, 441, 445, 446, 451, 452, 455, 459, 460, 468, 472,
        449, 7, 11,  419, 423, 442, 443, 444, 447, 448, 456, 461, 469, 473,
        450, 0, 4,  449, 451, 454, 455,
        450, 1, 7,  437, 452, 453, 456, 459, 460, 468,
        450, 2, 7,  436, 438, 457, 458, 461, 469, 472,
        450, 3, 12,  435, 440, 441, 462, 463, 464, 465, 466, 467, 470, 473, 474,
        450, 4, 2,  450, 454,
        450, 5, 11,  437, 438, 449, 451, 452, 453, 455, 459, 460, 468, 472,
        450, 6, 6,  436, 440, 456, 461, 469, 473,
        450, 7, 22,  420, 421, 435, 439, 441, 445, 446, 457, 458, 462, 463, 464, 465, 466, 467, 470, 471, 474, 479, 480, 490, 494,
        451, 0, 3,  450, 452, 468,
        451, 1, 5,  449, 454, 455, 469, 472,
        451, 2, 8,  437, 453, 456, 459, 460, 470, 473, 474,
        451, 3, 10,  436, 438, 457, 458, 461, 471, 475, 479, 480, 490,
        451, 4, 4,  451, 452, 468, 472,
        451, 5, 4,  450, 454, 469, 473,
        451, 6, 14,  437, 438, 449, 453, 455, 459, 460, 470, 471, 474, 479, 480, 490, 494,
        451, 7, 9,  436, 440, 456, 461, 475, 481, 482, 491, 495,
        452, 0, 1,  451,
        452, 1, 2,  450, 468,
        452, 2, 5,  449, 454, 455, 469, 472,
        452, 3, 8,  437, 453, 456, 459, 460, 470, 473, 474,
        452, 4, 4,  451, 452, 468, 472,
        452, 5, 4,  450, 454, 469, 473,
        452, 6, 14,  437, 438, 449, 453, 455, 459, 460, 470, 471, 474, 479, 480, 490, 494,
        452, 7, 9,  436, 440, 456, 461, 475, 481, 482, 491, 495,
        453, 0, 1,  449,
        453, 1, 2,  437, 450,
        453, 2, 5,  436, 438, 451, 454, 455,
        453, 3, 8,  435, 440, 441, 452, 456, 459, 460, 468,
        453, 4, 4,  437, 438, 449, 453,
        453, 5, 4,  436, 440, 450, 454,
        453, 6, 14,  420, 421, 435, 439, 441, 445, 446, 451, 452, 455, 459, 460, 468, 472,
        453, 7, 11,  419, 423, 442, 443, 444, 447, 448, 456, 461, 469, 473,
        454, 0, 1,  450,
        454, 1, 3,  449, 451, 455,
        454, 2, 7,  437, 452, 453, 456, 459, 460, 468,
        454, 3, 7,  436, 438, 457, 458, 461, 469, 472,
        454, 4, 2,  450, 454,
        454, 5, 11,  437, 438, 449, 451, 452, 453, 455, 459, 460, 468, 472,
        454, 6, 6,  436, 440, 456, 461, 469, 473,
        454, 7, 22,  420, 421, 435, 439, 441, 445, 446, 457, 458, 462, 463, 464, 465, 466, 467, 470, 471, 474, 479, 480, 490, 494,
        455, 0, 4,  450, 456, 459, 460,
        455, 1, 6,  449, 451, 454, 457, 458, 461,
        455, 2, 10,  437, 452, 453, 462, 463, 464, 465, 466, 467, 468,
        455, 3, 4,  436, 438, 469, 472,
        455, 4, 3,  455, 459, 460,
        455, 5, 4,  450, 454, 456, 461,
        455, 6, 16,  437, 438, 449, 451, 452, 453, 457, 458, 462, 463, 464, 465, 466, 467, 468, 472,
        455, 7, 4,  436, 440, 469, 473,
        456, 0, 4,  455, 457, 458, 461,
        456, 1, 9,  450, 459, 460, 462, 463, 464, 465, 466, 467,
        456, 2, 3,  449, 451, 454,
        456, 3, 4,  437, 452, 453, 468,
        456, 4, 2,  456, 461,
        456, 5, 11,  455, 457, 458, 459, 460, 462, 463, 464, 465, 466, 467,
        456, 6, 2,  450, 454,
        456, 7, 8,  437, 438, 449, 451, 452, 453, 468, 472,
        457, 0, 4,  456, 462, 463, 464,
        457, 1, 3,  455, 458, 461,
        457, 2, 6,  450, 459, 460, 465, 466, 467,
        457, 3, 3,  449, 451, 454,
        457, 4, 4,  457, 462, 463, 464,
        457, 5, 2,  456, 461,
        457, 6, 7,  455, 458, 459, 460, 465, 466, 467,
        457, 7, 2,  450, 454,
        458, 0, 4,  456, 465, 466, 467,
        458, 1, 3,  455, 457, 461,
        458, 2, 6,  450, 459, 460, 462, 463, 464,
        458, 3, 3,  449, 451, 454,
        458, 4, 4,  458, 465, 466, 467,
        458, 5, 2,  456, 461,
        458, 6, 7,  455, 457, 459, 460, 462, 463, 464,
        458, 7, 2,  450, 454,
        459, 0, 1,  455,
        459, 1, 3,  450, 456, 460,
        459, 2, 6,  449, 451, 454, 457, 458, 461,
        459, 3, 10,  437, 452, 453, 462, 463, 464, 465, 466, 467, 468,
        459, 4, 3,  455, 459, 460,
        459, 5, 4,  450, 454, 456, 461,
        459, 6, 16,  437, 438, 449, 451, 452, 453, 457, 458, 462, 463, 464, 465, 466, 467, 468, 472,
        459, 7, 4,  436, 440, 469, 473,
        460, 0, 1,  455,
        460, 1, 3,  450, 456, 459,
        460, 2, 6,  449, 451, 454, 457, 458, 461,
        460, 3, 10,  437, 452, 453, 462, 463, 464, 465, 466, 467, 468,
        460, 4, 3,  455, 459, 460,
        460, 5, 4,  450, 454, 456, 461,
        460, 6, 16,  437, 438, 449, 451, 452, 453, 457, 458, 462, 463, 464, 465, 466, 467, 468, 472,
        460, 7, 4,  436, 440, 469, 473,
        461, 0, 1,  456,
        461, 1, 3,  455, 457, 458,
        461, 2, 9,  450, 459, 460, 462, 463, 464, 465, 466, 467,
        461, 3, 3,  449, 451, 454,
        461, 4, 2,  456, 461,
        461, 5, 11,  455, 457, 458, 459, 460, 462, 463, 464, 465, 466, 467,
        461, 6, 2,  450, 454,
        461, 7, 8,  437, 438, 449, 451, 452, 453, 468, 472,
        462, 0, 1,  457,
        462, 1, 3,  456, 463, 464,
        462, 2, 3,  455, 458, 461,
        462, 3, 6,  450, 459, 460, 465, 466, 467,
        462, 4, 4,  457, 462, 463, 464,
        462, 5, 2,  456, 461,
        462, 6, 7,  455, 458, 459, 460, 465, 466, 467,
        462, 7, 2,  450, 454,
        463, 0, 1,  457,
        463, 1, 3,  456, 462, 464,
        463, 2, 3,  455, 458, 461,
        463, 3, 6,  450, 459, 460, 465, 466, 467,
        463, 4, 4,  457, 462, 463, 464,
        463, 5, 2,  456, 461,
        463, 6, 7,  455, 458, 459, 460, 465, 466, 467,
        463, 7, 2,  450, 454,
        464, 0, 1,  457,
        464, 1, 3,  456, 462, 463,
        464, 2, 3,  455, 458, 461,
        464, 3, 6,  450, 459, 460, 465, 466, 467,
        464, 4, 4,  457, 462, 463, 464,
        464, 5, 2,  456, 461,
        464, 6, 7,  455, 458, 459, 460, 465, 466, 467,
        464, 7, 2,  450, 454,
        465, 0, 1,  458,
        465, 1, 3,  456, 466, 467,
        465, 2, 3,  455, 457, 461,
        465, 3, 6,  450, 459, 460, 462, 463, 464,
        465, 4, 4,  458, 465, 466, 467,
        465, 5, 2,  456, 461,
        465, 6, 7,  455, 457, 459, 460, 462, 463, 464,
        465, 7, 2,  450, 454,
        466, 0, 1,  458,
        466, 1, 3,  456, 465, 467,
        466, 2, 3,  455, 457, 461,
        466, 3, 6,  450, 459, 460, 462, 463, 464,
        466, 4, 4,  458, 465, 466, 467,
        466, 5, 2,  456, 461,
        466, 6, 7,  455, 457, 459, 460, 462, 463, 464,
        466, 7, 2,  450, 454,
        467, 0, 1,  458,
        467, 1, 3,  456, 465, 466,
        467, 2, 3,  455, 457, 461,
        467, 3, 6,  450, 459, 460, 462, 463, 464,
        467, 4, 4,  458, 465, 466, 467,
        467, 5, 2,  456, 461,
        467, 6, 7,  455, 457, 459, 460, 462, 463, 464,
        467, 7, 2,  450, 454,
        468, 0, 3,  451, 469, 472,
        468, 1, 5,  450, 452, 470, 473, 474,
        468, 2, 8,  449, 454, 455, 471, 475, 479, 480, 490,
        468, 3, 10,  437, 453, 456, 459, 460, 476, 481, 482, 491, 494,
        468, 4, 4,  451, 452, 468, 472,
        468, 5, 4,  450, 454, 469, 473,
        468, 6, 14,  437, 438, 449, 453, 455, 459, 460, 470, 471, 474, 479, 480, 490, 494,
        468, 7, 9,  436, 440, 456, 461, 475, 481, 482, 491, 495,
        469, 0, 4,  468, 470, 473, 474,
        469, 1, 7,  451, 471, 472, 475, 479, 480, 490,
        469, 2, 7,  450, 452, 476, 481, 482, 491, 494,
        469, 3, 9,  449, 454, 455, 477, 483, 484, 492, 495, 496,
        469, 4, 2,  469, 473,
        469, 5, 11,  451, 452, 468, 470, 471, 472, 474, 479, 480, 490, 494,
        469, 6, 7,  450, 454, 475, 481, 482, 491, 495,
        469, 7, 17,  437, 438, 449, 453, 455, 459, 460, 476, 483, 484, 492, 493, 496, 501, 502, 512, 516,
        470, 0, 3,  469, 471, 490,
        470, 1, 5,  468, 473, 474, 491, 494,
        470, 2, 8,  451, 472, 475, 479, 480, 492, 495, 496,
        470, 3, 10,  450, 452, 476, 481, 482, 493, 497, 501, 502, 512,
        470, 4, 4,  470, 471, 490, 494,
        470, 5, 4,  469, 473, 491, 495,
        470, 6, 14,  451, 452, 468, 472, 474, 479, 480, 492, 493, 496, 501, 502, 512, 516,
        470, 7, 10,  450, 454, 475, 481, 482, 497, 503, 504, 513, 517,
        471, 0, 1,  470,
        471, 1, 2,  469, 490,
        471, 2, 5,  468, 473, 474, 491, 494,
        471, 3, 8,  451, 472, 475, 479, 480, 492, 495, 496,
        471, 4, 4,  470, 471, 490, 494,
        471, 5, 4,  469, 473, 491, 495,
        471, 6, 14,  451, 452, 468, 472, 474, 479, 480, 492, 493, 496, 501, 502, 512, 516,
        471, 7, 10,  450, 454, 475, 481, 482, 497, 503, 504, 513, 517,
        472, 0, 1,  468,
        472, 1, 2,  451, 469,
        472, 2, 5,  450, 452, 470, 473, 474,
        472, 3, 8,  449, 454, 455, 471, 475, 479, 480, 490,
        472, 4, 4,  451, 452, 468, 472,
        472, 5, 4,  450, 454, 469, 473,
        472, 6, 14,  437, 438, 449, 453, 455, 459, 460, 470, 471, 474, 479, 480, 490, 494,
        472, 7, 9,  436, 440, 456, 461, 475, 481, 482, 491, 495,
        473, 0, 1,  469,
        473, 1, 3,  468, 470, 474,
        473, 2, 7,  451, 471, 472, 475, 479, 480, 490,
        473, 3, 7,  450, 452, 476, 481, 482, 491, 494,
        473, 4, 2,  469, 473,
        473, 5, 11,  451, 452, 468, 470, 471, 472, 474, 479, 480, 490, 494,
        473, 6, 7,  450, 454, 475, 481, 482, 491, 495,
        473, 7, 17,  437, 438, 449, 453, 455, 459, 460, 476, 483, 484, 492, 493, 496, 501, 502, 512, 516,
        474, 0, 4,  469, 475, 479, 480,
        474, 1, 6,  468, 470, 473, 476, 481, 482,
        474, 2, 7,  451, 471, 472, 477, 483, 484, 490,
        474, 3, 7,  450, 452, 478, 485, 486, 491, 494,
        474, 4, 3,  474, 479, 480,
        474, 5, 5,  469, 473, 475, 481, 482,
        474, 6, 11,  451, 452, 468, 470, 471, 472, 476, 483, 484, 490, 494,
        474, 7, 7,  450, 454, 477, 485, 486, 491, 495,
        475, 0, 4,  474, 476, 481, 482,
        475, 1, 6,  469, 477, 479, 480, 483, 484,
        475, 2, 6,  468, 470, 473, 478, 485, 486,
        475, 3, 7,  451, 471, 472, 487, 488, 489, 490,
        475, 4, 3,  475, 481, 482,
        475, 5, 6,  474, 476, 479, 480, 483, 484,
        475, 6, 5,  469, 473, 477, 485, 486,
        475, 7, 12,  451, 452, 468, 470, 471, 472, 478, 487, 488, 489, 490, 494,
        476, 0, 4,  475, 477, 483, 484,
        476, 1, 6,  474, 478, 481, 482, 485, 486,
        476, 2, 6,  469, 479, 480, 487, 488, 489,
        476, 3, 3,  468, 470, 473,
        476, 4, 3,  476, 483, 484,
        476, 5, 6,  475, 477, 481, 482, 485, 486,
        476, 6, 7,  474, 478, 479, 480, 487, 488, 489,
        476, 7, 2,  469, 473,
        477, 0, 4,  476, 478, 485, 486,
        477, 1, 6,  475, 483, 484, 487, 488, 489,
        477, 2, 3,  474, 481, 482,
        477, 3, 3,  469, 479, 480,
        477, 4, 3,  477, 485, 486,
        477, 5, 7,  476, 478, 483, 484, 487, 488, 489,
        477, 6, 3,  475, 481, 482,
        477, 7, 3,  474, 479, 480,
        478, 0, 4,  477, 487, 488, 489,
        478, 1, 3,  476, 485, 486,
        478, 2, 3,  475, 483, 484,
        478, 3, 3,  474, 481, 482,
        478, 4, 4,  478, 487, 488, 489,
        478, 5, 3,  477, 485, 486,
        478, 6, 3,  476, 483, 484,
        478, 7, 3,  475, 481, 482,
        479, 0, 1,  474,
        479, 1, 3,  469, 475, 480,
        479, 2, 6,  468, 470, 473, 476, 481, 482,
        479, 3, 7,  451, 471, 472, 477, 483, 484, 490,
        479, 4, 3,  474, 479, 480,
        479, 5, 5,  469, 473, 475, 481, 482,
        479, 6, 11,  451, 452, 468, 470, 471, 472, 476, 483, 484, 490, 494,
        479, 7, 7,  450, 454, 477, 485, 486, 491, 495,
        480, 0, 1,  474,
        480, 1, 3,  469, 475, 479,
        480, 2, 6,  468, 470, 473, 476, 481, 482,
        480, 3, 7,  451, 471, 472, 477, 483, 484, 490,
        480, 4, 3,  474, 479, 480,
        480, 5, 5,  469, 473, 475, 481, 482,
        480, 6, 11,  451, 452, 468, 470, 471, 472, 476, 483, 484, 490, 494,
        480, 7, 7,  450, 454, 477, 485, 486, 491, 495,
        481, 0, 1,  475,
        481, 1, 3,  474, 476, 482,
        481, 2, 6,  469, 477, 479, 480, 483, 484,
        481, 3, 6,  468, 470, 473, 478, 485, 486,
        481, 4, 3,  475, 481, 482,
        481, 5, 6,  474, 476, 479, 480, 483, 484,
        481, 6, 5,  469, 473, 477, 485, 486,
        481, 7, 12,  451, 452, 468, 470, 471, 472, 478, 487, 488, 489, 490, 494,
        482, 0, 1,  475,
        482, 1, 3,  474, 476, 481,
        482, 2, 6,  469, 477, 479, 480, 483, 484,
        482, 3, 6,  468, 470, 473, 478, 485, 486,
        482, 4, 3,  475, 481, 482,
        482, 5, 6,  474, 476, 479, 480, 483, 484,
        482, 6, 5,  469, 473, 477, 485, 486,
        482, 7, 12,  451, 452, 468, 470, 471, 472, 478, 487, 488, 489, 490, 494,
        483, 0, 1,  476,
        483, 1, 3,  475, 477, 484,
        483, 2, 6,  474, 478, 481, 482, 485, 486,
        483, 3, 6,  469, 479, 480, 487, 488, 489,
        483, 4, 3,  476, 483, 484,
        483, 5, 6,  475, 477, 481, 482, 485, 486,
        483, 6, 7,  474, 478, 479, 480, 487, 488, 489,
        483, 7, 2,  469, 473,
        484, 0, 1,  476,
        484, 1, 3,  475, 477, 483,
        484, 2, 6,  474, 478, 481, 482, 485, 486,
        484, 3, 6,  469, 479, 480, 487, 488, 489,
        484, 4, 3,  476, 483, 484,
        484, 5, 6,  475, 477, 481, 482, 485, 486,
        484, 6, 7,  474, 478, 479, 480, 487, 488, 489,
        484, 7, 2,  469, 473,
        485, 0, 1,  477,
        485, 1, 3,  476, 478, 486,
        485, 2, 6,  475, 483, 484, 487, 488, 489,
        485, 3, 3,  474, 481, 482,
        485, 4, 3,  477, 485, 486,
        485, 5, 7,  476, 478, 483, 484, 487, 488, 489,
        485, 6, 3,  475, 481, 482,
        485, 7, 3,  474, 479, 480,
        486, 0, 1,  477,
        486, 1, 3,  476, 478, 485,
        486, 2, 6,  475, 483, 484, 487, 488, 489,
        486, 3, 3,  474, 481, 482,
        486, 4, 3,  477, 485, 486,
        486, 5, 7,  476, 478, 483, 484, 487, 488, 489,
        486, 6, 3,  475, 481, 482,
        486, 7, 3,  474, 479, 480,
        487, 0, 1,  478,
        487, 1, 3,  477, 488, 489,
        487, 2, 3,  476, 485, 486,
        487, 3, 3,  475, 483, 484,
        487, 4, 4,  478, 487, 488, 489,
        487, 5, 3,  477, 485, 486,
        487, 6, 3,  476, 483, 484,
        487, 7, 3,  475, 481, 482,
        488, 0, 1,  478,
        488, 1, 3,  477, 487, 489,
        488, 2, 3,  476, 485, 486,
        488, 3, 3,  475, 483, 484,
        488, 4, 4,  478, 487, 488, 489,
        488, 5, 3,  477, 485, 486,
        488, 6, 3,  476, 483, 484,
        488, 7, 3,  475, 481, 482,
        489, 0, 1,  478,
        489, 1, 3,  477, 487, 488,
        489, 2, 3,  476, 485, 486,
        489, 3, 3,  475, 483, 484,
        489, 4, 4,  478, 487, 488, 489,
        489, 5, 3,  477, 485, 486,
        489, 6, 3,  476, 483, 484,
        489, 7, 3,  475, 481, 482,
        490, 0, 3,  470, 491, 494,
        490, 1, 5,  469, 471, 492, 495, 496,
        490, 2, 8,  468, 473, 474, 493, 497, 501, 502, 512,
        490, 3, 10,  451, 472, 475, 479, 480, 498, 503, 504, 513, 516,
        490, 4, 4,  470, 471, 490, 494,
        490, 5, 4,  469, 473, 491, 495,
        490, 6, 14,  451, 452, 468, 472, 474, 479, 480, 492, 493, 496, 501, 502, 512, 516,
        490, 7, 10,  450, 454, 475, 481, 482, 497, 503, 504, 513, 517,
        491, 0, 4,  490, 492, 495, 496,
        491, 1, 7,  470, 493, 494, 497, 501, 502, 512,
        491, 2, 7,  469, 471, 498, 503, 504, 513, 516,
        491, 3, 9,  468, 473, 474, 499, 505, 506, 514, 517, 518,
        491, 4, 2,  491, 495,
        491, 5, 11,  470, 471, 490, 492, 493, 494, 496, 501, 502, 512, 516,
        491, 6, 7,  469, 473, 497, 503, 504, 513, 517,
        491, 7, 17,  451, 452, 468, 472, 474, 479, 480, 498, 505, 506, 514, 515, 518, 523, 524, 527, 531,
        492, 0, 3,  491, 493, 512,
        492, 1, 5,  490, 495, 496, 513, 516,
        492, 2, 8,  470, 494, 497, 501, 502, 514, 517, 518,
        492, 3, 10,  469, 471, 498, 503, 504, 515, 519, 523, 524, 527,
        492, 4, 4,  492, 493, 512, 516,
        492, 5, 4,  491, 495, 513, 517,
        492, 6, 14,  470, 471, 490, 494, 496, 501, 502, 514, 515, 518, 523, 524, 527, 531,
        492, 7, 10,  469, 473, 497, 503, 504, 519, 525, 526, 528, 532,
        493, 0, 1,  492,
        493, 1, 2,  491, 512,
        493, 2, 5,  490, 495, 496, 513, 516,
        493, 3, 8,  470, 494, 497, 501, 502, 514, 517, 518,
        493, 4, 4,  492, 493, 512, 516,
        493, 5, 4,  491, 495, 513, 517,
        493, 6, 14,  470, 471, 490, 494, 496, 501, 502, 514, 515, 518, 523, 524, 527, 531,
        493, 7, 10,  469, 473, 497, 503, 504, 519, 525, 526, 528, 532,
        494, 0, 1,  490,
        494, 1, 2,  470, 491,
        494, 2, 5,  469, 471, 492, 495, 496,
        494, 3, 8,  468, 473, 474, 493, 497, 501, 502, 512,
        494, 4, 4,  470, 471, 490, 494,
        494, 5, 4,  469, 473, 491, 495,
        494, 6, 14,  451, 452, 468, 472, 474, 479, 480, 492, 493, 496, 501, 502, 512, 516,
        494, 7, 10,  450, 454, 475, 481, 482, 497, 503, 504, 513, 517,
        495, 0, 1,  491,
        495, 1, 3,  490, 492, 496,
        495, 2, 7,  470, 493, 494, 497, 501, 502, 512,
        495, 3, 7,  469, 471, 498, 503, 504, 513, 516,
        495, 4, 2,  491, 495,
        495, 5, 11,  470, 471, 490, 492, 493, 494, 496, 501, 502, 512, 516,
        495, 6, 7,  469, 473, 497, 503, 504, 513, 517,
        495, 7, 17,  451, 452, 468, 472, 474, 479, 480, 498, 505, 506, 514, 515, 518, 523, 524, 527, 531,
        496, 0, 4,  491, 497, 501, 502,
        496, 1, 6,  490, 492, 495, 498, 503, 504,
        496, 2, 7,  470, 493, 494, 499, 505, 506, 512,
        496, 3, 7,  469, 471, 500, 507, 508, 513, 516,
        496, 4, 3,  496, 501, 502,
        496, 5, 5,  491, 495, 497, 503, 504,
        496, 6, 11,  470, 471, 490, 492, 493, 494, 498, 505, 506, 512, 516,
        496, 7, 7,  469, 473, 499, 507, 508, 513, 517,
        497, 0, 4,  496, 498, 503, 504,
        497, 1, 6,  491, 499, 501, 502, 505, 506,
        497, 2, 6,  490, 492, 495, 500, 507, 508,
        497, 3, 7,  470, 493, 494, 509, 510, 511, 512,
        497, 4, 3,  497, 503, 504,
        497, 5, 6,  496, 498, 501, 502, 505, 506,
        497, 6, 5,  491, 495, 499, 507, 508,
        497, 7, 12,  470, 471, 490, 492, 493, 494, 500, 509, 510, 511, 512, 516,
        498, 0, 4,  497, 499, 505, 506,
        498, 1, 6,  496, 500, 503, 504, 507, 508,
        498, 2, 6,  491, 501, 502, 509, 510, 511,
        498, 3, 3,  490, 492, 495,
        498, 4, 3,  498, 505, 506,
        498, 5, 6,  497, 499, 503, 504, 507, 508,
        498, 6, 7,  496, 500, 501, 502, 509, 510, 511,
        498, 7, 2,  491, 495,
        499, 0, 4,  498, 500, 507, 508,
        499, 1, 6,  497, 505, 506, 509, 510, 511,
        499, 2, 3,  496, 503, 504,
        499, 3, 3,  491, 501, 502,
        499, 4, 3,  499, 507, 508,
        499, 5, 7,  498, 500, 505, 506, 509, 510, 511,
        499, 6, 3,  497, 503, 504,
        499, 7, 3,  496, 501, 502,
        500, 0, 4,  499, 509, 510, 511,
        500, 1, 3,  498, 507, 508,
        500, 2, 3,  497, 505, 506,
        500, 3, 3,  496, 503, 504,
        500, 4, 4,  500, 509, 510, 511,
        500, 5, 3,  499, 507, 508,
        500, 6, 3,  498, 505, 506,
        500, 7, 3,  497, 503, 504,
        501, 0, 1,  496,
        501, 1, 3,  491, 497, 502,
        501, 2, 6,  490, 492, 495, 498, 503, 504,
        501, 3, 7,  470, 493, 494, 499, 505, 506, 512,
        501, 4, 3,  496, 501, 502,
        501, 5, 5,  491, 495, 497, 503, 504,
        501, 6, 11,  470, 471, 490, 492, 493, 494, 498, 505, 506, 512, 516,
        501, 7, 7,  469, 473, 499, 507, 508, 513, 517,
        502, 0, 1,  496,
        502, 1, 3,  491, 497, 501,
        502, 2, 6,  490, 492, 495, 498, 503, 504,
        502, 3, 7,  470, 493, 494, 499, 505, 506, 512,
        502, 4, 3,  496, 501, 502,
        502, 5, 5,  491, 495, 497, 503, 504,
        502, 6, 11,  470, 471, 490, 492, 493, 494, 498, 505, 506, 512, 516,
        502, 7, 7,  469, 473, 499, 507, 508, 513, 517,
        503, 0, 1,  497,
        503, 1, 3,  496, 498, 504,
        503, 2, 6,  491, 499, 501, 502, 505, 506,
        503, 3, 6,  490, 492, 495, 500, 507, 508,
        503, 4, 3,  497, 503, 504,
        503, 5, 6,  496, 498, 501, 502, 505, 506,
        503, 6, 5,  491, 495, 499, 507, 508,
        503, 7, 12,  470, 471, 490, 492, 493, 494, 500, 509, 510, 511, 512, 516,
        504, 0, 1,  497,
        504, 1, 3,  496, 498, 503,
        504, 2, 6,  491, 499, 501, 502, 505, 506,
        504, 3, 6,  490, 492, 495, 500, 507, 508,
        504, 4, 3,  497, 503, 504,
        504, 5, 6,  496, 498, 501, 502, 505, 506,
        504, 6, 5,  491, 495, 499, 507, 508,
        504, 7, 12,  470, 471, 490, 492, 493, 494, 500, 509, 510, 511, 512, 516,
        505, 0, 1,  498,
        505, 1, 3,  497, 499, 506,
        505, 2, 6,  496, 500, 503, 504, 507, 508,
        505, 3, 6,  491, 501, 502, 509, 510, 511,
        505, 4, 3,  498, 505, 506,
        505, 5, 6,  497, 499, 503, 504, 507, 508,
        505, 6, 7,  496, 500, 501, 502, 509, 510, 511,
        505, 7, 2,  491, 495,
        506, 0, 1,  498,
        506, 1, 3,  497, 499, 505,
        506, 2, 6,  496, 500, 503, 504, 507, 508,
        506, 3, 6,  491, 501, 502, 509, 510, 511,
        506, 4, 3,  498, 505, 506,
        506, 5, 6,  497, 499, 503, 504, 507, 508,
        506, 6, 7,  496, 500, 501, 502, 509, 510, 511,
        506, 7, 2,  491, 495,
        507, 0, 1,  499,
        507, 1, 3,  498, 500, 508,
        507, 2, 6,  497, 505, 506, 509, 510, 511,
        507, 3, 3,  496, 503, 504,
        507, 4, 3,  499, 507, 508,
        507, 5, 7,  498, 500, 505, 506, 509, 510, 511,
        507, 6, 3,  497, 503, 504,
        507, 7, 3,  496, 501, 502,
        508, 0, 1,  499,
        508, 1, 3,  498, 500, 507,
        508, 2, 6,  497, 505, 506, 509, 510, 511,
        508, 3, 3,  496, 503, 504,
        508, 4, 3,  499, 507, 508,
        508, 5, 7,  498, 500, 505, 506, 509, 510, 511,
        508, 6, 3,  497, 503, 504,
        508, 7, 3,  496, 501, 502,
        509, 0, 1,  500,
        509, 1, 3,  499, 510, 511,
        509, 2, 3,  498, 507, 508,
        509, 3, 3,  497, 505, 506,
        509, 4, 4,  500, 509, 510, 511,
        509, 5, 3,  499, 507, 508,
        509, 6, 3,  498, 505, 506,
        509, 7, 3,  497, 503, 504,
        510, 0, 1,  500,
        510, 1, 3,  499, 509, 511,
        510, 2, 3,  498, 507, 508,
        510, 3, 3,  497, 505, 506,
        510, 4, 4,  500, 509, 510, 511,
        510, 5, 3,  499, 507, 508,
        510, 6, 3,  498, 505, 506,
        510, 7, 3,  497, 503, 504,
        511, 0, 1,  500,
        511, 1, 3,  499, 509, 510,
        511, 2, 3,  498, 507, 508,
        511, 3, 3,  497, 505, 506,
        511, 4, 4,  500, 509, 510, 511,
        511, 5, 3,  499, 507, 508,
        511, 6, 3,  498, 505, 506,
        511, 7, 3,  497, 503, 504,
        512, 0, 3,  492, 513, 516,
        512, 1, 5,  491, 493, 514, 517, 518,
        512, 2, 8,  490, 495, 496, 515, 519, 523, 524, 527,
        512, 3, 10,  470, 494, 497, 501, 502, 520, 525, 526, 528, 531,
        512, 4, 4,  492, 493, 512, 516,
        512, 5, 4,  491, 495, 513, 517,
        512, 6, 14,  470, 471, 490, 494, 496, 501, 502, 514, 515, 518, 523, 524, 527, 531,
        512, 7, 10,  469, 473, 497, 503, 504, 519, 525, 526, 528, 532,
        513, 0, 4,  512, 514, 517, 518,
        513, 1, 7,  492, 515, 516, 519, 523, 524, 527,
        513, 2, 7,  491, 493, 520, 525, 526, 528, 531,
        513, 3, 8,  490, 495, 496, 521, 522, 529, 532, 533,
        513, 4, 2,  513, 517,
        513, 5, 11,  492, 493, 512, 514, 515, 516, 518, 523, 524, 527, 531,
        513, 6, 7,  491, 495, 519, 525, 526, 528, 532,
        513, 7, 17,  470, 471, 490, 494, 496, 501, 502, 520, 521, 522, 529, 530, 533, 538, 539, 549, 553,
        514, 0, 3,  513, 515, 527,
        514, 1, 5,  512, 517, 518, 528, 531,
        514, 2, 8,  492, 516, 519, 523, 524, 529, 532, 533,
        514, 3, 10,  491, 493, 520, 525, 526, 530, 534, 538, 539, 549,
        514, 4, 4,  514, 515, 527, 531,
        514, 5, 4,  513, 517, 528, 532,
        514, 6, 14,  492, 493, 512, 516, 518, 523, 524, 529, 530, 533, 538, 539, 549, 553,
        514, 7, 11,  491, 495, 519, 525, 526, 534, 540, 541, 550, 554, 555,
        515, 0, 1,  514,
        515, 1, 2,  513, 527,
        515, 2, 5,  512, 517, 518, 528, 531,
        515, 3, 8,  492, 516, 519, 523, 524, 529, 532, 533,
        515, 4, 4,  514, 515, 527, 531,
        515, 5, 4,  513, 517, 528, 532,
        515, 6, 14,  492, 493, 512, 516, 518, 523, 524, 529, 530, 533, 538, 539, 549, 553,
        515, 7, 11,  491, 495, 519, 525, 526, 534, 540, 541, 550, 554, 555,
        516, 0, 1,  512,
        516, 1, 2,  492, 513,
        516, 2, 5,  491, 493, 514, 517, 518,
        516, 3, 8,  490, 495, 496, 515, 519, 523, 524, 527,
        516, 4, 4,  492, 493, 512, 516,
        516, 5, 4,  491, 495, 513, 517,
        516, 6, 14,  470, 471, 490, 494, 496, 501, 502, 514, 515, 518, 523, 524, 527, 531,
        516, 7, 10,  469, 473, 497, 503, 504, 519, 525, 526, 528, 532,
        517, 0, 1,  513,
        517, 1, 3,  512, 514, 518,
        517, 2, 7,  492, 515, 516, 519, 523, 524, 527,
        517, 3, 7,  491, 493, 520, 525, 526, 528, 531,
        517, 4, 2,  513, 517,
        517, 5, 11,  492, 493, 512, 514, 515, 516, 518, 523, 524, 527, 531,
        517, 6, 7,  491, 495, 519, 525, 526, 528, 532,
        517, 7, 17,  470, 471, 490, 494, 496, 501, 502, 520, 521, 522, 529, 530, 533, 538, 539, 549, 553,
        518, 0, 4,  513, 519, 523, 524,
        518, 1, 6,  512, 514, 517, 520, 525, 526,
        518, 2, 6,  492, 515, 516, 521, 522, 527,
        518, 3, 4,  491, 493, 528, 531,
        518, 4, 3,  518, 523, 524,
        518, 5, 5,  513, 517, 519, 525, 526,
        518, 6, 11,  492, 493, 512, 514, 515, 516, 520, 521, 522, 527, 531,
        518, 7, 4,  491, 495, 528, 532,
        519, 0, 4,  518, 520, 525, 526,
        519, 1, 5,  513, 521, 522, 523, 524,
        519, 2, 3,  512, 514, 517,
        519, 3, 4,  492, 515, 516, 527,
        519, 4, 3,  519, 525, 526,
        519, 5, 6,  518, 520, 521, 522, 523, 524,
        519, 6, 2,  513, 517,
        519, 7, 8,  492, 493, 512, 514, 515, 516, 527, 531,
        520, 0, 3,  519, 521, 522,
        520, 1, 3,  518, 525, 526,
        520, 2, 3,  513, 523, 524,
        520, 3, 3,  512, 514, 517,
        520, 4, 3,  520, 521, 522,
        520, 5, 3,  519, 525, 526,
        520, 6, 3,  518, 523, 524,
        520, 7, 2,  513, 517,
        521, 0, 1,  520,
        521, 1, 2,  519, 522,
        521, 2, 3,  518, 525, 526,
        521, 3, 3,  513, 523, 524,
        521, 4, 3,  520, 521, 522,
        521, 5, 3,  519, 525, 526,
        521, 6, 3,  518, 523, 524,
        521, 7, 2,  513, 517,
        522, 0, 1,  520,
        522, 1, 2,  519, 521,
        522, 2, 3,  518, 525, 526,
        522, 3, 3,  513, 523, 524,
        522, 4, 3,  520, 521, 522,
        522, 5, 3,  519, 525, 526,
        522, 6, 3,  518, 523, 524,
        522, 7, 2,  513, 517,
        523, 0, 1,  518,
        523, 1, 3,  513, 519, 524,
        523, 2, 6,  512, 514, 517, 520, 525, 526,
        523, 3, 6,  492, 515, 516, 521, 522, 527,
        523, 4, 3,  518, 523, 524,
        523, 5, 5,  513, 517, 519, 525, 526,
        523, 6, 11,  492, 493, 512, 514, 515, 516, 520, 521, 522, 527, 531,
        523, 7, 4,  491, 495, 528, 532,
        524, 0, 1,  518,
        524, 1, 3,  513, 519, 523,
        524, 2, 6,  512, 514, 517, 520, 525, 526,
        524, 3, 6,  492, 515, 516, 521, 522, 527,
        524, 4, 3,  518, 523, 524,
        524, 5, 5,  513, 517, 519, 525, 526,
        524, 6, 11,  492, 493, 512, 514, 515, 516, 520, 521, 522, 527, 531,
        524, 7, 4,  491, 495, 528, 532,
        525, 0, 1,  519,
        525, 1, 3,  518, 520, 526,
        525, 2, 5,  513, 521, 522, 523, 524,
        525, 3, 3,  512, 514, 517,
        525, 4, 3,  519, 525, 526,
        525, 5, 6,  518, 520, 521, 522, 523, 524,
        525, 6, 2,  513, 517,
        525, 7, 8,  492, 493, 512, 514, 515, 516, 527, 531,
        526, 0, 1,  519,
        526, 1, 3,  518, 520, 525,
        526, 2, 5,  513, 521, 522, 523, 524,
        526, 3, 3,  512, 514, 517,
        526, 4, 3,  519, 525, 526,
        526, 5, 6,  518, 520, 521, 522, 523, 524,
        526, 6, 2,  513, 517,
        526, 7, 8,  492, 493, 512, 514, 515, 516, 527, 531,
        527, 0, 3,  514, 528, 531,
        527, 1, 5,  513, 515, 529, 532, 533,
        527, 2, 8,  512, 517, 518, 530, 534, 538, 539, 549,
        527, 3, 10,  492, 516, 519, 523, 524, 535, 540, 541, 550, 553,
        527, 4, 4,  514, 515, 527, 531,
        527, 5, 4,  513, 517, 528, 532,
        527, 6, 14,  492, 493, 512, 516, 518, 523, 524, 529, 530, 533, 538, 539, 549, 553,
        527, 7, 11,  491, 495, 519, 525, 526, 534, 540, 541, 550, 554, 555,
        528, 0, 4,  527, 529, 532, 533,
        528, 1, 7,  514, 530, 531, 534, 538, 539, 549,
        528, 2, 7,  513, 515, 535, 540, 541, 550, 553,
        528, 3, 9,  512, 517, 518, 536, 542, 543, 551, 554, 555,
        528, 4, 2,  528, 532,
        528, 5, 11,  514, 515, 527, 529, 530, 531, 533, 538, 539, 549, 553,
        528, 6, 8,  513, 517, 534, 540, 541, 550, 554, 555,
        528, 7, 14,  492, 493, 512, 516, 518, 523, 524, 535, 542, 543, 551, 552, 556, 560,
        529, 0, 3,  528, 530, 549,
        529, 1, 5,  527, 532, 533, 550, 553,
        529, 2, 8,  514, 531, 534, 538, 539, 551, 554, 555,
        529, 3, 7,  513, 515, 535, 540, 541, 552, 556,
        529, 4, 4,  529, 530, 549, 553,
        529, 5, 5,  528, 532, 550, 554, 555,
        529, 6, 11,  514, 515, 527, 531, 533, 538, 539, 551, 552, 556, 560,
        529, 7, 7,  513, 517, 534, 540, 541, 557, 561,
        530, 0, 1,  529,
        530, 1, 2,  528, 549,
        530, 2, 5,  527, 532, 533, 550, 553,
        530, 3, 8,  514, 531, 534, 538, 539, 551, 554, 555,
        530, 4, 4,  529, 530, 549, 553,
        530, 5, 5,  528, 532, 550, 554, 555,
        530, 6, 11,  514, 515, 527, 531, 533, 538, 539, 551, 552, 556, 560,
        530, 7, 7,  513, 517, 534, 540, 541, 557, 561,
        531, 0, 1,  527,
        531, 1, 2,  514, 528,
        531, 2, 5,  513, 515, 529, 532, 533,
        531, 3, 8,  512, 517, 518, 530, 534, 538, 539, 549,
        531, 4, 4,  514, 515, 527, 531,
        531, 5, 4,  513, 517, 528, 532,
        531, 6, 14,  492, 493, 512, 516, 518, 523, 524, 529, 530, 533, 538, 539, 549, 553,
        531, 7, 11,  491, 495, 519, 525, 526, 534, 540, 541, 550, 554, 555,
        532, 0, 1,  528,
        532, 1, 3,  527, 529, 533,
        532, 2, 7,  514, 530, 531, 534, 538, 539, 549,
        532, 3, 7,  513, 515, 535, 540, 541, 550, 553,
        532, 4, 2,  528, 532,
        532, 5, 11,  514, 515, 527, 529, 530, 531, 533, 538, 539, 549, 553,
        532, 6, 8,  513, 517, 534, 540, 541, 550, 554, 555,
        532, 7, 14,  492, 493, 512, 516, 518, 523, 524, 535, 542, 543, 551, 552, 556, 560,
        533, 0, 4,  528, 534, 538, 539,
        533, 1, 6,  527, 529, 532, 535, 540, 541,
        533, 2, 7,  514, 530, 531, 536, 542, 543, 549,
        533, 3, 7,  513, 515, 537, 544, 545, 550, 553,
        533, 4, 3,  533, 538, 539,
        533, 5, 5,  528, 532, 534, 540, 541,
        533, 6, 11,  514, 515, 527, 529, 530, 531, 535, 542, 543, 549, 553,
        533, 7, 8,  513, 517, 536, 544, 545, 550, 554, 555,
        534, 0, 4,  533, 535, 540, 541,
        534, 1, 6,  528, 536, 538, 539, 542, 543,
        534, 2, 6,  527, 529, 532, 537, 544, 545,
        534, 3, 7,  514, 530, 531, 546, 547, 548, 549,
        534, 4, 3,  534, 540, 541,
        534, 5, 6,  533, 535, 538, 539, 542, 543,
        534, 6, 5,  528, 532, 536, 544, 545,
        534, 7, 12,  514, 515, 527, 529, 530, 531, 537, 546, 547, 548, 549, 553,
        535, 0, 4,  534, 536, 542, 543,
        535, 1, 6,  533, 537, 540, 541, 544, 545,
        535, 2, 6,  528, 538, 539, 546, 547, 548,
        535, 3, 3,  527, 529, 532,
        535, 4, 3,  535, 542, 543,
        535, 5, 6,  534, 536, 540, 541, 544, 545,
        535, 6, 7,  533, 537, 538, 539, 546, 547, 548,
        535, 7, 2,  528, 532,
        536, 0, 4,  535, 537, 544, 545,
        536, 1, 6,  534, 542, 543, 546, 547, 548,
        536, 2, 3,  533, 540, 541,
        536, 3, 3,  528, 538, 539,
        536, 4, 3,  536, 544, 545,
        536, 5, 7,  535, 537, 542, 543, 546, 547, 548,
        536, 6, 3,  534, 540, 541,
        536, 7, 3,  533, 538, 539,
        537, 0, 4,  536, 546, 547, 548,
        537, 1, 3,  535, 544, 545,
        537, 2, 3,  534, 542, 543,
        537, 3, 3,  533, 540, 541,
        537, 4, 4,  537, 546, 547, 548,
        537, 5, 3,  536, 544, 545,
        537, 6, 3,  535, 542, 543,
        537, 7, 3,  534, 540, 541,
        538, 0, 1,  533,
        538, 1, 3,  528, 534, 539,
        538, 2, 6,  527, 529, 532, 535, 540, 541,
        538, 3, 7,  514, 530, 531, 536, 542, 543, 549,
        538, 4, 3,  533, 538, 539,
        538, 5, 5,  528, 532, 534, 540, 541,
        538, 6, 11,  514, 515, 527, 529, 530, 531, 535, 542, 543, 549, 553,
        538, 7, 8,  513, 517, 536, 544, 545, 550, 554, 555,
        539, 0, 1,  533,
        539, 1, 3,  528, 534, 538,
        539, 2, 6,  527, 529, 532, 535, 540, 541,
        539, 3, 7,  514, 530, 531, 536, 542, 543, 549,
        539, 4, 3,  533, 538, 539,
        539, 5, 5,  528, 532, 534, 540, 541,
        539, 6, 11,  514, 515, 527, 529, 530, 531, 535, 542, 543, 549, 553,
        539, 7, 8,  513, 517, 536, 544, 545, 550, 554, 555,
        540, 0, 1,  534,
        540, 1, 3,  533, 535, 541,
        540, 2, 6,  528, 536, 538, 539, 542, 543,
        540, 3, 6,  527, 529, 532, 537, 544, 545,
        540, 4, 3,  534, 540, 541,
        540, 5, 6,  533, 535, 538, 539, 542, 543,
        540, 6, 5,  528, 532, 536, 544, 545,
        540, 7, 12,  514, 515, 527, 529, 530, 531, 537, 546, 547, 548, 549, 553,
        541, 0, 1,  534,
        541, 1, 3,  533, 535, 540,
        541, 2, 6,  528, 536, 538, 539, 542, 543,
        541, 3, 6,  527, 529, 532, 537, 544, 545,
        541, 4, 3,  534, 540, 541,
        541, 5, 6,  533, 535, 538, 539, 542, 543,
        541, 6, 5,  528, 532, 536, 544, 545,
        541, 7, 12,  514, 515, 527, 529, 530, 531, 537, 546, 547, 548, 549, 553,
        542, 0, 1,  535,
        542, 1, 3,  534, 536, 543,
        542, 2, 6,  533, 537, 540, 541, 544, 545,
        542, 3, 6,  528, 538, 539, 546, 547, 548,
        542, 4, 3,  535, 542, 543,
        542, 5, 6,  534, 536, 540, 541, 544, 545,
        542, 6, 7,  533, 537, 538, 539, 546, 547, 548,
        542, 7, 2,  528, 532,
        543, 0, 1,  535,
        543, 1, 3,  534, 536, 542,
        543, 2, 6,  533, 537, 540, 541, 544, 545,
        543, 3, 6,  528, 538, 539, 546, 547, 548,
        543, 4, 3,  535, 542, 543,
        543, 5, 6,  534, 536, 540, 541, 544, 545,
        543, 6, 7,  533, 537, 538, 539, 546, 547, 548,
        543, 7, 2,  528, 532,
        544, 0, 1,  536,
        544, 1, 3,  535, 537, 545,
        544, 2, 6,  534, 542, 543, 546, 547, 548,
        544, 3, 3,  533, 540, 541,
        544, 4, 3,  536, 544, 545,
        544, 5, 7,  535, 537, 542, 543, 546, 547, 548,
        544, 6, 3,  534, 540, 541,
        544, 7, 3,  533, 538, 539,
        545, 0, 1,  536,
        545, 1, 3,  535, 537, 544,
        545, 2, 6,  534, 542, 543, 546, 547, 548,
        545, 3, 3,  533, 540, 541,
        545, 4, 3,  536, 544, 545,
        545, 5, 7,  535, 537, 542, 543, 546, 547, 548,
        545, 6, 3,  534, 540, 541,
        545, 7, 3,  533, 538, 539,
        546, 0, 1,  537,
        546, 1, 3,  536, 547, 548,
        546, 2, 3,  535, 544, 545,
        546, 3, 3,  534, 542, 543,
        546, 4, 4,  537, 546, 547, 548,
        546, 5, 3,  536, 544, 545,
        546, 6, 3,  535, 542, 543,
        546, 7, 3,  534, 540, 541,
        547, 0, 1,  537,
        547, 1, 3,  536, 546, 548,
        547, 2, 3,  535, 544, 545,
        547, 3, 3,  534, 542, 543,
        547, 4, 4,  537, 546, 547, 548,
        547, 5, 3,  536, 544, 545,
        547, 6, 3,  535, 542, 543,
        547, 7, 3,  534, 540, 541,
        548, 0, 1,  537,
        548, 1, 3,  536, 546, 547,
        548, 2, 3,  535, 544, 545,
        548, 3, 3,  534, 542, 543,
        548, 4, 4,  537, 546, 547, 548,
        548, 5, 3,  536, 544, 545,
        548, 6, 3,  535, 542, 543,
        548, 7, 3,  534, 540, 541,
        549, 0, 3,  529, 550, 553,
        549, 1, 5,  528, 530, 551, 554, 555,
        549, 2, 5,  527, 532, 533, 552, 556,
        549, 3, 7,  514, 531, 534, 538, 539, 557, 560,
        549, 4, 4,  529, 530, 549, 553,
        549, 5, 5,  528, 532, 550, 554, 555,
        549, 6, 11,  514, 515, 527, 531, 533, 538, 539, 551, 552, 556, 560,
        549, 7, 7,  513, 517, 534, 540, 541, 557, 561,
        550, 0, 4,  549, 551, 554, 555,
        550, 1, 4,  529, 552, 553, 556,
        550, 2, 4,  528, 530, 557, 560,
        550, 3, 6,  527, 532, 533, 558, 561, 562,
        550, 4, 3,  550, 554, 555,
        550, 5, 8,  529, 530, 549, 551, 552, 553, 556, 560,
        550, 6, 4,  528, 532, 557, 561,
        550, 7, 14,  514, 515, 527, 531, 533, 538, 539, 558, 559, 562, 566, 567, 575, 579,
        551, 0, 3,  550, 552, 556,
        551, 1, 5,  549, 554, 555, 557, 560,
        551, 2, 5,  529, 553, 558, 561, 562,
        551, 3, 7,  528, 530, 559, 563, 566, 567, 575,
        551, 4, 4,  551, 552, 556, 560,
        551, 5, 5,  550, 554, 555, 557, 561,
        551, 6, 11,  529, 530, 549, 553, 558, 559, 562, 566, 567, 575, 579,
        551, 7, 6,  528, 532, 563, 568, 576, 580,
        552, 0, 1,  551,
        552, 1, 2,  550, 556,
        552, 2, 5,  549, 554, 555, 557, 560,
        552, 3, 5,  529, 553, 558, 561, 562,
        552, 4, 4,  551, 552, 556, 560,
        552, 5, 5,  550, 554, 555, 557, 561,
        552, 6, 11,  529, 530, 549, 553, 558, 559, 562, 566, 567, 575, 579,
        552, 7, 6,  528, 532, 563, 568, 576, 580,
        553, 0, 1,  549,
        553, 1, 2,  529, 550,
        553, 2, 5,  528, 530, 551, 554, 555,
        553, 3, 5,  527, 532, 533, 552, 556,
        553, 4, 4,  529, 530, 549, 553,
        553, 5, 5,  528, 532, 550, 554, 555,
        553, 6, 11,  514, 515, 527, 531, 533, 538, 539, 551, 552, 556, 560,
        553, 7, 7,  513, 517, 534, 540, 541, 557, 561,
        554, 0, 1,  550,
        554, 1, 3,  549, 551, 555,
        554, 2, 4,  529, 552, 553, 556,
        554, 3, 4,  528, 530, 557, 560,
        554, 4, 3,  550, 554, 555,
        554, 5, 8,  529, 530, 549, 551, 552, 553, 556, 560,
        554, 6, 4,  528, 532, 557, 561,
        554, 7, 14,  514, 515, 527, 531, 533, 538, 539, 558, 559, 562, 566, 567, 575, 579,
        555, 0, 1,  550,
        555, 1, 3,  549, 551, 554,
        555, 2, 4,  529, 552, 553, 556,
        555, 3, 4,  528, 530, 557, 560,
        555, 4, 3,  550, 554, 555,
        555, 5, 8,  529, 530, 549, 551, 552, 553, 556, 560,
        555, 6, 4,  528, 532, 557, 561,
        555, 7, 14,  514, 515, 527, 531, 533, 538, 539, 558, 559, 562, 566, 567, 575, 579,
        556, 0, 3,  551, 557, 560,
        556, 1, 5,  550, 552, 558, 561, 562,
        556, 2, 8,  549, 554, 555, 559, 563, 566, 567, 575,
        556, 3, 7,  529, 553, 564, 565, 568, 576, 579,
        556, 4, 4,  551, 552, 556, 560,
        556, 5, 5,  550, 554, 555, 557, 561,
        556, 6, 11,  529, 530, 549, 553, 558, 559, 562, 566, 567, 575, 579,
        556, 7, 6,  528, 532, 563, 568, 576, 580,
        557, 0, 4,  556, 558, 561, 562,
        557, 1, 7,  551, 559, 560, 563, 566, 567, 575,
        557, 2, 7,  550, 552, 564, 565, 568, 576, 579,
        557, 3, 12,  549, 554, 555, 569, 570, 571, 572, 573, 574, 577, 580, 581,
        557, 4, 2,  557, 561,
        557, 5, 11,  551, 552, 556, 558, 559, 560, 562, 566, 567, 575, 579,
        557, 6, 7,  550, 554, 555, 563, 568, 576, 580,
        557, 7, 18,  529, 530, 549, 553, 564, 565, 569, 570, 571, 572, 573, 574, 577, 578, 581, 588, 589, 595,
        558, 0, 3,  557, 559, 575,
        558, 1, 5,  556, 561, 562, 576, 579,
        558, 2, 8,  551, 560, 563, 566, 567, 577, 580, 581,
        558, 3, 10,  550, 552, 564, 565, 568, 578, 582, 588, 589, 595,
        558, 4, 4,  558, 559, 575, 579,
        558, 5, 4,  557, 561, 576, 580,
        558, 6, 13,  551, 552, 556, 560, 562, 566, 567, 577, 578, 581, 588, 589, 595,
        558, 7, 16,  550, 554, 555, 563, 568, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        559, 0, 1,  558,
        559, 1, 2,  557, 575,
        559, 2, 5,  556, 561, 562, 576, 579,
        559, 3, 8,  551, 560, 563, 566, 567, 577, 580, 581,
        559, 4, 4,  558, 559, 575, 579,
        559, 5, 4,  557, 561, 576, 580,
        559, 6, 13,  551, 552, 556, 560, 562, 566, 567, 577, 578, 581, 588, 589, 595,
        559, 7, 16,  550, 554, 555, 563, 568, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        560, 0, 1,  556,
        560, 1, 2,  551, 557,
        560, 2, 5,  550, 552, 558, 561, 562,
        560, 3, 8,  549, 554, 555, 559, 563, 566, 567, 575,
        560, 4, 4,  551, 552, 556, 560,
        560, 5, 5,  550, 554, 555, 557, 561,
        560, 6, 11,  529, 530, 549, 553, 558, 559, 562, 566, 567, 575, 579,
        560, 7, 6,  528, 532, 563, 568, 576, 580,
        561, 0, 1,  557,
        561, 1, 3,  556, 558, 562,
        561, 2, 7,  551, 559, 560, 563, 566, 567, 575,
        561, 3, 7,  550, 552, 564, 565, 568, 576, 579,
        561, 4, 2,  557, 561,
        561, 5, 11,  551, 552, 556, 558, 559, 560, 562, 566, 567, 575, 579,
        561, 6, 7,  550, 554, 555, 563, 568, 576, 580,
        561, 7, 18,  529, 530, 549, 553, 564, 565, 569, 570, 571, 572, 573, 574, 577, 578, 581, 588, 589, 595,
        562, 0, 4,  557, 563, 566, 567,
        562, 1, 6,  556, 558, 561, 564, 565, 568,
        562, 2, 10,  551, 559, 560, 569, 570, 571, 572, 573, 574, 575,
        562, 3, 4,  550, 552, 576, 579,
        562, 4, 3,  562, 566, 567,
        562, 5, 4,  557, 561, 563, 568,
        562, 6, 16,  551, 552, 556, 558, 559, 560, 564, 565, 569, 570, 571, 572, 573, 574, 575, 579,
        562, 7, 5,  550, 554, 555, 576, 580,
        563, 0, 4,  562, 564, 565, 568,
        563, 1, 9,  557, 566, 567, 569, 570, 571, 572, 573, 574,
        563, 2, 3,  556, 558, 561,
        563, 3, 4,  551, 559, 560, 575,
        563, 4, 2,  563, 568,
        563, 5, 11,  562, 564, 565, 566, 567, 569, 570, 571, 572, 573, 574,
        563, 6, 2,  557, 561,
        563, 7, 8,  551, 552, 556, 558, 559, 560, 575, 579,
        564, 0, 4,  563, 569, 570, 571,
        564, 1, 3,  562, 565, 568,
        564, 2, 6,  557, 566, 567, 572, 573, 574,
        564, 3, 3,  556, 558, 561,
        564, 4, 4,  564, 569, 570, 571,
        564, 5, 2,  563, 568,
        564, 6, 7,  562, 565, 566, 567, 572, 573, 574,
        564, 7, 2,  557, 561,
        565, 0, 4,  563, 572, 573, 574,
        565, 1, 3,  562, 564, 568,
        565, 2, 6,  557, 566, 567, 569, 570, 571,
        565, 3, 3,  556, 558, 561,
        565, 4, 4,  565, 572, 573, 574,
        565, 5, 2,  563, 568,
        565, 6, 7,  562, 564, 566, 567, 569, 570, 571,
        565, 7, 2,  557, 561,
        566, 0, 1,  562,
        566, 1, 3,  557, 563, 567,
        566, 2, 6,  556, 558, 561, 564, 565, 568,
        566, 3, 10,  551, 559, 560, 569, 570, 571, 572, 573, 574, 575,
        566, 4, 3,  562, 566, 567,
        566, 5, 4,  557, 561, 563, 568,
        566, 6, 16,  551, 552, 556, 558, 559, 560, 564, 565, 569, 570, 571, 572, 573, 574, 575, 579,
        566, 7, 5,  550, 554, 555, 576, 580,
        567, 0, 1,  562,
        567, 1, 3,  557, 563, 566,
        567, 2, 6,  556, 558, 561, 564, 565, 568,
        567, 3, 10,  551, 559, 560, 569, 570, 571, 572, 573, 574, 575,
        567, 4, 3,  562, 566, 567,
        567, 5, 4,  557, 561, 563, 568,
        567, 6, 16,  551, 552, 556, 558, 559, 560, 564, 565, 569, 570, 571, 572, 573, 574, 575, 579,
        567, 7, 5,  550, 554, 555, 576, 580,
        568, 0, 1,  563,
        568, 1, 3,  562, 564, 565,
        568, 2, 9,  557, 566, 567, 569, 570, 571, 572, 573, 574,
        568, 3, 3,  556, 558, 561,
        568, 4, 2,  563, 568,
        568, 5, 11,  562, 564, 565, 566, 567, 569, 570, 571, 572, 573, 574,
        568, 6, 2,  557, 561,
        568, 7, 8,  551, 552, 556, 558, 559, 560, 575, 579,
        569, 0, 1,  564,
        569, 1, 3,  563, 570, 571,
        569, 2, 3,  562, 565, 568,
        569, 3, 6,  557, 566, 567, 572, 573, 574,
        569, 4, 4,  564, 569, 570, 571,
        569, 5, 2,  563, 568,
        569, 6, 7,  562, 565, 566, 567, 572, 573, 574,
        569, 7, 2,  557, 561,
        570, 0, 1,  564,
        570, 1, 3,  563, 569, 571,
        570, 2, 3,  562, 565, 568,
        570, 3, 6,  557, 566, 567, 572, 573, 574,
        570, 4, 4,  564, 569, 570, 571,
        570, 5, 2,  563, 568,
        570, 6, 7,  562, 565, 566, 567, 572, 573, 574,
        570, 7, 2,  557, 561,
        571, 0, 1,  564,
        571, 1, 3,  563, 569, 570,
        571, 2, 3,  562, 565, 568,
        571, 3, 6,  557, 566, 567, 572, 573, 574,
        571, 4, 4,  564, 569, 570, 571,
        571, 5, 2,  563, 568,
        571, 6, 7,  562, 565, 566, 567, 572, 573, 574,
        571, 7, 2,  557, 561,
        572, 0, 1,  565,
        572, 1, 3,  563, 573, 574,
        572, 2, 3,  562, 564, 568,
        572, 3, 6,  557, 566, 567, 569, 570, 571,
        572, 4, 4,  565, 572, 573, 574,
        572, 5, 2,  563, 568,
        572, 6, 7,  562, 564, 566, 567, 569, 570, 571,
        572, 7, 2,  557, 561,
        573, 0, 1,  565,
        573, 1, 3,  563, 572, 574,
        573, 2, 3,  562, 564, 568,
        573, 3, 6,  557, 566, 567, 569, 570, 571,
        573, 4, 4,  565, 572, 573, 574,
        573, 5, 2,  563, 568,
        573, 6, 7,  562, 564, 566, 567, 569, 570, 571,
        573, 7, 2,  557, 561,
        574, 0, 1,  565,
        574, 1, 3,  563, 572, 573,
        574, 2, 3,  562, 564, 568,
        574, 3, 6,  557, 566, 567, 569, 570, 571,
        574, 4, 4,  565, 572, 573, 574,
        574, 5, 2,  563, 568,
        574, 6, 7,  562, 564, 566, 567, 569, 570, 571,
        574, 7, 2,  557, 561,
        575, 0, 3,  558, 576, 579,
        575, 1, 5,  557, 559, 577, 580, 581,
        575, 2, 8,  556, 561, 562, 578, 582, 588, 589, 595,
        575, 3, 7,  551, 560, 563, 566, 567, 583, 584,
        575, 4, 4,  558, 559, 575, 579,
        575, 5, 4,  557, 561, 576, 580,
        575, 6, 13,  551, 552, 556, 560, 562, 566, 567, 577, 578, 581, 588, 589, 595,
        575, 7, 16,  550, 554, 555, 563, 568, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        576, 0, 4,  575, 577, 580, 581,
        576, 1, 7,  558, 578, 579, 582, 588, 589, 595,
        576, 2, 4,  557, 559, 583, 584,
        576, 3, 7,  556, 561, 562, 585, 586, 590, 591,
        576, 4, 2,  576, 580,
        576, 5, 10,  558, 559, 575, 577, 578, 579, 581, 588, 589, 595,
        576, 6, 13,  557, 561, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        576, 7, 7,  551, 552, 556, 560, 562, 566, 567,
        577, 0, 3,  576, 578, 595,
        577, 1, 3,  575, 580, 581,
        577, 2, 5,  558, 579, 582, 588, 589,
        577, 3, 4,  557, 559, 583, 584,
        577, 4, 3,  577, 578, 595,
        577, 5, 2,  576, 580,
        577, 6, 7,  558, 559, 575, 579, 581, 588, 589,
        577, 7, 13,  557, 561, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        578, 0, 1,  577,
        578, 1, 2,  576, 595,
        578, 2, 3,  575, 580, 581,
        578, 3, 5,  558, 579, 582, 588, 589,
        578, 4, 3,  577, 578, 595,
        578, 5, 2,  576, 580,
        578, 6, 7,  558, 559, 575, 579, 581, 588, 589,
        578, 7, 13,  557, 561, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        579, 0, 1,  575,
        579, 1, 2,  558, 576,
        579, 2, 5,  557, 559, 577, 580, 581,
        579, 3, 8,  556, 561, 562, 578, 582, 588, 589, 595,
        579, 4, 4,  558, 559, 575, 579,
        579, 5, 4,  557, 561, 576, 580,
        579, 6, 13,  551, 552, 556, 560, 562, 566, 567, 577, 578, 581, 588, 589, 595,
        579, 7, 16,  550, 554, 555, 563, 568, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        580, 0, 1,  576,
        580, 1, 3,  575, 577, 581,
        580, 2, 7,  558, 578, 579, 582, 588, 589, 595,
        580, 3, 4,  557, 559, 583, 584,
        580, 4, 2,  576, 580,
        580, 5, 10,  558, 559, 575, 577, 578, 579, 581, 588, 589, 595,
        580, 6, 13,  557, 561, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        580, 7, 7,  551, 552, 556, 560, 562, 566, 567,
        581, 0, 4,  576, 582, 588, 589,
        581, 1, 5,  575, 577, 580, 583, 584,
        581, 2, 8,  558, 578, 579, 585, 586, 590, 591, 595,
        581, 3, 6,  557, 559, 587, 587, 592, 593,
        581, 4, 3,  581, 588, 589,
        581, 5, 13,  576, 580, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        581, 6, 7,  558, 559, 575, 577, 578, 579, 595,
        581, 7, 2,  557, 561,
        582, 0, 3,  581, 583, 584,
        582, 1, 7,  576, 585, 586, 588, 589, 590, 591,
        582, 2, 7,  575, 577, 580, 587, 587, 592, 593,
        582, 3, 6,  558, 578, 579, 594, 594, 595,
        582, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        582, 5, 3,  581, 588, 589,
        582, 6, 2,  576, 580,
        582, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        583, 0, 3,  582, 585, 590,
        583, 1, 4,  581, 584, 587, 592,
        583, 2, 7,  576, 586, 586, 588, 589, 591, 594,
        583, 3, 5,  575, 577, 580, 593, 593,
        583, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        583, 5, 3,  581, 588, 589,
        583, 6, 2,  576, 580,
        583, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        584, 0, 3,  582, 586, 591,
        584, 1, 4,  581, 583, 587, 593,
        584, 2, 7,  576, 585, 585, 588, 589, 590, 594,
        584, 3, 5,  575, 577, 580, 592, 592,
        584, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        584, 5, 3,  581, 588, 589,
        584, 6, 2,  576, 580,
        584, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        585, 0, 3,  583, 587, 592,
        585, 1, 4,  582, 586, 590, 594,
        585, 2, 4,  581, 584, 584, 593,
        585, 3, 5,  576, 588, 589, 591, 591,
        585, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        585, 5, 3,  581, 588, 589,
        585, 6, 2,  576, 580,
        585, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        586, 0, 3,  584, 587, 593,
        586, 1, 4,  582, 585, 591, 594,
        586, 2, 4,  581, 583, 583, 592,
        586, 3, 5,  576, 588, 589, 590, 590,
        586, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        586, 5, 3,  581, 588, 589,
        586, 6, 2,  576, 580,
        586, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        587, 0, 3,  585, 586, 594,
        587, 1, 4,  583, 584, 592, 593,
        587, 2, 4,  582, 582, 590, 591,
        587, 3, 2,  581, 581,
        587, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        587, 5, 3,  581, 588, 589,
        587, 6, 2,  576, 580,
        587, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        588, 0, 1,  581,
        588, 1, 3,  576, 582, 589,
        588, 2, 5,  575, 577, 580, 583, 584,
        588, 3, 8,  558, 578, 579, 585, 586, 590, 591, 595,
        588, 4, 3,  581, 588, 589,
        588, 5, 13,  576, 580, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        588, 6, 7,  558, 559, 575, 577, 578, 579, 595,
        588, 7, 2,  557, 561,
        589, 0, 1,  581,
        589, 1, 3,  576, 582, 588,
        589, 2, 5,  575, 577, 580, 583, 584,
        589, 3, 8,  558, 578, 579, 585, 586, 590, 591, 595,
        589, 4, 3,  581, 588, 589,
        589, 5, 13,  576, 580, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        589, 6, 7,  558, 559, 575, 577, 578, 579, 595,
        589, 7, 2,  557, 561,
        590, 0, 1,  583,
        590, 1, 2,  582, 585,
        590, 2, 4,  581, 584, 587, 592,
        590, 3, 7,  576, 586, 586, 588, 589, 591, 594,
        590, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        590, 5, 3,  581, 588, 589,
        590, 6, 2,  576, 580,
        590, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        591, 0, 1,  584,
        591, 1, 2,  582, 586,
        591, 2, 4,  581, 583, 587, 593,
        591, 3, 7,  576, 585, 585, 588, 589, 590, 594,
        591, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        591, 5, 3,  581, 588, 589,
        591, 6, 2,  576, 580,
        591, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        592, 0, 1,  585,
        592, 1, 2,  583, 587,
        592, 2, 4,  582, 586, 590, 594,
        592, 3, 4,  581, 584, 584, 593,
        592, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        592, 5, 3,  581, 588, 589,
        592, 6, 2,  576, 580,
        592, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        593, 0, 1,  586,
        593, 1, 2,  584, 587,
        593, 2, 4,  582, 585, 591, 594,
        593, 3, 4,  581, 583, 583, 592,
        593, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        593, 5, 3,  581, 588, 589,
        593, 6, 2,  576, 580,
        593, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        594, 0, 1,  587,
        594, 1, 2,  585, 586,
        594, 2, 4,  583, 584, 592, 593,
        594, 3, 4,  582, 582, 590, 591,
        594, 4, 11,  582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594,
        594, 5, 3,  581, 588, 589,
        594, 6, 2,  576, 580,
        594, 7, 7,  558, 559, 575, 577, 578, 579, 595,
        595, 0, 1,  577,
        595, 1, 2,  576, 578,
        595, 2, 3,  575, 580, 581,
        595, 3, 5,  558, 579, 582, 588, 589,
        595, 4, 3,  577, 578, 595,
        595, 5, 2,  576, 580,
        595, 6, 7,  558, 559, 575, 579, 581, 588, 589,
        595, 7, 13,  557, 561, 582, 583, 584, 585, 586, 587, 590, 591, 592, 593, 594
    };
    unsigned int covalentMapDataSize = sizeof(covalentMapData)/sizeof(int);

    unsigned int covalentIndex       = 0;
    while (covalentIndex < covalentMapDataSize) { 
        int particleIndex = covalentMapData[covalentIndex++];
        int typeIndex     = covalentMapData[covalentIndex++];
        int entries       = covalentMapData[covalentIndex++];
        std::vector< int > covalentMap(entries);
        for (unsigned int ii = 0; ii < entries; ii++) {
            covalentMap[ii] =  covalentMapData[covalentIndex++];
        }
        amoebaMultipoleForce->setCovalentMap(particleIndex, static_cast<OpenMM::AmoebaMultipoleForce::CovalentType>(typeIndex), covalentMap);
    }
    system.addForce(amoebaMultipoleForce);

    // end of Multipole setup

    // GK force

    AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce  = new AmoebaGeneralizedKirkwoodForce();
    amoebaGeneralizedKirkwoodForce->setSolventDielectric(  7.8300000e+01);
    amoebaGeneralizedKirkwoodForce->setSoluteDielectric(   1.0000000e+00);
    amoebaGeneralizedKirkwoodForce->setIncludeCavityTerm(includeCavityTerm);

    // addParticle: charge, radius, scalingFactor

    static const double generalizedKirkwoodData[] = {
      1.1164000e-01,   1.5965000e-01,   6.9000000e-01,
      1.8220000e-02,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      2.1240000e-01,   1.2360000e-01,   6.9000000e-01,
      2.1240000e-01,   1.2360000e-01,   6.9000000e-01,
      2.1240000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -2.9490000e-02,   1.7510000e-01,   6.9000000e-01,
     -7.2850000e-02,   1.7510000e-01,   6.9000000e-01,
      6.9690000e-02,   1.8540000e-01,   6.9000000e-01,
     -1.5553000e-01,   1.7510000e-01,   6.9000000e-01,
      6.3730000e-02,   1.2360000e-01,   6.9000000e-01,
      6.3730000e-02,   1.2360000e-01,   6.9000000e-01,
      2.5100000e-03,   1.2360000e-01,   6.9000000e-01,
      2.5100000e-03,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8800000e-02,   1.7510000e-01,   6.9000000e-01,
      2.4250000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      8.4100000e-03,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
     -7.7000000e-02,   1.5965000e-01,   6.9000000e-01,
     -3.1424000e-01,   1.7510000e-01,   6.9000000e-01,
      8.6830000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.8568000e-01,   1.5656000e-01,   6.9000000e-01,
      1.3014000e-01,   1.2360000e-01,   6.9000000e-01,
      9.0300000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8740000e-01,   1.7510000e-01,   6.9000000e-01,
     -4.1295000e-01,   1.5656000e-01,   6.9000000e-01,
      1.9820000e-02,   1.2360000e-01,   6.9000000e-01,
      1.9820000e-02,   1.2360000e-01,   6.9000000e-01,
      2.7409000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -3.3040000e-01,   1.7510000e-01,   6.9000000e-01,
      1.0164400e+00,   1.7510000e-01,   6.9000000e-01,
     -8.5689000e-01,   1.5656000e-01,   6.9000000e-01,
     -8.5689000e-01,   1.5656000e-01,   6.9000000e-01,
      4.8930000e-02,   1.2360000e-01,   6.9000000e-01,
      4.8930000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1000000e-03,   1.7510000e-01,   6.9000000e-01,
     -3.6014000e-01,   1.7510000e-01,   6.9000000e-01,
      1.1459600e+00,   1.7510000e-01,   6.9000000e-01,
     -8.9716000e-01,   1.5656000e-01,   6.9000000e-01,
     -8.9716000e-01,   1.5656000e-01,   6.9000000e-01,
      4.0820000e-02,   1.2360000e-01,   6.9000000e-01,
      4.0820000e-02,   1.2360000e-01,   6.9000000e-01,
     -3.5600000e-03,   1.2360000e-01,   6.9000000e-01,
     -3.5600000e-03,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -3.3040000e-01,   1.7510000e-01,   6.9000000e-01,
      1.0164400e+00,   1.7510000e-01,   6.9000000e-01,
     -8.5689000e-01,   1.5656000e-01,   6.9000000e-01,
     -8.5689000e-01,   1.5656000e-01,   6.9000000e-01,
      4.8930000e-02,   1.2360000e-01,   6.9000000e-01,
      4.8930000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.8950000e-02,   1.7510000e-01,   6.9000000e-01,
     -6.2750000e-02,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.8260000e-02,   1.7510000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5880000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4170000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.4658000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.2517000e-01,   1.7510000e-01,   6.9000000e-01,
     -6.3700000e-03,   1.7510000e-01,   6.9000000e-01,
      1.0679000e-01,   1.5965000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.5440000e-01,   1.7510000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4820000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.7773000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.7773000e-01,   1.7510000e-01,   6.9000000e-01,
      6.6180000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
      5.7430000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.8950000e-02,   1.7510000e-01,   6.9000000e-01,
     -6.2750000e-02,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.8260000e-02,   1.7510000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5880000e-02,   1.2360000e-01,   6.9000000e-01,
     -2.2620000e-01,   1.5965000e-01,   6.9000000e-01,
     -1.5245000e-01,   1.7510000e-01,   6.9000000e-01,
      8.4374000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.3597000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2752000e-01,   1.2360000e-01,   6.9000000e-01,
      7.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.1680000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -2.9490000e-02,   1.7510000e-01,   6.9000000e-01,
     -7.2850000e-02,   1.7510000e-01,   6.9000000e-01,
      6.9690000e-02,   1.8540000e-01,   6.9000000e-01,
     -1.5553000e-01,   1.7510000e-01,   6.9000000e-01,
      6.3730000e-02,   1.2360000e-01,   6.9000000e-01,
      6.3730000e-02,   1.2360000e-01,   6.9000000e-01,
      2.5100000e-03,   1.2360000e-01,   6.9000000e-01,
      2.5100000e-03,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1940000e-02,   1.2360000e-01,   6.9000000e-01,
     -7.7000000e-02,   1.5965000e-01,   6.9000000e-01,
     -3.1424000e-01,   1.7510000e-01,   6.9000000e-01,
      8.6830000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.8568000e-01,   1.5656000e-01,   6.9000000e-01,
      1.3014000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9720000e-02,   1.2360000e-01,   6.9000000e-01,
      1.7854000e-01,   1.7510000e-01,   6.9000000e-01,
     -3.9938000e-01,   1.5656000e-01,   6.9000000e-01,
     -1.5307000e-01,   1.7510000e-01,   6.9000000e-01,
      2.1340000e-02,   1.2360000e-01,   6.9000000e-01,
      2.2557000e-01,   1.2360000e-01,   6.9000000e-01,
      7.1920000e-02,   1.2360000e-01,   6.9000000e-01,
      7.1920000e-02,   1.2360000e-01,   6.9000000e-01,
      7.1920000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8100000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.2460000e-01,   1.7510000e-01,   6.9000000e-01,
     -5.0400000e-03,   1.7510000e-01,   6.9000000e-01,
     -2.6789000e-01,   1.5965000e-01,   6.9000000e-01,
      9.0680000e-01,   1.7510000e-01,   6.9000000e-01,
     -3.0169000e-01,   1.5965000e-01,   6.9000000e-01,
     -3.0169000e-01,   1.5965000e-01,   6.9000000e-01,
      7.3320000e-02,   1.2360000e-01,   6.9000000e-01,
      7.3320000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0013000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0013000e-01,   1.2360000e-01,   6.9000000e-01,
      6.3860000e-02,   1.2360000e-01,   6.9000000e-01,
      6.3860000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4895000e-01,   1.2360000e-01,   6.9000000e-01,
      1.4969000e-01,   1.2360000e-01,   6.9000000e-01,
      1.4969000e-01,   1.2360000e-01,   6.9000000e-01,
      1.4969000e-01,   1.2360000e-01,   6.9000000e-01,
      1.4969000e-01,   1.2360000e-01,   6.9000000e-01,
     -7.7000000e-02,   1.5965000e-01,   6.9000000e-01,
     -3.1424000e-01,   1.7510000e-01,   6.9000000e-01,
      8.6830000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.8568000e-01,   1.5656000e-01,   6.9000000e-01,
      1.3014000e-01,   1.2360000e-01,   6.9000000e-01,
      9.0300000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8740000e-01,   1.7510000e-01,   6.9000000e-01,
     -4.1295000e-01,   1.5656000e-01,   6.9000000e-01,
      1.9820000e-02,   1.2360000e-01,   6.9000000e-01,
      1.9820000e-02,   1.2360000e-01,   6.9000000e-01,
      2.7409000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.5440000e-01,   1.7510000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.8950000e-02,   1.7510000e-01,   6.9000000e-01,
     -6.2750000e-02,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.8260000e-02,   1.7510000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5880000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.5440000e-01,   1.7510000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
      7.4840000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.5072000e-01,   1.7510000e-01,   6.9000000e-01,
      7.6960000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.2950000e-01,   1.5656000e-01,   6.9000000e-01,
     -3.8020000e-01,   1.5965000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8800000e-02,   1.7510000e-01,   6.9000000e-01,
      2.4250000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      8.4100000e-03,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.1893000e-01,   1.5965000e-01,   6.9000000e-01,
     -3.8385000e-01,   1.7510000e-01,   6.9000000e-01,
      9.8874000e-01,   1.7510000e-01,   6.9000000e-01,
     -8.2816000e-01,   1.5656000e-01,   6.9000000e-01,
      1.0715000e-01,   1.2360000e-01,   6.9000000e-01,
     -9.4950000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1624000e-01,   1.7510000e-01,   6.9000000e-01,
     -2.0160000e-02,   1.7510000e-01,   6.9000000e-01,
      9.2750000e-02,   1.2360000e-01,   6.9000000e-01,
      9.2750000e-02,   1.2360000e-01,   6.9000000e-01,
      9.2170000e-02,   1.2360000e-01,   6.9000000e-01,
      9.2170000e-02,   1.2360000e-01,   6.9000000e-01,
      4.8280000e-02,   1.2360000e-01,   6.9000000e-01,
      4.8280000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8800000e-02,   1.7510000e-01,   6.9000000e-01,
      2.4250000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      8.4100000e-03,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.6392000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.0518000e-01,   1.7510000e-01,   6.9000000e-01,
      6.3630000e-02,   1.7510000e-01,   6.9000000e-01,
      6.7000000e-03,   1.7510000e-01,   6.9000000e-01,
     -4.0140000e-02,   1.5965000e-01,   6.9000000e-01,
      2.7213000e-01,   1.7510000e-01,   6.9000000e-01,
     -2.1794000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1761000e-01,   1.7510000e-01,   6.9000000e-01,
      1.0201000e-01,   1.7510000e-01,   6.9000000e-01,
     -5.7640000e-02,   1.7510000e-01,   6.9000000e-01,
      9.3070000e-02,   1.2360000e-01,   6.9000000e-01,
      9.3070000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8580000e-02,   1.2360000e-01,   6.9000000e-01,
      8.9430000e-02,   1.2360000e-01,   6.9000000e-01,
      8.0100000e-03,   1.2360000e-01,   6.9000000e-01,
      4.4000000e-03,   1.2360000e-01,   6.9000000e-01,
      7.7500000e-03,   1.2360000e-01,   6.9000000e-01,
      3.7700000e-03,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4170000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.4658000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.2517000e-01,   1.7510000e-01,   6.9000000e-01,
     -6.3700000e-03,   1.7510000e-01,   6.9000000e-01,
      1.0679000e-01,   1.5965000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.2108000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.4858000e-01,   1.7510000e-01,   6.9000000e-01,
      7.6960000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.2950000e-01,   1.5656000e-01,   6.9000000e-01,
     -3.8020000e-01,   1.5965000e-01,   6.9000000e-01,
      5.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      5.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.2108000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.4858000e-01,   1.7510000e-01,   6.9000000e-01,
      7.6960000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.2950000e-01,   1.5656000e-01,   6.9000000e-01,
     -3.8020000e-01,   1.5965000e-01,   6.9000000e-01,
      5.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      5.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.5072000e-01,   1.7510000e-01,   6.9000000e-01,
      7.6960000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.2950000e-01,   1.5656000e-01,   6.9000000e-01,
     -3.8020000e-01,   1.5965000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      9.6790000e-02,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
      1.8368000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8800000e-02,   1.7510000e-01,   6.9000000e-01,
      2.4250000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      8.4100000e-03,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4170000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.4658000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.2517000e-01,   1.7510000e-01,   6.9000000e-01,
     -6.3700000e-03,   1.7510000e-01,   6.9000000e-01,
      1.0679000e-01,   1.5965000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4170000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.4658000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.2517000e-01,   1.7510000e-01,   6.9000000e-01,
     -6.3700000e-03,   1.7510000e-01,   6.9000000e-01,
      1.0679000e-01,   1.5965000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      4.1000000e-03,   1.7510000e-01,   6.9000000e-01,
     -3.6014000e-01,   1.7510000e-01,   6.9000000e-01,
      1.1459600e+00,   1.7510000e-01,   6.9000000e-01,
     -8.9716000e-01,   1.5656000e-01,   6.9000000e-01,
     -8.9716000e-01,   1.5656000e-01,   6.9000000e-01,
      4.0820000e-02,   1.2360000e-01,   6.9000000e-01,
      4.0820000e-02,   1.2360000e-01,   6.9000000e-01,
     -3.5600000e-03,   1.2360000e-01,   6.9000000e-01,
     -3.5600000e-03,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
      1.4170000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.4658000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.2517000e-01,   1.7510000e-01,   6.9000000e-01,
     -6.3700000e-03,   1.7510000e-01,   6.9000000e-01,
      1.0679000e-01,   1.5965000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      6.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9140000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      7.9470000e-02,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.0424000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
      1.9274000e-01,   1.2360000e-01,   6.9000000e-01,
     -2.2620000e-01,   1.5965000e-01,   6.9000000e-01,
     -1.5245000e-01,   1.7510000e-01,   6.9000000e-01,
      8.4374000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.3597000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2752000e-01,   1.2360000e-01,   6.9000000e-01,
      7.1680000e-02,   1.2360000e-01,   6.9000000e-01,
      7.1680000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -2.1238000e-01,   1.7510000e-01,   6.9000000e-01,
      8.5068000e-01,   1.7510000e-01,   6.9000000e-01,
     -7.7770000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -5.8800000e-02,   1.7510000e-01,   6.9000000e-01,
      2.4250000e-02,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
     -1.1850000e-01,   1.7510000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      4.6960000e-02,   1.2360000e-01,   6.9000000e-01,
      8.4100000e-03,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
      3.9890000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.4985000e-01,   1.5965000e-01,   6.9000000e-01,
     -3.5660000e-01,   1.7510000e-01,   6.9000000e-01,
      1.0266900e+00,   1.7510000e-01,   6.9000000e-01,
     -9.0443000e-01,   1.5656000e-01,   6.9000000e-01,
      1.2992000e-01,   1.2360000e-01,   6.9000000e-01,
      8.9210000e-02,   1.2360000e-01,   6.9000000e-01,
     -1.8950000e-02,   1.7510000e-01,   6.9000000e-01,
     -6.2750000e-02,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
      8.9000000e-03,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.4020000e-02,   1.7510000e-01,   6.9000000e-01,
     -5.8260000e-02,   1.7510000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      6.8390000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      2.8180000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5650000e-02,   1.2360000e-01,   6.9000000e-01,
      3.5880000e-02,   1.2360000e-01,   6.9000000e-01,
     -9.0443000e-01,   1.5656000e-01,   6.9000000e-01
    };

    const double* gkData = generalizedKirkwoodData;
    for (unsigned int ii = 0; ii < numberOfParticles; ii++) {
        amoebaGeneralizedKirkwoodForce->addParticle(gkData[0], gkData[1], gkData[2]);
        gkData += 3;
    }
    system.addForce(amoebaGeneralizedKirkwoodForce);

    // end of GeneralizedKirkwood setup

    std::vector<Vec3> positions(numberOfParticles);

    static const double positionsData[] = {
      1.4717950e-01,  -1.0081856e+00,  -3.8562320e-01,
      4.8537500e-02,  -9.0291270e-01,  -3.5820030e-01,
     -2.6019700e-02,  -9.3169790e-01,  -2.2723420e-01,
     -4.5244300e-02,  -1.0465546e+00,  -1.8474070e-01,
      1.0388820e-01,  -1.0995290e+00,  -3.8977830e-01,
      1.9721800e-01,  -9.9460900e-01,  -4.7245640e-01,
      2.1713110e-01,  -1.0142324e+00,  -3.1262690e-01,
      1.0656620e-01,  -8.0838750e-01,  -3.4304340e-01,
     -5.2870900e-02,  -8.8667920e-01,  -4.7667050e-01,
      3.2290000e-03,  -8.0096810e-01,  -5.9288490e-01,
     -3.2176000e-02,  -8.7432710e-01,  -7.5555050e-01,
     -1.0836400e-01,  -7.3282860e-01,  -8.3940590e-01,
     -8.4130800e-02,  -9.8792480e-01,  -5.1145110e-01,
     -1.4685810e-01,  -8.3847960e-01,  -4.4018780e-01,
     -3.9015600e-02,  -6.9799340e-01,  -5.8586510e-01,
      1.1350270e-01,  -7.8858260e-01,  -5.8264020e-01,
     -3.6694900e-02,  -6.4798630e-01,  -8.4263390e-01,
     -1.3562780e-01,  -7.6202110e-01,  -9.4310570e-01,
     -1.9994440e-01,  -7.0148680e-01,  -7.8468360e-01,
     -6.7911500e-02,  -8.2136210e-01,  -1.6052400e-01,
     -1.4457370e-01,  -8.1989360e-01,  -3.6966500e-02,
     -2.9301550e-01,  -8.3814140e-01,  -7.3732900e-02,
     -3.5529330e-01,  -7.4944980e-01,  -1.3501020e-01,
     -5.5968900e-02,  -7.2972960e-01,  -2.0543440e-01,
     -1.1446630e-01,  -9.0762490e-01,   2.5702300e-02,
     -1.1994260e-01,  -6.8812360e-01,   4.5033600e-02,
     -1.4173400e-02,  -7.0349310e-01,   1.5826690e-01,
      1.2541120e-01,  -7.4146540e-01,   1.0232850e-01,
     -2.1287000e-03,  -5.7251290e-01,   2.4071090e-01,
     -9.2964300e-02,  -6.0284320e-01,  -2.1507400e-02,
     -2.1643910e-01,  -6.5906530e-01,   9.3197400e-02,
     -4.9112200e-02,  -7.8425460e-01,   2.2710570e-01,
      1.2678840e-01,  -8.4276540e-01,   5.6066200e-02,
      1.5978170e-01,  -6.6941240e-01,   2.4674200e-02,
      2.0235310e-01,  -7.4213750e-01,   1.8292520e-01,
     -1.0020520e-01,  -5.2155410e-01,   2.5414990e-01,
      3.6696100e-02,  -5.9370080e-01,   3.4318640e-01,
      6.6954200e-02,  -4.9881930e-01,   1.9357120e-01,
     -3.5080860e-01,  -9.5532840e-01,  -3.7174500e-02,
     -4.9549820e-01,  -9.7853520e-01,  -3.2239700e-02,
     -5.5571330e-01,  -8.8470520e-01,   7.3723100e-02,
     -4.8330620e-01,  -8.2459290e-01,   1.5387390e-01,
     -2.9117430e-01,  -1.0350153e+00,  -1.1544900e-02,
     -5.3944100e-01,  -9.4993440e-01,  -1.3149480e-01,
     -5.2933610e-01,  -1.1275369e+00,  -2.1309000e-03,
     -4.7924410e-01,  -1.1654181e+00,   1.2574110e-01,
     -6.3917160e-01,  -1.1460307e+00,  -4.8881000e-03,
     -4.8452420e-01,  -1.1938119e+00,  -7.9651700e-02,
     -4.5571360e-01,  -1.2573206e+00,   1.2086610e-01,
     -6.9055700e-01,  -8.7214170e-01,   7.5835100e-02,
     -7.6006360e-01,  -7.7948480e-01,   1.6431800e-01,
     -7.4346140e-01,  -8.1251960e-01,   3.1435450e-01,
     -7.4721590e-01,  -7.2063140e-01,   3.9650230e-01,
     -7.4810960e-01,  -9.1600180e-01,   2.7571000e-03,
     -7.0628740e-01,  -6.8135980e-01,   1.5488670e-01,
     -9.0915860e-01,  -7.5422170e-01,   1.2882700e-01,
     -9.3948840e-01,  -6.9889590e-01,  -1.1368700e-02,
     -8.5444940e-01,  -7.1126550e-01,  -1.0383220e-01,
     -1.0504229e+00,  -6.4180170e-01,  -2.9924800e-02,
     -9.6650890e-01,  -8.4935780e-01,   1.4136330e-01,
     -9.5183310e-01,  -6.8203990e-01,   2.0328130e-01,
     -7.2364840e-01,  -9.4153470e-01,   3.5319260e-01,
     -7.0186630e-01,  -9.8485730e-01,   4.9087390e-01,
     -5.5912970e-01,  -9.6277140e-01,   5.4337780e-01,
     -5.4202250e-01,  -9.3958020e-01,   6.6355640e-01,
     -7.2289260e-01,  -1.0160788e+00,   2.8286060e-01,
     -7.6273350e-01,  -9.1807080e-01,   5.5673240e-01,
     -7.4551420e-01,  -1.1318830e+00,   5.1913680e-01,
     -8.9752130e-01,  -1.1563516e+00,   5.0268300e-01,
     -9.5263780e-01,  -1.2758749e+00,   5.8170320e-01,
     -8.8328540e-01,  -1.3276614e+00,   6.7314250e-01,
     -1.0679408e+00,  -1.3182197e+00,   5.5342520e-01,
     -6.8852990e-01,  -1.2029038e+00,   4.5424600e-01,
     -7.1600880e-01,  -1.1568517e+00,   6.2430510e-01,
     -9.5383120e-01,  -1.0656025e+00,   5.3622130e-01,
     -9.2172910e-01,  -1.1694655e+00,   3.9418330e-01,
     -4.5571240e-01,  -9.6049550e-01,   4.5544950e-01,
     -3.1846930e-01,  -9.2586860e-01,   4.9243280e-01,
     -2.9699900e-01,  -7.7387630e-01,   4.9875590e-01,
     -2.1390990e-01,  -7.2694070e-01,   5.7666040e-01,
     -4.6958240e-01,  -9.9199380e-01,   3.5861150e-01,
     -3.0229090e-01,  -9.5846860e-01,   5.9838170e-01,
     -2.0855020e-01,  -9.9126560e-01,   4.0473220e-01,
     -1.9916820e-01,  -1.1443703e+00,   4.0628350e-01,
     -2.9326860e-01,  -1.2119551e+00,   4.5651300e-01,
     -9.6056700e-02,  -1.1978985e+00,   3.5732090e-01,
     -2.2180120e-01,  -9.5919080e-01,   2.9849510e-01,
     -1.0875960e-01,  -9.5156270e-01,   4.3629180e-01,
     -3.7491510e-01,  -6.9414110e-01,   4.2129200e-01,
     -3.6867950e-01,  -5.4844630e-01,   4.2209290e-01,
     -4.2109270e-01,  -4.8669060e-01,   5.5318400e-01,
     -3.6855290e-01,  -3.8629360e-01,   6.0316190e-01,
     -4.3866130e-01,  -7.3587040e-01,   3.5183170e-01,
     -2.6095740e-01,  -5.1914410e-01,   4.1902440e-01,
     -4.3912630e-01,  -4.8944120e-01,   2.9615930e-01,
     -4.1149370e-01,  -3.4194150e-01,   2.7870990e-01,
     -3.1342090e-01,  -2.9671720e-01,   1.8835310e-01,
     -4.7914060e-01,  -2.4781930e-01,   3.5782430e-01,
     -2.8316440e-01,  -1.6065630e-01,   1.7853540e-01,
     -4.4347210e-01,  -1.1348310e-01,   3.5259590e-01,
     -3.4781890e-01,  -6.8893800e-02,   2.6156760e-01,
     -4.0605470e-01,  -5.4288670e-01,   2.0424840e-01,
     -5.4926500e-01,  -5.0574760e-01,   3.0106110e-01,
     -2.5717880e-01,  -3.6874180e-01,   1.2650110e-01,
     -5.5816760e-01,  -2.7933580e-01,   4.2829740e-01,
     -2.0347530e-01,  -1.2626410e-01,   1.1031440e-01,
     -4.9073870e-01,  -4.2622300e-02,   4.2166620e-01,
     -3.2239530e-01,   3.7971100e-02,   2.5764470e-01,
     -5.2455310e-01,  -5.5111820e-01,   6.1235070e-01,
     -6.0067730e-01,  -5.0912240e-01,   7.2930650e-01,
     -5.2546450e-01,  -5.1933630e-01,   8.6298770e-01,
     -5.6204880e-01,  -4.5198430e-01,   9.5992890e-01,
     -5.6255550e-01,  -6.3433320e-01,   5.6552380e-01,
     -6.2059040e-01,  -4.0001720e-01,   7.1793530e-01,
     -7.3576220e-01,  -5.8866890e-01,   7.3574680e-01,
     -8.5893710e-01,  -5.0477940e-01,   7.7727660e-01,
     -9.9194550e-01,  -5.6358760e-01,   7.2219400e-01,
     -1.1182542e+00,  -4.8092310e-01,   7.5702060e-01,
     -1.1609806e+00,  -4.9059110e-01,   8.9672360e-01,
     -7.5596070e-01,  -6.3192860e-01,   6.3473800e-01,
     -7.2581530e-01,  -6.7738300e-01,   8.0282620e-01,
     -8.6169420e-01,  -4.9733850e-01,   8.8858910e-01,
     -8.4751850e-01,  -4.0017870e-01,   7.3992450e-01,
     -9.8535320e-01,  -5.6778260e-01,   6.1061640e-01,
     -1.0054610e+00,  -6.6929810e-01,   7.5582250e-01,
     -1.1021778e+00,  -3.7346890e-01,   7.3224950e-01,
     -1.2033971e+00,  -5.1485220e-01,   6.9395970e-01,
     -1.0877007e+00,  -4.6685980e-01,   9.6244430e-01,
     -1.2377589e+00,  -4.2708780e-01,   9.1530270e-01,
     -1.1942968e+00,  -5.8320600e-01,   9.2084000e-01,
     -4.1690620e-01,  -5.9983170e-01,   8.6898140e-01,
     -3.2798440e-01,  -6.1684520e-01,   9.8321240e-01,
     -1.8925180e-01,  -5.5244190e-01,   9.6333340e-01,
     -9.7213100e-02,  -5.7776730e-01,   1.0423348e+00,
     -3.8950510e-01,  -6.5486640e-01,   7.8645240e-01,
     -3.7094860e-01,  -5.6363310e-01,   1.0715024e+00,
     -3.1672950e-01,  -7.6770340e-01,   1.0177810e+00,
     -4.1253290e-01,  -8.0860630e-01,   1.0567028e+00,
     -2.8871960e-01,  -8.2775080e-01,   9.2849760e-01,
     -2.3935590e-01,  -7.8606740e-01,   1.0953935e+00,
     -1.7327600e-01,  -4.6909380e-01,   8.5700210e-01,
     -5.1506200e-02,  -3.9989160e-01,   8.1679460e-01,
     -7.8402300e-02,  -2.4755020e-01,   8.2195970e-01,
     -5.0286500e-02,  -1.8471460e-01,   9.2469690e-01,
     -2.5358130e-01,  -4.4556600e-01,   7.9689840e-01,
      2.9498200e-02,  -4.1610920e-01,   8.9242380e-01,
     -1.6189000e-03,  -4.6160930e-01,   6.8084040e-01,
      1.1066920e-01,  -3.7920180e-01,   6.1444610e-01,
      5.6341000e-02,  -6.0535300e-01,   7.0180870e-01,
     -8.8565000e-02,  -4.6665840e-01,   6.1094770e-01,
      8.3159500e-02,  -2.7286620e-01,   5.9946710e-01,
      2.0332350e-01,  -3.8098450e-01,   6.7521340e-01,
      1.3665200e-01,  -4.1941800e-01,   5.1403120e-01,
     -1.7822600e-02,  -6.7455220e-01,   7.4751020e-01,
      8.7968200e-02,  -6.5131200e-01,   6.0526050e-01,
      1.4603060e-01,  -6.0527550e-01,   7.6773670e-01,
     -1.2828610e-01,  -1.8441250e-01,   7.1176500e-01,
     -1.3701690e-01,  -3.8730000e-02,   6.9854050e-01,
     -2.5843890e-01,   2.6862900e-02,   7.6680300e-01,
     -2.4239580e-01,   1.2701160e-01,   8.3754590e-01,
     -1.6619380e-01,  -2.4014880e-01,   6.3397860e-01,
     -4.9692900e-02,   4.3045000e-03,   7.5348650e-01,
     -1.3125860e-01,   1.3982000e-02,   5.5247370e-01,
     -1.9452400e-02,  -3.1193500e-02,   4.5985540e-01,
      8.7370100e-02,   5.5006100e-02,   4.2800390e-01,
     -3.3247100e-02,  -1.4865810e-01,   3.8336170e-01,
      1.7507650e-01,   2.5321200e-02,   3.2243520e-01,
      5.4070800e-02,  -1.7793930e-01,   2.7775070e-01,
      1.5739160e-01,  -9.0067500e-02,   2.4590380e-01,
     -2.2701790e-01,  -1.2506700e-02,   5.0232230e-01,
     -1.3096230e-01,   1.2565270e-01,   5.5545350e-01,
      1.0111830e-01,   1.4950380e-01,   4.8318600e-01,
     -1.1723370e-01,  -2.1736540e-01,   4.0176080e-01,
      2.5696510e-01,   9.4377700e-02,   2.9641670e-01,
      3.8496800e-02,  -2.6813660e-01,   2.1621660e-01,
      2.2449740e-01,  -1.1188190e-01,   1.6087280e-01,
     -3.8089140e-01,  -2.8283100e-02,   7.4655830e-01,
     -5.0508540e-01,   1.5350100e-02,   8.0702420e-01,
     -6.2672700e-01,   5.5628300e-02,   7.2240300e-01,
     -7.3941440e-01,   4.9944000e-02,   7.7314930e-01,
     -3.8899280e-01,  -1.1453580e-01,   6.9157010e-01,
     -5.4065780e-01,  -6.5380200e-02,   8.7500210e-01,
     -4.8546070e-01,   1.0172640e-01,   8.7441730e-01,
     -6.0860660e-01,   1.0098440e-01,   5.9524160e-01,
     -7.1800200e-01,   1.4038090e-01,   5.0425760e-01,
     -7.3854140e-01,   2.9957800e-02,   3.9801110e-01,
     -7.7435220e-01,  -8.3108200e-02,   4.3344590e-01,
     -5.1361220e-01,   1.0898970e-01,   5.5661460e-01,
     -8.1417230e-01,   1.3928000e-01,   5.6146880e-01,
     -6.9916660e-01,   2.8432150e-01,   4.4679100e-01,
     -6.9421550e-01,   3.9685470e-01,   5.5220700e-01,
     -7.2577440e-01,   5.5818670e-01,   4.7320310e-01,
     -6.3759670e-01,   6.7078770e-01,   5.8473020e-01,
     -6.0601610e-01,   2.8995900e-01,   3.8547650e-01,
     -7.8461500e-01,   3.0394680e-01,   3.7724520e-01,
     -7.6994660e-01,   3.8021330e-01,   6.3220720e-01,
     -5.9427240e-01,   3.9554520e-01,   6.0119260e-01,
     -5.2790880e-01,   6.6304180e-01,   5.6811590e-01,
     -6.6950170e-01,   7.7512690e-01,   5.6368850e-01,
     -6.6077430e-01,   6.4581060e-01,   6.9044170e-01,
     -7.1475810e-01,   6.0041100e-02,   2.6743590e-01,
     -7.1411150e-01,  -3.1161000e-02,   1.5175910e-01,
     -5.9747220e-01,   1.0365600e-02,   5.9400000e-02,
     -5.5704360e-01,   1.2695880e-01,   6.3161600e-02,
     -6.8360320e-01,   1.5451050e-01,   2.4187750e-01,
     -6.9421330e-01,  -1.3487950e-01,   1.8885200e-01,
     -8.4916830e-01,  -2.7939500e-02,   7.0550900e-02,
     -9.4824200e-01,   4.5536100e-02,   1.4235830e-01,
     -9.0493780e-01,  -1.7061640e-01,   4.9406800e-02,
     -8.3368430e-01,   2.3818800e-02,  -2.7161400e-02,
     -1.0076032e+00,   8.5396300e-02,   7.9710400e-02,
     -9.3277290e-01,  -2.1925390e-01,   1.4610590e-01,
     -9.9607740e-01,  -1.6895460e-01,  -1.4829100e-02,
     -8.2892970e-01,  -2.3477480e-01,  -1.4358000e-03,
     -5.4691980e-01,  -7.9001000e-02,  -3.0042400e-02,
     -4.3608280e-01,  -4.6857300e-02,  -1.2117740e-01,
     -4.6933510e-01,   6.6427900e-02,  -2.2026940e-01,
     -3.8208170e-01,   1.4766620e-01,  -2.5250410e-01,
     -5.7990240e-01,  -1.7645960e-01,  -2.8866200e-02,
     -3.5675490e-01,  -2.7845000e-03,  -5.6079300e-02,
     -3.7324280e-01,  -1.6697170e-01,  -2.0052240e-01,
     -2.1739110e-01,  -1.5882120e-01,  -2.0468250e-01,
     -1.4070960e-01,  -2.3498250e-01,  -3.1639470e-01,
     -2.2394920e-01,  -3.3110010e-01,  -3.8955720e-01,
     -2.7345700e-01,  -4.4721940e-01,  -3.4433960e-01,
     -2.4427540e-01,  -4.9613260e-01,  -2.2331450e-01,
     -3.5689690e-01,  -5.1605040e-01,  -4.2314660e-01,
     -4.0314570e-01,  -2.6392940e-01,  -1.5417540e-01,
     -4.1426610e-01,  -1.6851700e-01,  -3.0445370e-01,
     -1.8862320e-01,  -5.1400000e-02,  -2.1404000e-01,
     -1.7768890e-01,  -1.8993110e-01,  -1.0513970e-01,
     -1.0084320e-01,  -1.6046850e-01,  -3.8952690e-01,
     -4.8566500e-02,  -2.8302480e-01,  -2.7511140e-01,
     -2.4906530e-01,  -2.9937080e-01,  -4.8453900e-01,
     -1.8099590e-01,  -4.4636870e-01,  -1.5972380e-01,
     -2.8382690e-01,  -5.8516870e-01,  -1.8933770e-01,
     -3.8112300e-01,  -4.8243080e-01,  -5.1755210e-01,
     -3.9905750e-01,  -6.0417630e-01,  -3.9115100e-01,
     -5.9724600e-01,   7.4583500e-02,  -2.6483410e-01,
     -6.5010250e-01,   1.7751990e-01,  -3.5377270e-01,
     -6.6188260e-01,   3.1735990e-01,  -2.8961230e-01,
     -6.5334490e-01,   4.1754350e-01,  -3.6099920e-01,
     -6.6324890e-01,  -2.1790000e-04,  -2.3978920e-01,
     -5.7442200e-01,   1.9293260e-01,  -4.3460570e-01,
     -7.8490080e-01,   1.3629040e-01,  -4.1980670e-01,
     -7.6691110e-01,   1.1700220e-01,  -5.5982590e-01,
     -8.2766560e-01,   4.3109400e-02,  -3.7647530e-01,
     -8.6206620e-01,   2.1506500e-01,  -4.0528910e-01,
     -8.3944460e-01,   1.5997730e-01,  -6.0437380e-01,
     -6.7901340e-01,   3.2629700e-01,  -1.5502340e-01,
     -6.7998730e-01,   4.4971290e-01,  -7.5807400e-02,
     -5.4142110e-01,   4.8924870e-01,  -2.0999500e-02,
     -5.1993050e-01,   6.0484850e-01,   1.7353600e-02,
     -6.9014110e-01,   2.4032050e-01,  -9.9930600e-02,
     -7.0630300e-01,   5.3446240e-01,  -1.4406050e-01,
     -7.8576540e-01,   4.4581340e-01,   3.8151400e-02,
     -8.4547270e-01,   3.5182830e-01,   3.7982300e-02,
     -7.3917470e-01,   4.5489380e-01,   1.3900000e-01,
     -8.5728110e-01,   5.3057980e-01,   2.8004600e-02,
     -4.4565180e-01,   3.9321590e-01,  -2.0399800e-02,
     -3.0542840e-01,   4.0779190e-01,   1.5797100e-02,
     -2.2414380e-01,   4.8251350e-01,  -8.9902600e-02,
     -1.3490430e-01,   5.6049570e-01,  -5.5622700e-02,
     -4.7314500e-01,   2.9563270e-01,  -3.7545000e-02,
     -2.9908440e-01,   4.7307130e-01,   1.0641460e-01,
     -2.4121640e-01,   2.6987160e-01,   4.7702200e-02,
     -9.1645900e-02,   2.7164590e-01,   6.7518200e-02,
     -3.8062400e-02,   3.3617820e-01,   1.7989690e-01,
     -3.1968000e-03,   2.1519010e-01,  -2.6717300e-02,
      9.9970200e-02,   3.5024740e-01,   1.9440650e-01,
      1.3481290e-01,   2.2578550e-01,  -9.8342000e-03,
      1.8678680e-01,   2.9484700e-01,   9.9992300e-02,
     -2.8847360e-01,   2.2636080e-01,   1.3867070e-01,
     -2.6339030e-01,   1.9937800e-01,  -3.5524500e-02,
     -1.0522840e-01,   3.8032950e-01,   2.5562850e-01,
     -4.1239400e-02,   1.6409340e-01,  -1.1682440e-01,
      1.3982770e-01,   4.0693980e-01,   2.7973300e-01,
      2.0239750e-01,   1.8156090e-01,  -8.4979300e-02,
      2.9561880e-01,   3.0716050e-01,   1.1113900e-01,
     -2.6236740e-01,   4.7091760e-01,  -2.1976810e-01,
     -2.0413600e-01,   5.4702240e-01,  -3.2976650e-01,
     -2.6089520e-01,   6.9003800e-01,  -3.4528390e-01,
     -2.5043410e-01,   7.4968010e-01,  -4.5319680e-01,
     -3.3688960e-01,   4.0455100e-01,  -2.4537430e-01,
     -9.6294200e-02,   5.6465740e-01,  -3.0620670e-01,
     -2.0992540e-01,   4.6640330e-01,  -4.6242550e-01,
     -3.1405540e-01,   4.5883630e-01,  -5.0094980e-01,
     -1.4844000e-01,   5.1584910e-01,  -5.4114220e-01,
     -1.6992000e-01,   3.6306390e-01,  -4.4958540e-01,
     -3.1531880e-01,   7.4936720e-01,  -2.3546890e-01,
     -3.5892560e-01,   8.8810990e-01,  -2.2406590e-01,
     -2.6859640e-01,   9.7388650e-01,  -1.3289540e-01,
     -2.8198530e-01,   1.0971128e+00,  -1.3112940e-01,
     -3.3622850e-01,   6.9387320e-01,  -1.5159080e-01,
     -3.4979620e-01,   9.3640080e-01,  -3.2442790e-01,
     -5.0645570e-01,   9.0144610e-01,  -1.7663730e-01,
     -6.1026110e-01,   8.4494100e-01,  -2.7416800e-01,
     -5.9102330e-01,   7.4033950e-01,  -3.3696810e-01,
     -7.2585220e-01,   9.1542580e-01,  -2.8717570e-01,
     -5.2017370e-01,   8.5295480e-01,  -7.7098900e-02,
     -5.2759810e-01,   1.0097482e+00,  -1.6065220e-01,
     -8.0005020e-01,   8.7877360e-01,  -3.4871570e-01,
     -7.4331630e-01,   1.0057407e+00,  -2.3990920e-01,
     -1.7622350e-01,   9.1174860e-01,  -5.4660000e-02,
     -7.4951200e-02,   9.8111810e-01,   2.5853000e-02,
      3.5510300e-02,   1.0287464e+00,  -7.2964000e-02,
      3.7982500e-02,   9.8405940e-01,  -1.8833770e-01,
     -1.6009180e-01,   8.1041360e-01,  -6.3086800e-02,
     -1.2271260e-01,   1.0704657e+00,   7.3576500e-02,
     -1.8555800e-02,   8.8683660e-01,   1.3789470e-01,
     -1.2809960e-01,   8.1854610e-01,   2.2715970e-01,
     -8.1395000e-02,   6.7819960e-01,   2.7344100e-01,
     -1.6104460e-01,   9.0537900e-01,   3.5159050e-01,
      4.3143000e-02,   8.0844970e-01,   8.7613100e-02,
      5.2588400e-02,   9.4292630e-01,   2.0265910e-01,
     -2.2206100e-01,   8.0458310e-01,   1.6845140e-01,
      1.4345800e-02,   6.8170120e-01,   3.3000930e-01,
     -1.5669270e-01,   6.2817610e-01,   3.3882190e-01,
     -6.5262000e-02,   6.1161160e-01,   1.8558830e-01,
     -7.8839500e-02,   9.0325840e-01,   4.2708720e-01,
     -1.7780030e-01,   1.0119338e+00,   3.2363370e-01,
     -2.5389560e-01,   8.6914270e-01,   4.0184930e-01,
      1.2841240e-01,   1.1210089e+00,  -3.2446900e-02,
      2.4044900e-01,   1.1628009e+00,  -1.2003770e-01,
      3.3017120e-01,   1.0441571e+00,  -1.5834150e-01,
      3.3177360e-01,   9.4458030e-01,  -8.5819600e-02,
      1.9933720e-01,   1.2113830e+00,  -2.1211040e-01,
      3.1995770e-01,   1.2617182e+00,  -3.0979600e-02,
      2.1378500e-01,   1.3240711e+00,   5.9367000e-02,
      1.0936730e-01,   1.2135619e+00,   7.8184700e-02,
      3.9859270e-01,   1.2106811e+00,   2.9148500e-02,
      3.7272380e-01,   1.3386318e+00,  -9.1985700e-02,
      2.5509300e-01,   1.3598928e+00,   1.5644880e-01,
      1.6883610e-01,   1.4122013e+00,   8.5086000e-03,
      1.2239420e-01,   1.1621950e+00,   1.7594400e-01,
      7.4880000e-03,   1.2581044e+00,   7.8714400e-02,
      4.0900580e-01,   1.0538519e+00,  -2.6824400e-01,
      4.9432620e-01,   9.4537420e-01,  -3.1949030e-01,
      6.1198910e-01,   9.0936770e-01,  -2.2632040e-01,
      6.4542960e-01,   7.9126150e-01,  -2.1332350e-01,
      4.1097940e-01,   1.1427279e+00,  -3.2008470e-01,
      4.3079320e-01,   8.5334320e-01,  -3.1919810e-01,
      5.4216940e-01,   9.7159120e-01,  -4.6582630e-01,
      5.4283770e-01,   8.5216660e-01,  -5.6702020e-01,
      6.0875270e-01,   7.2263580e-01,  -5.1329390e-01,
      3.9874190e-01,   8.2128660e-01,  -6.1687970e-01,
      4.7743560e-01,   1.0505397e+00,  -5.1039650e-01,
      6.4319120e-01,   1.0189966e+00,  -4.6318070e-01,
      6.0256610e-01,   8.8456010e-01,  -6.5581810e-01,
      7.1611720e-01,   7.3753340e-01,  -4.8831400e-01,
      5.6050320e-01,   6.8518130e-01,  -4.2094920e-01,
      6.0364460e-01,   6.3860790e-01,  -5.8609710e-01,
      3.3473470e-01,   7.7981390e-01,  -5.3587680e-01,
      3.4796220e-01,   9.1269250e-01,  -6.5574250e-01,
      3.9954230e-01,   7.4726320e-01,  -7.0036190e-01,
      6.6920180e-01,   1.0072257e+00,  -1.5103670e-01,
      7.6835100e-01,   9.7937900e-01,  -4.6318100e-02,
      7.0361300e-01,   9.1718790e-01,   7.9156000e-02,
      7.6931080e-01,   8.4123100e-01,   1.5086500e-01,
      6.4308770e-01,   1.1056156e+00,  -1.6532010e-01,
      8.3559950e-01,   8.9801850e-01,  -8.4248300e-02,
      8.6072490e-01,   1.0996465e+00,  -9.6731000e-03,
      7.8689960e-01,   1.2257689e+00,   3.4152200e-02,
      7.4082630e-01,   1.3255314e+00,  -5.2495800e-02,
      7.3431160e-01,   1.2549401e+00,   1.5905420e-01,
      6.6330670e-01,   1.4121584e+00,   2.0134800e-02,
      6.5834120e-01,   1.3697759e+00,   1.4950630e-01,
      7.5222310e-01,   1.1893452e+00,   2.8143570e-01,
      5.9043850e-01,   1.4246600e+00,   2.5799270e-01,
      6.8595720e-01,   1.2436208e+00,   3.9405700e-01,
      6.0476620e-01,   1.3589120e+00,   3.8186550e-01,
      9.3350480e-01,   1.0698771e+00,   6.9430800e-02,
      9.2336260e-01,   1.1249071e+00,  -9.8298400e-02,
      7.5777320e-01,   1.3370297e+00,  -1.6111750e-01,
      6.1636330e-01,   1.4951744e+00,  -1.8953400e-02,
      8.1672720e-01,   1.0999700e+00,   2.9187140e-01,
      5.2599660e-01,   1.5138803e+00,   2.4750480e-01,
      6.9654500e-01,   1.1949722e+00,   4.9297130e-01,
      5.5297770e-01,   1.3989559e+00,   4.7123290e-01,
      5.7251650e-01,   9.4357450e-01,   1.0323330e-01,
      4.9007680e-01,   8.8266060e-01,   2.0762730e-01,
      4.4699470e-01,   7.3944620e-01,   1.7112460e-01,
      4.3608140e-01,   6.5507060e-01,   2.6039380e-01,
      5.2397040e-01,   1.0119895e+00,   4.3893800e-02,
      5.5439470e-01,   8.6866840e-01,   2.9800720e-01,
      3.6598370e-01,   9.7065040e-01,   2.4518290e-01,
      3.6118490e-01,   1.0241081e+00,   3.9061490e-01,
      2.9060140e-01,   9.3065670e-01,   4.9217910e-01,
      3.7959490e-01,   8.2747840e-01,   5.6509490e-01,
      4.7489660e-01,   8.8566260e-01,   6.5999840e-01,
      3.6368320e-01,   1.0597767e+00,   1.7876690e-01,
      2.7109760e-01,   9.1801620e-01,   2.1976640e-01,
      4.6308270e-01,   1.0535677e+00,   4.2458870e-01,
      3.0408980e-01,   1.1200270e+00,   3.8945130e-01,
      2.4102310e-01,   9.9350950e-01,   5.6985060e-01,
      2.0650090e-01,   8.7694710e-01,   4.4195250e-01,
      3.1373390e-01,   7.5969200e-01,   6.2397470e-01,
      4.3436430e-01,   7.6381370e-01,   4.9193540e-01,
      4.2919010e-01,   9.4480580e-01,   7.2843000e-01,
      5.2321340e-01,   8.1372660e-01,   7.1257620e-01,
      5.4668710e-01,   9.4138210e-01,   6.1529850e-01,
      4.3005860e-01,   7.0743700e-01,   3.9864300e-02,
      4.0135100e-01,   5.7411110e-01,  -1.0714700e-02,
      5.2156780e-01,   4.8090740e-01,   5.5637000e-03,
      5.0318970e-01,   3.6594870e-01,   4.5838100e-02,
      4.2986230e-01,   7.8189290e-01,  -3.0685800e-02,
      3.2159150e-01,   5.3039390e-01,   5.3770100e-02,
      3.5719220e-01,   5.6447950e-01,  -1.6064810e-01,
      2.1505230e-01,   6.1292080e-01,  -1.9694940e-01,
      1.8083700e-01,   6.1857820e-01,  -3.4630010e-01,
      1.2256990e-01,   7.1673190e-01,  -3.9272420e-01,
      2.1447240e-01,   5.1266540e-01,  -4.2435550e-01,
      4.3213640e-01,   6.1768180e-01,  -2.2387460e-01,
      3.6320930e-01,   4.5652160e-01,  -1.8991020e-01,
      1.3842180e-01,   5.4832890e-01,  -1.4831980e-01,
      1.9787870e-01,   7.1581880e-01,  -1.5792820e-01,
      1.9410910e-01,   5.1540570e-01,  -5.2552250e-01,
      2.6173360e-01,   4.2905600e-01,  -3.8706680e-01,
      6.4443970e-01,   5.2714230e-01,  -2.9472600e-02,
      7.6492660e-01,   4.4528850e-01,  -2.3679700e-02,
      8.1267200e-01,   4.2189770e-01,   1.1949190e-01,
      8.5613460e-01,   3.1152580e-01,   1.5131940e-01,
      6.5564860e-01,   6.2337320e-01,  -6.4222200e-02,
      7.3773890e-01,   3.4271780e-01,  -5.9094100e-02,
      8.8525000e-01,   4.9474480e-01,  -1.0902580e-01,
      8.6638090e-01,   4.7454560e-01,  -2.6121010e-01,
      9.9059470e-01,   5.0875020e-01,  -3.4389920e-01,
      1.1043645e+00,   5.0743090e-01,  -2.9566890e-01,
      9.6821120e-01,   5.3663670e-01,  -4.7456760e-01,
      9.0849540e-01,   6.0125350e-01,  -8.5851000e-02,
      9.7649320e-01,   4.3775400e-01,  -7.9019800e-02,
      8.4156480e-01,   3.6793310e-01,  -2.8313810e-01,
      7.8061420e-01,   5.3636490e-01,  -2.9638290e-01,
      1.0471382e+00,   5.6199100e-01,  -5.3567900e-01,
      8.7377490e-01,   5.3050340e-01,  -5.1637530e-01,
      8.0165120e-01,   5.2376810e-01,   2.0863920e-01,
      8.3929430e-01,   5.0825560e-01,   3.4833870e-01,
      7.5022360e-01,   4.0535750e-01,   4.1733500e-01,
      8.0202530e-01,   3.1601190e-01,   4.8429990e-01,
      7.7501130e-01,   6.1795630e-01,   1.7801170e-01,
      9.4102400e-01,   4.6122630e-01,   3.5141250e-01,
      8.4386270e-01,   6.4043280e-01,   4.3031960e-01,
      9.6367380e-01,   7.2981650e-01,   3.9688100e-01,
      1.0669959e+00,   6.8397780e-01,   3.4682830e-01,
      9.5287240e-01,   8.6059380e-01,   4.3183100e-01,
      7.4855910e-01,   6.9684800e-01,   4.1627260e-01,
      8.5175810e-01,   6.1675910e-01,   5.3925470e-01,
      1.0324007e+00,   9.2426680e-01,   4.1508240e-01,
      8.6836360e-01,   8.9898360e-01,   4.7721000e-01,
      6.1605550e-01,   4.0996770e-01,   3.9525860e-01,
      5.2484700e-01,   3.1108820e-01,   4.5028830e-01,
      5.4551330e-01,   1.7112030e-01,   3.9075870e-01,
      5.4586580e-01,   7.5823500e-02,   4.6854670e-01,
      5.7342180e-01,   4.8688930e-01,   3.4156400e-01,
      5.5439990e-01,   2.9647810e-01,   5.5717880e-01,
      3.7450150e-01,   3.5256220e-01,   4.4898160e-01,
      3.3151180e-01,   4.4205270e-01,   5.7008650e-01,
      1.9811110e-01,   5.1488990e-01,   5.3770400e-01,
      3.1074260e-01,   3.5776020e-01,   6.9954310e-01,
      3.5411410e-01,   4.0498450e-01,   3.5265220e-01,
      3.1018650e-01,   2.6142340e-01,   4.4756990e-01,
      4.1056750e-01,   5.1896960e-01,   5.8879500e-01,
      1.1621480e-01,   4.4335210e-01,   5.1458220e-01,
      1.6147940e-01,   5.7747960e-01,   6.2207950e-01,
      2.0780280e-01,   5.8296010e-01,   4.5023600e-01,
      4.0010850e-01,   2.9492280e-01,   7.2223710e-01,
      2.9367490e-01,   4.2341500e-01,   7.8879620e-01,
      2.2261370e-01,   2.8932120e-01,   6.9042990e-01,
      5.7224160e-01,   1.5500520e-01,   2.5823570e-01,
      5.9432320e-01,   2.3814100e-02,   1.9623650e-01,
      7.1301660e-01,  -5.1589900e-02,   2.5832040e-01,
      7.0281260e-01,  -1.7282990e-01,   2.7769800e-01,
      5.7188060e-01,   2.3611520e-01,   1.9462520e-01,
      5.0717760e-01,  -4.0653200e-02,   2.2477590e-01,
      6.0492470e-01,   2.2664300e-02,   3.9752000e-02,
      4.6908550e-01,   4.7016000e-02,  -3.0898400e-02,
      4.7705120e-01,   7.3896600e-02,  -1.8311310e-01,
      4.9957700e-01,  -4.9161900e-02,  -2.7423070e-01,
      4.8635300e-01,  -1.1543200e-02,  -4.1535290e-01,
      6.7946620e-01,   9.9238200e-02,   7.6740000e-03,
      6.4570940e-01,  -7.5885300e-02,   6.0694000e-03,
      3.9982350e-01,  -3.8362800e-02,  -1.1261400e-02,
      4.2154570e-01,   1.3629290e-01,   1.5825900e-02,
      3.8094930e-01,   1.2115970e-01,  -2.1457750e-01,
      5.5588170e-01,   1.5040690e-01,  -2.0378490e-01,
      6.0166160e-01,  -9.1069700e-02,  -2.5865810e-01,
      4.2604660e-01,  -1.3054550e-01,  -2.5329440e-01,
      3.9019950e-01,   8.0212000e-03,  -4.4129080e-01,
      5.4012220e-01,   7.0837500e-02,  -4.3950080e-01,
      5.1756610e-01,  -8.5103000e-02,  -4.7707190e-01,
      8.2381630e-01,   1.8609900e-02,   2.9484510e-01,
      9.3806320e-01,  -3.9249400e-02,   3.6419050e-01,
      9.2411000e-01,  -4.6476300e-02,   5.1839130e-01,
      1.0049152e+00,  -1.1554340e-01,   5.8129270e-01,
      8.3344160e-01,   1.1672500e-01,   2.6524400e-01,
      9.4123210e-01,  -1.4761530e-01,   3.3679720e-01,
      1.0775636e+00,   1.7610300e-02,   3.2359120e-01,
      1.1227200e+00,  -2.3648000e-03,   1.7593140e-01,
      1.0819502e+00,   1.1154940e-01,   7.8942900e-02,
      1.1701714e+00,   1.2857790e-01,  -4.7715900e-02,
      1.1533803e+00,   2.4644000e-02,  -1.4922820e-01,
      1.0859128e+00,   1.2487400e-01,   3.5276500e-01,
      1.1528982e+00,  -3.5467500e-02,   3.8700710e-01,
      1.2342212e+00,  -8.5730000e-03,   1.7651980e-01,
      1.0877723e+00,  -1.0106050e-01,   1.3703770e-01,
      9.7506050e-01,   1.0012950e-01,   4.8899800e-02,
      1.0873723e+00,   2.0864690e-01,   1.3356980e-01,
      1.1461199e+00,   2.2618260e-01,  -9.5883500e-02,
      1.2781596e+00,   1.3443610e-01,  -2.1116600e-02,
      1.1901092e+00,  -6.4888800e-02,  -1.1962710e-01,
      1.0568444e+00,   9.6706000e-03,  -1.7595550e-01,
      1.2028383e+00,   4.8784900e-02,  -2.3420850e-01,
      8.2310190e-01,   1.8833900e-02,   5.8260040e-01,
      7.9253150e-01,   7.7510000e-03,   7.2603570e-01,
      6.9659030e-01,  -1.0886810e-01,   7.5813830e-01,
      7.0484560e-01,  -1.6782760e-01,   8.6681000e-01,
      7.6547160e-01,   8.6303700e-02,   5.3068350e-01,
      8.8634640e-01,  -1.5797400e-02,   7.8284800e-01,
      7.2840110e-01,   1.3689560e-01,   7.8729060e-01,
      8.1939560e-01,   2.6161360e-01,   7.9812550e-01,
      8.2792630e-01,   3.2140360e-01,   9.3909230e-01,
      8.4194350e-01,   2.4440470e-01,   1.0379413e+00,
      8.2131750e-01,   4.4684810e-01,   9.5178530e-01,
      6.3548880e-01,   1.6325520e-01,   7.3123490e-01,
      6.9269250e-01,   1.1056850e-01,   8.8989970e-01,
      9.2265100e-01,   2.3857210e-01,   7.6226400e-01,
      7.8040600e-01,   3.4092450e-01,   7.2973180e-01,
      6.0114420e-01,  -1.3899150e-01,   6.6654730e-01,
      4.9029810e-01,  -2.3287470e-01,   6.8020600e-01,
      4.2682350e-01,  -2.5091660e-01,   5.4007190e-01,
      3.7946310e-01,  -1.5515410e-01,   4.7893260e-01,
      5.9684360e-01,  -8.6678600e-02,   5.7803980e-01,
      5.3104760e-01,  -3.3115170e-01,   7.1493560e-01,
      3.8769540e-01,  -1.7939010e-01,   7.8719350e-01,
      3.1942980e-01,  -4.3320900e-02,   7.5821020e-01,
      2.2487020e-01,   8.9690000e-03,   8.6880130e-01,
      2.9345170e-01,   6.0432300e-02,   9.9759500e-01,
      1.9486760e-01,   9.7170100e-02,   1.0998808e+00,
      3.0819250e-01,  -2.5521550e-01,   8.0307230e-01,
      4.3864030e-01,  -1.7513140e-01,   8.8600400e-01,
      3.9529950e-01,   3.4129500e-02,   7.3387380e-01,
      2.5950160e-01,  -5.5097800e-02,   6.6518850e-01,
      1.6310750e-01,   9.1928100e-02,   8.2654320e-01,
      1.5286480e-01,  -7.1758000e-02,   8.9527090e-01,
      3.6011380e-01,  -1.7789900e-02,   1.0405722e+00,
      3.5654600e-01,   1.4994340e-01,   9.7624130e-01,
      1.3804390e-01,   1.7680540e-01,   1.0733447e+00,
      1.3101240e-01,   2.1578300e-02,   1.1216283e+00,
      2.3995470e-01,   1.2182900e-01,   1.1869873e+00,
      4.3154850e-01,  -3.7570610e-01,   4.8568600e-01,
      3.7998430e-01,  -4.0984870e-01,   3.5397140e-01,
      4.7686720e-01,  -4.5962120e-01,   2.4663120e-01,
      4.3726970e-01,  -4.7332260e-01,   1.3020350e-01,
      4.5827550e-01,  -4.5578520e-01,   5.4485500e-01,
      3.0324710e-01,  -4.9032270e-01,   3.6459090e-01,
      3.2516550e-01,  -3.2326550e-01,   3.0965240e-01,
      6.0191410e-01,  -4.9634090e-01,   2.8536810e-01,
      6.9991750e-01,  -5.6669990e-01,   2.0215210e-01,
      6.9409280e-01,  -7.2034850e-01,   2.2238040e-01,
      7.8113210e-01,  -7.9396900e-01,   1.7435070e-01,
      6.3083590e-01,  -4.8086600e-01,   3.8286300e-01,
      6.7335420e-01,  -5.4891650e-01,   9.4909500e-02,
      8.4593740e-01,  -5.1626330e-01,   2.2827870e-01,
      8.9997550e-01,  -5.5272300e-01,   3.7127100e-01,
      1.0200414e+00,  -6.5117550e-01,   3.6332300e-01,
      9.4276780e-01,  -4.2703490e-01,   4.5115280e-01,
      9.1154540e-01,  -5.6113880e-01,   1.4999340e-01,
      8.4977590e-01,  -4.0620940e-01,   2.1028200e-01,
      8.1835040e-01,  -6.0206250e-01,   4.2947910e-01,
      1.1120110e+00,  -6.0247770e-01,   3.2413290e-01,
      1.0451642e+00,  -6.9363510e-01,   4.6316180e-01,
      9.9738500e-01,  -7.3734650e-01,   2.9651290e-01,
      1.0357670e+00,  -3.8001410e-01,   4.1262440e-01,
      8.6484390e-01,  -3.4800300e-01,   4.4969570e-01,
      9.6110330e-01,  -4.5022750e-01,   5.5840210e-01,
      5.9643160e-01,  -7.6806880e-01,   3.0410080e-01,
      5.8704010e-01,  -9.0045250e-01,   3.6436890e-01,
      4.4133160e-01,  -9.5106810e-01,   3.7101380e-01,
      3.4722080e-01,  -8.6670950e-01,   3.7457010e-01,
      5.0697340e-01,  -7.1801080e-01,   3.0877740e-01,
      6.4490400e-01,  -9.7348450e-01,   3.0275600e-01,
      6.4354700e-01,  -8.9008370e-01,   5.1020020e-01,
      7.6341760e-01,  -9.7823400e-01,   5.3976370e-01,
      7.5290460e-01,  -1.1181550e+00,   5.3192130e-01,
      8.8563180e-01,  -9.2253210e-01,   5.8187120e-01,
      8.6187760e-01,  -1.1997086e+00,   5.6482950e-01,
      9.9505290e-01,  -1.0042641e+00,   6.1286710e-01,
      9.8309970e-01,  -1.1430553e+00,   6.0478560e-01,
      6.6915780e-01,  -7.8472170e-01,   5.3665410e-01,
      5.6413480e-01,  -9.1830040e-01,   5.8314440e-01,
      6.5793860e-01,  -1.1652637e+00,   5.0118910e-01,
      8.9667310e-01,  -8.1328420e-01,   5.9209770e-01,
      8.5177230e-01,  -1.3092726e+00,   5.5907860e-01,
      1.0904713e+00,  -9.5872680e-01,   6.4428120e-01,
      1.0688544e+00,  -1.2075179e+00,   6.3074730e-01,
      4.2022320e-01,  -1.0755661e+00,   3.7395090e-01,
    };

    const double* positionsDataPtr = positionsData;
    for (unsigned int ii = 0; ii < numberOfParticles; ii++) { 
        positions[ii]     = Vec3(positionsDataPtr[0], positionsDataPtr[1], positionsDataPtr[2]);
        positionsDataPtr += 3;
    }

    LangevinIntegrator integrator(0.0, 0.1, 0.01);
    Context context(system, integrator, platform);

    context.setPositions(positions);
    State state                      = context.getState(State::Forces | State::Energy);
    forces                           = state.getForces();
    energy                           = state.getPotentialEnergy();
    
    // Check that the forces and energy are consistent.
    
    checkFiniteDifferences(forces, context, positions);
}

// compare forces and energies 

static void compareForcesEnergy(std::string& testName, double expectedEnergy, double energy,
                                const std::vector<Vec3>& expectedForces,
                                const std::vector<Vec3>& forces, double tolerance) {
    for (unsigned int ii = 0; ii < forces.size(); ii++) {
        ASSERT_EQUAL_VEC(expectedForces[ii], forces[ii], tolerance);
    }
    ASSERT_EQUAL_TOL_MOD(expectedEnergy, energy, tolerance, testName);
}

// compare relative differences in force norms and energies 

static void compareForceNormsEnergy(std::string& testName, double expectedEnergy, double energy,
                                     std::vector<Vec3>& expectedForces,
                                     const std::vector<Vec3>& forces, double tolerance) {
    for (unsigned int ii = 0; ii < forces.size(); ii++) {
        double expectedNorm = sqrt(expectedForces[ii][0]*expectedForces[ii][0] +
                                   expectedForces[ii][1]*expectedForces[ii][1] +
                                   expectedForces[ii][2]*expectedForces[ii][2]);

        double norm         = sqrt(forces[ii][0]*forces[ii][0] + forces[ii][1]*forces[ii][1] + forces[ii][2]*forces[ii][2]);
        double absDiff      = fabs(norm - expectedNorm);
        double relDiff      = 2.0*absDiff/(fabs(norm) + fabs(expectedNorm) + 1.0e-08);

        if (relDiff > tolerance && absDiff > 0.001) {
            std::stringstream details;
            details << testName << "Relative difference in norms " << relDiff << " larger than allowed tolerance at particle=" << ii;
            details << ": norms=" << norm << " expected norm=" << expectedNorm; 
            throwException(__FILE__, __LINE__, details.str());
        }
    }
    double energyAbsDiff = fabs(expectedEnergy - energy);   
    double energyRelDiff =  2.0*energyAbsDiff/(fabs(expectedEnergy) + fabs(energy) + 1.0e-08);   
    if (energyRelDiff > tolerance) {
        std::stringstream details;
        details << testName << "Relative difference in energies " << energyRelDiff << " larger than allowed tolerance.";
        details << "Energies=" << energy << " expected energy=" << expectedEnergy; 
        throwException(__FILE__, __LINE__, details.str());
    }
}

// test GK direct polarization for system comprised of two ammonia molecules

static void testGeneralizedKirkwoodAmmoniaDirectPolarization() {

    std::string testName      = "testGeneralizedKirkwoodAmmoniaDirectPolarization";

    int numberOfParticles     = 8;
    std::vector<Vec3> forces;
    double energy;

    System system;
    AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce  = new AmoebaGeneralizedKirkwoodForce();
    setupMultipoleAmmonia(system, amoebaGeneralizedKirkwoodForce, AmoebaMultipoleForce::Direct, 0);
    LangevinIntegrator integrator(0.0, 0.1, 0.01);
    Context context(system, integrator, platform);
    getForcesEnergyMultipoleAmmonia(context, forces, energy);
    std::vector<Vec3> expectedForces(numberOfParticles);

    double expectedEnergy     = -7.6636680e+01;

    expectedForces[0]         = Vec3( -6.9252994e+02,  -8.9085133e+00,   9.6489739e+01);
    expectedForces[1]         = Vec3(  1.5593797e+02,  -6.0331931e+01,   1.5104507e+01);
    expectedForces[2]         = Vec3(  1.5870088e+02,   6.1702809e+01,   6.7708985e+00);
    expectedForces[3]         = Vec3(  1.4089885e+02,   7.5870617e+00,  -1.1362294e+02);
    expectedForces[4]         = Vec3( -1.8916205e+02,   2.1465549e-01,  -4.3433152e+02);
    expectedForces[5]         = Vec3(  1.0208290e+01,   6.2676753e+01,   1.4987953e+02);
    expectedForces[6]         = Vec3(  4.0621859e+02,   1.8962203e-01,   1.3021956e+02);
    expectedForces[7]         = Vec3(  9.7274235e+00,  -6.3130458e+01,   1.4949024e+02);

    double tolerance          = 1.0e-04;
    compareForcesEnergy(testName, expectedEnergy, energy, expectedForces, forces, tolerance);
}

static void testGeneralizedKirkwoodAmmoniaExtrapolatedPolarization() {

    std::string testName      = "testGeneralizedKirkwoodAmmoniaExtrapolatedPolarization";

    int numberOfParticles     = 8;
    std::vector<Vec3> forces;
    double energy;

    System system;
    AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce  = new AmoebaGeneralizedKirkwoodForce();
    setupMultipoleAmmonia(system, amoebaGeneralizedKirkwoodForce, AmoebaMultipoleForce::Direct, 0);
    LangevinIntegrator integrator(0.0, 0.1, 0.01);
    Context context(system, integrator, platform);

    // We don't have reference values for this case, but at least check that force and energy are consistent.

    getForcesEnergyMultipoleAmmonia(context, forces, energy);
}

// test GK mutual polarization for system comprised of two ammonia molecules

static void testGeneralizedKirkwoodAmmoniaMutualPolarization() {

    std::string testName      = "testGeneralizedKirkwoodAmmoniaMutualPolarization";

    int numberOfParticles     = 8;
    std::vector<Vec3> forces;
    double energy;

    System system;
    AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce  = new AmoebaGeneralizedKirkwoodForce();
    setupMultipoleAmmonia(system, amoebaGeneralizedKirkwoodForce, AmoebaMultipoleForce::Mutual, 0);
    LangevinIntegrator integrator(0.0, 0.1, 0.01);
    Context context(system, integrator, platform);
    getForcesEnergyMultipoleAmmonia(context, forces, energy);
    std::vector<Vec3> expectedForces(numberOfParticles);

    double expectedEnergy     =  -7.8018875e+01;

    expectedForces[0]         = Vec3( -7.6820301e+02,  -1.0102760e+01,   1.0094389e+02);
    expectedForces[1]         = Vec3(  1.7037307e+02,  -7.5621857e+01,   2.3320365e+01);
    expectedForces[2]         = Vec3(  1.7353828e+02,   7.7199741e+01,   1.3965379e+01);
    expectedForces[3]         = Vec3(  1.5045244e+02,   8.5784569e+00,  -1.3377619e+02);
    expectedForces[4]         = Vec3( -2.1811615e+02,  -1.6818022e-01,  -4.6103163e+02);
    expectedForces[5]         = Vec3(  6.2091942e+00,   7.6748687e+01,   1.5883463e+02);
    expectedForces[6]         = Vec3(  4.8035662e+02,   4.9704902e-01,   1.3948083e+02);
    expectedForces[7]         = Vec3(  5.3895456e+00,  -7.7131137e+01,   1.5826273e+02);

    double tolerance          = 2.0e-04;
    compareForcesEnergy(testName, expectedEnergy, energy, expectedForces, forces, tolerance);
}

// test GK mutual polarization for system comprised of two ammonia molecules
// including cavity term

static void testGeneralizedKirkwoodAmmoniaMutualPolarizationWithCavityTerm() {

    std::string testName      = "testGeneralizedKirkwoodAmmoniaMutualPolarizationWithCavityTerm";

    int numberOfParticles     = 8;
    std::vector<Vec3> forces;
    double energy;

    System system;
    AmoebaGeneralizedKirkwoodForce* amoebaGeneralizedKirkwoodForce  = new AmoebaGeneralizedKirkwoodForce();
    setupMultipoleAmmonia(system, amoebaGeneralizedKirkwoodForce, AmoebaMultipoleForce::Mutual, 1);
    LangevinIntegrator integrator(0.0, 0.1, 0.01);
    Context context(system, integrator, platform);
    getForcesEnergyMultipoleAmmonia(context, forces, energy);
    std::vector<Vec3> expectedForces(numberOfParticles);

    double expectedEnergy     = -6.0434582e+01;

    expectedForces[0]         = Vec3( -7.8323218e+02,  -1.0097644e+01,   1.0256890e+02);
    expectedForces[1]         = Vec3(  1.7078480e+02,  -7.1896701e+01,   2.0840172e+01);
    expectedForces[2]         = Vec3(  1.7394089e+02,   7.3488594e+01,   1.1484648e+01);
    expectedForces[3]         = Vec3(  1.5169364e+02,   8.5611299e+00,  -1.2968050e+02);
    expectedForces[4]         = Vec3( -2.1669693e+02,  -1.5926823e-01,  -4.6636274e+02);
    expectedForces[5]         = Vec3(  8.7397444e+00,   7.3330990e+01,   1.6016898e+02);
    expectedForces[6]         = Vec3(  4.8684950e+02,   4.8937161e-01,   1.4137061e+02);
    expectedForces[7]         = Vec3(  7.9205382e+00,  -7.3716473e+01,   1.5960993e+02);

    double tolerance          = 1.0e-04;
    compareForcesEnergy(testName, expectedEnergy, energy, expectedForces, forces, tolerance);
    
    // Try changing the particle parameters and make sure it's still correct.
    
    for (int i = 0; i < numberOfParticles; i++) {
        double charge, radius, scale;
        amoebaGeneralizedKirkwoodForce->getParticleParameters(i, charge, radius, scale);
        amoebaGeneralizedKirkwoodForce->setParticleParameters(i, charge, 0.9*radius, 1.1*scale);
    }
    LangevinIntegrator integrator2(0.0, 0.1, 0.01);
    Context context2(system, integrator2, context.getPlatform());
    context2.setPositions(context.getState(State::Positions).getPositions());
    State state1 = context.getState(State::Forces | State::Energy);
    State state2 = context2.getState(State::Forces | State::Energy);
    bool exceptionThrown = false;
    try {
        // This should throw an exception.
        compareForcesEnergy(testName, state2.getPotentialEnergy(), state1.getPotentialEnergy(), state2.getForces(), state1.getForces(), tolerance);
    }
    catch (std::exception ex) {
        exceptionThrown = true;
    }
    ASSERT(exceptionThrown);
    amoebaGeneralizedKirkwoodForce->updateParametersInContext(context);
    state1 = context.getState(State::Forces | State::Energy);
    compareForcesEnergy(testName, state2.getPotentialEnergy(), state1.getPotentialEnergy(), state2.getForces(), state1.getForces(), tolerance);
}

// test GK direct polarization for villin system

static void testGeneralizedKirkwoodVillinDirectPolarization() {

    std::string testName      = "testGeneralizedKirkwoodVillinDirectPolarization";

    int numberOfParticles     = 596;
    std::vector<Vec3> forces;
    double energy;

    setupAndGetForcesEnergyMultipoleVillin(AmoebaMultipoleForce::Direct, 0, forces, energy);
    std::vector<Vec3> expectedForces(numberOfParticles);

    double expectedEnergy            = -8.4281157e+03;

    static const double forceData[] = {
      1.4000030e+02,   6.5353942e+01,  -1.5232364e+02,
     -2.0189390e+01,  -8.9048971e+01,   1.6728524e+02,
      1.5879992e+02,   6.1585467e+02,  -4.3582690e+02,
     -9.4397437e+01,  -4.9012290e+02,   2.8622327e+02,
      9.7540437e+01,   1.7059207e+02,   1.3220725e+02,
     -4.1003215e+01,  -1.2087264e+02,   2.1160810e+02,
     -1.6641387e+02,  -7.6466832e+01,  -1.2896755e+02,
      4.5541157e+01,  -6.5763629e+00,  -1.3201541e+01,
     -1.4634834e+02,   6.7671906e+01,  -1.9498260e+02,
      5.7508243e+00,  -1.9056737e+01,  -8.5842367e+01,
      2.0617180e+01,   1.8161043e+01,   1.2222388e+01,
     -1.0069398e+01,  -2.6866960e+01,  -3.6282043e+01,
      1.6369919e+01,  -1.7790945e+01,   4.9122751e+01,
      1.7591311e+01,  -4.1589085e+00,   3.1746846e+01,
     -1.2445639e+00,   1.1963558e+01,  -1.2269262e+00,
      1.4676422e+01,   1.4069838e+01,   1.9992267e+00,
     -2.1582184e+00,   2.0154671e+01,   8.6462734e+00,
     -1.3596918e+01,   9.8368612e+00,  -3.2313480e+01,
     -5.5508526e+00,   2.2207441e+01,   8.1710039e+00,
      1.3469339e+02,  -2.4291299e+02,   1.0232561e+02,
     -2.2198013e+02,   1.6315708e+02,  -1.1005164e+02,
      3.6801227e+02,  -6.5703855e+02,   7.4641162e+02,
     -1.5977900e+02,   9.0842552e+02,  -6.9524064e+02,
     -4.3680341e+01,  -5.3652465e+01,  -1.2230089e+01,
      1.9031197e+01,  -3.3808608e+01,   1.5736268e+01,
      6.8055240e+01,   1.7442807e+02,   4.9831944e+01,
     -3.9646089e+01,  -1.6959202e+01,   8.7332612e+00,
     -9.9780657e+00,   2.3735107e+01,  -7.8230715e+00,
      1.5740440e+01,   2.4704610e+00,  -1.9459046e+01,
      2.6687182e+01,  -1.4420451e+01,   1.8603458e+01,
     -1.5252752e+01,  -4.2373176e+01,  -1.6440064e+01,
      3.9188915e+00,  -7.4464241e+00,   1.0387341e+01,
     -1.7432370e+00,  -1.6718638e+01,  -8.9559356e+00,
      1.1114716e+01,   9.9589573e+00,  -2.2863253e+01,
      3.1340802e+01,  -1.9795450e+01,   2.7037626e+01,
     -2.0911805e+01,   4.0303061e+00,   1.4784916e+01,
      9.4007856e-01,  -8.1214867e+00,   2.7327458e+01,
      1.9783699e+01,   1.0438279e+01,  -1.1174353e+01,
      8.7008455e+01,   1.4552822e+02,  -2.1186400e+02,
      3.6805750e+01,   1.2518355e+02,   3.1287438e+02,
     -5.3053977e+02,  -2.2455742e+02,  -5.0295123e+02,
      3.7417173e+02,   1.6300800e+02,   2.3257179e+02,
     -3.9040319e+01,   2.0019745e+01,  -4.0700873e+01,
     -2.1457694e+01,   2.4209024e+01,  -4.8872811e+01,
     -8.7269544e+01,  -9.2227028e+01,  -1.3292053e+02,
      7.2208729e+01,   1.2816445e+02,   2.1008199e+02,
     -1.1016274e+01,   1.0798074e+00,  -1.3585421e+01,
     -2.0034600e+00,  -1.4851278e+01,  -1.9817283e+01,
     -1.7709907e-01,  -1.4750550e+02,  -4.8650586e+01,
      1.9355993e+02,  -5.5808320e+01,   2.1682408e+02,
     -1.8889405e+02,  -2.9256324e+01,   7.5746603e+01,
      2.0663820e+02,  -4.6433770e+02,  -4.6497356e+02,
     -3.6963757e+01,   4.0607640e+02,   3.0048540e+02,
      2.9428461e+01,   1.4115272e+02,  -6.3331956e+01,
      4.0708380e+01,   1.4076074e+01,   9.0939909e+00,
     -7.8932461e+01,   1.2348647e+02,  -2.8659563e+02,
      1.4954841e+02,  -3.4247585e+02,   8.0432415e+02,
     -2.2943343e+02,   1.3359207e+02,  -1.7996883e+02,
      1.1879030e+02,   1.0175199e+02,  -4.3868455e+02,
     -1.5818498e+01,  -4.1655954e+01,   5.3614847e+01,
      2.0229869e+01,  -1.0894737e+01,   4.1283706e+01,
     -1.7120668e+02,   1.4536093e+02,   1.8814506e+02,
      1.6848188e+02,   3.6819794e+01,  -6.0856163e+01,
     -1.0466706e+02,  -2.3174214e+02,  -6.1868995e+02,
      7.3073476e+00,   5.7125415e+01,   4.4488291e+02,
      2.4330554e+01,   4.0760693e+01,  -1.7894461e+00,
      2.3648644e+01,   2.8129980e+01,  -1.2922394e+00,
     -7.6629304e+01,  -7.1037688e+01,   1.3867825e+01,
     -1.2493867e+02,  -2.0607152e+02,   1.2279087e+02,
      3.3127420e+02,   5.5963754e+02,  -3.5508408e+02,
     -2.9111339e+02,  -1.4139900e+02,  -2.4495020e+01,
      1.2211045e+02,  -2.7730883e+02,   3.2750559e+02,
      3.5393186e+01,  -5.2304358e+01,   4.7193396e-02,
     -1.9570297e+01,  -5.1284799e+01,   9.5940583e+00,
     -1.4872038e+00,   9.4617863e+01,  -1.5254162e+01,
      3.7872301e+00,   3.8523881e+01,  -7.2948839e+01,
     -9.8524843e+01,   9.9175920e+01,   2.4268707e+02,
      1.3636866e+02,   1.4775829e+02,  -1.8128883e+02,
     -4.5480233e+02,  -3.0522449e+02,  -3.2388172e+02,
      2.4236878e+02,   1.4155564e+02,   2.3941615e+02,
     -5.2935517e+00,  -6.4182390e+01,   2.1074525e+01,
     -3.1246345e+01,   1.1951498e-01,   3.4142506e+01,
      6.1044962e+01,  -3.1481157e+02,   3.4253842e+01,
     -9.2877239e+01,   8.7462878e+02,  -7.4287135e+01,
      2.3153728e+02,  -2.8349661e+02,  -5.5740190e+01,
     -1.4579569e+02,  -4.3091067e+02,   1.2021746e+02,
     -1.5998859e-01,   5.3399978e+01,  -2.9907025e+01,
     -5.9185068e+00,   4.4733968e+01,   1.4697301e+01,
      1.5863026e+02,   9.0815140e+01,   1.2908163e+02,
     -1.2795117e+02,   1.7919025e+01,   1.3591912e+02,
     -1.9791271e+02,  -5.4319704e+02,  -3.2185076e+02,
      2.9502998e+02,   3.6129241e+02,   2.9571349e+02,
      2.9380401e+01,  -5.5510105e+01,  -8.7772337e+01,
      1.2424464e+01,  -4.9878478e-01,   3.7909147e+01,
     -1.2143128e+01,   5.0290393e+01,  -1.8035721e+02,
      9.8155999e+00,  -1.6469131e+01,   2.0393257e+01,
     -3.3936915e+01,   3.2089976e+01,   2.7191570e+01,
      3.1349780e+01,   2.9090255e+01,  -7.5946001e+01,
     -6.4328625e+00,  -1.7361160e+01,   1.1080106e+01,
      2.3227120e+01,  -1.0925778e+01,  -5.6151998e+00,
     -5.4997351e+00,  -2.9281763e+00,   1.1255536e+01,
     -1.6974066e+01,  -3.2286284e+01,  -1.1713997e+01,
     -2.7164340e+01,  -1.2466184e+01,   3.9898025e+01,
      1.2483124e+01,  -7.8191653e+00,  -1.7048420e+01,
     -3.9606058e+01,   2.0431528e+01,   1.3407056e+01,
      1.4872100e+01,   1.6385246e+01,   1.7888571e+00,
     -3.1500767e+00,  -2.9533724e+01,  -6.0179250e+00,
      8.0052803e+00,   1.2438449e-01,   3.3748541e+00,
      8.4060418e+01,   2.5471916e+02,   6.6896840e+01,
      9.1878680e+01,  -1.4469507e+02,   8.7246189e+01,
      1.7048966e+02,  -2.8831698e+02,  -5.7031052e+02,
     -1.3923221e+02,   2.4766624e+02,   3.7236286e+02,
      1.7698918e+01,   1.4982282e+01,  -3.1122759e+01,
      1.9999433e+01,   1.4974576e+01,   7.7943928e+00,
     -1.4557826e+02,  -3.6412029e+01,  -2.6410946e+01,
      3.6926253e+01,   1.6497102e+01,  -1.2173012e+01,
      1.1820992e+02,  -6.7261414e+01,  -1.0921796e+02,
      5.1648833e+01,   1.0928439e+01,  -1.7376342e+02,
     -1.0206392e+02,   2.4227501e+01,   5.4376335e+01,
     -3.9367403e+00,  -2.5581826e+01,  -2.0876132e+01,
      3.2197131e+01,   2.8138042e+00,   6.9882166e+00,
      3.1177282e+01,   8.7078224e+00,   1.5375218e+01,
     -8.2404071e+00,   9.7116623e+00,   7.0805493e+00,
     -2.8246783e+01,   8.2637599e+00,   5.9782615e+00,
     -3.0136964e+01,   5.1640335e+00,   1.0232237e+01,
     -4.7222992e+01,  -2.5080809e+01,   6.5882729e+01,
     -7.3439623e+00,   1.1566649e+01,   8.0365879e+01,
     -1.7865530e+02,  -2.6952325e+01,  -2.6287441e+01,
      9.7597931e+01,  -1.0925912e+02,   4.3237798e+01,
      8.3241075e+00,   1.5724371e+02,   4.1268453e+01,
     -1.8821625e+02,   1.2959892e+02,   2.5441914e+02,
      2.7349408e+02,   3.5610728e+01,  -1.5261797e+02,
     -5.6910531e+02,   2.8511431e+02,  -3.7000936e+02,
      3.4422161e+02,  -1.7143496e+02,   3.2051471e+02,
      9.3383973e+00,   4.9007425e+01,  -1.7827871e+01,
     -2.6001820e+01,   3.4953008e+01,  -9.9788546e+00,
     -8.3418265e+01,  -2.4961604e+02,   9.2937718e+01,
     -2.5908106e+01,  -6.7558559e-02,   1.4130526e+01,
      3.3265678e+01,   7.2635063e+00,  -2.7513816e+01,
      4.6614788e+01,   7.2575857e+01,  -1.4962613e+01,
      2.2543415e+02,  -2.8887421e+02,   1.5563832e+02,
     -1.3545780e+02,   2.2774344e+02,   2.4950166e+01,
     -2.2769173e+02,  -3.0997943e+02,  -5.0125865e+02,
      1.5981991e+02,   1.7711418e+02,   3.3693513e+02,
     -2.7740422e+01,   6.1264470e+01,   1.7700527e+01,
     -2.2177665e+01,   1.1134910e+00,   2.3875119e+01,
     -3.3679323e+01,   5.4672271e+01,  -3.1598009e+01,
      4.3653322e+01,  -7.0031742e+01,  -2.0602697e+01,
      8.8920376e+01,  -3.5944769e+01,  -3.6377050e+00,
      1.5047214e+01,  -2.7107111e+01,  -1.6526834e+01,
     -4.2691925e+01,   4.4677732e+01,   4.2855629e+00,
      7.9031621e+00,  -2.8455099e+00,   4.3216590e+00,
      1.2044240e+01,   9.8123471e+00,  -2.5336044e+01,
     -2.6150742e+01,  -1.6692444e+01,  -1.8230336e+01,
      1.3747821e+00,  -1.9809569e+01,  -1.6919914e+01,
      1.8608912e+01,  -2.0692743e+00,   2.1612729e+01,
      2.6026755e+02,   1.0216338e+02,   1.2619385e+02,
     -1.6378340e+02,  -3.1704129e+01,   5.5911179e+01,
     -2.8633841e+01,  -5.8178799e+02,  -3.9314305e+02,
      3.8112399e+01,   3.4837330e+02,   2.3640650e+02,
     -7.9585402e+01,  -4.1676229e+01,   6.7039573e+01,
     -1.4519696e+01,  -1.0722733e+01,   3.8738152e+01,
     -1.7548163e+01,   1.2958394e+02,  -2.2120200e+02,
      2.6680714e+01,  -5.5558346e+01,   7.2778356e+01,
     -1.5293880e+01,  -3.9829682e+01,  -1.8617704e+01,
      4.8432794e+01,   4.9642719e+01,  -1.5183591e+01,
     -2.5749201e+01,   4.2479310e+00,   8.1167424e+00,
     -3.6710657e+00,   1.2107987e+01,   5.2837887e+00,
     -1.7823824e+01,  -4.6512214e+00,  -1.5591513e+01,
     -3.0888260e+01,   3.5661830e+00,   3.1839294e+00,
     -3.7761614e+01,   7.5677091e+00,   4.5146589e+01,
      3.2250371e+00,   2.0356276e+01,   1.1263316e+01,
     -1.8784006e+01,  -2.1974713e+01,  -2.4161712e+01,
      8.2508424e+00,   6.4011359e-01,  -3.1933307e+01,
     -2.6580756e+00,  -1.4359021e+01,  -7.6334715e+00,
      8.1035680e+00,   5.8170147e-01,  -6.1540995e+00,
      1.3921176e+02,   2.6944643e+02,   2.0130772e+02,
     -5.8789728e+01,  -6.4394875e+01,  -1.1705271e+02,
      5.6333667e+02,  -1.1086977e+02,  -3.1605004e+02,
     -4.6490965e+02,   4.5464460e+01,   2.3943661e+02,
     -2.5692105e+01,  -7.0670968e-01,   1.0303890e+01,
     -1.2861659e+01,   3.2401134e+00,   2.0042209e+00,
      3.6405711e+01,   7.4913336e-01,  -1.6462419e+01,
     -1.3207259e+02,   8.8154682e+01,   1.9865442e+02,
      2.9441118e+01,  -2.4206087e+02,  -1.4109047e+02,
      1.7083445e+02,   7.8121902e+02,  -2.8200275e+02,
     -1.0827067e+02,  -4.3453364e+02,   2.0580121e+02,
     -2.5611369e+01,  -9.4881949e-01,   6.7172643e+01,
     -2.8945663e-01,  -1.3356301e+01,   3.1279379e+01,
      4.4138130e+00,   1.0481968e+02,  -7.9705077e+01,
     -1.9494704e+01,   5.5650633e+01,  -2.3886336e+00,
     -1.1400765e+01,  -3.2427083e+01,   1.9569122e+01,
     -6.3773140e+00,   3.6730705e+01,  -3.7248333e+01,
      3.6444252e+01,  -8.4145500e+00,   1.2330941e+01,
     -3.5925645e+01,  -4.9912236e+01,  -1.0535562e+00,
     -4.2908957e+00,  -1.0747732e+01,   1.2858163e+01,
      4.5359822e+00,  -2.7222287e+00,   1.0692303e+01,
      2.0286964e+01,  -9.0569942e-01,   2.1567689e+01,
      2.9595321e+00,   3.5574308e+01,   1.0187023e+01,
      9.2818598e+00,  -1.2677467e+01,   1.1071709e+01,
     -1.3591435e+02,  -1.4123739e+02,   2.7584935e+02,
      1.9553568e+02,   2.7539021e+02,  -1.3587169e+02,
     -1.3186737e+02,  -6.4855676e+02,  -2.6732527e+01,
     -9.4005752e+01,   4.6306226e+02,   9.7706920e+01,
      1.4668791e+02,  -1.3338307e+02,  -1.4093522e+02,
      1.2815327e+01,  -4.8246914e+01,   3.8536622e+01,
     -5.2125297e+01,  -5.2051930e+01,  -2.0491477e+02,
     -6.0076441e+01,  -3.0494621e+01,   1.9476556e+02,
     -1.8177063e+01,  -3.6084718e+01,   1.8946474e+01,
      1.1898101e+01,  -8.3186762e+00,   1.0878117e+01,
     -4.5405594e+01,   7.5428695e+01,  -1.0682497e+02,
     -5.0054588e+00,  -6.3038574e+00,   2.6355807e+01,
     -2.2389455e+01,   9.0702339e+00,  -2.3114849e+01,
      1.5292871e+01,  -1.7452764e+01,  -2.3883137e+01,
      5.3737980e+01,   1.9235152e+02,   2.6697696e+01,
     -9.1578263e+01,   8.2138692e+01,  -8.9846578e+01,
     -4.6937464e+02,  -3.6581045e+02,   1.6542077e+02,
      4.0216338e+02,   2.7462757e+02,  -1.5486708e+01,
      8.4090809e+00,   3.2513413e+01,   3.2933639e+01,
     -1.2285427e+01,   3.0977349e+01,   6.1815424e+00,
      3.6120317e+01,  -1.6454604e+02,  -4.3047115e+01,
      3.4056908e+01,  -1.7809398e+01,   5.8526924e+01,
      1.0309191e+02,   7.2479344e+01,   3.4494379e+01,
     -5.1500506e+01,  -5.6599531e+01,   7.0442474e+01,
     -4.3464283e+01,  -6.3055594e+00,  -8.8798624e+01,
      3.3378260e+00,  -2.4236730e+01,  -2.1535753e+01,
     -4.7812862e+00,  -2.2804456e+01,  -1.8557857e+01,
     -1.0159266e+01,  -3.1915604e+00,   1.0874852e+01,
      1.6444510e+00,   3.7534298e+01,  -6.6276593e+00,
     -6.4536555e+00,   2.3150719e+01,  -1.3647830e+01,
      1.1082208e+00,   1.6464762e+01,  -1.7644091e+01,
     -4.6543574e+00,  -1.7665480e+01,  -2.9294635e+01,
      2.8917967e+01,  -2.4484990e+00,  -2.8392335e+01,
     -3.4567618e+00,  -4.3015871e+01,   2.2701607e+01,
     -5.5457810e+01,   1.4658764e+00,  -2.9815242e+01,
     -1.0650036e+02,  -4.1626432e+02,   2.1619937e+02,
      1.5496809e+00,  -1.4194650e+01,   3.1941612e+01,
      1.0866963e+01,   1.7823313e+01,   1.1195647e+01,
      2.0388557e+02,   9.0545705e+01,  -1.2306449e+02,
     -1.4245507e+02,   8.4168114e+01,   1.1156065e+02,
      2.2357635e+01,  -5.2153456e+02,   3.1163372e+02,
      9.5760348e+00,   2.6030335e+02,  -2.5632187e+02,
      4.6029581e+01,   9.1350944e-01,  -8.6635586e+00,
      3.9564015e+00,   1.7951177e+01,   1.2513075e+01,
     -6.6691183e+01,   6.1996293e+00,   6.4117591e+01,
      1.3888684e+02,  -1.4673761e+02,  -1.2127299e+02,
     -3.8789119e+01,  -1.2397102e+01,   2.6227275e+01,
      1.9201304e+01,   1.3886645e+01,   2.6517459e+01,
     -1.1085638e+02,   8.7730354e+01,  -7.0762079e+01,
     -1.2504763e+02,   1.8195398e+02,  -2.0239160e+02,
      2.4558795e+02,  -5.5583697e+01,   1.4837003e+02,
     -1.1541573e+02,  -5.9476552e+02,  -2.4232479e+02,
      6.8110749e+01,   4.1968723e+02,   1.1978571e+02,
      6.2187118e+01,   4.1077507e+00,   2.1439830e+00,
      2.4011514e+01,  -1.6804897e+00,  -4.0615632e+01,
     -2.3512768e+02,  -1.4705005e+01,   1.0426947e+02,
     -6.7084722e+00,  -2.7037270e+01,   6.0493404e+00,
      5.5199853e+01,   3.0820608e+01,   1.8719536e+00,
      1.1626042e+01,   3.3726703e+01,  -2.9501831e+00,
     -6.3727794e-02,   2.4224522e+02,   7.5825416e+01,
      3.9252839e+01,   2.8288963e+01,  -2.0274336e+02,
     -3.9616266e+02,  -4.9669320e+02,  -4.8355900e+01,
      2.0305604e+02,   3.3502938e+02,   8.9470408e+01,
     -4.5683857e+01,  -5.5945916e+01,   1.3695400e+02,
     -8.5366246e-01,   3.8067700e+01,  -8.3678815e+00,
      5.6650892e+01,  -1.1171370e+02,   1.1671310e+02,
     -1.7538425e+01,  -1.6184638e+01,  -1.3571456e+01,
      2.9692755e+01,  -4.3046098e+01,  -1.8901137e+01,
      2.0477449e+01,   1.5122565e+01,   5.4724553e+01,
     -1.4045535e+01,  -3.5878293e+00,  -1.5615802e+01,
     -1.8606102e+01,   7.7247413e+00,  -1.0512434e+00,
     -3.1217125e+00,  -6.0343484e+00,   1.3026243e+01,
     -2.8843133e+01,   1.3655426e+01,  -8.9887589e+00,
     -6.9905727e+00,   7.1711531e+00,  -7.9207346e+01,
     -9.3912574e+00,   1.3808951e+01,   2.3213858e+01,
     -3.6338021e+00,  -1.6866408e+01,  -2.7878374e+01,
      1.1844538e+01,   9.4534382e+00,   1.1099442e+01,
      1.1897010e+01,  -5.9062019e+00,   5.7539869e-01,
      1.2239861e+02,   6.0662895e+01,  -6.0221236e+01,
      2.9176380e+02,   9.6417790e+01,   1.2570824e+02,
     -2.1083278e+02,   1.3755255e+02,  -5.5034720e+01,
     -6.5627912e+01,  -2.6429808e+02,   6.4741061e+02,
     -1.6998180e+00,   1.9138542e+02,  -4.6075196e+02,
     -1.0207215e+01,   2.1065126e+00,   4.6736578e+01,
      9.5943240e+00,   2.2652640e+01,   4.5949085e+01,
      6.0254225e+01,  -2.1306584e+02,  -1.9798092e+02,
     -2.8765755e+01,   4.4264764e+01,   5.6658884e+00,
     -1.8807301e+01,   7.1655387e+01,   2.0128801e+01,
      8.5576495e+00,  -2.5291594e+01,   1.2358651e+01,
      1.7712230e+02,   5.8171888e+01,  -2.3324960e+02,
      9.3265456e+01,   8.0646879e+01,   2.5054928e+02,
      2.3618076e+02,  -7.6980463e+02,  -2.6419460e+02,
     -1.7391540e+02,   5.4148942e+02,   1.2396707e+02,
     -6.7892523e+01,   4.0105242e+01,  -3.7861001e+01,
     -6.4026886e+00,  -3.0595797e+01,  -1.5128816e+01,
     -1.1419537e+02,  -3.0753155e+00,  -7.4305942e+01,
      8.3429198e+01,   3.0507628e+02,   3.2094098e+02,
      4.9569474e+01,  -2.7304112e+02,  -2.3185923e+02,
      1.4970868e+02,  -1.2968886e+02,  -3.0243031e+01,
      3.2164440e+01,  -2.0050894e+00,   2.8219381e+01,
      1.0172424e+02,  -1.3981837e+01,  -2.4279894e+01,
     -1.3418897e+02,   1.2847447e+01,  -6.0034373e+01,
     -2.0167737e+02,   1.8166612e+02,   2.6309896e+01,
     -3.3731722e+02,   1.6113799e+02,   1.5897874e+02,
      1.9162183e+02,  -4.7450926e+01,  -2.1261995e+02,
      1.0459257e+02,   1.1218275e+02,   6.0762639e+02,
     -7.0663453e+01,  -5.3438451e+01,  -3.4974352e+02,
      4.3216438e+01,   2.3365049e+01,  -9.1776515e+01,
     -3.7975877e+01,   2.7304340e+01,  -1.0831482e+01,
      4.0614559e+01,  -7.3741981e+01,   1.5446457e+02,
      7.1874189e+00,  -1.9724253e+00,  -3.6669186e+01,
     -6.7142256e+00,   2.0190678e+01,   4.0293490e+01,
     -1.8502216e+00,  -1.5216883e+01,   9.2196849e+00,
      6.2278560e+00,   1.4266791e+01,  -2.5627738e+01,
     -2.2829499e+00,   1.1240453e+01,  -1.6509910e+01,
     -1.1202333e+01,   9.9313114e-01,   3.2876591e+00,
      2.2513333e+01,  -6.5462593e+00,   7.4419002e+00,
     -2.2854338e+01,  -1.6634923e+01,   1.6554929e+01,
     -1.0974134e+01,  -2.5723796e+01,  -6.1893822e+01,
      1.7878338e+01,   8.8866944e+00,   2.3451492e+01,
     -4.4204479e+00,   2.2577602e+01,  -9.2188013e+00,
     -3.2038374e+01,  -4.0511183e+00,   1.5422710e+01,
     -1.5860645e+02,  -9.1526679e+01,  -1.2045090e+02,
      1.6557718e+02,  -3.5174024e+02,   4.6470001e+01,
      4.1470420e-01,   4.7669595e+02,  -6.5933575e+02,
      6.6917256e+01,  -2.9476291e+02,   4.6702615e+02,
     -4.0901819e+01,  -1.5321851e+00,  -3.9885343e+01,
      9.2680962e+01,   1.9972958e+02,   1.0315309e+02,
      1.0738593e+01,   7.6216461e+01,   5.8132497e+01,
     -3.4001807e+01,   8.0655814e+01,   2.7343538e-01,
     -7.0707444e+00,  -3.9039843e+00,  -1.9724955e+01,
      1.5853756e+01,  -2.5026096e+01,  -1.2635194e+01,
     -8.0835734e+00,  -7.4202965e+00,   4.7055096e+00,
     -8.9791239e+00,  -1.7412201e+01,  -8.4177961e+00,
     -1.4013169e+01,  -1.9220450e+01,  -1.0019443e+01,
     -1.6921942e+01,  -2.1593709e+01,   5.5755193e-01,
     -6.5797049e+01,  -1.6141473e+02,   7.7999934e+01,
      1.5339610e+02,   4.5136065e+01,   8.0431600e+01,
     -2.3029478e+02,   5.3695032e+02,  -1.0766715e+01,
      6.6557945e+01,  -4.3733486e+02,   7.5154460e+01,
     -1.0887844e+01,  -1.4983541e+01,   6.8271370e+01,
      8.6525660e+00,  -1.5010921e+01,   4.3219235e+01,
     -1.7656390e+01,   4.1549487e+01,  -1.3854357e+02,
      1.2056779e+01,   1.2946120e+01,   3.9503810e+01,
     -1.4726723e+01,  -3.4157673e+01,  -9.4721288e+01,
      1.4586559e+01,  -1.1611540e+01,  -1.4259522e+01,
      7.7141561e-01,   4.8555395e+00,  -1.0278798e+01,
      1.0027233e+01,  -8.6289219e+00,   1.5878979e+01,
      1.2401681e+01,   9.9271459e+00,  -5.0397861e+00,
      1.6628410e+01,   1.2887091e+01,   1.7762022e+01,
      3.2056532e+01,   9.4384620e+00,   5.2968956e+01,
     -9.9017660e-01,  -2.9076244e+01,  -2.1301170e+01,
     -2.7284609e+01,   6.3759954e+00,   1.3534726e+01,
     -1.3882806e+01,   1.9863363e+01,  -6.1289165e+00,
     -6.8243752e+00,  -1.7922404e+01,  -3.1390633e+01,
      8.3826118e+01,  -1.7955058e+02,  -8.8701779e+01,
     -1.2230192e+02,  -1.0244307e+01,   1.9879188e+02,
     -4.0147823e+02,   2.7193688e+02,  -4.3254268e+02,
      2.8864422e+02,  -3.0705517e+02,   2.3090305e+02,
      7.2217829e+00,  -2.3294282e+01,  -1.5177831e+00,
     -2.3681408e+01,  -4.9297563e+01,  -1.0121412e+01,
      1.9787456e+02,   2.0390133e+02,  -2.2722851e+01,
      4.1436528e+01,  -1.0771628e+02,  -7.9344795e+00,
     -2.5901079e+00,  -1.6259902e+00,   3.3450875e+00,
      9.8133618e+00,  -2.8301848e+01,   3.2645985e+01,
     -6.2402546e+01,   6.8173314e+01,   3.5848163e+01,
     -3.8951790e+01,   1.6634550e+02,  -2.0134965e+02,
      7.8232560e+01,  -1.2002034e+02,   6.6996717e+00,
     -4.4812951e+01,   4.3490273e+01,   4.5626763e+01,
     -2.0641599e+00,  -1.8931221e+01,   9.2844871e+01,
     -9.6655093e+00,  -5.6845557e-01,   1.6999827e+01,
     -2.4430722e+01,  -5.1595403e+01,  -1.0113977e+01,
      5.6530047e+00,  -3.4836085e+01,   1.2414210e+01,
     -5.2725891e-01,   4.1333802e+01,  -5.1097809e+01,
      7.9032222e+00,  -1.4738583e+00,  -2.8198033e+00,
      1.0777994e+01,  -1.0837118e+01,   6.3599236e+00,
     -1.0469573e+01,   1.9792562e+01,  -8.1280266e+00,
      5.5521224e+00,   8.3688096e+00,   1.8621459e+01,
     -1.3768927e+01,   1.3231123e+01,   2.1363078e+01,
      2.2995932e+02,  -1.8216823e+01,   2.2298308e+02,
     -1.1082068e+02,  -1.5645177e+02,  -1.7068594e+02,
      6.7692738e+01,   4.6552663e+02,  -4.4465196e+02,
      3.2197699e+01,  -3.1249437e+02,   3.5801734e+02,
      4.0639625e+01,  -8.6701360e+01,  -2.1868545e+01,
      1.5161337e+01,  -2.4677502e+01,  -1.0851707e+01,
     -8.9065681e+01,   1.4360225e+02,   8.9711154e+01,
     -9.0231915e+00,   6.6911036e+01,  -5.6804234e+00,
     -1.1601556e+02,   5.8728509e+01,  -1.1605903e+02,
     -1.0840848e+02,  -5.3468812e+01,  -7.9295780e+01,
      5.1265973e+01,  -3.8903244e+01,   8.3912701e+01,
     -2.9898271e+01,  -2.0139986e+01,  -1.4967265e+01,
      2.9938524e+01,  -3.3526728e+01,  -3.2712156e+01,
      5.9983252e+00,   2.7734381e+01,  -1.9448162e+01,
     -2.4606353e+00,   2.3908640e+00,   3.7732028e+00,
      1.0706019e+01,  -1.3667399e+01,   2.3274072e+01,
      1.4589694e+01,  -1.7480329e+01,   2.8265035e+01,
      7.0755202e+01,   3.3141437e+01,   2.3866198e+01,
      1.5350930e+01,   2.6924099e+00,   2.3089360e+01,
      1.3868713e+02,  -8.4923160e+01,  -3.9634774e+01,
     -2.7131201e+01,   1.4375114e+02,  -3.2794016e+01,
     -5.8725752e+01,  -9.2196938e+01,   1.5078722e+02,
     -4.8007964e+01,  -1.6208267e+02,   1.9075411e+02,
      1.8569112e+02,  -6.8266848e+01,   2.4832757e+01,
      7.1645833e+01,   6.4327938e+02,  -1.1715964e+02,
     -2.1546311e+02,  -4.8727320e+02,   2.6862532e+02,
     -1.7990680e+02,   1.9161673e+02,  -1.6278005e+01,
      2.9210832e+01,  -9.1282921e+00,   2.3803403e+01,
     -4.2345511e+01,  -5.1875472e+01,  -1.0772462e+02,
      4.1892069e+01,   3.2452748e+01,   4.1397417e+01,
      1.3886287e+02,  -2.0534936e+02,   1.7450973e+02,
     -1.2224986e+02,   2.0640305e+02,  -7.5026744e+01,
     -1.0207351e+02,   1.4624299e+02,   4.2466686e+01,
      1.8120359e+01,   1.7879437e+00,  -3.1558267e+00,
      4.7377524e+01,  -1.0136045e+01,   6.4389838e+01,
     -5.4769900e+01,  -1.1064546e+01,  -2.1575090e+01,
     -3.9555423e+01,   1.9891230e+01,  -8.1349455e+01,
      8.5720486e+00,  -6.4941402e+01,  -1.3718334e+02,
      6.2006411e+01,  -1.4447508e+02,  -5.3871008e+01,
     -7.3920998e+01,  -2.2405463e+02,  -3.7593218e+01,
     -5.4387810e+00,   1.6492289e+01,   1.7714716e+02,
     -2.0876771e+02,   5.2303789e+02,  -8.7362363e+01,
      1.2270913e+02,  -3.6940671e+02,   9.7835074e+00,
     -4.0525055e+01,   4.2885660e+01,  -6.4250770e+01,
     -1.2001963e+01,  -9.3531016e+00,   4.5974357e+01,
      3.6978652e+01,   3.9568691e+01,   6.2221272e+01,
     -5.0442136e+01,  -6.3416987e+01,   2.4453171e+01,
     -2.8883938e+02,   1.9919915e+01,  -5.2360882e+01,
      3.4203345e+02,  -3.2713258e+01,   1.0716078e+02,
      4.9774829e+01,  -3.9000591e+01,   2.0909013e+02,
      3.0990742e+01,   3.6916704e+00,  -1.4138066e+00,
      1.5679787e+01,   2.5578848e+01,  -3.3504566e+01,
      3.9888255e+01,  -2.6091817e+01,  -2.9564057e+01,
      1.8126441e+01,   4.1509555e+01,  -1.1745145e+01,
      5.5229613e+01,   3.8726087e+01,  -1.3335472e+02,
     -1.1663420e+02,   9.7981368e+00,  -1.9960424e+02,
      3.4431497e+01,  -1.7310456e+02,  -6.7824647e+01,
     -1.4497206e+02,  -1.1250492e+02,   8.1046150e+01,
     -1.3751235e+02,   4.1183669e+02,  -4.4758976e+02,
      1.5756998e+02,  -2.9218290e+02,   2.1069146e+02,
      8.3691996e+01,   6.8519349e+01,  -2.0774096e+01,
     -5.4322610e+00,  -1.3189771e+01,  -2.7874809e-01,
      8.0421683e+01,   4.6966389e+01,   7.6639904e+01,
     -4.7464033e+02,   2.5642814e+01,   1.2524201e+02,
      3.4702247e+02,  -1.3932999e+02,  -1.6734175e+02,
      9.4008962e+01,  -1.2404891e+02,  -4.2428681e+01,
     -9.5046595e+00,  -2.0808575e+01,  -3.1863161e+01,
      1.6764218e+01,  -4.2002794e+01,  -1.2066862e+01,
      7.0992473e+01,   1.3029096e+02,   1.1217903e+01,
     -8.1360901e+01,   1.6946046e+02,   9.5669670e+01,
      8.9788411e+01,  -4.5847493e+01,   1.8085007e+02,
     -1.0959859e+01,  -1.0648174e+02,  -1.3345938e+02,
      4.6362083e+01,   4.1820536e+02,  -3.7068967e+02,
      2.0047129e+02,  -2.5335298e+02,   3.5463786e+02,
     -1.9217710e+01,   3.7851026e+01,   1.8348302e+00,
      2.6247353e+01,  -4.3440932e+01,  -1.9753346e+01,
     -1.6545485e+02,   3.5553665e+01,   3.5470526e+01,
      2.7533561e+01,  -8.5605763e+00,  -3.2517037e+01,
      8.5083784e+00,  -9.6010755e+00,   1.0410974e+01,
     -2.2139260e+01,   2.3953413e+01,   7.0613108e+00,
      2.2418982e+01,   1.3856224e+01,  -8.7802921e+00,
      3.7160875e+01,  -1.6439656e+01,   6.2372196e+00,
      1.6066874e+00,   4.4058035e+00,   1.2390653e+01,
     -2.7885134e+01,  -8.4811080e+00,   2.0586627e+00,
     -2.4726353e+01,   1.3302263e+01,   2.1186550e+01,
      1.6997713e+01,   1.7211088e+01,  -2.0303286e+01,
      1.6741293e+01,  -9.1931485e+00,   1.9813991e+00,
      2.8784081e-01,   1.6084534e+01,   3.2112658e+01,
     -1.3428401e+01,  -1.9814895e+01,   4.4555865e+00,
     -1.3110530e+02,  -1.4205359e+02,   1.5663299e+02,
      1.5867224e+02,   1.9717586e+01,   2.9331327e+01,
      1.2634550e+01,   6.2252924e+02,   1.3321073e+01,
     -6.7132725e+01,  -3.9437396e+02,  -8.7893090e+00,
     -5.3976276e+01,   5.4061604e+01,  -9.8110923e+01,
      1.1529459e+01,  -1.5797579e+01,   3.8147949e+01,
     -2.2135922e+01,  -2.2819260e+01,  -1.6262487e+02,
     -7.7003544e+01,  -2.6692048e+00,   8.6336398e+01,
     -1.3740216e+01,   7.3511514e+01,   1.5149437e+02,
      1.6657190e+01,  -3.8559174e+01,   1.5939858e+02,
      1.0878584e+01,  -4.8095672e+01,  -8.6098147e+01,
      2.4791867e+01,  -4.2055144e+00,   2.0115208e+01,
      2.4720140e+01,  -5.6507451e+00,   4.0529543e+01,
      5.8276661e+00,  -6.7574626e+00,  -1.6383091e+01,
      2.3427282e+01,   2.3420286e+01,  -1.1594049e+01,
      1.5022447e+01,  -1.0326470e+01,  -4.2731252e+01,
     -6.9578323e-01,  -3.0682006e+00,  -2.3264711e+01,
     -3.3892874e+01,   1.7917452e+01,  -8.2712400e+01,
      1.8055403e+01,   2.6841248e+01,  -8.5220314e+01,
      1.5342082e+02,  -3.3399363e+01,  -5.7519831e+01,
     -8.9165852e+01,  -1.3251479e+02,  -6.5515462e+01,
     -6.2510423e+01,   1.3763261e+02,   2.1383867e+01,
     -4.4586320e+01,  -2.8049640e+02,  -7.4627681e+01,
     -1.7737962e+01,   1.5578424e+02,   1.8159479e+02,
     -4.7251973e+02,   2.9452167e+02,  -3.4142628e+02,
      3.4194769e+02,  -2.1282913e+02,   1.9132610e+02,
      1.6359237e+01,   4.6312961e+01,  -4.7257658e+01,
     -3.1093309e+01,  -1.3480374e+01,  -9.8122537e-02,
      1.5032257e+02,   3.1819914e+01,  -9.1766913e+01,
      9.6834011e+00,  -5.2646834e+01,   2.4586139e+01,
     -1.4787619e+01,   2.9904441e+00,   1.3990144e+02,
      2.0344412e+01,   1.3428733e+02,   1.2551424e+02,
      2.4796143e+01,  -5.9670168e+00,  -1.1183195e+02,
     -1.4031438e+01,  -3.4564775e+00,   3.6466554e+01,
     -4.6633018e+01,  -6.6272893e+00,   5.3447279e+01,
      5.2973662e+00,   3.6460056e+00,   1.1128312e+00,
      1.7055845e+00,  -9.9201261e+00,  -2.0136426e-01,
      6.4769418e+00,   2.9090207e+01,  -1.4849865e+01,
     -1.5462752e+01,   1.7407134e+00,  -3.4399240e+01,
      1.0263681e+01,  -7.0695303e+01,  -4.2698926e+01,
     -2.8276939e+01,  -3.3191538e+01,  -7.4248002e+01,
     -5.3989670e+01,   1.0062843e+02,  -1.2299216e+02,
      1.6301811e+02,  -2.8202541e+01,  -3.3374044e+01,
     -8.3454485e+01,  -9.6198652e+01,   8.3202304e+01,
      2.4982166e+02,  -1.3262643e+02,   1.2964620e+02,
     -1.4527116e+02,  -1.3759471e+02,  -6.3999676e+01,
     -6.9073129e+01,   4.9488138e+02,  -6.7424918e+02,
      5.8507792e+01,  -2.4682843e+02,   4.4194971e+02,
      2.0044554e+01,   2.7366312e+01,   1.8906327e+01,
      7.4924266e+00,  -2.6523678e+01,  -1.3403341e+01,
     -2.3054311e+00,   1.3924454e+02,   6.4355776e+01,
      7.2488047e+01,   1.1672214e+02,   2.2481023e+02,
     -1.1187427e+02,  -3.6179604e+02,  -6.7557757e+02,
     -5.3767327e+01,   3.6159821e+02,   1.5682427e+02,
      3.7381080e+01,  -1.1965552e+02,   4.1149359e+02,
     -4.6417684e+01,  -5.5405801e+00,  -2.0475081e+01,
      4.3569401e+01,  -3.4490016e+01,   3.7322249e+01,
      4.5216904e+01,  -3.1454287e+01,  -6.8785835e+01,
     -1.3861342e+01,   3.9042293e+01,  -3.0658945e+01,
      1.5445731e+02,  -1.1584713e+02,   3.7158932e+02,
     -5.7733558e+01,   4.1231779e+01,  -3.0059505e+02,
      2.6088589e+02,  -4.1672478e+02,   2.9083280e+02,
     -1.2637756e+02,   1.4305132e+02,  -1.6581965e+02,
     -1.6672497e+02,   7.0128712e+01,  -6.5017402e+01,
      2.6996856e+01,  -1.0168596e+00,   3.0303500e+01,
     -5.9340786e+01,   4.5660155e+01,   1.1643671e+02,
     -4.3056667e+01,  -9.1876848e+00,   9.9855266e+00,
      4.7734862e+01,  -5.5290828e+01,  -1.5359715e+02,
      1.0069449e+02,  -4.3094795e+01,  -1.0913938e+02,
      5.3627419e+00,   3.3448523e+01,   9.3108747e+01,
      1.0798909e+01,   1.3449096e+00,  -9.9212409e+00,
      2.9926744e+01,  -2.0355447e+01,  -6.2446899e+00,
     -7.0095557e+00,  -3.3783870e+00,  -2.6559847e+01,
      3.4188576e+01,   1.5963035e+00,  -4.6773215e+01,
     -7.1816460e+00,   6.3000479e+00,   2.9545595e+01,
     -4.0448006e+01,   1.2215700e+01,   3.2879249e+01,
     -6.2501462e+01,   4.8725710e+01,   5.0088789e+01,
     -5.8962617e+01,  -1.8096432e+00,   6.9044816e+01,
      4.4713912e+01,  -1.0277850e+02,   1.2137770e+02,
      5.0137400e+01,   1.4585149e+02,   4.0974801e+01,
     -1.2607432e+02,  -1.8853620e+01,  -8.3433032e+01,
     -1.1359082e+02,   2.4180886e+02,  -8.2942189e+01,
      1.1280258e+02,  -1.0934845e+02,  -4.3093754e+00,
      1.3845947e+02,   1.2530665e+02,   6.3352586e+02,
     -1.0971075e+02,  -6.0666701e+01,  -4.7517019e+02,
     -4.4650674e+01,   5.9130053e+00,  -1.2334173e+01,
     -2.6324454e+00,  -8.8330724e+00,  -2.0225665e+00,
      1.8964775e+01,   2.2887779e+01,   3.7414359e+01,
     -1.4003349e+02,   4.8291333e+01,  -2.9685501e+02,
      9.0669393e+01,  -2.0889686e+02,   1.7290216e+02,
     -6.0144363e+02,   4.6822954e+02,   8.6116253e+01,
      3.5893483e+02,  -2.5413796e+02,  -1.7527940e+02,
     -5.4386458e+01,  -1.7408694e+01,  -3.0728757e+01,
     -4.3718140e+01,   5.1452821e+00,  -6.4617034e+00,
      1.3474460e+02,   9.4349685e+01,  -9.0581023e+00,
     -2.9276375e+01,  -3.2794528e+01,  -5.5888737e+01,
      3.1769364e+01,   3.7176173e+01,   3.7024431e+01,
      1.0441512e-01,  -4.2268750e+01,  -2.0713462e+00,
     -2.9922296e+01,  -3.6204457e+01,   5.7721109e+00,
     -1.2125818e+01,   1.2596314e+01,   1.2291885e+01,
     -9.7641366e-01,  -2.2569702e+00,   1.5224543e+01,
      2.6370969e+01,   3.1906442e+00,  -1.4777788e+01,
      2.1122168e+01,  -1.3007032e+01,   1.7042767e+01,
     -2.0565762e+01,  -2.1860348e+01,  -2.5974919e+01,
      1.8675810e+01,   2.4304453e+01,  -1.7603937e+00,
     -1.5665998e+01,   2.0831815e+01,  -1.1705634e+01,
      4.9186635e+00,   3.2158246e+00,   3.3784074e+01,
      4.1389884e+02,  -8.5476970e+01,  -9.8462433e+00,
     -5.0739374e+02,  -2.7708759e+01,   1.5716962e+01,
      1.0118522e+03,   3.4579889e+02,  -2.7474039e+01,
     -1.8073941e+02,  -2.2711598e+02,  -3.3178747e+00,
     -9.1312671e+01,  -2.4805049e+02,   3.7663697e+00,
      3.9649525e+01,   4.9206573e+00,  -1.9187394e+01,
      7.7628258e+01,   3.4654943e+01,   1.0044112e+02,
     -1.8492220e+01,  -5.3130525e+00,  -1.8597859e+01,
      5.2898855e+01,   3.2817427e+01,   9.7096646e+00,
     -4.4356279e+00,  -6.6176543e+01,  -3.3086969e+00,
     -7.4092319e+00,   1.4741076e+01,   1.6495935e+00,
     -1.4767994e+01,  -1.6740364e-01,  -3.4085059e+00,
      1.7073171e+00,  -2.1531092e+00,   1.5090681e+00,
     -1.2288035e+01,   4.0199849e+01,   1.8213925e+01,
     -5.9729140e+01,  -1.3729306e+01,  -4.7347162e+01,
     -3.9123914e+01,  -1.0547538e+01,  -1.8782548e+01,
      2.8544066e+00,   1.7662692e+01,   9.2912276e+00,
      1.2684987e+00,  -2.2862991e+01,  -6.6851605e-01,
      2.3150374e+01,   5.1807265e+00,   7.5975381e+00,
      1.2234039e+01,  -8.8210387e+00,   3.6308193e+00,
     -4.8728150e+02,   3.3969945e+01,   3.6799170e+01,
    };

    const double* forceDataPtr = forceData;
    for (unsigned int ii = 0; ii < numberOfParticles; ii++) { 
        expectedForces[ii]    = Vec3(forceDataPtr[0], forceDataPtr[1], forceDataPtr[2]);
        forceDataPtr         += 3;
    }

    double tolerance          = 1.0e-05;
    compareForceNormsEnergy(testName, expectedEnergy, energy, expectedForces, forces, tolerance);
}

static void testGeneralizedKirkwoodVillinExtrapolatedPolarization() {

    std::string testName      = "testGeneralizedKirkwoodVillinExtrapolatedPolarization";

    int numberOfParticles     = 596;
    std::vector<Vec3> forces;
    double energy;

    // We don't have reference values for this case, but at least check that force and energy are consistent.
    
    setupAndGetForcesEnergyMultipoleVillin(AmoebaMultipoleForce::Extrapolated, 0, forces, energy);
}

// test GK mutual polarization for villin system

static void testGeneralizedKirkwoodVillinMutualPolarization() {

    std::string testName      = "testGeneralizedKirkwoodVillinMutualPolarization";

    int numberOfParticles     = 596;
    std::vector<Vec3> forces;
    double energy;

    setupAndGetForcesEnergyMultipoleVillin(AmoebaMultipoleForce::Mutual, 0, forces, energy);
    std::vector<Vec3> expectedForces(numberOfParticles);

    double expectedEnergy            = -8.6477811e+03;

    static const double forceData[] = {
      1.2966353e+02,   8.5572073e+01,  -1.4934563e+02,
     -3.6953516e+01,  -5.4877368e+01,   1.5127299e+02,
      1.9708505e+02,   6.2850891e+02,  -5.0909467e+02,
     -9.9954717e+01,  -5.2655323e+02,   3.2517396e+02,
      9.2773631e+01,   1.6763129e+02,   1.3452410e+02,
     -3.9605887e+01,  -1.2405492e+02,   2.0132480e+02,
     -1.6305897e+02,  -8.5270797e+01,  -1.2414105e+02,
      4.4666422e+01,  -3.1281818e+01,  -2.6245401e+01,
     -1.0702463e+02,   6.2230726e+01,  -1.8295313e+02,
      2.2761673e+00,  -2.1690714e+01,  -8.6393411e+01,
      1.8190727e+01,   1.8826270e+01,   1.3545200e+01,
     -7.9749941e+00,  -3.4354493e+01,  -3.8811506e+01,
      2.0156315e+01,  -1.9221969e+01,   4.9269404e+01,
      1.0336264e+01,  -3.5414449e+00,   3.3711725e+01,
     -1.3985413e+00,   1.5354841e+01,  -2.0828563e+00,
      1.7552049e+01,   1.6663297e+01,   2.3889984e+00,
     -1.0699064e+00,   2.2036440e+01,   8.3354660e+00,
     -1.4856508e+01,   1.2917536e+01,  -3.3831626e+01,
     -7.0798125e+00,   2.3711674e+01,   9.3398409e+00,
      8.2579244e+01,  -2.3021505e+02,   2.0236870e+02,
     -1.6792134e+02,   1.7929771e+02,  -1.4332735e+02,
      4.1181464e+02,  -7.0228872e+02,   8.1044182e+02,
     -1.6607507e+02,   1.0677262e+03,  -8.0996978e+02,
     -4.2454854e+01,  -6.0154471e+01,  -1.4143803e+01,
      3.5857448e+01,  -3.1369887e+01,   1.2992615e+01,
      8.9067456e+01,   1.8182825e+02,   3.3894361e+01,
     -4.9580508e+01,  -2.5324292e+01,   1.5549981e+01,
     -1.1539787e+01,   2.8108476e+01,  -8.9724073e+00,
      1.5820588e+01,  -5.8763509e-01,  -2.5573987e+01,
      2.5880400e+01,  -1.2553449e+01,   1.4911469e+01,
     -2.1483509e+01,  -4.1174080e+01,  -1.4136591e+01,
      4.1676568e+00,  -8.0815697e+00,   1.3054781e+01,
     -2.8542456e-01,  -1.9944761e+01,  -1.0443563e+01,
      1.1005912e+01,   1.0749139e+01,  -2.3059792e+01,
      3.2053224e+01,  -2.0030573e+01,   2.9301875e+01,
     -2.2752916e+01,   4.0681158e+00,   1.6345554e+01,
      1.1622046e+00,  -9.7307602e+00,   2.8153485e+01,
      2.0499440e+01,   8.3330980e+00,  -1.2475268e+01,
     -1.2685684e+01,   1.2906418e+02,  -1.9660979e+02,
      9.4592211e+01,   9.4918099e+01,   2.5903307e+02,
     -5.4433994e+02,  -2.7955919e+02,  -5.2922124e+02,
      4.0153985e+02,   2.2183469e+02,   3.0703244e+02,
     -3.6112354e+01,   2.5930754e+01,  -4.7511838e+01,
     -1.0829309e+01,   2.8527112e+01,  -4.7409003e+01,
     -8.6860701e+01,  -8.7720733e+01,  -1.4165213e+02,
      6.8809494e+01,   1.5707412e+02,   2.2470068e+02,
     -1.0964685e+01,  -2.8767095e+00,  -1.5406734e+01,
     -1.6300177e-01,  -1.6694871e+01,  -2.3378991e+01,
     -1.5646677e-01,  -1.6758904e+02,  -5.0106428e+01,
      1.4585535e+02,   9.1003785e+00,   2.3788891e+02,
     -2.3840138e+02,  -6.0573391e+01,  -2.8226740e+01,
      1.9972025e+02,  -4.6019677e+02,  -4.5077872e+02,
     -2.4216533e+01,   4.3575254e+02,   3.3310464e+02,
      3.8346007e+01,   1.4535976e+02,  -6.1041703e+01,
      3.3278973e+01,  -2.0814985e+01,   2.9406293e+01,
      5.4865139e+01,   7.4560707e+01,  -1.4933283e+02,
      9.5093180e+01,  -3.4438122e+02,   7.1983429e+02,
     -1.7860104e+02,   1.5263548e+02,  -2.6408228e+02,
      7.5306414e+01,   1.6031578e+02,  -4.9616711e+02,
     -2.7965676e+01,  -7.2336746e+01,   9.6531597e+01,
     -1.3912514e-01,   2.1996759e+00,   9.9710194e+01,
     -1.6739175e+02,   9.7564407e+01,   2.8747801e+02,
      1.3346903e+02,   4.4742046e+01,  -1.2941925e+02,
     -1.6130402e+02,  -2.6081638e+02,  -6.0370874e+02,
      5.4929396e+01,   7.5847770e+01,   4.6926694e+02,
      1.9342285e+01,   4.8350786e+01,  -5.8708877e+00,
      4.9721997e+01,   1.0207404e+01,  -1.1139732e+01,
     -1.1721403e+02,  -4.1863858e+01,  -4.1600966e+01,
      2.8783280e+00,  -9.7306901e+01,   7.0764634e+01,
      2.9166723e+02,   5.3891607e+02,  -3.3225326e+02,
     -2.6957000e+02,  -1.9593683e+02,   3.1192255e+01,
      6.4165193e+01,  -3.4382307e+02,   3.4759571e+02,
      3.1229710e+01,  -5.0621533e+01,   4.2294181e+00,
     -1.8767237e+01,  -5.6431420e+01,   2.7692542e+01,
      2.2708722e+00,   1.0769951e+02,  -3.0873392e+01,
     -1.1377881e+01,   5.3402569e+01,  -9.1621544e+01,
     -1.8627250e+01,   8.1765177e+01,   2.6354466e+02,
      9.5074406e+01,   5.6099318e+01,  -2.5605658e+02,
     -4.4893184e+02,  -3.0456286e+02,  -3.2283934e+02,
      2.6232511e+02,   1.7785994e+02,   2.6016631e+02,
     -9.5056700e+00,  -6.6412473e+01,   2.0451063e+01,
     -3.9035309e+01,   2.0975230e+01,   1.4607951e+01,
     -1.2470405e+01,  -1.5762071e+02,   1.0826097e+02,
     -8.6421495e+01,   7.8020478e+02,  -1.2167297e+02,
      1.8078866e+02,  -3.5518754e+02,  -7.5223903e+00,
     -9.8173521e+01,  -5.1039573e+02,   1.1710822e+02,
      3.0470445e+00,   1.0217402e+02,  -5.3428477e+01,
      3.5468618e+01,   9.4771802e+01,   1.7434326e+01,
      1.6200766e+02,   2.0466901e+02,   1.1196053e+02,
     -1.2942151e+02,  -6.4172221e+01,   7.3941426e+01,
     -2.0098556e+02,  -5.4100177e+02,  -3.7372267e+02,
      3.2712229e+02,   4.0460836e+02,   3.6283741e+02,
      3.8562864e+01,  -5.7885373e+01,  -1.1309310e+02,
     -2.6423563e+01,  -6.2975281e+00,   5.1059324e+01,
      1.9825061e+01,   3.0144179e+01,  -1.5806317e+02,
      8.5322355e+00,  -2.6729150e+01,   2.6984234e+01,
     -3.6827759e+01,   3.2525307e+01,   2.6652876e+01,
      3.2903913e+01,   2.4658260e+01,  -8.1897223e+01,
     -1.0826715e+01,  -2.0713520e+01,   1.4996812e+01,
      2.7429108e+01,  -1.3776460e+01,  -7.2875988e+00,
     -7.0648176e+00,  -9.0060594e+00,   1.2911020e+01,
     -1.7182188e+01,  -3.1338597e+01,  -1.1016200e+01,
     -2.5198591e+01,  -5.6311892e+00,   4.2939929e+01,
      1.3755355e+01,  -1.1634098e+01,  -2.0595212e+01,
     -4.8809313e+01,   2.1040690e+01,   1.8467670e+01,
      2.3175411e+01,   2.0166155e+01,  -3.9012767e+00,
     -8.3922767e+00,  -2.6826712e+01,  -5.4420665e+00,
      9.6641127e+00,   7.8118054e+00,   2.4362326e+00,
      1.4512497e+01,   2.8729787e+02,   1.5821444e+02,
      8.7013233e+01,  -1.7125970e+02,  -3.0259976e+01,
      1.2487547e+02,  -2.9334468e+02,  -5.7441741e+02,
     -1.2117208e+02,   2.6545717e+02,   4.1702893e+02,
      1.0712827e+01,   2.3214759e+01,  -3.4854116e+01,
      3.2530187e+01,  -3.0391827e+01,   2.2448569e+01,
     -1.2152368e+02,  -1.0019499e+01,  -3.5729384e+01,
      4.8451718e+01,   1.4869144e+00,  -1.4726809e+01,
      1.2321088e+02,  -6.0184158e+01,  -9.8155078e+01,
      6.2686911e+01,   1.5297386e+01,  -1.9154187e+02,
     -1.1069693e+02,   2.2480609e+01,   5.6440410e+01,
      1.4220205e+00,  -2.2901391e+01,  -2.5874261e+01,
      3.2061821e+01,   7.9626014e+00,   7.3664609e+00,
      3.1358595e+01,   7.8279279e+00,   1.4711313e+01,
     -1.1729796e+01,   8.4892715e+00,   6.8085399e+00,
     -3.0168654e+01,   9.7436540e+00,   5.6452469e+00,
     -2.9983039e+01,   8.3203769e+00,   1.1315775e+01,
     -4.8850724e+01,  -3.0692116e+01,   6.9531244e+01,
     -4.9419418e+00,   1.1420579e+01,   8.5866701e+01,
     -1.7825230e+02,  -2.8175317e+01,  -2.7566206e+01,
      9.9570739e+01,  -1.0946954e+02,   4.4026657e+01,
      1.1206807e+01,   1.5762896e+02,   4.0657791e+01,
     -1.4038276e+02,   1.1931675e+02,   3.3987641e+02,
      1.9441192e+02,   7.7056646e-01,  -1.9979208e+02,
     -5.7071428e+02,   2.7917817e+02,  -3.5933815e+02,
      3.9811237e+02,  -1.6077088e+02,   3.2593749e+02,
      5.6072029e+00,   5.0242051e+01,  -2.4553400e+01,
     -2.8567296e+01,   2.1538750e+01,  -2.6926519e+01,
     -1.0640170e+02,  -2.0939476e+02,   8.0679885e+01,
     -2.0391209e+01,  -8.0251289e-01,   1.4105813e+01,
      3.3894723e+01,   8.6878003e+00,  -2.3968920e+01,
      5.1469978e+01,   7.3372638e+01,  -1.0761297e+01,
      3.2764441e+02,  -2.6496931e+02,   1.2541114e+02,
     -1.7705735e+02,   1.6085687e+02,   4.2500013e+01,
     -2.1151900e+02,  -3.5065682e+02,  -5.4205202e+02,
      1.7932014e+02,   2.2939655e+02,   3.7895751e+02,
     -3.7478909e+01,   6.8594292e+01,   1.0469748e+01,
     -5.5883868e+01,   8.5345998e+00,   6.1507729e+00,
     -4.6839182e+01,   8.7425507e+01,  -1.9352559e+01,
      2.6700878e+01,  -7.9876130e+01,  -3.0341105e+00,
      1.0656288e+02,  -4.0153852e+01,   3.0578444e+00,
      1.4438996e+01,  -3.5369045e+01,  -1.8966882e+01,
     -4.5827091e+01,   4.9782780e+01,   2.4862850e+00,
      9.4177000e+00,  -8.1527352e-01,   6.2559235e+00,
      1.4937092e+01,   4.9255492e+00,  -3.0974165e+01,
     -3.3416509e+01,  -2.0394811e+01,  -1.9408395e+01,
      2.8123258e+00,  -2.2555963e+01,  -2.0979985e+01,
      2.0427658e+01,  -3.8580303e+00,   2.1554470e+01,
      2.7193313e+02,   2.0022360e+02,   1.0625517e+02,
     -1.2231128e+02,  -1.1255621e+02,   3.0449272e+01,
     -1.6624296e+01,  -5.8022886e+02,  -4.1210186e+02,
      1.6783893e+01,   3.6324378e+02,   2.6954298e+02,
     -8.3138346e+01,  -5.7898280e+01,   7.8372604e+01,
     -4.3448092e+01,  -2.2758600e+01,   3.3670243e+01,
     -3.0881078e+01,   1.2902499e+02,  -2.1818354e+02,
      3.7082345e+01,  -6.4236439e+01,   8.6884774e+01,
     -1.5068794e+01,  -4.1549373e+01,  -1.8741419e+01,
      5.4363765e+01,   4.8826023e+01,  -1.5249000e+01,
     -3.1511763e+01,   2.6318650e+00,   9.5718246e+00,
     -3.3726521e+00,   1.6863676e+01,   7.3877960e+00,
     -2.1709790e+01,  -4.8156425e+00,  -1.1864000e+01,
     -3.4532237e+01,   4.5746300e+00,  -1.4481537e+00,
     -3.6289056e+01,   1.2016791e+01,   4.4563883e+01,
      3.9590505e+00,   2.5941639e+01,   1.4062390e+01,
     -2.4627015e+01,  -2.5556389e+01,  -2.4951912e+01,
      1.6243669e+01,   6.9154603e+00,  -3.6076605e+01,
     -4.6665038e+00,  -2.3316535e+01,  -1.3890689e+01,
      1.3225927e+01,   1.1567652e-01,  -1.4239295e+01,
      3.9801460e+01,   3.1080393e+02,   2.6745484e+02,
      8.4391816e+01,  -1.0882362e+02,  -1.2909009e+02,
      5.7216681e+02,  -1.2876206e+02,  -3.0690705e+02,
     -5.2598599e+02,   5.5333722e+01,   2.3769399e+02,
     -2.1453115e+01,  -3.1001716e+00,   1.1536944e+01,
     -1.0514579e+01,   4.4707457e+01,  -2.6970055e+01,
      2.8465151e+01,  -1.7750531e+01,  -1.9351656e+01,
     -2.0082145e+02,   1.3034063e+02,   1.5114436e+02,
      5.9467833e+01,  -2.3786277e+02,  -1.0616860e+02,
      1.7046910e+02,   8.4543202e+02,  -2.4839529e+02,
     -1.1669290e+02,  -4.8787876e+02,   1.8578287e+02,
     -2.4408643e+01,  -1.1746636e+01,   7.3310836e+01,
      2.8440892e+01,  -2.0202913e+01,   3.1206055e+01,
      6.6497179e+00,   9.1228503e+01,  -7.9405857e+01,
     -2.3315728e+01,   6.3404670e+01,   6.2497251e-01,
     -1.0034592e+01,  -3.6348044e+01,   1.9713468e+01,
     -1.1442711e+01,   3.8768097e+01,  -4.4634286e+01,
      4.1166666e+01,  -4.9905906e+00,   7.4379464e+00,
     -4.1516419e+01,  -4.5428859e+01,  -2.0117479e+00,
     -6.6177730e+00,  -9.4881317e+00,   1.6561006e+01,
      7.7220202e+00,  -1.4350247e+00,   1.3647720e+01,
      2.2529572e+01,  -1.0979843e+00,   2.2719458e+01,
      4.7287175e+00,   3.7623124e+01,   1.3009561e+01,
      9.1477691e+00,  -1.2890418e+01,   1.3625953e+01,
     -1.3590607e+02,  -2.1149723e+02,   1.9315613e+02,
      1.6380738e+02,   2.6025280e+02,  -4.2420857e+01,
     -1.8016822e+02,  -6.9070413e+02,  -2.5218347e+01,
     -4.6150105e+01,   5.6427542e+02,   7.7108033e+01,
      1.5731186e+02,  -1.2423234e+02,  -1.4399782e+02,
      3.4439143e+00,  -4.3160174e+01,   5.0964227e+01,
     -6.1310563e+01,  -4.9169106e+01,  -2.0841970e+02,
     -4.5963141e+01,  -3.4070300e+01,   2.0872664e+02,
     -2.0804203e+00,  -4.8137498e+01,   1.9226523e+01,
      7.2211673e+00,  -5.7962890e+00,   3.2161086e+00,
     -5.8064789e+01,   8.3564820e+01,  -1.1670026e+02,
     -7.0188836e+00,  -2.4981736e+00,   2.5934388e+01,
     -2.3348324e+01,   1.1049777e+01,  -2.2526439e+01,
      1.3922240e+01,  -1.8934980e+01,  -2.5332262e+01,
      1.4469586e+02,   1.9462996e+02,  -4.2505304e+01,
     -1.1564950e+02,   1.5387458e+01,  -2.0469571e+01,
     -4.3245145e+02,  -4.0062669e+02,   1.6175365e+02,
      4.1177250e+02,   3.2637786e+02,  -2.2553446e+01,
      9.4675963e+00,   4.1950777e+01,   3.6810224e+01,
     -5.5473078e+01,   3.0064101e+01,  -2.9631287e+01,
      1.5068659e+01,  -1.6829567e+02,  -2.2925110e+01,
      3.0769610e+01,  -5.5121975e-01,   5.7221513e+01,
      1.0102830e+02,   7.0981015e+01,   3.7107536e+01,
     -4.4789877e+01,  -4.6607184e+01,   6.9224469e+01,
     -5.9549354e+01,  -3.2089239e+01,  -7.2853757e+01,
      1.3608931e+01,  -1.0661819e+01,  -3.9206964e+01,
     -4.3317736e-01,  -1.2559044e+01,  -2.7154151e+01,
     -9.8421637e+00,   7.1612999e-01,   1.0838790e+01,
      5.2341847e+00,   3.9892035e+01,  -4.7211459e+00,
     -2.2189010e+00,   2.6093300e+01,  -1.7967039e+01,
      2.7475836e+00,   1.2766799e+01,  -1.8182318e+01,
     -4.2683869e+00,  -1.4997385e+01,  -2.8294169e+01,
      2.9177608e+01,  -3.7178088e+00,  -2.7930160e+01,
     -6.8952622e+00,  -4.5976185e+01,   2.0514204e+01,
     -5.6216730e+01,  -7.2643929e+00,  -2.0251422e+01,
     -1.5574852e+02,  -5.1908291e+02,   2.5905224e+02,
     -2.4340395e+00,  -1.6823669e+01,   2.8392155e+01,
      1.4208983e+01,   1.6271373e+01,   2.1863062e+01,
      1.5737893e+02,   1.6107664e+02,  -2.0392359e+02,
     -9.5523470e+01,  -3.3265039e+01,   1.2266365e+02,
     -2.0473541e+00,  -5.0934557e+02,   2.8182637e+02,
      1.9315663e+01,   2.9289075e+02,  -2.5116116e+02,
      5.3545758e+01,  -5.9796307e+00,  -3.8458674e+00,
     -3.3729677e+01,   2.0368449e+01,   5.6757517e+01,
     -5.6863210e+01,   1.3913397e+01,   7.7233322e+01,
      1.6348971e+02,  -1.5013857e+02,  -1.1868948e+02,
     -3.7195934e+01,  -1.8799434e+01,   2.7542513e+01,
      1.4060870e+01,   1.5641295e+01,   2.4987956e+01,
     -1.2794136e+02,   9.5460937e+01,  -7.8971705e+01,
     -1.4087817e+02,   2.5733919e+02,  -1.5979367e+02,
      1.9184301e+02,  -1.3218076e+02,   1.1409108e+02,
     -1.3545146e+02,  -5.9454419e+02,  -2.2245748e+02,
      1.1026832e+02,   4.5975629e+02,   1.2344335e+02,
      6.4078020e+01,   5.9882898e+00,   3.0794584e+00,
      3.4404045e+01,  -2.3091970e+01,  -2.5448736e+01,
     -2.2767482e+02,  -2.2180242e+01,   6.7351887e+01,
     -4.8574658e+00,  -2.1748525e+01,   7.4593164e+00,
      5.6303373e+01,   3.4360877e+01,   3.8791913e+00,
      1.2259865e+01,   3.3908219e+01,  -2.1320624e+00,
      1.0737668e+02,   2.4627001e+02,   1.1387045e+02,
     -5.9323121e+01,  -3.1098401e+01,  -1.8242147e+02,
     -3.8868465e+02,  -5.4567547e+02,  -1.6487209e+01,
      2.2490919e+02,   3.9847383e+02,   7.6334769e+01,
     -4.8423905e+01,  -7.5646564e+01,   1.6750656e+02,
      7.0306206e+00,   2.5678179e+01,  -5.5923209e+01,
      3.4640619e+01,  -7.3473132e+01,   1.1931505e+02,
     -2.5162633e+01,  -1.6489559e+01,  -1.9389509e+01,
      2.6651556e+01,  -4.5281155e+01,  -2.3068055e+01,
      1.7675807e+01,   1.8116506e+01,   5.6525151e+01,
     -1.8774345e+01,  -7.8280856e+00,  -2.1091752e+01,
     -2.5034377e+01,   9.8217505e+00,   2.6650820e+00,
     -1.8597607e+01,  -6.1524482e+00,   1.3752711e+01,
     -2.7213183e+01,   1.4450289e+01,  -9.2740173e+00,
     -9.1639219e-02,   1.6518907e+01,  -8.1310922e+01,
     -1.3706740e+01,   1.5766498e+01,   3.0465641e+01,
     -6.3165660e+00,  -2.0584963e+01,  -3.3725550e+01,
      1.6595051e+01,   1.4255544e+01,   2.0384606e+01,
      1.9871295e+01,  -1.0715997e+01,  -5.3041236e+00,
      1.6889231e+02,   6.5831394e+01,  -6.7701671e+01,
      3.5481688e+02,   1.4103977e+02,   4.8594115e+01,
     -2.3353275e+02,   4.9568541e+01,  -2.5613733e+01,
     -5.1822736e+01,  -2.8518553e+02,   6.3421081e+02,
     -1.9485221e+01,   2.4322938e+02,  -4.8613997e+02,
     -1.0673833e+01,  -6.4157301e+00,   5.2365005e+01,
     -6.5713631e+00,   1.7319500e+01,   4.9005752e+01,
      6.3782629e+01,  -2.0537630e+02,  -1.5658863e+02,
     -2.6960359e+01,   4.5365158e+01,   5.5666463e+00,
     -1.6196972e+01,   7.3743745e+01,   1.8289982e+01,
      5.9326605e+00,  -1.8276643e+01,   9.0326700e+00,
      1.3223126e+02,   1.3415998e+02,  -2.4195862e+02,
      7.6208678e+01,  -1.5271024e+01,   2.3622078e+02,
      2.2187055e+02,  -7.8790961e+02,  -2.7332300e+02,
     -1.6280593e+02,   5.9780818e+02,   1.5321407e+02,
     -6.7026162e+01,   2.5731504e+01,  -4.0517133e+01,
     -1.9777150e+01,  -4.7208497e+01,  -2.4350136e-01,
     -6.0626483e+01,   2.5149530e+01,  -6.6104864e+01,
      5.9114288e+01,   3.1694950e+02,   3.2024472e+02,
      6.3180315e+01,  -3.2981235e+02,  -2.7376225e+02,
      1.3916947e+02,  -1.2456450e+02,  -3.9784103e+01,
      4.4775903e+01,  -5.9221629e-01,   4.5273252e+01,
      1.0658006e+02,   1.0074081e+00,  -1.6157223e+01,
     -1.4368321e+02,   1.0695026e+01,  -6.4456274e+01,
     -2.0966453e+02,   1.9478939e+02,   3.3718784e+01,
     -2.6615720e+02,   2.1153746e+02,   2.1665789e+02,
      1.5029433e+02,  -6.9685805e+01,  -2.2586069e+02,
      7.7365900e+01,   1.1591818e+02,   6.7622680e+02,
     -7.1309709e+01,  -5.9081060e+01,  -4.0772774e+02,
      4.6469608e+01,   1.7086578e+01,  -9.6540468e+01,
     -4.9728033e+01,   1.9712287e+01,  -1.6329181e+01,
      4.6458125e+01,  -7.4674882e+01,   1.7256451e+02,
      4.2324173e+00,  -3.9979540e+00,  -4.9597923e+01,
     -9.1351363e+00,   2.7573921e+01,   4.2723360e+01,
     -6.5366436e-01,  -1.6123327e+01,   1.1882875e+01,
      8.3193121e+00,   1.2637625e+01,  -2.9047597e+01,
      4.5609031e-01,   1.3832574e+01,  -1.3788797e+01,
     -1.8356218e+01,  -1.8646071e+00,   6.1903289e+00,
      2.1779614e+01,  -5.4906091e+00,   7.6356249e+00,
     -2.3690438e+01,  -1.5341628e+01,   1.8123950e+01,
     -1.1402596e+01,  -2.8167749e+01,  -7.3629852e+01,
      1.8330607e+01,   8.1405031e+00,   2.3813048e+01,
     -4.8063072e+00,   2.5370158e+01,  -9.3037180e+00,
     -3.2828019e+01,  -5.4510168e+00,   1.5688503e+01,
     -1.2221960e+02,  -8.4096601e+01,  -1.6741947e+02,
      1.4575011e+02,  -2.9313861e+02,   5.2533129e+01,
      4.8217953e+00,   5.3672422e+02,  -6.8330227e+02,
      1.2928234e+02,  -4.3066864e+02,   5.2387327e+02,
     -5.3865559e+01,  -7.3987278e-02,  -2.2296832e+01,
      8.2219523e+01,   1.9259844e+02,   9.3924436e+01,
     -4.1761503e+00,   7.0126480e+01,   4.6565681e+01,
     -3.0554830e+01,   7.3912875e+01,   1.3304142e+01,
     -8.1209545e+00,  -3.9229340e+00,  -1.4265712e+01,
      1.9298137e+01,  -1.8892413e+01,  -1.1880570e+01,
     -6.2558149e+00,  -1.0687763e+01,   3.1305329e+00,
     -7.2720335e+00,  -1.3015752e+01,  -6.9941100e+00,
     -1.2604797e+01,  -1.9636250e+01,  -1.0784977e+01,
     -1.4331411e+01,  -2.0057019e+01,  -6.9176675e-01,
     -2.1847246e+01,  -2.3772071e+02,   2.8376055e+01,
      1.0647606e+02,   1.0768227e+02,   5.3907508e+01,
     -2.2041313e+02,   5.4251339e+02,  -5.0437107e+01,
      7.8557629e+01,  -4.9455328e+02,   1.2244968e+02,
     -1.5161065e+01,  -2.1571895e+01,   7.7903476e+01,
      3.2634541e+01,   1.2703927e+01,   6.1282875e+01,
     -5.2385124e+01,   4.9681767e+01,  -1.1936343e+02,
      7.8104976e+00,   1.8067222e+01,   3.8909997e+01,
     -1.2641638e+01,  -4.0905613e+01,  -9.2798114e+01,
      2.2095598e+01,  -9.9069010e+00,  -1.5758533e+01,
      3.7317343e+00,   1.6828288e+00,  -6.4131355e+00,
      1.0941432e+01,  -9.3913021e+00,   1.5634820e+01,
      1.4255051e+01,   8.9375578e+00,  -8.6836960e+00,
      1.9556386e+01,   1.3075170e+01,   1.8476974e+01,
      3.4642499e+01,   6.1114012e+00,   5.6971340e+01,
     -1.7903778e+00,  -2.9537640e+01,  -2.2342061e+01,
     -2.9216602e+01,   5.6492095e+00,   1.5121001e+01,
     -1.4926547e+01,   2.2282990e+01,  -6.3803545e+00,
     -6.4549325e+00,  -1.8588066e+01,  -3.1766861e+01,
      1.5363850e+02,  -1.8826929e+02,  -3.6657367e+01,
     -1.5438680e+02,   4.2058826e+01,   1.3286193e+02,
     -3.8261221e+02,   3.0307927e+02,  -4.3722734e+02,
      2.8498010e+02,  -3.7638881e+02,   2.6526507e+02,
      8.9260004e+00,  -2.5605109e+01,  -4.9740232e+00,
     -4.7648685e+01,  -3.5553216e+01,   6.4303050e+00,
      1.9616322e+02,   1.8593114e+02,  -5.8230667e+01,
      4.0420435e+01,  -1.1149320e+02,  -8.9747572e+00,
     -1.4992007e+01,   5.1512231e+00,   9.6296716e+00,
      6.7989744e+00,  -3.0979398e+01,   2.8975840e+01,
     -5.9291673e+01,   7.0075487e+01,   3.1507973e+01,
     -3.9011985e+01,   1.6289178e+02,  -1.9807165e+02,
      7.9046425e+01,  -1.2112183e+02,   3.4616918e+00,
     -4.3934826e+01,   4.3231990e+01,   4.2977626e+01,
     -4.0715967e+00,  -1.3276653e+01,   8.7863093e+01,
     -6.7311494e+00,  -4.1278035e+00,   1.3786439e+01,
     -1.6185047e+01,  -5.8779435e+01,  -1.1557713e+00,
      9.0341904e+00,  -3.4652445e+01,   1.4915186e+01,
      3.0442263e+00,   4.1686503e+01,  -5.5662472e+01,
      6.1592279e+00,   1.0759612e+00,  -5.8451058e+00,
      1.6031958e+01,  -1.6908282e+01,   7.1645617e+00,
     -1.4554418e+01,   2.5358204e+01,  -8.5143182e+00,
      7.0847384e+00,   3.4877304e+00,   3.0357521e+01,
     -1.8027215e+01,   1.6742059e+01,   2.8647355e+01,
      1.7332516e+02,  -4.2161717e+01,   2.9793304e+02,
     -6.6973552e+01,  -6.8020980e+01,  -1.8448708e+02,
      4.1124447e+01,   5.0480142e+02,  -4.4635995e+02,
      5.3610897e+01,  -3.7268678e+02,   3.8248460e+02,
      4.7269047e+01,  -9.2100139e+01,  -1.7968255e+01,
     -1.3434904e+01,  -3.5765602e+01,  -5.2375886e+01,
     -6.8957180e+01,   1.2915844e+02,   7.1932936e+01,
      2.5151019e+00,   4.2384382e+01,  -7.1288924e-01,
     -1.1966408e+02,   6.3708772e+01,  -1.1963846e+02,
     -1.1347684e+02,  -5.1864510e+01,  -9.5456288e+01,
      5.5124664e+01,  -4.0718928e+01,   9.6577321e+01,
     -2.5646091e+01,  -1.5454519e+01,  -1.5282509e+01,
      3.2537218e+01,  -3.0630847e+01,  -3.0559619e+01,
      6.8073772e+00,   3.1668247e+01,  -1.5223650e+01,
     -6.4476666e+00,   8.2602229e+00,   8.1990777e+00,
      1.4342617e+01,  -1.3800110e+01,   2.1823986e+01,
      1.6670502e+01,  -1.8912604e+01,   3.0466792e+01,
      7.4851759e+01,   3.6144186e+01,   2.3548772e+01,
      1.2123683e+01,   1.9252298e+00,   2.6590684e+01,
      1.3691862e+02,  -8.4669417e+01,  -4.2576288e+01,
     -2.8919259e+01,   1.4369882e+02,  -3.5131247e+01,
     -6.1462982e+01,  -9.1516825e+01,   1.4909075e+02,
     -8.0251996e+01,  -2.5072812e+02,   1.5789156e+02,
      1.7288959e+02,   2.4931326e+01,   5.5684718e+01,
      5.8119864e+01,   6.6721597e+02,  -1.4144554e+02,
     -2.1548613e+02,  -5.7007641e+02,   3.3914518e+02,
     -2.2975073e+02,   2.4027286e+02,  -3.2212126e+01,
      6.5372114e+01,  -7.7129036e-01,   8.7876782e+00,
     -4.9837255e+01,  -2.4422199e+01,  -1.1080959e+02,
      5.9794989e+01,   9.5957490e+00,   6.4530107e+01,
      1.5640852e+02,  -2.2533487e+02,   1.7345664e+02,
     -1.4486475e+02,   2.3712527e+02,  -8.5306992e+01,
     -1.1024290e+02,   1.4976467e+02,   2.8625967e+01,
      1.4169550e+01,  -2.5657137e+00,   4.4321633e-01,
      4.6082565e+01,  -1.2663622e+01,   6.5065610e+01,
     -5.3454763e+01,  -8.0267553e+00,  -1.8852972e+01,
     -4.3959161e+01,   1.6753154e+01,  -9.1715282e+01,
      8.0134524e+00,  -6.5164269e+01,  -1.4348219e+02,
      6.8981427e+01,  -1.5760350e+02,  -4.4220406e+01,
     -3.0006646e+00,  -2.7828923e+02,  -4.9386470e+01,
     -7.9390698e+01,   6.1101065e+01,   1.4277132e+02,
     -2.0035517e+02,   5.3317274e+02,  -1.2730779e+02,
      1.3348374e+02,  -4.2398260e+02,   6.0041671e+01,
     -4.2929044e+01,   5.6046219e+01,  -8.2966915e+01,
     -6.5264904e-01,   1.5601325e+01,   7.4267819e+01,
      1.1345328e+01,   6.2013833e+01,   5.2702883e+01,
     -6.3377292e+01,  -7.5953177e+01,   5.1933292e+01,
     -3.0963293e+02,   2.2981719e+01,  -7.3778959e+01,
      3.9088435e+02,  -3.5716426e+01,   1.3236274e+02,
      5.3414875e+01,  -3.3253851e+01,   2.0158990e+02,
      3.1890861e+01,  -3.7327789e+00,  -6.8970620e+00,
      2.5222228e+01,   2.0000080e+01,  -3.7496053e+01,
      3.7979668e+01,  -2.5566492e+01,  -2.9134569e+01,
      1.9010320e+01,   4.2500309e+01,  -1.4307691e+01,
      6.0874345e+01,   4.1286801e+01,  -1.4247252e+02,
     -1.2955328e+02,   6.0202273e+00,  -1.9993627e+02,
      8.0186356e+01,  -1.5616304e+02,  -5.8084026e+00,
     -1.5876075e+02,  -6.1187158e+01,   2.5372478e+01,
     -1.2278418e+02,   4.3853230e+02,  -4.5140921e+02,
      1.5109118e+02,  -3.4850044e+02,   2.3758136e+02,
      9.0253029e+01,   8.5737828e+01,  -1.8649250e+01,
     -3.4452516e+01,  -2.3574091e+01,   8.8917764e-01,
      6.3941200e+01,   1.7164221e+01,   7.5324484e+01,
     -4.9650339e+02,   6.6005763e+01,   1.1873157e+02,
      4.0372761e+02,  -1.7820120e+02,  -1.9843441e+02,
      1.0642104e+02,  -1.0142934e+02,  -3.7445582e+01,
     -1.9218103e+01,  -2.4457348e+01,  -2.9661085e+01,
      9.8317629e+00,  -5.2443611e+01,  -6.0287867e-01,
      7.5913096e+01,   1.3926806e+02,   1.0820633e+01,
     -9.1152402e+01,   1.6884305e+02,   1.0019364e+02,
      1.8174070e+01,  -9.9910652e+01,   2.3105268e+02,
      2.7557915e+01,  -3.3789470e+01,  -1.5559940e+02,
      4.9416665e+00,   4.6470989e+02,  -3.8501342e+02,
      2.6440299e+02,  -3.5206043e+02,   4.2777824e+02,
     -2.6171227e+01,   4.6310882e+01,   5.8162580e-01,
      3.1564179e+01,  -5.7854835e+01,  -5.7634686e+01,
     -1.7448647e+02,   4.0243129e+01,   3.7319362e+01,
      3.1529416e+01,  -1.2884884e+00,  -3.9844683e+01,
      1.5876431e+01,  -1.3574779e+01,   1.0092116e+01,
     -2.4974987e+01,   1.8684427e+01,   1.0708551e+01,
      2.5356759e+01,   1.5962865e+01,  -8.1516821e+00,
      3.5621268e+01,  -2.0604211e+01,   5.1876379e+00,
     -2.9150983e-01,   1.1034757e+01,   1.9584229e+01,
     -2.7930987e+01,  -8.3996901e+00,   3.9087557e+00,
     -2.4518483e+01,   1.2905391e+01,   2.1962075e+01,
      1.9725913e+01,   1.9112465e+01,  -2.1881410e+01,
      1.7617820e+01,  -1.0344327e+01,   5.2041854e+00,
     -3.1127470e-01,   1.7037949e+01,   3.2667202e+01,
     -1.4507996e+01,  -1.9962485e+01,   5.1615155e+00,
     -1.2790482e+02,  -2.3154000e+02,   1.1757090e+02,
      9.5183762e+01,   1.0698025e+02,   2.1535882e+01,
      3.2325370e+00,   6.1634075e+02,  -1.7824777e+01,
     -6.0634874e+01,  -4.1856719e+02,   1.6118857e+01,
     -7.1121761e+01,   6.6744536e+01,  -1.3011560e+02,
      6.7794707e+01,   1.1765336e-01,   4.2690048e+01,
     -2.1508412e+01,  -1.4178418e+01,  -1.6289341e+02,
     -7.2944880e+01,  -1.6097308e+01,   7.5399669e+01,
     -7.0464795e+00,   6.1896952e+01,   1.5548969e+02,
      1.5950972e+01,  -3.8544977e+01,   1.8808315e+02,
      1.1421879e+01,  -4.7629823e+01,  -9.0581110e+01,
      2.2962567e+01,  -5.7087791e+00,   1.8272456e+01,
      1.8038997e+01,  -3.0668032e-01,   4.4514209e+01,
      5.3482352e+00,  -7.7169140e+00,  -1.8742750e+01,
      2.2006238e+01,   3.0028471e+01,  -1.5805128e+01,
      1.7352395e+01,  -1.2372365e+01,  -3.8120553e+01,
     -2.5710098e+00,  -6.2029044e+00,  -1.9748604e+01,
     -3.8389273e+01,   2.1103649e+01,  -8.7990101e+01,
      2.0646020e+01,   3.2988004e+01,  -8.9685879e+01,
      1.5460238e+02,  -3.2564189e+01,  -5.5796338e+01,
     -9.0871162e+01,  -1.3244755e+02,  -6.3273718e+01,
     -6.2800274e+01,   1.3910281e+02,   2.2481858e+01,
      4.7136911e+01,  -3.2398725e+02,  -3.5683451e+01,
     -6.2150419e+01,   1.9286563e+02,   7.6578056e+01,
     -4.4682654e+02,   3.0319951e+02,  -3.5304418e+02,
      3.5344682e+02,  -2.4415816e+02,   2.2907571e+02,
      2.2115339e+01,   6.3553661e+01,  -6.1289974e+01,
     -3.1117553e+01,   2.0433521e+01,   1.6163496e+01,
      1.0807958e+02,   4.0726016e+01,  -9.5901622e+01,
      9.1243782e+00,  -6.3353582e+01,   3.2951112e+01,
     -8.5862463e+00,  -9.7079768e+00,   1.3877649e+02,
      2.8454120e+01,   1.4487639e+02,   1.4035460e+02,
      2.1637202e+01,  -6.9779048e+00,  -1.2090387e+02,
     -9.9584228e+00,  -5.0602737e+00,   3.7463151e+01,
     -5.0339905e+01,  -5.8537495e+00,   4.7251107e+01,
      8.6217576e+00,   4.8370562e+00,  -3.9302561e-01,
      3.1525503e+00,  -1.1047934e+01,  -4.7241712e+00,
      6.7915343e+00,   3.0135731e+01,  -1.4279134e+01,
     -1.6958093e+01,   1.9372720e+00,  -3.5369739e+01,
      9.3338463e+00,  -7.6835016e+01,  -4.2154581e+01,
     -3.4110401e+01,  -3.5291708e+01,  -7.6535039e+01,
     -5.5775238e+01,   1.0227261e+02,  -1.2035804e+02,
      1.6347944e+02,  -2.7736772e+01,  -2.8724444e+01,
     -8.3691665e+01,  -9.6291447e+01,   8.5732091e+01,
      2.2940575e+02,  -1.3257670e+02,   2.3658876e+02,
     -9.7876366e+01,  -9.5568861e+01,  -1.4042286e+02,
     -6.1734071e+01,   5.1496689e+02,  -6.5011844e+02,
      3.7073852e+01,  -2.8251883e+02,   4.4360865e+02,
      1.5452333e+01,   3.9788064e+01,   1.8757624e+01,
     -2.6708679e+01,  -2.7593120e+01,  -2.5562573e+01,
      4.2807478e+01,   1.5579995e+02,   4.6933618e+00,
      9.4450812e+00,   4.0840985e+00,   9.3645192e+01,
     -7.9780675e+01,  -3.4519193e+02,  -6.4248260e+02,
     -5.5745247e+01,   3.4667428e+02,   2.5360487e+02,
      2.6138546e+01,  -4.5816017e+01,   4.7082606e+02,
     -3.5839286e+01,  -4.2322367e+00,  -1.1927510e+01,
      3.9949677e+01,  -3.5295550e+01,   4.8105030e+01,
      4.7322060e+01,  -3.4243566e+01,  -8.6987816e+01,
     -1.2769744e+01,   6.1814286e+01,  -5.0389331e+01,
      9.8781608e+01,  -1.9618018e+02,   4.2551584e+02,
      1.6886040e+01,   7.5848188e+01,  -2.5415723e+02,
      2.5370421e+02,  -4.0346371e+02,   3.3233610e+02,
     -1.6082665e+02,   1.4559594e+02,  -2.2800533e+02,
     -2.0548842e+02,   1.2981476e+02,  -1.1933633e+02,
      3.8081286e+01,   1.2910212e+01,   4.9252234e+01,
     -3.8854631e+01,   3.9366754e+01,   1.0126563e+02,
     -6.9277697e+01,  -2.0271389e+01,   2.5561584e+01,
      7.0616461e+01,  -5.2415083e+01,  -1.4590503e+02,
      1.1005519e+02,  -4.9897034e+01,  -1.2964342e+02,
      3.1494230e+00,   3.4444142e+01,   9.8167636e+01,
      1.0456529e+01,   4.5517479e+00,  -1.0480706e+01,
      2.8274785e+01,  -1.7636057e+01,   1.0648185e+00,
     -8.1792934e+00,  -7.3417958e-01,  -2.7407412e+01,
      3.6406771e+01,   1.4529232e+00,  -5.4330296e+01,
     -3.0396039e+00,   6.0146485e+00,   2.8019429e+01,
     -4.4193408e+01,   1.0041200e+01,   3.1878548e+01,
     -6.8321723e+01,   5.2813226e+01,   5.1888340e+01,
     -6.5549668e+01,  -4.8250425e+00,   7.2024312e+01,
      4.4938566e+01,  -1.0530041e+02,   1.1947566e+02,
      5.0140121e+01,   1.4649013e+02,   3.7579025e+01,
     -1.2711580e+02,  -1.9697818e+01,  -8.4687746e+01,
     -1.7008532e+02,   2.0236126e+02,  -1.8881296e+02,
      1.1448751e+02,  -6.9685595e+01,   1.1918129e+02,
      1.0669098e+02,   1.4325513e+02,   6.4865062e+02,
     -9.6267077e+01,  -7.6612374e+01,  -5.2888437e+02,
     -4.9965429e+01,   1.7461759e+01,  -1.1640873e+01,
      2.0759911e+01,   2.6288492e+01,  -9.0311457e+00,
      2.6452811e+01,  -4.6835529e+00,   4.0111940e+01,
     -7.7461119e+01,   2.4940882e+00,  -3.6420297e+02,
      6.9301306e+01,  -1.0602064e+02,   1.9037673e+02,
     -5.8338570e+02,   4.9301556e+02,   6.0897708e+01,
      3.6971747e+02,  -3.0727368e+02,  -1.6077104e+02,
     -6.2862958e+01,  -1.0904492e+01,  -2.7843155e+01,
     -4.0200023e+01,   1.8612557e+00,   2.1241119e+01,
      1.1540883e+02,   9.5140800e+01,  -8.2625244e+00,
     -3.5646078e+01,  -2.5528630e+01,  -5.1553518e+01,
      3.5001456e+01,   4.5437776e+01,   3.8612973e+01,
     -3.7141849e-01,  -5.4019124e+01,  -1.1314921e+01,
     -2.8303867e+01,  -3.6467283e+01,   1.4972636e+00,
     -1.1923395e+01,   3.6800399e+00,   1.2190962e+01,
     -5.9977862e-01,  -3.3467742e+00,   1.3313731e+01,
      2.7971821e+01,   4.0003398e+00,  -1.4819540e+01,
      2.2205655e+01,  -1.3081256e+01,   1.7913450e+01,
     -2.4495259e+01,  -2.6861706e+01,  -3.0753142e+01,
      1.9984629e+01,   2.2365604e+01,  -2.0481392e+00,
     -1.9331921e+01,   2.0065723e+01,  -1.2751849e+01,
      5.2210605e+00,   2.6081813e+00,   3.4280291e+01,
      4.3659586e+02,  -1.9700374e+02,   3.4584900e+01,
     -3.7166227e+02,   9.7148697e+01,  -2.4754869e+01,
      9.5789104e+02,   3.0644618e+02,   2.4756687e+00,
     -2.8023430e+02,  -1.8803401e+02,  -7.7809228e+00,
     -9.9388784e+01,  -2.5193393e+02,   7.4667488e+00,
      6.4752499e+01,   1.2751188e+01,  -1.1577336e+01,
      9.8913540e+01,   5.9555441e+01,   7.8835407e+01,
     -2.2494125e+01,  -1.7000028e+01,  -2.2122572e+01,
      5.6606616e+01,   3.9166248e+01,   1.3351039e+01,
     -1.4666563e+00,  -6.7284508e+01,  -2.9569084e+00,
     -7.6007124e+00,   2.1687443e+01,   1.8580634e+00,
     -1.9916341e+01,  -2.0421467e+00,  -4.0546709e+00,
     -3.7168947e+00,   1.9477580e+00,  -4.1076952e-01,
     -7.4950107e+00,   4.2093482e+01,   3.0351358e+01,
     -5.7380220e+01,  -1.3809564e+01,  -5.3766814e+01,
     -5.2396267e+01,  -1.9168275e+01,  -2.2402881e+01,
      1.4116491e+00,   2.2165994e+01,   1.0041847e+01,
      6.2188765e-01,  -3.5345999e+01,  -1.2943777e+00,
      3.4163512e+01,   1.0047516e+01,   1.1060459e+01,
      2.0856655e+01,  -1.5009262e+01,   6.3702684e+00,
     -5.4092846e+02,  -4.6549548e+01,   2.9199345e+01
    };

    const double* forceDataPtr = forceData;
    for (unsigned int ii = 0; ii < numberOfParticles; ii++) { 
        expectedForces[ii]    = Vec3(forceDataPtr[0], forceDataPtr[1], forceDataPtr[2]);
        forceDataPtr         += 3;
    }

    double tolerance          = 1.0e-05;
    compareForceNormsEnergy(testName, expectedEnergy, energy, expectedForces, forces, tolerance);
}

void setupKernels(int argc, char* argv[]);
void runPlatformTests();

int main(int argc, char* argv[]) {
    try {
        setupKernels(argc, argv);

        // test direct and mutual polarization cases and
        // mutual polarization w/ the cavity term

        testGeneralizedKirkwoodAmmoniaDirectPolarization();
        testGeneralizedKirkwoodAmmoniaMutualPolarization();
        testGeneralizedKirkwoodAmmoniaExtrapolatedPolarization();
        testGeneralizedKirkwoodAmmoniaMutualPolarizationWithCavityTerm();
        testGeneralizedKirkwoodVillinDirectPolarization();
        testGeneralizedKirkwoodVillinExtrapolatedPolarization();
        testGeneralizedKirkwoodVillinMutualPolarization();

        runPlatformTests();
    }
    catch(const std::exception& e) {
        std::cout << "exception: " << e.what() << std::endl;
        std::cout << "FAIL - ERROR.  Test failed." << std::endl;
        return 1;
    }
    std::cout << "Done" << std::endl;
    return 0;
}
