﻿' Licensed to the .NET Foundation under one or more agreements.
' The .NET Foundation licenses this file to you under the MIT license.
' See the LICENSE file in the project root for more information.

Imports System.Threading.Tasks
Imports Microsoft.CodeAnalysis.CodeFixes
Imports Microsoft.CodeAnalysis.Diagnostics

Namespace Microsoft.CodeAnalysis.Editor.UnitTests.Diagnostics.GenerateEvent
    Public Class GenerateEventCrossLanguageTests
        Inherits AbstractCrossLanguageUserDiagnosticTest

        Friend Overrides Function CreateDiagnosticProviderAndFixer(workspace As Workspace, language As String) As (DiagnosticAnalyzer, CodeFixProvider)
            Return (Nothing, New Microsoft.CodeAnalysis.VisualBasic.CodeFixes.GenerateEvent.GenerateEventCodeFixProvider())
        End Function

        <Fact, Trait(Traits.Feature, Traits.Features.CodeActionsGenerateEvent)>
        Public Async Function TestGenerateEventInCSharpFileFromImplementsWithParameterList() As Task
            Dim input =
        <Workspace>
            <Project Language="Visual Basic" AssemblyName="VBAssembly1" CommonReferences="true">
                <ProjectReference>CSAssembly1</ProjectReference>
                <Document>
Class c
    Implements i
    Event a(x As Integer) Implements $$i.goo
End Class
                </Document>
            </Project>
            <Project Language="C#" AssemblyName="CSAssembly1" CommonReferences="true">
                <Document FilePath=<%= DestinationDocument %>>
public interface i
{
}
                </Document>
            </Project>
        </Workspace>

            Dim expected =
                <text>public interface i
{
    event gooEventHandler goo;
}

public delegate void gooEventHandler(int x);
                </text>.Value.Trim()

            Await TestAsync(input, expected)
        End Function

        <Fact, Trait(Traits.Feature, Traits.Features.CodeActionsGenerateEvent)>
        Public Async Function TestGenerateEventInCSharpFileFromImplementsWithType() As Task
            Dim input =
        <Workspace>
            <Project Language="Visual Basic" AssemblyName="VBAssembly1" CommonReferences="true">
                <ProjectReference>CSAssembly1</ProjectReference>
                <Document>
Class c
    Implements i
    Event a as EventHandler Implements $$i.goo
End Class
                </Document>
            </Project>
            <Project Language="C#" AssemblyName="CSAssembly1" CommonReferences="true">
                <Document FilePath=<%= DestinationDocument %>>
public interface i
{
}
                </Document>
            </Project>
        </Workspace>

            Dim expected =
                <text>using System;

public interface i
{
    event EventHandler goo;
}</text>.Value.Trim()

            Await TestAsync(input, expected)
        End Function

        <Fact, Trait(Traits.Feature, Traits.Features.CodeActionsGenerateEvent)>
        <WorkItem(737021, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/737021")>
        Public Async Function TestGenerateEventInCSharpFileFromHandles() As Task
            Dim input =
        <Workspace>
            <Project Language="Visual Basic" AssemblyName="VBAssembly1" CommonReferences="true">
                <ProjectReference>CSAssembly1</ProjectReference>
                <Document>
Class c
    WithEvents field as Program
    Sub Goo(x as Integer) Handles field.Ev$$
    End Sub
End Class
                </Document>
            </Project>
            <Project Language="C#" AssemblyName="CSAssembly1" CommonReferences="true">
                <Document FilePath=<%= DestinationDocument %>>
public class Program
{
}
                </Document>
            </Project>
        </Workspace>

            Dim expected =
                <text>public class Program
{
    public event EvHandler Ev;
}

public delegate void EvHandler(int x);
                </text>.Value.Trim()

            Await TestAsync(input, expected)
        End Function

#Disable Warning CA2243 ' Attribute string literals should parse correctly
        <WorkItem(144843, "Generate method stub generates into *.Designer.cs")>
        <Fact, Trait(Traits.Feature, Traits.Features.CodeActionsGenerateEvent)>
        Public Async Function PreferNormalFileOverAutoGeneratedFile_Basic() As Task
#Enable Warning CA2243 ' Attribute string literals should parse correctly
            Dim input =
        <Workspace>
            <Project Language="Visual Basic" AssemblyName="VBAssembly1" CommonReferences="true">
                <Document>
Class c
    WithEvents field as UserControl1
    Sub Goo(x as Integer) Handles field.Ev$$
    End Sub
End Class
                </Document>
                <Document FilePath="UserControl1.Designer.vb">
' This file is auto-generated
Partial Class UserControl1
End Class
                </Document>
                <Document FilePath="UserControl1.vb">
Partial Public Class UserControl1
End Class
                </Document>
            </Project>
        </Workspace>

            Dim expectedFileWithText =
                 New Dictionary(Of String, String) From {
                    {"UserControl1.vb",
<Text>
Partial Public Class UserControl1
    Public Event Ev(x As Integer)
End Class
</Text>.Value.Trim()},
                    {"UserControl1.Designer.vb",
<Text>
' This file is auto-generated
Partial Class UserControl1
End Class
</Text>.Value.Trim()}
                }

            Await TestAsync(input, fileNameToExpected:=expectedFileWithText)
        End Function
    End Class
End Namespace
