/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Shaders.h"

#include <epoxy/gl.h>
#include <glm/gtc/type_ptr.hpp>

#include <cstring>
#include <iostream>

namespace libgltf
{

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const int iValue)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1i(iLoc, iValue);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const float fValue)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform1fv(iLoc, 1, &fValue);
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec2& vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform2fv(iLoc, 1, glm::value_ptr(vVector));
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec3& vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform3fv(iLoc, 1, glm::value_ptr(vVector));
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::vec4& vVector)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniform4fv(iLoc, 1, glm::value_ptr(vVector));
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::mat3& mMatrix)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix3fv(iLoc, 1, false, glm::value_ptr(mMatrix));
}

void ShaderProgram::setUniform(unsigned int uProgId, const char* name,
                               const glm::mat4& mMatrix)
{
    int iLoc = glGetUniformLocation(uProgId, name);
    glUniformMatrix4fv(iLoc, 1, false, glm::value_ptr(mMatrix));
}

unsigned int ShaderProgram::createProgram(const char* pvShader,
                                          size_t ivShaderSize,
                                          const char* pfShader,
                                          size_t ifShaderSize)
{
    unsigned int programId = glCreateProgram();
    if (!loadShader(programId, pvShader, ivShaderSize, GL_VERTEX_SHADER))
    {
        return 0;
    }
    if (!loadShader(programId, pfShader, ifShaderSize, GL_FRAGMENT_SHADER))
    {
        return 0;
    }

    return programId;
}

void ShaderProgram::deleteProgram(unsigned int programId)
{
    glDeleteShader(programId);
}

void ShaderProgram::useProgram(unsigned int programId)
{
    glUseProgram(programId);
}

bool ShaderProgram::loadShader(unsigned int programId, const char* pShader,
                               size_t iSize, int type)
{
    unsigned int shaderId = glCreateShader(type);

    if (!compileShader(pShader, iSize, shaderId))
    {
        std::cerr << "compileShader : compileShader failed." << std::endl;
        return false;
    }

    if (!linkProgram(programId, shaderId))
    {
        std::cerr << "compileShader : linkProgram failed." << std::endl;
        return false;
    }

    deleteShader(shaderId);
    return true;
}

bool ShaderProgram::compileShader(const char* pShader, size_t iSize,
                                  unsigned int shaderId)
{
    GLint iGLSize = static_cast<GLint>(iSize);
    if( strstr(pShader,"#version") == 0 )
    {
        const GLchar* aSources[] = {
            "#version 130\n",
            pShader,
        };

        const GLint aSizes[] = {
            strlen("#version 130\n"),
            iGLSize,
        };

        glShaderSource(shaderId, 2, &aSources[0], &aSizes[0]);
    }
    else
    {
        glShaderSource(shaderId, 1, &pShader, &iGLSize);
    }
    glCompileShader(shaderId);
    int iStatus = 0;
    glGetShaderiv(shaderId, GL_COMPILE_STATUS, &iStatus);
    if (GL_FALSE == iStatus)
    {
        int iLogLength;
        char sInfoLog[1024] = {0};
        glGetShaderInfoLog(shaderId, 1024, &iLogLength, sInfoLog);
        std::cerr << sInfoLog << std::endl;
        return false;
    }
    return true;
}

bool ShaderProgram::linkProgram(unsigned int programId,
                                unsigned int shaderId)
{
    int iStatus = 0;
    glAttachShader(programId, shaderId);
    glLinkProgram(programId);
    glGetProgramiv(programId, GL_LINK_STATUS, &iStatus);
    if (GL_FALSE == iStatus)
    {
        int iLogLength;
        char sInfoLog[1024] = {0};
        glGetShaderInfoLog(shaderId, 1024, &iLogLength, sInfoLog);
        std::cerr << sInfoLog << std::endl;
        return false;
    }
    return true;
}

void ShaderProgram::deleteShader(unsigned int shaderId)
{
    glDeleteShader(shaderId);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
