/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */

#include "verify_x509.h"
#include "_verify_x509.h"
#include "_verify_log.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <openssl/err.h>
#include <openssl/evp.h>


/*****************************************************************************/
/* Internal prototypes                                                       */
/*****************************************************************************/

/**
 * Process internal verification data to get to the fundamental types e.g.
 * STACK_OF (X509) * and EVP_PKEY from PEM files. 
 * Returns 0 on success or result of ERR_peek_error() on error
 */ 
static unsigned long process_internal_verify_data (internal_verify_x509_data_t ** verify_x509_data);

/*****************************************************************************/
/* Function definitions                                                      */
/*****************************************************************************/

/**
 * initializes verify_x509_data and loads OpenSSL_add_all_algorithms and
 * ERR_load_crypto_strings
 * Returns 0 on success, 1 on error
 */
int verify_X509_init (internal_verify_x509_data_t ** verify_x509_data)
{
    OpenSSL_add_all_algorithms();
    ERR_load_crypto_strings();

    /* Initialize our own library with number and error strings */
    verify_init_library();

    if ( (*verify_x509_data = calloc ((size_t)1, sizeof (internal_verify_x509_data_t))) == NULL)
        return 1;

    (*verify_x509_data)->is_initialized = 1;

    return 0;
}

/**
 * Parses key/value pair verify_x509_option/... and stores in the internal data
 * verify_x509_data. Returns VER_R_* error code or
 * VER_R_X509_PARAMS_OK on success
 */
verify_x509_error_t verify_X509_setParameter(
	internal_verify_x509_data_t ** verify_x509_data,
	verify_x509_option_t verify_x509_option,
	...)
{
    const char *oper = "Setting verify parameters";
    internal_verify_x509_data_t *internal_verify_x509_data;
    struct stat		my_stat;
    va_list		ap;
    verify_x509_error_t retval = VER_R_X509_PARAMS_OK; /* all ok */

    if (!verify_x509_data || !(*verify_x509_data))
        return VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_CONTAINER_FAILURE);

    internal_verify_x509_data = *verify_x509_data;

    va_start (ap, verify_x509_option);
    switch (verify_x509_option) {
        case VERIFY_X509_CA_PATH :
            if (internal_verify_x509_data->capath) {
		retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->capath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->capath) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_DATA_EMPTY);
                goto finalize;
            }
            /* Check if we can access the data */
            if (stat(internal_verify_x509_data->capath, &my_stat) != 0) {
                verify_error (oper, "Error: unable to stat() CA directory: %s",
                       internal_verify_x509_data->capath);
                internal_verify_x509_data->capath = NULL;
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ACCESS_FAILURE);
                goto finalize;
            }
            break;
        case VERIFY_X509_CERTIFICATE_FILEPATH :
            if (internal_verify_x509_data -> certificate_filepath) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->certificate_filepath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->certificate_filepath) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_DATA_EMPTY);
                goto finalize;
            }
            /* Check if we can access the data */
            if (stat(internal_verify_x509_data->certificate_filepath, &my_stat) != 0) {
                verify_error (oper, "Error: unable to stat() Certificate File: %s",
                       internal_verify_x509_data->certificate_filepath);
                internal_verify_x509_data->certificate_filepath = NULL;
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ACCESS_FAILURE);
                goto finalize;
            }
            break;
        case VERIFY_X509_CERTIFICATE_F_HANDLE :
            if (internal_verify_x509_data->certificate_f_handle) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->certificate_f_handle = va_arg(ap, FILE *);
            break;
        case VERIFY_X509_CERTIFICATE_PEM :
            if (internal_verify_x509_data->certificate_pem_str) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->certificate_pem_str = va_arg(ap, char *);
            break;
        case VERIFY_X509_PRIVATEKEY_FILE :
            if (internal_verify_x509_data->private_key_filepath) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->private_key_filepath = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->private_key_filepath) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_DATA_EMPTY);
                goto finalize;
            }
            /* Check if we can access the data */
            if (stat(internal_verify_x509_data->private_key_filepath, &my_stat) != 0) {
                verify_error (oper, "Error: unable to stat() Private Key File: %s",
                       internal_verify_x509_data->private_key_filepath);
                internal_verify_x509_data->private_key_filepath = NULL;
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ACCESS_FAILURE);
                goto finalize;
            }
            break;
        case VERIFY_X509_PRIVATEKEY_PEM :
            if (internal_verify_x509_data->private_key_pem) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->private_key_pem = va_arg(ap, char *);
            break;
        case VERIFY_X509_CRL_PATH :
            if (internal_verify_x509_data->crl_path) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->crl_path = va_arg(ap, char *);

            /* Check of not NULL */
            if (!internal_verify_x509_data->crl_path) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_DATA_EMPTY);
                goto finalize;
            }
            /* Check if we can access the data */
            if (stat(internal_verify_x509_data->crl_path, &my_stat) != 0) {
                verify_error (oper, "Error: unable to stat() CRL path: %s",
                       internal_verify_x509_data->crl_path);
                internal_verify_x509_data->crl_path = NULL;
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ACCESS_FAILURE);
                goto finalize;
            }
            break;
        case VERIFY_X509_OCSP_RESPONDER_URI :
            if (internal_verify_x509_data->ocsp_responder_uri) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->ocsp_responder_uri = va_arg(ap, char *);
            break;
        case VERIFY_X509_OPTIONS_NO_CRL_CHECK :
            internal_verify_x509_data->no_crl_check           = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY :
            internal_verify_x509_data->allow_limited_proxy    = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_REQUIRE_LIMITED_PROXY :
            internal_verify_x509_data->require_limited_proxy  = va_arg(ap, unsigned int);
            break;
        case VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY:
            internal_verify_x509_data->must_have_priv_key     = va_arg(ap, unsigned int);
            break;
	case VERIFY_X509_OPTIONS_VERIFY_AT_NOTBEFORE :
            internal_verify_x509_data->verify_at_notbefore    = va_arg(ap,  unsigned int);
            break;
        case VERIFY_X509_STACK_OF_X509 :
            if (internal_verify_x509_data -> stack_of_x509) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->stack_of_x509 = va_arg(ap, STACK_OF (X509) *);
            break;
        case VERIFY_X509_EVP_PKEY :
            if (internal_verify_x509_data->evp_pkey) {
                retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_ALREADY_SET);
                goto finalize;
            }
            internal_verify_x509_data->evp_pkey = va_arg(ap, EVP_PKEY *);
            break;
        default :
            verify_error (oper, "Unsupported datatype option specified: %s",
                   "the datatype and data specified is not supported and will not be used in the process");
            retval = VERIFY_reasonval(VER_F_SET_PARAM,VER_R_X509_PARAMS_UNSUPPORTED_DATATYPE);
            goto finalize;
    }
finalize:
    va_end(ap);
    return retval;
}



/**
 * This is the verification function that will first process the provided
 * setParameters into usable datatypes for the actual verification. Then the
 * verification of the certificate chain (includes GT2 and GT4/RFC proxies) will
 * start. First the chain will be checked and then (if there is a private key)
 * the private key must match the certificate chain.
 *
 * Returns 0 on success or error code as from ERR_peek_error()
 */
long unsigned verify_X509_verify (internal_verify_x509_data_t **verify_x509_data)
{
    const char *oper = "Verifying data";
    internal_verify_x509_data_t * internal_verify_x509_data = NULL;
    STACK_OF (X509) * stack;
    EVP_PKEY        * pkey;
    unsigned long result;
    int i = 0;
    int depth = 0;
    X509 *cert = NULL;
    const char *reason = NULL;

    /* Input check */
    if (!verify_x509_data || !(*verify_x509_data))
	return VERIFY_errval(VER_F_VERIFY_X509_VERIFY, VER_R_PARAMETER_EMPTY);

    internal_verify_x509_data = *verify_x509_data;


    /*
     * Crunch on the supplied information to get to the principle pieces of
     * information required for the verification based on the STACK_OF (X509) *
     * and the EVP_PKEY *
     */
    if ( (result = process_internal_verify_data(&internal_verify_x509_data)) )
        return result;

    /* Get the right stack and private key */
    stack = internal_verify_x509_data -> stack_of_x509 ?
                internal_verify_x509_data -> stack_of_x509 :
		internal_verify_x509_data -> derived_data.stack_of_x509;
    pkey =  internal_verify_x509_data -> evp_pkey ?
                internal_verify_x509_data -> evp_pkey :
		internal_verify_x509_data -> derived_data.evp_pkey;

    /* first, verify the certificate chain */
    if (!stack) {
        verify_error(oper, "No certificate chain present: %s",
              "There was no STACK_OF(X509) provided, "
	      "nor a PEM string to be transcoded into a STACK_OF(X509)");
	return VERIFY_errval(VER_F_VERIFY_X509_VERIFY, VER_R_CERTSTACK_EMPTY);
    }

    /* Verify the certificate chain and its content */
    /* returns 0 on success */
    if ( (result=verify_verifyCert(internal_verify_x509_data->capath, stack, internal_verify_x509_data->verify_at_notbefore)) )
        return result;

    /*
     * When set to have the private key, then this means that the certificate
     * chain MUST be accompanied by a private key (and in other test this one
     * must fit the chain)
     */
    if (!pkey)	{
	if (internal_verify_x509_data->must_have_priv_key == VERIFY_ENABLE) {
	    verify_error (oper, "No private key provided: %s",
		   "the configuration (by default or by explicit statement) "
		   "demands its presence");
	    return VERIFY_errval(VER_F_VERIFY_X509_VERIFY, VER_R_NOPRIVATEKEY_DISABLED);
	} else {
	    verify_log (L_INFO, "Verification of chain without private key is OK");
	    result = 0; /* Good */
	}
    } else {
        /* still OK? then match the proxy public and private keys */
	/* returns 0 on success */
        if ( (result = verify_verifyPrivateKey (stack, pkey)) ) {
	    if ( (reason=ERR_reason_error_string(result))==NULL )
		reason=ERR_error_string(result,NULL);
	    verify_error(oper, "Verifying private key: %s", reason);
            return result;
        }
        /* aaah, nirwana */
        verify_log (L_INFO, "Verification of chain with private key is OK");
    }

    /* Test for Limited proxies in the certificate stack, and fail when found.
     * It's annoying we need to go through the list again, but we cannot pass
     * this flag to grid_X509_verify_callback() which calls grid_verifyProxy()
     * and grid_verifyPathLenConstraints() which go through the list the
     * first time to figure out the type of certificate */
    if (internal_verify_x509_data->allow_limited_proxy == VERIFY_DISABLE) {
        depth = sk_X509_num(stack);

        for (i = 0; i < depth; i++) {
            if ( (cert = sk_X509_value(stack, i)) &&
		 (verify_type_of_proxy(cert) & LIMITED) == LIMITED )
	    {
		verify_error(oper, "Checking for limited proxy usage: %s",
			"Found a limited proxy in the certificate chain "
			"but this is disallowed by configuration.");
		return VERIFY_errval(VER_F_VERIFY_X509_VERIFY,
				     VER_R_LIMITED_DISABLED);
            }
        }
    }

    return result;
}


/**
 * Terminates the verification library by freeing the internal memory
 * Returns 0 on success, 1 on failure.
 */
int verify_X509_term (internal_verify_x509_data_t **verify_x509_data)
{
    internal_verify_x509_data_t *internal_verify_x509_data = NULL;

    if (!verify_x509_data || !(*verify_x509_data)) {
        return 1;
    }
    internal_verify_x509_data = *verify_x509_data;

    /* if nothing initialized, or already cleaned, nothing to do */
    if (!internal_verify_x509_data->is_initialized) {
        return 0;
    }

    /* Liberate derived/acquired/extracted information */
    if (internal_verify_x509_data->derived_data.certificate_pem_str) {
        free(internal_verify_x509_data->derived_data.certificate_pem_str);
        internal_verify_x509_data->derived_data.certificate_pem_str = NULL;
    }
    if (internal_verify_x509_data->derived_data.private_key_pem) {
        free(internal_verify_x509_data->derived_data.private_key_pem);
        internal_verify_x509_data->derived_data.private_key_pem = NULL;
    }
    if (internal_verify_x509_data->derived_data.ocsp_responder_uri) {
        free(internal_verify_x509_data->derived_data.ocsp_responder_uri);
        internal_verify_x509_data->derived_data.ocsp_responder_uri = NULL;
    }
    if (internal_verify_x509_data->derived_data.stack_of_x509) {
        /* popper-de-pop */
        sk_X509_pop_free(internal_verify_x509_data->derived_data.stack_of_x509, X509_free);
        internal_verify_x509_data->derived_data.stack_of_x509 = NULL;
    }
    if (internal_verify_x509_data->derived_data.evp_pkey) {
        EVP_PKEY_free(internal_verify_x509_data->derived_data.evp_pkey);
        internal_verify_x509_data->derived_data.evp_pkey = NULL;
    }

    /* Nullify memory */
    memset(internal_verify_x509_data, 0, sizeof(internal_verify_x509_data_t));

    /* Free and clean the struct */
    free(internal_verify_x509_data);
    internal_verify_x509_data = NULL;
    *verify_x509_data         = NULL;

    return 0;
}


/******************************************************************************
 * Internal functions
 *****************************************************************************/

/**
 * Process internal verification data to get to the fundamental types e.g.
 * STACK_OF (X509) * and EVP_PKEY
 * Returns 0 on success or result of ERR_peek_error() on error
 */ 
static unsigned long process_internal_verify_data (internal_verify_x509_data_t ** verify_x509_data)
{
    const char *oper = "Processing verification data";
    internal_verify_x509_data_t * internal_verify_x509_data = NULL;
    /* Note: OpenSSL X509_V error codes are int, see openssl/x509_vfy.h */
    unsigned long result;

    /* Input check */
    if (!verify_x509_data || !(*verify_x509_data))
	return VERIFY_errval(VER_F_PROCESS_INTERNAL, VER_R_PARAMETER_EMPTY);

    internal_verify_x509_data = *verify_x509_data;

    /* If the private key is already set, don't look for it */
    if (!(internal_verify_x509_data->evp_pkey))
    {
        /* To EVP Private Key from Source Private Key PEM String */
        if (internal_verify_x509_data->private_key_pem) {
	    /* returns 0 on success or value from ERR_get_error() */
            result = verify_x509_readPrivateKeyFromPEM(internal_verify_x509_data->private_key_pem,
                                                       &(internal_verify_x509_data->derived_data.evp_pkey));
            if (result) {
                verify_error (oper, "Failed to read the private key in file: %s",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Certificate PEM String */
        if (internal_verify_x509_data->certificate_pem_str) {
	    /* returns 0 on success or value from ERR_get_error() */
            result = verify_x509_readPrivateKeyFromPEM(internal_verify_x509_data->certificate_pem_str,
                                                       &(internal_verify_x509_data->derived_data.evp_pkey));
            if (result) {
                verify_error (oper, "Failed to read the private key in file: %s",
                        internal_verify_x509_data->certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Private Key File at path */
        else if (internal_verify_x509_data->private_key_filepath) {
	    /* returns 0 on success or value from ERR_get_error() */
            result = verify_x509_readPrivateKeyFromFile(internal_verify_x509_data->private_key_filepath,
                                                        &(internal_verify_x509_data->derived_data.evp_pkey));
            if (result) {
                verify_error (oper, "Failed to read the private key in file: %s",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }

        /* To EVP Private Key from Source Certificate File at path */
        else if (internal_verify_x509_data->certificate_filepath) {
	    /* returns 0 on success or value from ERR_get_error() */
            result = verify_x509_readPrivateKeyFromFile(internal_verify_x509_data->certificate_filepath,
                                                        &(internal_verify_x509_data->derived_data.evp_pkey));
            if (result) {
                verify_error (oper, "Failed to read the private key in file: %s",
                        internal_verify_x509_data -> certificate_filepath);
                return result;
            }
        }
    }

    /* If the certificate stack is already provided, don't go look for it in the certificate file */
    if (!(internal_verify_x509_data->stack_of_x509)) {
	/* returns 0 on success or value from ERR_get_error() */
        result = verify_x509_readPublicCertChain(internal_verify_x509_data->certificate_filepath,
                                                 &(internal_verify_x509_data->derived_data.stack_of_x509));
        if (result) {
            verify_error (oper, "Failed to read the certificate stack in file: %s",
                    internal_verify_x509_data -> certificate_filepath);
            return result;
        }
    }

    return 0;
}


