/*****************************************************************************
 * tables.c: const tables
 *****************************************************************************
 * Copyright (C) 2003-2021 x264 project
 *
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 *          Loren Merritt <lorenm@u.washington.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111, USA.
 *
 * This program is also available under a commercial proprietary license.
 * For more information, contact us at licensing@x264.com.
 *****************************************************************************/

#include "base.h"

const x264_level_t x264_levels[] =
{
    { 10,     1485,     99,    396,     64,    175,   64, 64,  0, 2, 0, 0, 1 },
    {  9,     1485,     99,    396,    128,    350,   64, 64,  0, 2, 0, 0, 1 }, /* "1b" */
    { 11,     3000,    396,    900,    192,    500,  128, 64,  0, 2, 0, 0, 1 },
    { 12,     6000,    396,   2376,    384,   1000,  128, 64,  0, 2, 0, 0, 1 },
    { 13,    11880,    396,   2376,    768,   2000,  128, 64,  0, 2, 0, 0, 1 },
    { 20,    11880,    396,   2376,   2000,   2000,  128, 64,  0, 2, 0, 0, 1 },
    { 21,    19800,    792,   4752,   4000,   4000,  256, 64,  0, 2, 0, 0, 0 },
    { 22,    20250,   1620,   8100,   4000,   4000,  256, 64,  0, 2, 0, 0, 0 },
    { 30,    40500,   1620,   8100,  10000,  10000,  256, 32, 22, 2, 0, 1, 0 },
    { 31,   108000,   3600,  18000,  14000,  14000,  512, 16, 60, 4, 1, 1, 0 },
    { 32,   216000,   5120,  20480,  20000,  20000,  512, 16, 60, 4, 1, 1, 0 },
    { 40,   245760,   8192,  32768,  20000,  25000,  512, 16, 60, 4, 1, 1, 0 },
    { 41,   245760,   8192,  32768,  50000,  62500,  512, 16, 24, 2, 1, 1, 0 },
    { 42,   522240,   8704,  34816,  50000,  62500,  512, 16, 24, 2, 1, 1, 1 },
    { 50,   589824,  22080, 110400, 135000, 135000,  512, 16, 24, 2, 1, 1, 1 },
    { 51,   983040,  36864, 184320, 240000, 240000,  512, 16, 24, 2, 1, 1, 1 },
    { 52,  2073600,  36864, 184320, 240000, 240000,  512, 16, 24, 2, 1, 1, 1 },
    { 60,  4177920, 139264, 696320, 240000, 240000, 8192, 16, 24, 2, 1, 1, 1 },
    { 61,  8355840, 139264, 696320, 480000, 480000, 8192, 16, 24, 2, 1, 1, 1 },
    { 62, 16711680, 139264, 696320, 800000, 800000, 8192, 16, 24, 2, 1, 1, 1 },
    { 0 }
};

/*****************************************************************************
 * MATH
 *****************************************************************************/

const uint8_t x264_exp2_lut[64] =
{
      0,   3,   6,   8,  11,  14,  17,  20,  23,  26,  29,  32,  36,  39,  42,  45,
     48,  52,  55,  58,  62,  65,  69,  72,  76,  80,  83,  87,  91,  94,  98, 102,
    106, 110, 114, 118, 122, 126, 130, 135, 139, 143, 147, 152, 156, 161, 165, 170,
    175, 179, 184, 189, 194, 198, 203, 208, 214, 219, 224, 229, 234, 240, 245, 250
};

const float x264_log2_lut[128] =
{
    0.00000, 0.01123, 0.02237, 0.03342, 0.04439, 0.05528, 0.06609, 0.07682,
    0.08746, 0.09803, 0.10852, 0.11894, 0.12928, 0.13955, 0.14975, 0.15987,
    0.16993, 0.17991, 0.18982, 0.19967, 0.20945, 0.21917, 0.22882, 0.23840,
    0.24793, 0.25739, 0.26679, 0.27612, 0.28540, 0.29462, 0.30378, 0.31288,
    0.32193, 0.33092, 0.33985, 0.34873, 0.35755, 0.36632, 0.37504, 0.38370,
    0.39232, 0.40088, 0.40939, 0.41785, 0.42626, 0.43463, 0.44294, 0.45121,
    0.45943, 0.46761, 0.47573, 0.48382, 0.49185, 0.49985, 0.50779, 0.51570,
    0.52356, 0.53138, 0.53916, 0.54689, 0.55459, 0.56224, 0.56986, 0.57743,
    0.58496, 0.59246, 0.59991, 0.60733, 0.61471, 0.62205, 0.62936, 0.63662,
    0.64386, 0.65105, 0.65821, 0.66534, 0.67243, 0.67948, 0.68650, 0.69349,
    0.70044, 0.70736, 0.71425, 0.72110, 0.72792, 0.73471, 0.74147, 0.74819,
    0.75489, 0.76155, 0.76818, 0.77479, 0.78136, 0.78790, 0.79442, 0.80090,
    0.80735, 0.81378, 0.82018, 0.82655, 0.83289, 0.83920, 0.84549, 0.85175,
    0.85798, 0.86419, 0.87036, 0.87652, 0.88264, 0.88874, 0.89482, 0.90087,
    0.90689, 0.91289, 0.91886, 0.92481, 0.93074, 0.93664, 0.94251, 0.94837,
    0.95420, 0.96000, 0.96578, 0.97154, 0.97728, 0.98299, 0.98868, 0.99435,
};

/* Avoid an int/float conversion. */
const float x264_log2_lz_lut[32] =
{
    31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1,0
};

/*****************************************************************************
 * ANALYSE
 *****************************************************************************/

/* lambda = pow(2,qp/6-2) */
const uint16_t x264_lambda_tab[QP_MAX_MAX+1] =
{
   1,   1,   1,   1,   1,   1,   1,   1, /*  0- 7 */
   1,   1,   1,   1,   1,   1,   1,   1, /*  8-15 */
   2,   2,   2,   2,   3,   3,   3,   4, /* 16-23 */
   4,   4,   5,   6,   6,   7,   8,   9, /* 24-31 */
  10,  11,  13,  14,  16,  18,  20,  23, /* 32-39 */
  25,  29,  32,  36,  40,  45,  51,  57, /* 40-47 */
  64,  72,  81,  91, 102, 114, 128, 144, /* 48-55 */
 161, 181, 203, 228, 256, 287, 323, 362, /* 56-63 */
 406, 456, 512, 575, 645, 724, 813, 912, /* 64-71 */
1024,1149,1290,1448,1625,1825,2048,2299, /* 72-79 */
2580,2896,                               /* 80-81 */
};

/* lambda2 = pow(lambda,2) * .9 * 256 */
/* Capped to avoid overflow */
const int x264_lambda2_tab[QP_MAX_MAX+1] =
{
       14,       18,       22,       28,       36,       45,      57,      72, /*  0- 7 */
       91,      115,      145,      182,      230,      290,     365,     460, /*  8-15 */
      580,      731,      921,     1161,     1462,     1843,    2322,    2925, /* 16-23 */
     3686,     4644,     5851,     7372,     9289,    11703,   14745,   18578, /* 24-31 */
    23407,    29491,    37156,    46814,    58982,    74313,   93628,  117964, /* 32-39 */
   148626,   187257,   235929,   297252,   374514,   471859,  594505,  749029, /* 40-47 */
   943718,  1189010,  1498059,  1887436,  2378021,  2996119, 3774873, 4756042, /* 48-55 */
  5992238,  7549747,  9512085, 11984476, 15099494, 19024170,23968953,30198988, /* 56-63 */
 38048341, 47937906, 60397977, 76096683, 95875813,120795955,                   /* 64-69 */
134217727,134217727,134217727,134217727,134217727,134217727,                   /* 70-75 */
134217727,134217727,134217727,134217727,134217727,134217727,                   /* 76-81 */
};

// should the intra and inter lambdas be different?
// I'm just matching the behaviour of deadzone quant.
const int x264_trellis_lambda2_tab[2][QP_MAX_MAX+1] =
{
    // inter lambda = .85 * .85 * 2**(qp/3. + 10 - LAMBDA_BITS)
    {
               46,       58,       73,       92,      117,      147,
              185,      233,      294,      370,      466,      587,
              740,      932,     1174,     1480,     1864,     2349,
             2959,     3728,     4697,     5918,     7457,     9395,
            11837,    14914,    18790,    23674,    29828,    37581,
            47349,    59656,    75163,    94699,   119313,   150326,
           189399,   238627,   300652,   378798,   477255,   601304,
           757596,   954511,  1202608,  1515192,  1909022,  2405217,
          3030384,  3818045,  4810435,  6060769,  7636091,  9620872,
         12121539, 15272182, 19241743, 24243077, 30544363, 38483486,
         48486154, 61088726, 76966972, 96972308,
        122177453,134217727,134217727,134217727,134217727,134217727,
        134217727,134217727,134217727,134217727,134217727,134217727,
    },
    // intra lambda = .65 * .65 * 2**(qp/3. + 10 - LAMBDA_BITS)
    {
               27,       34,       43,       54,       68,       86,
              108,      136,      172,      216,      273,      343,
              433,      545,      687,      865,     1090,     1374,
             1731,     2180,     2747,     3461,     4361,     5494,
             6922,     8721,    10988,    13844,    17442,    21976,
            27688,    34885,    43953,    55377,    69771,    87906,
           110755,   139543,   175813,   221511,   279087,   351627,
           443023,   558174,   703255,   886046,  1116348,  1406511,
          1772093,  2232697,  2813022,  3544186,  4465396,  5626046,
          7088374,  8930791, 11252092, 14176748, 17861583, 22504184,
         28353495, 35723165, 45008368, 56706990,
         71446330, 90016736,113413980,134217727,134217727,134217727,
        134217727,134217727,134217727,134217727,134217727,134217727,
        134217727,134217727,134217727,134217727,134217727,134217727,
    }
};

const uint16_t x264_chroma_lambda2_offset_tab[MAX_CHROMA_LAMBDA_OFFSET+1] =
{
       16,    20,    25,    32,    40,    50,
       64,    80,   101,   128,   161,   203,
      256,   322,   406,   512,   645,   812,
     1024,  1290,  1625,  2048,  2580,  3250,
     4096,  5160,  6501,  8192, 10321, 13003,
    16384, 20642, 26007, 32768, 41285, 52015,
    65535
};

/*****************************************************************************
 * MC
 *****************************************************************************/

const uint8_t x264_hpel_ref0[16] = {0,1,1,1,0,1,1,1,2,3,3,3,0,1,1,1};
const uint8_t x264_hpel_ref1[16] = {0,0,1,0,2,2,3,2,2,2,3,2,2,2,3,2};

/*****************************************************************************
 * CQM
 *****************************************************************************/

/* default quant matrices */
const uint8_t x264_cqm_jvt4i[16] =
{
      6,13,20,28,
     13,20,28,32,
     20,28,32,37,
     28,32,37,42
};
const uint8_t x264_cqm_jvt4p[16] =
{
    10,14,20,24,
    14,20,24,27,
    20,24,27,30,
    24,27,30,34
};
const uint8_t x264_cqm_jvt8i[64] =
{
     6,10,13,16,18,23,25,27,
    10,11,16,18,23,25,27,29,
    13,16,18,23,25,27,29,31,
    16,18,23,25,27,29,31,33,
    18,23,25,27,29,31,33,36,
    23,25,27,29,31,33,36,38,
    25,27,29,31,33,36,38,40,
    27,29,31,33,36,38,40,42
};
const uint8_t x264_cqm_jvt8p[64] =
{
     9,13,15,17,19,21,22,24,
    13,13,17,19,21,22,24,25,
    15,17,19,21,22,24,25,27,
    17,19,21,22,24,25,27,28,
    19,21,22,24,25,27,28,30,
    21,22,24,25,27,28,30,32,
    22,24,25,27,28,30,32,33,
    24,25,27,28,30,32,33,35
};
const uint8_t x264_cqm_flat16[64] =
{
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16,
    16,16,16,16,16,16,16,16
};
const uint8_t * const x264_cqm_jvt[8] =
{
    x264_cqm_jvt4i, x264_cqm_jvt4p,
    x264_cqm_jvt4i, x264_cqm_jvt4p,
    x264_cqm_jvt8i, x264_cqm_jvt8p,
    x264_cqm_jvt8i, x264_cqm_jvt8p
};

// 720p_avci50, 1080i_avci50, 1080p_avci50
const uint8_t x264_cqm_avci50_4ic[16] =
{
    16,22,28,40,
    22,28,40,44,
    28,40,44,48,
    40,44,48,60
};

// 720p_avci50, 1080p_avci50
const uint8_t x264_cqm_avci50_p_8iy[64] =
{
    16,18,19,21,24,27,30,33,
    18,19,21,24,27,30,33,78,
    19,21,24,27,30,33,78,81,
    21,24,27,30,33,78,81,84,
    24,27,30,33,78,81,84,87,
    27,30,33,78,81,84,87,90,
    30,33,78,81,84,87,90,93,
    33,78,81,84,87,90,93,96
};

// 1080i_avci50
const uint8_t x264_cqm_avci50_1080i_8iy[64] =
{
    16,18,19,21,27,33,81,87,
    18,19,21,24,30,33,81,87,
    19,21,24,27,30,78,84,90,
    21,24,27,30,33,78,84,90,
    24,27,30,33,78,81,84,90,
    24,27,30,33,78,81,84,93,
    27,30,33,78,78,81,87,93,
    30,33,33,78,81,84,87,96
};

// 720p_avci100
const uint8_t x264_cqm_avci100_720p_4ic[16] =
{
    16,21,27,34,
    21,27,34,41,
    27,34,41,46,
    34,41,46,54
};

// 720p_avci100
const uint8_t x264_cqm_avci100_720p_8iy[64] =
{
    16,18,19,21,22,24,26,32,
    18,19,19,21,22,24,26,32,
    19,19,21,22,22,24,26,32,
    21,21,22,22,23,24,26,34,
    22,22,22,23,24,25,26,34,
    24,24,24,24,25,26,34,36,
    26,26,26,26,26,34,36,38,
    32,32,32,34,34,36,38,42
};

// 1080i_avci100, 1080p_avci100
const uint8_t x264_cqm_avci100_1080_4ic[16] =
{
    16,20,26,32,
    20,26,32,38,
    26,32,38,44,
    32,38,44,50
};

// 1080i_avci100
const uint8_t x264_cqm_avci100_1080i_8iy[64] =
{
    16,19,20,23,24,26,32,42,
    18,19,22,24,26,32,36,42,
    18,20,23,24,26,32,36,63,
    19,20,23,26,32,36,42,63,
    20,22,24,26,32,36,59,63,
    22,23,24,26,32,36,59,68,
    22,23,24,26,32,42,59,68,
    22,23,24,26,36,42,59,72
};

// 1080p_avci100
const uint8_t x264_cqm_avci100_1080p_8iy[64] =
{
    16,18,19,20,22,23,24,26,
    18,19,20,22,23,24,26,32,
    19,20,22,23,24,26,32,36,
    20,22,23,24,26,32,36,42,
    22,23,24,26,32,36,42,59,
    23,24,26,32,36,42,59,63,
    24,26,32,36,42,59,63,68,
    26,32,36,42,59,63,68,72
};

// 2160p_avci300
const uint8_t x264_cqm_avci300_2160p_4iy[16] =
{
    12,16,19,20,
    16,19,20,24,
    19,20,24,33,
    20,24,33,39
};

// 2160p_avci300
const uint8_t x264_cqm_avci300_2160p_4ic[16] =
{
    28,39,56,67,
    39,56,67,77,
    56,67,77,104,
    67,77,104,133
};

// 2160p_avci300
const uint8_t x264_cqm_avci300_2160p_8iy[64] =
{
    12,14,16,17,19,20,20,24,
    14,16,17,19,20,20,24,30,
    16,17,19,20,20,24,30,42,
    17,19,20,20,24,30,42,56,
    19,20,20,24,30,42,56,72,
    20,20,24,30,42,56,72,76,
    20,24,30,42,56,72,76,80,
    24,30,42,56,72,76,80,84
};

/*****************************************************************************
 * QUANT
 *****************************************************************************/

const uint8_t x264_decimate_table4[16] =
{
    3,2,2,1,1,1,0,0,0,0,0,0,0,0,0,0
};
const uint8_t x264_decimate_table8[64] =
{
    3,3,3,3,2,2,2,2,2,2,2,2,1,1,1,1,
    1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};

/*****************************************************************************
 * DCT
 *****************************************************************************/

/* the inverse of the scaling factors introduced by 8x8 fdct */
/* uint32 is for the asm implementation of trellis. the actual values fit in uint16. */
#define W(i) (i==0 ? FIX8(1.0000) :\
              i==1 ? FIX8(0.8859) :\
              i==2 ? FIX8(1.6000) :\
              i==3 ? FIX8(0.9415) :\
              i==4 ? FIX8(1.2651) :\
              i==5 ? FIX8(1.1910) :0)
const uint32_t x264_dct8_weight_tab[64] = {
    W(0), W(3), W(4), W(3),  W(0), W(3), W(4), W(3),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),
    W(4), W(5), W(2), W(5),  W(4), W(5), W(2), W(5),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),

    W(0), W(3), W(4), W(3),  W(0), W(3), W(4), W(3),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),
    W(4), W(5), W(2), W(5),  W(4), W(5), W(2), W(5),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1)
};
#undef W

#define W(i) (i==0 ? FIX8(1.76777) :\
              i==1 ? FIX8(1.11803) :\
              i==2 ? FIX8(0.70711) :0)
const uint32_t x264_dct4_weight_tab[16] = {
    W(0), W(1), W(0), W(1),
    W(1), W(2), W(1), W(2),
    W(0), W(1), W(0), W(1),
    W(1), W(2), W(1), W(2)
};
#undef W

/* inverse squared */
#define W(i) (i==0 ? FIX8(3.125) :\
              i==1 ? FIX8(1.25) :\
              i==2 ? FIX8(0.5) :0)
const uint32_t x264_dct4_weight2_tab[16] = {
    W(0), W(1), W(0), W(1),
    W(1), W(2), W(1), W(2),
    W(0), W(1), W(0), W(1),
    W(1), W(2), W(1), W(2)
};
#undef W

#define W(i) (i==0 ? FIX8(1.00000) :\
              i==1 ? FIX8(0.78487) :\
              i==2 ? FIX8(2.56132) :\
              i==3 ? FIX8(0.88637) :\
              i==4 ? FIX8(1.60040) :\
              i==5 ? FIX8(1.41850) :0)
const uint32_t x264_dct8_weight2_tab[64] = {
    W(0), W(3), W(4), W(3),  W(0), W(3), W(4), W(3),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),
    W(4), W(5), W(2), W(5),  W(4), W(5), W(2), W(5),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),

    W(0), W(3), W(4), W(3),  W(0), W(3), W(4), W(3),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1),
    W(4), W(5), W(2), W(5),  W(4), W(5), W(2), W(5),
    W(3), W(1), W(5), W(1),  W(3), W(1), W(5), W(1)
};
#undef W

/*****************************************************************************
 * CABAC
 *****************************************************************************/

const int8_t x264_cabac_context_init_I[1024][2] =
{
    /* 0 - 10 */
    { 20, -15 }, {  2, 54 },  {  3,  74 }, { 20, -15 },
    {  2,  54 }, {  3, 74 },  { -28,127 }, { -23, 104 },
    { -6,  53 }, { -1, 54 },  {  7,  51 },

    /* 11 - 23 unused for I */
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },

    /* 24- 39 */
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },

    /* 40 - 53 */
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },

    /* 54 - 59 */
    { 0, 0 },    { 0, 0 },    { 0, 0 },      { 0, 0 },
    { 0, 0 },    { 0, 0 },

    /* 60 - 69 */
    { 0, 41 },   { 0, 63 },   { 0, 63 },     { 0, 63 },
    { -9, 83 },  { 4, 86 },   { 0, 97 },     { -7, 72 },
    { 13, 41 },  { 3, 62 },

    /* 70 -> 87 */
    { 0, 11 },   { 1, 55 },   { 0, 69 },     { -17, 127 },
    { -13, 102 },{ 0, 82 },   { -7, 74 },    { -21, 107 },
    { -27, 127 },{ -31, 127 },{ -24, 127 },  { -18, 95 },
    { -27, 127 },{ -21, 114 },{ -30, 127 },  { -17, 123 },
    { -12, 115 },{ -16, 122 },

    /* 88 -> 104 */
    { -11, 115 },{ -12, 63 }, { -2, 68 },    { -15, 84 },
    { -13, 104 },{ -3, 70 },  { -8, 93 },    { -10, 90 },
    { -30, 127 },{ -1, 74 },  { -6, 97 },    { -7, 91 },
    { -20, 127 },{ -4, 56 },  { -5, 82 },    { -7, 76 },
    { -22, 125 },

    /* 105 -> 135 */
    { -7, 93 },  { -11, 87 }, { -3, 77 },    { -5, 71 },
    { -4, 63 },  { -4, 68 },  { -12, 84 },   { -7, 62 },
    { -7, 65 },  { 8, 61 },   { 5, 56 },     { -2, 66 },
    { 1, 64 },   { 0, 61 },   { -2, 78 },    { 1, 50 },
    { 7, 52 },   { 10, 35 },  { 0, 44 },     { 11, 38 },
    { 1, 45 },   { 0, 46 },   { 5, 44 },     { 31, 17 },
    { 1, 51 },   { 7, 50 },   { 28, 19 },    { 16, 33 },
    { 14, 62 },  { -13, 108 },{ -15, 100 },

    /* 136 -> 165 */
    { -13, 101 },{ -13, 91 }, { -12, 94 },   { -10, 88 },
    { -16, 84 }, { -10, 86 }, { -7, 83 },    { -13, 87 },
    { -19, 94 }, { 1, 70 },   { 0, 72 },     { -5, 74 },
    { 18, 59 },  { -8, 102 }, { -15, 100 },  { 0, 95 },
    { -4, 75 },  { 2, 72 },   { -11, 75 },   { -3, 71 },
    { 15, 46 },  { -13, 69 }, { 0, 62 },     { 0, 65 },
    { 21, 37 },  { -15, 72 }, { 9, 57 },     { 16, 54 },
    { 0, 62 },   { 12, 72 },

    /* 166 -> 196 */
    { 24, 0 },   { 15, 9 },   { 8, 25 },     { 13, 18 },
    { 15, 9 },   { 13, 19 },  { 10, 37 },    { 12, 18 },
    { 6, 29 },   { 20, 33 },  { 15, 30 },    { 4, 45 },
    { 1, 58 },   { 0, 62 },   { 7, 61 },     { 12, 38 },
    { 11, 45 },  { 15, 39 },  { 11, 42 },    { 13, 44 },
    { 16, 45 },  { 12, 41 },  { 10, 49 },    { 30, 34 },
    { 18, 42 },  { 10, 55 },  { 17, 51 },    { 17, 46 },
    { 0, 89 },   { 26, -19 }, { 22, -17 },

    /* 197 -> 226 */
    { 26, -17 }, { 30, -25 }, { 28, -20 },   { 33, -23 },
    { 37, -27 }, { 33, -23 }, { 40, -28 },   { 38, -17 },
    { 33, -11 }, { 40, -15 }, { 41, -6 },    { 38, 1 },
    { 41, 17 },  { 30, -6 },  { 27, 3 },     { 26, 22 },
    { 37, -16 }, { 35, -4 },  { 38, -8 },    { 38, -3 },
    { 37, 3 },   { 38, 5 },   { 42, 0 },     { 35, 16 },
    { 39, 22 },  { 14, 48 },  { 27, 37 },    { 21, 60 },
    { 12, 68 },  { 2, 97 },

    /* 227 -> 251 */
    { -3, 71 },  { -6, 42 },  { -5, 50 },    { -3, 54 },
    { -2, 62 },  { 0, 58 },   { 1, 63 },     { -2, 72 },
    { -1, 74 },  { -9, 91 },  { -5, 67 },    { -5, 27 },
    { -3, 39 },  { -2, 44 },  { 0, 46 },     { -16, 64 },
    { -8, 68 },  { -10, 78 }, { -6, 77 },    { -10, 86 },
    { -12, 92 }, { -15, 55 }, { -10, 60 },   { -6, 62 },
    { -4, 65 },

    /* 252 -> 275 */
    { -12, 73 }, { -8, 76 },  { -7, 80 },    { -9, 88 },
    { -17, 110 },{ -11, 97 }, { -20, 84 },   { -11, 79 },
    { -6, 73 },  { -4, 74 },  { -13, 86 },   { -13, 96 },
    { -11, 97 }, { -19, 117 },{ -8, 78 },    { -5, 33 },
    { -4, 48 },  { -2, 53 },  { -3, 62 },    { -13, 71 },
    { -10, 79 }, { -12, 86 }, { -13, 90 },   { -14, 97 },

    /* 276 a bit special (not used, x264_cabac_encode_bypass is used instead) */
    { 0, 0 },

    /* 277 -> 307 */
    { -6, 93 },  { -6, 84 },  { -8, 79 },    { 0, 66 },
    { -1, 71 },  { 0, 62 },   { -2, 60 },    { -2, 59 },
    { -5, 75 },  { -3, 62 },  { -4, 58 },    { -9, 66 },
    { -1, 79 },  { 0, 71 },   { 3, 68 },     { 10, 44 },
    { -7, 62 },  { 15, 36 },  { 14, 40 },    { 16, 27 },
    { 12, 29 },  { 1, 44 },   { 20, 36 },    { 18, 32 },
    { 5, 42 },   { 1, 48 },   { 10, 62 },    { 17, 46 },
    { 9, 64 },   { -12, 104 },{ -11, 97 },

    /* 308 -> 337 */
    { -16, 96 }, { -7, 88 },  { -8, 85 },    { -7, 85 },
    { -9, 85 },  { -13, 88 }, { 4, 66 },     { -3, 77 },
    { -3, 76 },  { -6, 76 },  { 10, 58 },    { -1, 76 },
    { -1, 83 },  { -7, 99 },  { -14, 95 },   { 2, 95 },
    { 0, 76 },   { -5, 74 },  { 0, 70 },     { -11, 75 },
    { 1, 68 },   { 0, 65 },   { -14, 73 },   { 3, 62 },
    { 4, 62 },   { -1, 68 },  { -13, 75 },   { 11, 55 },
    { 5, 64 },   { 12, 70 },

    /* 338 -> 368 */
    { 15, 6 },   { 6, 19 },   { 7, 16 },     { 12, 14 },
    { 18, 13 },  { 13, 11 },  { 13, 15 },    { 15, 16 },
    { 12, 23 },  { 13, 23 },  { 15, 20 },    { 14, 26 },
    { 14, 44 },  { 17, 40 },  { 17, 47 },    { 24, 17 },
    { 21, 21 },  { 25, 22 },  { 31, 27 },    { 22, 29 },
    { 19, 35 },  { 14, 50 },  { 10, 57 },    { 7, 63 },
    { -2, 77 },  { -4, 82 },  { -3, 94 },    { 9, 69 },
    { -12, 109 },{ 36, -35 }, { 36, -34 },

    /* 369 -> 398 */
    { 32, -26 }, { 37, -30 }, { 44, -32 },   { 34, -18 },
    { 34, -15 }, { 40, -15 }, { 33, -7 },    { 35, -5 },
    { 33, 0 },   { 38, 2 },   { 33, 13 },    { 23, 35 },
    { 13, 58 },  { 29, -3 },  { 26, 0 },     { 22, 30 },
    { 31, -7 },  { 35, -15 }, { 34, -3 },    { 34, 3 },
    { 36, -1 },  { 34, 5 },   { 32, 11 },    { 35, 5 },
    { 34, 12 },  { 39, 11 },  { 30, 29 },    { 34, 26 },
    { 29, 39 },  { 19, 66 },

    /* 399 -> 435 */
    {  31,  21 }, {  31,  31 }, {  25,  50 },
    { -17, 120 }, { -20, 112 }, { -18, 114 }, { -11,  85 },
    { -15,  92 }, { -14,  89 }, { -26,  71 }, { -15,  81 },
    { -14,  80 }, {   0,  68 }, { -14,  70 }, { -24,  56 },
    { -23,  68 }, { -24,  50 }, { -11,  74 }, {  23, -13 },
    {  26, -13 }, {  40, -15 }, {  49, -14 }, {  44,   3 },
    {  45,   6 }, {  44,  34 }, {  33,  54 }, {  19,  82 },
    {  -3,  75 }, {  -1,  23 }, {   1,  34 }, {   1,  43 },
    {   0,  54 }, {  -2,  55 }, {   0,  61 }, {   1,  64 },
    {   0,  68 }, {  -9,  92 },

    /* 436 -> 459 */
    { -14, 106 }, { -13,  97 }, { -15,  90 }, { -12,  90 },
    { -18,  88 }, { -10,  73 }, {  -9,  79 }, { -14,  86 },
    { -10,  73 }, { -10,  70 }, { -10,  69 }, {  -5,  66 },
    {  -9,  64 }, {  -5,  58 }, {   2,  59 }, {  21, -10 },
    {  24, -11 }, {  28,  -8 }, {  28,  -1 }, {  29,   3 },
    {  29,   9 }, {  35,  20 }, {  29,  36 }, {  14,  67 },

    /* 460 -> 1024 */
    { -17, 123 }, { -12, 115 }, { -16, 122 }, { -11, 115 },
    { -12,  63 }, {  -2,  68 }, { -15,  84 }, { -13, 104 },
    {  -3,  70 }, {  -8,  93 }, { -10,  90 }, { -30, 127 },
    { -17, 123 }, { -12, 115 }, { -16, 122 }, { -11, 115 },
    { -12,  63 }, {  -2,  68 }, { -15,  84 }, { -13, 104 },
    {  -3,  70 }, {  -8,  93 }, { -10,  90 }, { -30, 127 },
    {  -7,  93 }, { -11,  87 }, {  -3,  77 }, {  -5,  71 },
    {  -4,  63 }, {  -4,  68 }, { -12,  84 }, {  -7,  62 },
    {  -7,  65 }, {   8,  61 }, {   5,  56 }, {  -2,  66 },
    {   1,  64 }, {   0,  61 }, {  -2,  78 }, {   1,  50 },
    {   7,  52 }, {  10,  35 }, {   0,  44 }, {  11,  38 },
    {   1,  45 }, {   0,  46 }, {   5,  44 }, {  31,  17 },
    {   1,  51 }, {   7,  50 }, {  28,  19 }, {  16,  33 },
    {  14,  62 }, { -13, 108 }, { -15, 100 }, { -13, 101 },
    { -13,  91 }, { -12,  94 }, { -10,  88 }, { -16,  84 },
    { -10,  86 }, {  -7,  83 }, { -13,  87 }, { -19,  94 },
    {   1,  70 }, {   0,  72 }, {  -5,  74 }, {  18,  59 },
    {  -7,  93 }, { -11,  87 }, {  -3,  77 }, {  -5,  71 },
    {  -4,  63 }, {  -4,  68 }, { -12,  84 }, {  -7,  62 },
    {  -7,  65 }, {   8,  61 }, {   5,  56 }, {  -2,  66 },
    {   1,  64 }, {   0,  61 }, {  -2,  78 }, {   1,  50 },
    {   7,  52 }, {  10,  35 }, {   0,  44 }, {  11,  38 },
    {   1,  45 }, {   0,  46 }, {   5,  44 }, {  31,  17 },
    {   1,  51 }, {   7,  50 }, {  28,  19 }, {  16,  33 },
    {  14,  62 }, { -13, 108 }, { -15, 100 }, { -13, 101 },
    { -13,  91 }, { -12,  94 }, { -10,  88 }, { -16,  84 },
    { -10,  86 }, {  -7,  83 }, { -13,  87 }, { -19,  94 },
    {   1,  70 }, {   0,  72 }, {  -5,  74 }, {  18,  59 },
    {  24,   0 }, {  15,   9 }, {   8,  25 }, {  13,  18 },
    {  15,   9 }, {  13,  19 }, {  10,  37 }, {  12,  18 },
    {   6,  29 }, {  20,  33 }, {  15,  30 }, {   4,  45 },
    {   1,  58 }, {   0,  62 }, {   7,  61 }, {  12,  38 },
    {  11,  45 }, {  15,  39 }, {  11,  42 }, {  13,  44 },
    {  16,  45 }, {  12,  41 }, {  10,  49 }, {  30,  34 },
    {  18,  42 }, {  10,  55 }, {  17,  51 }, {  17,  46 },
    {   0,  89 }, {  26, -19 }, {  22, -17 }, {  26, -17 },
    {  30, -25 }, {  28, -20 }, {  33, -23 }, {  37, -27 },
    {  33, -23 }, {  40, -28 }, {  38, -17 }, {  33, -11 },
    {  40, -15 }, {  41,  -6 }, {  38,   1 }, {  41,  17 },
    {  24,   0 }, {  15,   9 }, {   8,  25 }, {  13,  18 },
    {  15,   9 }, {  13,  19 }, {  10,  37 }, {  12,  18 },
    {   6,  29 }, {  20,  33 }, {  15,  30 }, {   4,  45 },
    {   1,  58 }, {   0,  62 }, {   7,  61 }, {  12,  38 },
    {  11,  45 }, {  15,  39 }, {  11,  42 }, {  13,  44 },
    {  16,  45 }, {  12,  41 }, {  10,  49 }, {  30,  34 },
    {  18,  42 }, {  10,  55 }, {  17,  51 }, {  17,  46 },
    {   0,  89 }, {  26, -19 }, {  22, -17 }, {  26, -17 },
    {  30, -25 }, {  28, -20 }, {  33, -23 }, {  37, -27 },
    {  33, -23 }, {  40, -28 }, {  38, -17 }, {  33, -11 },
    {  40, -15 }, {  41,  -6 }, {  38,   1 }, {  41,  17 },
    { -17, 120 }, { -20, 112 }, { -18, 114 }, { -11,  85 },
    { -15,  92 }, { -14,  89 }, { -26,  71 }, { -15,  81 },
    { -14,  80 }, {   0,  68 }, { -14,  70 }, { -24,  56 },
    { -23,  68 }, { -24,  50 }, { -11,  74 }, { -14, 106 },
    { -13,  97 }, { -15,  90 }, { -12,  90 }, { -18,  88 },
    { -10,  73 }, {  -9,  79 }, { -14,  86 }, { -10,  73 },
    { -10,  70 }, { -10,  69 }, {  -5,  66 }, {  -9,  64 },
    {  -5,  58 }, {   2,  59 }, {  23, -13 }, {  26, -13 },
    {  40, -15 }, {  49, -14 }, {  44,   3 }, {  45,   6 },
    {  44,  34 }, {  33,  54 }, {  19,  82 }, {  21, -10 },
    {  24, -11 }, {  28,  -8 }, {  28,  -1 }, {  29,   3 },
    {  29,   9 }, {  35,  20 }, {  29,  36 }, {  14,  67 },
    {  -3,  75 }, {  -1,  23 }, {   1,  34 }, {   1,  43 },
    {   0,  54 }, {  -2,  55 }, {   0,  61 }, {   1,  64 },
    {   0,  68 }, {  -9,  92 }, { -17, 120 }, { -20, 112 },
    { -18, 114 }, { -11,  85 }, { -15,  92 }, { -14,  89 },
    { -26,  71 }, { -15,  81 }, { -14,  80 }, {   0,  68 },
    { -14,  70 }, { -24,  56 }, { -23,  68 }, { -24,  50 },
    { -11,  74 }, { -14, 106 }, { -13,  97 }, { -15,  90 },
    { -12,  90 }, { -18,  88 }, { -10,  73 }, {  -9,  79 },
    { -14,  86 }, { -10,  73 }, { -10,  70 }, { -10,  69 },
    {  -5,  66 }, {  -9,  64 }, {  -5,  58 }, {   2,  59 },
    {  23, -13 }, {  26, -13 }, {  40, -15 }, {  49, -14 },
    {  44,   3 }, {  45,   6 }, {  44,  34 }, {  33,  54 },
    {  19,  82 }, {  21, -10 }, {  24, -11 }, {  28,  -8 },
    {  28,  -1 }, {  29,   3 }, {  29,   9 }, {  35,  20 },
    {  29,  36 }, {  14,  67 }, {  -3,  75 }, {  -1,  23 },
    {   1,  34 }, {   1,  43 }, {   0,  54 }, {  -2,  55 },
    {   0,  61 }, {   1,  64 }, {   0,  68 }, {  -9,  92 },
    {  -6,  93 }, {  -6,  84 }, {  -8,  79 }, {   0,  66 },
    {  -1,  71 }, {   0,  62 }, {  -2,  60 }, {  -2,  59 },
    {  -5,  75 }, {  -3,  62 }, {  -4,  58 }, {  -9,  66 },
    {  -1,  79 }, {   0,  71 }, {   3,  68 }, {  10,  44 },
    {  -7,  62 }, {  15,  36 }, {  14,  40 }, {  16,  27 },
    {  12,  29 }, {   1,  44 }, {  20,  36 }, {  18,  32 },
    {   5,  42 }, {   1,  48 }, {  10,  62 }, {  17,  46 },
    {   9,  64 }, { -12, 104 }, { -11,  97 }, { -16,  96 },
    {  -7,  88 }, {  -8,  85 }, {  -7,  85 }, {  -9,  85 },
    { -13,  88 }, {   4,  66 }, {  -3,  77 }, {  -3,  76 },
    {  -6,  76 }, {  10,  58 }, {  -1,  76 }, {  -1,  83 },
    {  -6,  93 }, {  -6,  84 }, {  -8,  79 }, {   0,  66 },
    {  -1,  71 }, {   0,  62 }, {  -2,  60 }, {  -2,  59 },
    {  -5,  75 }, {  -3,  62 }, {  -4,  58 }, {  -9,  66 },
    {  -1,  79 }, {   0,  71 }, {   3,  68 }, {  10,  44 },
    {  -7,  62 }, {  15,  36 }, {  14,  40 }, {  16,  27 },
    {  12,  29 }, {   1,  44 }, {  20,  36 }, {  18,  32 },
    {   5,  42 }, {   1,  48 }, {  10,  62 }, {  17,  46 },
    {   9,  64 }, { -12, 104 }, { -11,  97 }, { -16,  96 },
    {  -7,  88 }, {  -8,  85 }, {  -7,  85 }, {  -9,  85 },
    { -13,  88 }, {   4,  66 }, {  -3,  77 }, {  -3,  76 },
    {  -6,  76 }, {  10,  58 }, {  -1,  76 }, {  -1,  83 },
    {  15,   6 }, {   6,  19 }, {   7,  16 }, {  12,  14 },
    {  18,  13 }, {  13,  11 }, {  13,  15 }, {  15,  16 },
    {  12,  23 }, {  13,  23 }, {  15,  20 }, {  14,  26 },
    {  14,  44 }, {  17,  40 }, {  17,  47 }, {  24,  17 },
    {  21,  21 }, {  25,  22 }, {  31,  27 }, {  22,  29 },
    {  19,  35 }, {  14,  50 }, {  10,  57 }, {   7,  63 },
    {  -2,  77 }, {  -4,  82 }, {  -3,  94 }, {   9,  69 },
    { -12, 109 }, {  36, -35 }, {  36, -34 }, {  32, -26 },
    {  37, -30 }, {  44, -32 }, {  34, -18 }, {  34, -15 },
    {  40, -15 }, {  33,  -7 }, {  35,  -5 }, {  33,   0 },
    {  38,   2 }, {  33,  13 }, {  23,  35 }, {  13,  58 },
    {  15,   6 }, {   6,  19 }, {   7,  16 }, {  12,  14 },
    {  18,  13 }, {  13,  11 }, {  13,  15 }, {  15,  16 },
    {  12,  23 }, {  13,  23 }, {  15,  20 }, {  14,  26 },
    {  14,  44 }, {  17,  40 }, {  17,  47 }, {  24,  17 },
    {  21,  21 }, {  25,  22 }, {  31,  27 }, {  22,  29 },
    {  19,  35 }, {  14,  50 }, {  10,  57 }, {   7,  63 },
    {  -2,  77 }, {  -4,  82 }, {  -3,  94 }, {   9,  69 },
    { -12, 109 }, {  36, -35 }, {  36, -34 }, {  32, -26 },
    {  37, -30 }, {  44, -32 }, {  34, -18 }, {  34, -15 },
    {  40, -15 }, {  33,  -7 }, {  35,  -5 }, {  33,   0 },
    {  38,   2 }, {  33,  13 }, {  23,  35 }, {  13,  58 },
    {  -3,  71 }, {  -6,  42 }, {  -5,  50 }, {  -3,  54 },
    {  -2,  62 }, {   0,  58 }, {   1,  63 }, {  -2,  72 },
    {  -1,  74 }, {  -9,  91 }, {  -5,  67 }, {  -5,  27 },
    {  -3,  39 }, {  -2,  44 }, {   0,  46 }, { -16,  64 },
    {  -8,  68 }, { -10,  78 }, {  -6,  77 }, { -10,  86 },
    { -12,  92 }, { -15,  55 }, { -10,  60 }, {  -6,  62 },
    {  -4,  65 }, { -12,  73 }, {  -8,  76 }, {  -7,  80 },
    {  -9,  88 }, { -17, 110 }, {  -3,  71 }, {  -6,  42 },
    {  -5,  50 }, {  -3,  54 }, {  -2,  62 }, {   0,  58 },
    {   1,  63 }, {  -2,  72 }, {  -1,  74 }, {  -9,  91 },
    {  -5,  67 }, {  -5,  27 }, {  -3,  39 }, {  -2,  44 },
    {   0,  46 }, { -16,  64 }, {  -8,  68 }, { -10,  78 },
    {  -6,  77 }, { -10,  86 }, { -12,  92 }, { -15,  55 },
    { -10,  60 }, {  -6,  62 }, {  -4,  65 }, { -12,  73 },
    {  -8,  76 }, {  -7,  80 }, {  -9,  88 }, { -17, 110 },
    {  -3,  70 }, {  -8,  93 }, { -10,  90 }, { -30, 127 },
    {  -3,  70 }, {  -8,  93 }, { -10,  90 }, { -30, 127 },
    {  -3,  70 }, {  -8,  93 }, { -10,  90 }, { -30, 127 }
};

const int8_t x264_cabac_context_init_PB[3][1024][2] =
{
    /* i_cabac_init_idc == 0 */
    {
        /* 0 - 10 */
        {  20, -15 }, {   2,  54 }, {   3,  74 }, {  20, -15 },
        {   2,  54 }, {   3,  74 }, { -28, 127 }, { -23, 104 },
        {  -6,  53 }, {  -1,  54 }, {   7,  51 },

        /* 11 - 23 */
        {  23,  33 }, {  23,   2 }, {  21,   0 }, {   1,   9 },
        {   0,  49 }, { -37, 118 }, {   5,  57 }, { -13,  78 },
        { -11,  65 }, {   1,  62 }, {  12,  49 }, {  -4,  73 },
        {  17,  50 },

        /* 24 - 39 */
        {  18,  64 }, {   9,  43 }, {  29,   0 }, {  26,  67 },
        {  16,  90 }, {   9, 104 }, { -46, 127 }, { -20, 104 },
        {   1,  67 }, { -13,  78 }, { -11,  65 }, {   1,  62 },
        {  -6,  86 }, { -17,  95 }, {  -6,  61 }, {   9,  45 },

        /* 40 - 53 */
        {  -3,  69 }, {  -6,  81 }, { -11,  96 }, {   6,  55 },
        {   7,  67 }, {  -5,  86 }, {   2,  88 }, {   0,  58 },
        {  -3,  76 }, { -10,  94 }, {   5,  54 }, {   4,  69 },
        {  -3,  81 }, {   0,  88 },

        /* 54 - 59 */
        {  -7,  67 }, {  -5,  74 }, {  -4,  74 }, {  -5,  80 },
        {  -7,  72 }, {   1,  58 },

        /* 60 - 69 */
        {   0,  41 }, {   0,  63 }, {   0,  63 }, { 0, 63 },
        {  -9,  83 }, {   4,  86 }, {   0,  97 }, { -7, 72 },
        {  13,  41 }, {   3,  62 },

        /* 70 - 87 */
        {   0,  45 }, {  -4,  78 }, {  -3,  96 }, { -27,  126 },
        { -28,  98 }, { -25, 101 }, { -23,  67 }, { -28,  82 },
        { -20,  94 }, { -16,  83 }, { -22, 110 }, { -21,  91 },
        { -18, 102 }, { -13,  93 }, { -29, 127 }, {  -7,  92 },
        {  -5,  89 }, {  -7,  96 }, { -13, 108 }, {  -3,  46 },
        {  -1,  65 }, {  -1,  57 }, {  -9,  93 }, {  -3,  74 },
        {  -9,  92 }, {  -8,  87 }, { -23, 126 }, {   5,  54 },
        {   6,  60 }, {   6,  59 }, {   6,  69 }, {  -1,  48 },
        {   0,  68 }, {  -4,  69 }, {  -8,  88 },

        /* 105 -> 165 */
        {  -2,  85 }, {  -6,  78 }, {  -1,  75 }, {  -7,  77 },
        {   2,  54 }, {   5,  50 }, {  -3,  68 }, {   1,  50 },
        {   6,  42 }, {  -4,  81 }, {   1,  63 }, {  -4,  70 },
        {   0,  67 }, {   2,  57 }, {  -2,  76 }, {  11,  35 },
        {   4,  64 }, {   1,  61 }, {  11,  35 }, {  18,  25 },
        {  12,  24 }, {  13,  29 }, {  13,  36 }, { -10,  93 },
        {  -7,  73 }, {  -2,  73 }, {  13,  46 }, {   9,  49 },
        {  -7, 100 }, {   9,  53 }, {   2,  53 }, {   5,  53 },
        {  -2,  61 }, {   0,  56 }, {   0,  56 }, { -13,  63 },
        {  -5,  60 }, {  -1,  62 }, {   4,  57 }, {  -6,  69 },
        {   4,  57 }, {  14,  39 }, {   4,  51 }, {  13,  68 },
        {   3,  64 }, {   1,  61 }, {   9,  63 }, {   7,  50 },
        {  16,  39 }, {   5,  44 }, {   4,  52 }, {  11,  48 },
        {  -5,  60 }, {  -1,  59 }, {   0,  59 }, {  22,  33 },
        {   5,  44 }, {  14,  43 }, {  -1,  78 }, {   0,  60 },
        {   9,  69 },

        /* 166 - 226 */
        {  11,  28 }, {   2,  40 }, {   3,  44 }, {   0,  49 },
        {   0,  46 }, {   2,  44 }, {   2,  51 }, {   0,  47 },
        {   4,  39 }, {   2,  62 }, {   6,  46 }, {   0,  54 },
        {   3,  54 }, {   2,  58 }, {   4,  63 }, {   6,  51 },
        {   6,  57 }, {   7,  53 }, {   6,  52 }, {   6,  55 },
        {  11,  45 }, {  14,  36 }, {   8,  53 }, {  -1,  82 },
        {   7,  55 }, {  -3,  78 }, {  15,  46 }, {  22,  31 },
        {  -1,  84 }, {  25,   7 }, {  30,  -7 }, {  28,   3 },
        {  28,   4 }, {  32,   0 }, {  34,  -1 }, {  30,   6 },
        {  30,   6 }, {  32,   9 }, {  31,  19 }, {  26,  27 },
        {  26,  30 }, {  37,  20 }, {  28,  34 }, {  17,  70 },
        {   1,  67 }, {   5,  59 }, {   9,  67 }, {  16,  30 },
        {  18,  32 }, {  18,  35 }, {  22,  29 }, {  24,  31 },
        {  23,  38 }, {  18,  43 }, {  20,  41 }, {  11,  63 },
        {   9,  59 }, {   9,  64 }, {  -1,  94 }, {  -2,  89 },
        {  -9, 108 },

        /* 227 - 275 */
        {  -6,  76 }, {  -2,  44 }, {   0,  45 }, {   0,  52 },
        {  -3,  64 }, {  -2,  59 }, {  -4,  70 }, {  -4,  75 },
        {  -8,  82 }, { -17, 102 }, {  -9,  77 }, {   3,  24 },
        {   0,  42 }, {   0,  48 }, {   0,  55 }, {  -6,  59 },
        {  -7,  71 }, { -12,  83 }, { -11,  87 }, { -30, 119 },
        {   1,  58 }, {  -3,  29 }, {  -1,  36 }, {   1,  38 },
        {   2,  43 }, {  -6,  55 }, {   0,  58 }, {   0,  64 },
        {  -3,  74 }, { -10,  90 }, {   0,  70 }, {  -4,  29 },
        {   5,  31 }, {   7,  42 }, {   1,  59 }, {  -2,  58 },
        {  -3,  72 }, {  -3,  81 }, { -11,  97 }, {   0,  58 },
        {   8,   5 }, {  10,  14 }, {  14,  18 }, {  13,  27 },
        {   2,  40 }, {   0,  58 }, {  -3,  70 }, {  -6,  79 },
        {  -8,  85 },

        /* 276 a bit special (not used, x264_cabac_encode_bypass is used instead) */
        { 0, 0 },

        /* 277 - 337 */
        { -13, 106 }, { -16, 106 }, { -10,  87 }, { -21, 114 },
        { -18, 110 }, { -14,  98 }, { -22, 110 }, { -21, 106 },
        { -18, 103 }, { -21, 107 }, { -23, 108 }, { -26, 112 },
        { -10,  96 }, { -12,  95 }, {  -5,  91 }, {  -9,  93 },
        { -22,  94 }, {  -5,  86 }, {   9,  67 }, {  -4,  80 },
        { -10,  85 }, {  -1,  70 }, {   7,  60 }, {   9,  58 },
        {   5,  61 }, {  12,  50 }, {  15,  50 }, {  18,  49 },
        {  17,  54 }, {  10,  41 }, {   7,  46 }, {  -1,  51 },
        {   7,  49 }, {   8,  52 }, {   9,  41 }, {   6,  47 },
        {   2,  55 }, {  13,  41 }, {  10,  44 }, {   6,  50 },
        {   5,  53 }, {  13,  49 }, {   4,  63 }, {   6,  64 },
        {  -2,  69 }, {  -2,  59 }, {   6,  70 }, {  10,  44 },
        {   9,  31 }, {  12,  43 }, {   3,  53 }, {  14,  34 },
        {  10,  38 }, {  -3,  52 }, {  13,  40 }, {  17,  32 },
        {   7,  44 }, {   7,  38 }, {  13,  50 }, {  10,  57 },
        {  26,  43 },

        /* 338 - 398 */
        {  14,  11 }, {  11,  14 }, {   9,  11 }, {  18,  11 },
        {  21,   9 }, {  23,  -2 }, {  32, -15 }, {  32, -15 },
        {  34, -21 }, {  39, -23 }, {  42, -33 }, {  41, -31 },
        {  46, -28 }, {  38, -12 }, {  21,  29 }, {  45, -24 },
        {  53, -45 }, {  48, -26 }, {  65, -43 }, {  43, -19 },
        {  39, -10 }, {  30,   9 }, {  18,  26 }, {  20,  27 },
        {   0,  57 }, { -14,  82 }, {  -5,  75 }, { -19,  97 },
        { -35, 125 }, {  27,   0 }, {  28,   0 }, {  31,  -4 },
        {  27,   6 }, {  34,   8 }, {  30,  10 }, {  24,  22 },
        {  33,  19 }, {  22,  32 }, {  26,  31 }, {  21,  41 },
        {  26,  44 }, {  23,  47 }, {  16,  65 }, {  14,  71 },
        {   8,  60 }, {   6,  63 }, {  17,  65 }, {  21,  24 },
        {  23,  20 }, {  26,  23 }, {  27,  32 }, {  28,  23 },
        {  28,  24 }, {  23,  40 }, {  24,  32 }, {  28,  29 },
        {  23,  42 }, {  19,  57 }, {  22,  53 }, {  22,  61 },
        {  11,  86 },

        /* 399 -> 435 */
        {  12,  40 }, {  11,  51 }, {  14,  59 },
        {  -4,  79 }, {  -7,  71 }, {  -5,  69 }, {  -9,  70 },
        {  -8,  66 }, { -10,  68 }, { -19,  73 }, { -12,  69 },
        { -16,  70 }, { -15,  67 }, { -20,  62 }, { -19,  70 },
        { -16,  66 }, { -22,  65 }, { -20,  63 }, {   9,  -2 },
        {  26,  -9 }, {  33,  -9 }, {  39,  -7 }, {  41,  -2 },
        {  45,   3 }, {  49,   9 }, {  45,  27 }, {  36,  59 },
        {  -6,  66 }, {  -7,  35 }, {  -7,  42 }, {  -8,  45 },
        {  -5,  48 }, { -12,  56 }, {  -6,  60 }, {  -5,  62 },
        {  -8,  66 }, {  -8,  76 },

        /* 436 -> 459 */
        {  -5,  85 }, {  -6,  81 }, { -10,  77 }, {  -7,  81 },
        { -17,  80 }, { -18,  73 }, {  -4,  74 }, { -10,  83 },
        {  -9,  71 }, {  -9,  67 }, {  -1,  61 }, {  -8,  66 },
        { -14,  66 }, {   0,  59 }, {   2,  59 }, {  21, -13 },
        {  33, -14 }, {  39,  -7 }, {  46,  -2 }, {  51,   2 },
        {  60,   6 }, {  61,  17 }, {  55,  34 }, {  42,  62 },

        /* 460 - 1024 */
        {  -7,  92 }, {  -5,  89 }, {  -7,  96 }, { -13, 108 },
        {  -3,  46 }, {  -1,  65 }, {  -1,  57 }, {  -9,  93 },
        {  -3,  74 }, {  -9,  92 }, {  -8,  87 }, { -23, 126 },
        {  -7,  92 }, {  -5,  89 }, {  -7,  96 }, { -13, 108 },
        {  -3,  46 }, {  -1,  65 }, {  -1,  57 }, {  -9,  93 },
        {  -3,  74 }, {  -9,  92 }, {  -8,  87 }, { -23, 126 },
        {  -2,  85 }, {  -6,  78 }, {  -1,  75 }, {  -7,  77 },
        {   2,  54 }, {   5,  50 }, {  -3,  68 }, {   1,  50 },
        {   6,  42 }, {  -4,  81 }, {   1,  63 }, {  -4,  70 },
        {   0,  67 }, {   2,  57 }, {  -2,  76 }, {  11,  35 },
        {   4,  64 }, {   1,  61 }, {  11,  35 }, {  18,  25 },
        {  12,  24 }, {  13,  29 }, {  13,  36 }, { -10,  93 },
        {  -7,  73 }, {  -2,  73 }, {  13,  46 }, {   9,  49 },
        {  -7, 100 }, {   9,  53 }, {   2,  53 }, {   5,  53 },
        {  -2,  61 }, {   0,  56 }, {   0,  56 }, { -13,  63 },
        {  -5,  60 }, {  -1,  62 }, {   4,  57 }, {  -6,  69 },
        {   4,  57 }, {  14,  39 }, {   4,  51 }, {  13,  68 },
        {  -2,  85 }, {  -6,  78 }, {  -1,  75 }, {  -7,  77 },
        {   2,  54 }, {   5,  50 }, {  -3,  68 }, {   1,  50 },
        {   6,  42 }, {  -4,  81 }, {   1,  63 }, {  -4,  70 },
        {   0,  67 }, {   2,  57 }, {  -2,  76 }, {  11,  35 },
        {   4,  64 }, {   1,  61 }, {  11,  35 }, {  18,  25 },
        {  12,  24 }, {  13,  29 }, {  13,  36 }, { -10,  93 },
        {  -7,  73 }, {  -2,  73 }, {  13,  46 }, {   9,  49 },
        {  -7, 100 }, {   9,  53 }, {   2,  53 }, {   5,  53 },
        {  -2,  61 }, {   0,  56 }, {   0,  56 }, { -13,  63 },
        {  -5,  60 }, {  -1,  62 }, {   4,  57 }, {  -6,  69 },
        {   4,  57 }, {  14,  39 }, {   4,  51 }, {  13,  68 },
        {  11,  28 }, {   2,  40 }, {   3,  44 }, {   0,  49 },
        {   0,  46 }, {   2,  44 }, {   2,  51 }, {   0,  47 },
        {   4,  39 }, {   2,  62 }, {   6,  46 }, {   0,  54 },
        {   3,  54 }, {   2,  58 }, {   4,  63 }, {   6,  51 },
        {   6,  57 }, {   7,  53 }, {   6,  52 }, {   6,  55 },
        {  11,  45 }, {  14,  36 }, {   8,  53 }, {  -1,  82 },
        {   7,  55 }, {  -3,  78 }, {  15,  46 }, {  22,  31 },
        {  -1,  84 }, {  25,   7 }, {  30,  -7 }, {  28,   3 },
        {  28,   4 }, {  32,   0 }, {  34,  -1 }, {  30,   6 },
        {  30,   6 }, {  32,   9 }, {  31,  19 }, {  26,  27 },
        {  26,  30 }, {  37,  20 }, {  28,  34 }, {  17,  70 },
        {  11,  28 }, {   2,  40 }, {   3,  44 }, {   0,  49 },
        {   0,  46 }, {   2,  44 }, {   2,  51 }, {   0,  47 },
        {   4,  39 }, {   2,  62 }, {   6,  46 }, {   0,  54 },
        {   3,  54 }, {   2,  58 }, {   4,  63 }, {   6,  51 },
        {   6,  57 }, {   7,  53 }, {   6,  52 }, {   6,  55 },
        {  11,  45 }, {  14,  36 }, {   8,  53 }, {  -1,  82 },
        {   7,  55 }, {  -3,  78 }, {  15,  46 }, {  22,  31 },
        {  -1,  84 }, {  25,   7 }, {  30,  -7 }, {  28,   3 },
        {  28,   4 }, {  32,   0 }, {  34,  -1 }, {  30,   6 },
        {  30,   6 }, {  32,   9 }, {  31,  19 }, {  26,  27 },
        {  26,  30 }, {  37,  20 }, {  28,  34 }, {  17,  70 },
        {  -4,  79 }, {  -7,  71 }, {  -5,  69 }, {  -9,  70 },
        {  -8,  66 }, { -10,  68 }, { -19,  73 }, { -12,  69 },
        { -16,  70 }, { -15,  67 }, { -20,  62 }, { -19,  70 },
        { -16,  66 }, { -22,  65 }, { -20,  63 }, {  -5,  85 },
        {  -6,  81 }, { -10,  77 }, {  -7,  81 }, { -17,  80 },
        { -18,  73 }, {  -4,  74 }, { -10,  83 }, {  -9,  71 },
        {  -9,  67 }, {  -1,  61 }, {  -8,  66 }, { -14,  66 },
        {   0,  59 }, {   2,  59 }, {   9,  -2 }, {  26,  -9 },
        {  33,  -9 }, {  39,  -7 }, {  41,  -2 }, {  45,   3 },
        {  49,   9 }, {  45,  27 }, {  36,  59 }, {  21, -13 },
        {  33, -14 }, {  39,  -7 }, {  46,  -2 }, {  51,   2 },
        {  60,   6 }, {  61,  17 }, {  55,  34 }, {  42,  62 },
        {  -6,  66 }, {  -7,  35 }, {  -7,  42 }, {  -8,  45 },
        {  -5,  48 }, { -12,  56 }, {  -6,  60 }, {  -5,  62 },
        {  -8,  66 }, {  -8,  76 }, {  -4,  79 }, {  -7,  71 },
        {  -5,  69 }, {  -9,  70 }, {  -8,  66 }, { -10,  68 },
        { -19,  73 }, { -12,  69 }, { -16,  70 }, { -15,  67 },
        { -20,  62 }, { -19,  70 }, { -16,  66 }, { -22,  65 },
        { -20,  63 }, {  -5,  85 }, {  -6,  81 }, { -10,  77 },
        {  -7,  81 }, { -17,  80 }, { -18,  73 }, {  -4,  74 },
        { -10,  83 }, {  -9,  71 }, {  -9,  67 }, {  -1,  61 },
        {  -8,  66 }, { -14,  66 }, {   0,  59 }, {   2,  59 },
        {   9,  -2 }, {  26,  -9 }, {  33,  -9 }, {  39,  -7 },
        {  41,  -2 }, {  45,   3 }, {  49,   9 }, {  45,  27 },
        {  36,  59 }, {  21, -13 }, {  33, -14 }, {  39,  -7 },
        {  46,  -2 }, {  51,   2 }, {  60,   6 }, {  61,  17 },
        {  55,  34 }, {  42,  62 }, {  -6,  66 }, {  -7,  35 },
        {  -7,  42 }, {  -8,  45 }, {  -5,  48 }, { -12,  56 },
        {  -6,  60 }, {  -5,  62 }, {  -8,  66 }, {  -8,  76 },
        { -13, 106 }, { -16, 106 }, { -10,  87 }, { -21, 114 },
        { -18, 110 }, { -14,  98 }, { -22, 110 }, { -21, 106 },
        { -18, 103 }, { -21, 107 }, { -23, 108 }, { -26, 112 },
        { -10,  96 }, { -12,  95 }, {  -5,  91 }, {  -9,  93 },
        { -22,  94 }, {  -5,  86 }, {   9,  67 }, {  -4,  80 },
        { -10,  85 }, {  -1,  70 }, {   7,  60 }, {   9,  58 },
        {   5,  61 }, {  12,  50 }, {  15,  50 }, {  18,  49 },
        {  17,  54 }, {  10,  41 }, {   7,  46 }, {  -1,  51 },
        {   7,  49 }, {   8,  52 }, {   9,  41 }, {   6,  47 },
        {   2,  55 }, {  13,  41 }, {  10,  44 }, {   6,  50 },
        {   5,  53 }, {  13,  49 }, {   4,  63 }, {   6,  64 },
        { -13, 106 }, { -16, 106 }, { -10,  87 }, { -21, 114 },
        { -18, 110 }, { -14,  98 }, { -22, 110 }, { -21, 106 },
        { -18, 103 }, { -21, 107 }, { -23, 108 }, { -26, 112 },
        { -10,  96 }, { -12,  95 }, {  -5,  91 }, {  -9,  93 },
        { -22,  94 }, {  -5,  86 }, {   9,  67 }, {  -4,  80 },
        { -10,  85 }, {  -1,  70 }, {   7,  60 }, {   9,  58 },
        {   5,  61 }, {  12,  50 }, {  15,  50 }, {  18,  49 },
        {  17,  54 }, {  10,  41 }, {   7,  46 }, {  -1,  51 },
        {   7,  49 }, {   8,  52 }, {   9,  41 }, {   6,  47 },
        {   2,  55 }, {  13,  41 }, {  10,  44 }, {   6,  50 },
        {   5,  53 }, {  13,  49 }, {   4,  63 }, {   6,  64 },
        {  14,  11 }, {  11,  14 }, {   9,  11 }, {  18,  11 },
        {  21,   9 }, {  23,  -2 }, {  32, -15 }, {  32, -15 },
        {  34, -21 }, {  39, -23 }, {  42, -33 }, {  41, -31 },
        {  46, -28 }, {  38, -12 }, {  21,  29 }, {  45, -24 },
        {  53, -45 }, {  48, -26 }, {  65, -43 }, {  43, -19 },
        {  39, -10 }, {  30,   9 }, {  18,  26 }, {  20,  27 },
        {   0,  57 }, { -14,  82 }, {  -5,  75 }, { -19,  97 },
        { -35, 125 }, {  27,   0 }, {  28,   0 }, {  31,  -4 },
        {  27,   6 }, {  34,   8 }, {  30,  10 }, {  24,  22 },
        {  33,  19 }, {  22,  32 }, {  26,  31 }, {  21,  41 },
        {  26,  44 }, {  23,  47 }, {  16,  65 }, {  14,  71 },
        {  14,  11 }, {  11,  14 }, {   9,  11 }, {  18,  11 },
        {  21,   9 }, {  23,  -2 }, {  32, -15 }, {  32, -15 },
        {  34, -21 }, {  39, -23 }, {  42, -33 }, {  41, -31 },
        {  46, -28 }, {  38, -12 }, {  21,  29 }, {  45, -24 },
        {  53, -45 }, {  48, -26 }, {  65, -43 }, {  43, -19 },
        {  39, -10 }, {  30,   9 }, {  18,  26 }, {  20,  27 },
        {   0,  57 }, { -14,  82 }, {  -5,  75 }, { -19,  97 },
        { -35, 125 }, {  27,   0 }, {  28,   0 }, {  31,  -4 },
        {  27,   6 }, {  34,   8 }, {  30,  10 }, {  24,  22 },
        {  33,  19 }, {  22,  32 }, {  26,  31 }, {  21,  41 },
        {  26,  44 }, {  23,  47 }, {  16,  65 }, {  14,  71 },
        {  -6,  76 }, {  -2,  44 }, {   0,  45 }, {   0,  52 },
        {  -3,  64 }, {  -2,  59 }, {  -4,  70 }, {  -4,  75 },
        {  -8,  82 }, { -17, 102 }, {  -9,  77 }, {   3,  24 },
        {   0,  42 }, {   0,  48 }, {   0,  55 }, {  -6,  59 },
        {  -7,  71 }, { -12,  83 }, { -11,  87 }, { -30, 119 },
        {   1,  58 }, {  -3,  29 }, {  -1,  36 }, {   1,  38 },
        {   2,  43 }, {  -6,  55 }, {   0,  58 }, {   0,  64 },
        {  -3,  74 }, { -10,  90 }, {  -6,  76 }, {  -2,  44 },
        {   0,  45 }, {   0,  52 }, {  -3,  64 }, {  -2,  59 },
        {  -4,  70 }, {  -4,  75 }, {  -8,  82 }, { -17, 102 },
        {  -9,  77 }, {   3,  24 }, {   0,  42 }, {   0,  48 },
        {   0,  55 }, {  -6,  59 }, {  -7,  71 }, { -12,  83 },
        { -11,  87 }, { -30, 119 }, {   1,  58 }, {  -3,  29 },
        {  -1,  36 }, {   1,  38 }, {   2,  43 }, {  -6,  55 },
        {   0,  58 }, {   0,  64 }, {  -3,  74 }, { -10,  90 },
        {  -3,  74 }, {  -9,  92 }, {  -8,  87 }, { -23, 126 },
        {  -3,  74 }, {  -9,  92 }, {  -8,  87 }, { -23, 126 },
        {  -3,  74 }, {  -9,  92 }, {  -8,  87 }, { -23, 126 }
    },

    /* i_cabac_init_idc == 1 */
    {
        /* 0 - 10 */
        {  20, -15 }, {   2,  54 }, {   3,  74 }, {  20, -15 },
        {   2,  54 }, {   3,  74 }, { -28, 127 }, { -23, 104 },
        {  -6,  53 }, {  -1,  54 }, {   7,  51 },

        /* 11 - 23 */
        {  22,  25 }, {  34,   0 }, {  16,   0 }, {  -2,   9 },
        {   4,  41 }, { -29, 118 }, {   2,  65 }, {  -6,  71 },
        { -13,  79 }, {   5,  52 }, {   9,  50 }, {  -3,  70 },
        {  10,  54 },

        /* 24 - 39 */
        {  26,  34 }, {  19,  22 }, {  40,   0 }, {  57,   2 },
        {  41,  36 }, {  26,  69 }, { -45, 127 }, { -15, 101 },
        {  -4,  76 }, {  -6,  71 }, { -13,  79 }, {   5,  52 },
        {   6,  69 }, { -13,  90 }, {   0,  52 }, {   8,  43 },

        /* 40 - 53 */
        {  -2,  69 },{  -5,  82 },{ -10,  96 },{   2,  59 },
        {   2,  75 },{  -3,  87 },{  -3,  100 },{   1,  56 },
        {  -3,  74 },{  -6,  85 },{   0,  59 },{  -3,  81 },
        {  -7,  86 },{  -5,  95 },

        /* 54 - 59 */
        {  -1,  66 },{  -1,  77 },{   1,  70 },{  -2,  86 },
        {  -5,  72 },{   0,  61 },

        /* 60 - 69 */
        { 0, 41 },   { 0, 63 },   { 0, 63 },     { 0, 63 },
        { -9, 83 },  { 4, 86 },   { 0, 97 },     { -7, 72 },
        { 13, 41 },  { 3, 62 },

        /* 70 - 104 */
        {  13,  15 }, {   7,  51 }, {   2,  80 }, { -39, 127 },
        { -18,  91 }, { -17,  96 }, { -26,  81 }, { -35,  98 },
        { -24, 102 }, { -23,  97 }, { -27, 119 }, { -24,  99 },
        { -21, 110 }, { -18, 102 }, { -36, 127 }, {   0,  80 },
        {  -5,  89 }, {  -7,  94 }, {  -4,  92 }, {   0,  39 },
        {   0,  65 }, { -15,  84 }, { -35, 127 }, {  -2,  73 },
        { -12, 104 }, {  -9,  91 }, { -31, 127 }, {   3,  55 },
        {   7,  56 }, {   7,  55 }, {   8,  61 }, {  -3,  53 },
        {   0,  68 }, {  -7,  74 }, {  -9,  88 },

        /* 105 -> 165 */
        { -13, 103 }, { -13,  91 }, {  -9,  89 }, { -14,  92 },
        {  -8,  76 }, { -12,  87 }, { -23, 110 }, { -24, 105 },
        { -10,  78 }, { -20, 112 }, { -17,  99 }, { -78, 127 },
        { -70, 127 }, { -50, 127 }, { -46, 127 }, {  -4,  66 },
        {  -5,  78 }, {  -4,  71 }, {  -8,  72 }, {   2,  59 },
        {  -1,  55 }, {  -7,  70 }, {  -6,  75 }, {  -8,  89 },
        { -34, 119 }, {  -3,  75 }, {  32,  20 }, {  30,  22 },
        { -44, 127 }, {   0,  54 }, {  -5,  61 }, {   0,  58 },
        {  -1,  60 }, {  -3,  61 }, {  -8,  67 }, { -25,  84 },
        { -14,  74 }, {  -5,  65 }, {   5,  52 }, {   2,  57 },
        {   0,  61 }, {  -9,  69 }, { -11,  70 }, {  18,  55 },
        {  -4,  71 }, {   0,  58 }, {   7,  61 }, {   9,  41 },
        {  18,  25 }, {   9,  32 }, {   5,  43 }, {   9,  47 },
        {   0,  44 }, {   0,  51 }, {   2,  46 }, {  19,  38 },
        {  -4,  66 }, {  15,  38 }, {  12,  42 }, {   9,  34 },
        {   0,  89 },

        /* 166 - 226 */
        {   4,  45 }, {  10,  28 }, {  10,  31 }, {  33, -11 },
        {  52, -43 }, {  18,  15 }, {  28,   0 }, {  35, -22 },
        {  38, -25 }, {  34,   0 }, {  39, -18 }, {  32, -12 },
        { 102, -94 }, {   0,   0 }, {  56, -15 }, {  33,  -4 },
        {  29,  10 }, {  37,  -5 }, {  51, -29 }, {  39,  -9 },
        {  52, -34 }, {  69, -58 }, {  67, -63 }, {  44,  -5 },
        {  32,   7 }, {  55, -29 }, {  32,   1 }, {   0,   0 },
        {  27,  36 }, {  33, -25 }, {  34, -30 }, {  36, -28 },
        {  38, -28 }, {  38, -27 }, {  34, -18 }, {  35, -16 },
        {  34, -14 }, {  32,  -8 }, {  37,  -6 }, {  35,   0 },
        {  30,  10 }, {  28,  18 }, {  26,  25 }, {  29,  41 },
        {   0,  75 }, {   2,  72 }, {   8,  77 }, {  14,  35 },
        {  18,  31 }, {  17,  35 }, {  21,  30 }, {  17,  45 },
        {  20,  42 }, {  18,  45 }, {  27,  26 }, {  16,  54 },
        {   7,  66 }, {  16,  56 }, {  11,  73 }, {  10,  67 },
        { -10, 116 },

        /* 227 - 275 */
        { -23, 112 }, { -15,  71 }, {  -7,  61 }, {   0,  53 },
        {  -5,  66 }, { -11,  77 }, {  -9,  80 }, {  -9,  84 },
        { -10,  87 }, { -34, 127 }, { -21, 101 }, {  -3,  39 },
        {  -5,  53 }, {  -7,  61 }, { -11,  75 }, { -15,  77 },
        { -17,  91 }, { -25, 107 }, { -25, 111 }, { -28, 122 },
        { -11,  76 }, { -10,  44 }, { -10,  52 }, { -10,  57 },
        {  -9,  58 }, { -16,  72 }, {  -7,  69 }, {  -4,  69 },
        {  -5,  74 }, {  -9,  86 }, {   2,  66 }, {  -9,  34 },
        {   1,  32 }, {  11,  31 }, {   5,  52 }, {  -2,  55 },
        {  -2,  67 }, {   0,  73 }, {  -8,  89 }, {   3,  52 },
        {   7,   4 }, {  10,   8 }, {  17,   8 }, {  16,  19 },
        {   3,  37 }, {  -1,  61 }, {  -5,  73 }, {  -1,  70 },
        {  -4,  78 },

        /* 276 a bit special (not used, x264_cabac_encode_bypass is used instead) */
        { 0, 0 },

        /* 277 - 337 */
        { -21, 126 }, { -23, 124 }, { -20, 110 }, { -26, 126 },
        { -25, 124 }, { -17, 105 }, { -27, 121 }, { -27, 117 },
        { -17, 102 }, { -26, 117 }, { -27, 116 }, { -33, 122 },
        { -10,  95 }, { -14, 100 }, {  -8,  95 }, { -17, 111 },
        { -28, 114 }, {  -6,  89 }, {  -2,  80 }, {  -4,  82 },
        {  -9,  85 }, {  -8,  81 }, {  -1,  72 }, {   5,  64 },
        {   1,  67 }, {   9,  56 }, {   0,  69 }, {   1,  69 },
        {   7,  69 }, {  -7,  69 }, {  -6,  67 }, { -16,  77 },
        {  -2,  64 }, {   2,  61 }, {  -6,  67 }, {  -3,  64 },
        {   2,  57 }, {  -3,  65 }, {  -3,  66 }, {   0,  62 },
        {   9,  51 }, {  -1,  66 }, {  -2,  71 }, {  -2,  75 },
        {  -1,  70 }, {  -9,  72 }, {  14,  60 }, {  16,  37 },
        {   0,  47 }, {  18,  35 }, {  11,  37 }, {  12,  41 },
        {  10,  41 }, {   2,  48 }, {  12,  41 }, {  13,  41 },
        {   0,  59 }, {   3,  50 }, {  19,  40 }, {   3,  66 },
        {  18,  50 },

        /* 338 - 398 */
        {  19,  -6 }, {  18,  -6 }, {  14,   0 }, {  26, -12 },
        {  31, -16 }, {  33, -25 }, {  33, -22 }, {  37, -28 },
        {  39, -30 }, {  42, -30 }, {  47, -42 }, {  45, -36 },
        {  49, -34 }, {  41, -17 }, {  32,   9 }, {  69, -71 },
        {  63, -63 }, {  66, -64 }, {  77, -74 }, {  54, -39 },
        {  52, -35 }, {  41, -10 }, {  36,   0 }, {  40,  -1 },
        {  30,  14 }, {  28,  26 }, {  23,  37 }, {  12,  55 },
        {  11,  65 }, {  37, -33 }, {  39, -36 }, {  40, -37 },
        {  38, -30 }, {  46, -33 }, {  42, -30 }, {  40, -24 },
        {  49, -29 }, {  38, -12 }, {  40, -10 }, {  38,  -3 },
        {  46,  -5 }, {  31,  20 }, {  29,  30 }, {  25,  44 },
        {  12,  48 }, {  11,  49 }, {  26,  45 }, {  22,  22 },
        {  23,  22 }, {  27,  21 }, {  33,  20 }, {  26,  28 },
        {  30,  24 }, {  27,  34 }, {  18,  42 }, {  25,  39 },
        {  18,  50 }, {  12,  70 }, {  21,  54 }, {  14,  71 },
        {  11,  83 },

        /* 399 -> 435 */
        {  25,  32 }, {  21,  49 }, {  21,  54 },
        {  -5,  85 }, {  -6,  81 }, { -10,  77 }, {  -7,  81 },
        { -17,  80 }, { -18,  73 }, {  -4,  74 }, { -10,  83 },
        {  -9,  71 }, {  -9,  67 }, {  -1,  61 }, {  -8,  66 },
        { -14,  66 }, {   0,  59 }, {   2,  59 }, {  17, -10 },
        {  32, -13 }, {  42,  -9 }, {  49,  -5 }, {  53,   0 },
        {  64,   3 }, {  68,  10 }, {  66,  27 }, {  47,  57 },
        {  -5,  71 }, {   0,  24 }, {  -1,  36 }, {  -2,  42 },
        {  -2,  52 }, {  -9,  57 }, {  -6,  63 }, {  -4,  65 },
        {  -4,  67 }, {  -7,  82 },

        /* 436 -> 459 */
        {  -3,  81 }, {  -3,  76 }, {  -7,  72 }, {  -6,  78 },
        { -12,  72 }, { -14,  68 }, {  -3,  70 }, {  -6,  76 },
        {  -5,  66 }, {  -5,  62 }, {   0,  57 }, {  -4,  61 },
        {  -9,  60 }, {   1,  54 }, {   2,  58 }, {  17, -10 },
        {  32, -13 }, {  42,  -9 }, {  49,  -5 }, {  53,   0 },
        {  64,   3 }, {  68,  10 }, {  66,  27 }, {  47,  57 },

        /* 460 - 1024 */
        {   0,  80 }, {  -5,  89 }, {  -7,  94 }, {  -4,  92 },
        {   0,  39 }, {   0,  65 }, { -15,  84 }, { -35, 127 },
        {  -2,  73 }, { -12, 104 }, {  -9,  91 }, { -31, 127 },
        {   0,  80 }, {  -5,  89 }, {  -7,  94 }, {  -4,  92 },
        {   0,  39 }, {   0,  65 }, { -15,  84 }, { -35, 127 },
        {  -2,  73 }, { -12, 104 }, {  -9,  91 }, { -31, 127 },
        { -13, 103 }, { -13,  91 }, {  -9,  89 }, { -14,  92 },
        {  -8,  76 }, { -12,  87 }, { -23, 110 }, { -24, 105 },
        { -10,  78 }, { -20, 112 }, { -17,  99 }, { -78, 127 },
        { -70, 127 }, { -50, 127 }, { -46, 127 }, {  -4,  66 },
        {  -5,  78 }, {  -4,  71 }, {  -8,  72 }, {   2,  59 },
        {  -1,  55 }, {  -7,  70 }, {  -6,  75 }, {  -8,  89 },
        { -34, 119 }, {  -3,  75 }, {  32,  20 }, {  30,  22 },
        { -44, 127 }, {   0,  54 }, {  -5,  61 }, {   0,  58 },
        {  -1,  60 }, {  -3,  61 }, {  -8,  67 }, { -25,  84 },
        { -14,  74 }, {  -5,  65 }, {   5,  52 }, {   2,  57 },
        {   0,  61 }, {  -9,  69 }, { -11,  70 }, {  18,  55 },
        { -13, 103 }, { -13,  91 }, {  -9,  89 }, { -14,  92 },
        {  -8,  76 }, { -12,  87 }, { -23, 110 }, { -24, 105 },
        { -10,  78 }, { -20, 112 }, { -17,  99 }, { -78, 127 },
        { -70, 127 }, { -50, 127 }, { -46, 127 }, {  -4,  66 },
        {  -5,  78 }, {  -4,  71 }, {  -8,  72 }, {   2,  59 },
        {  -1,  55 }, {  -7,  70 }, {  -6,  75 }, {  -8,  89 },
        { -34, 119 }, {  -3,  75 }, {  32,  20 }, {  30,  22 },
        { -44, 127 }, {   0,  54 }, {  -5,  61 }, {   0,  58 },
        {  -1,  60 }, {  -3,  61 }, {  -8,  67 }, { -25,  84 },
        { -14,  74 }, {  -5,  65 }, {   5,  52 }, {   2,  57 },
        {   0,  61 }, {  -9,  69 }, { -11,  70 }, {  18,  55 },
        {   4,  45 }, {  10,  28 }, {  10,  31 }, {  33, -11 },
        {  52, -43 }, {  18,  15 }, {  28,   0 }, {  35, -22 },
        {  38, -25 }, {  34,   0 }, {  39, -18 }, {  32, -12 },
        { 102, -94 }, {   0,   0 }, {  56, -15 }, {  33,  -4 },
        {  29,  10 }, {  37,  -5 }, {  51, -29 }, {  39,  -9 },
        {  52, -34 }, {  69, -58 }, {  67, -63 }, {  44,  -5 },
        {  32,   7 }, {  55, -29 }, {  32,   1 }, {   0,   0 },
        {  27,  36 }, {  33, -25 }, {  34, -30 }, {  36, -28 },
        {  38, -28 }, {  38, -27 }, {  34, -18 }, {  35, -16 },
        {  34, -14 }, {  32,  -8 }, {  37,  -6 }, {  35,   0 },
        {  30,  10 }, {  28,  18 }, {  26,  25 }, {  29,  41 },
        {   4,  45 }, {  10,  28 }, {  10,  31 }, {  33, -11 },
        {  52, -43 }, {  18,  15 }, {  28,   0 }, {  35, -22 },
        {  38, -25 }, {  34,   0 }, {  39, -18 }, {  32, -12 },
        { 102, -94 }, {   0,   0 }, {  56, -15 }, {  33,  -4 },
        {  29,  10 }, {  37,  -5 }, {  51, -29 }, {  39,  -9 },
        {  52, -34 }, {  69, -58 }, {  67, -63 }, {  44,  -5 },
        {  32,   7 }, {  55, -29 }, {  32,   1 }, {   0,   0 },
        {  27,  36 }, {  33, -25 }, {  34, -30 }, {  36, -28 },
        {  38, -28 }, {  38, -27 }, {  34, -18 }, {  35, -16 },
        {  34, -14 }, {  32,  -8 }, {  37,  -6 }, {  35,   0 },
        {  30,  10 }, {  28,  18 }, {  26,  25 }, {  29,  41 },
        {  -5,  85 }, {  -6,  81 }, { -10,  77 }, {  -7,  81 },
        { -17,  80 }, { -18,  73 }, {  -4,  74 }, { -10,  83 },
        {  -9,  71 }, {  -9,  67 }, {  -1,  61 }, {  -8,  66 },
        { -14,  66 }, {   0,  59 }, {   2,  59 }, {  -3,  81 },
        {  -3,  76 }, {  -7,  72 }, {  -6,  78 }, { -12,  72 },
        { -14,  68 }, {  -3,  70 }, {  -6,  76 }, {  -5,  66 },
        {  -5,  62 }, {   0,  57 }, {  -4,  61 }, {  -9,  60 },
        {   1,  54 }, {   2,  58 }, {  17, -10 }, {  32, -13 },
        {  42,  -9 }, {  49,  -5 }, {  53,   0 }, {  64,   3 },
        {  68,  10 }, {  66,  27 }, {  47,  57 }, {  17, -10 },
        {  32, -13 }, {  42,  -9 }, {  49,  -5 }, {  53,   0 },
        {  64,   3 }, {  68,  10 }, {  66,  27 }, {  47,  57 },
        {  -5,  71 }, {   0,  24 }, {  -1,  36 }, {  -2,  42 },
        {  -2,  52 }, {  -9,  57 }, {  -6,  63 }, {  -4,  65 },
        {  -4,  67 }, {  -7,  82 }, {  -5,  85 }, {  -6,  81 },
        { -10,  77 }, {  -7,  81 }, { -17,  80 }, { -18,  73 },
        {  -4,  74 }, { -10,  83 }, {  -9,  71 }, {  -9,  67 },
        {  -1,  61 }, {  -8,  66 }, { -14,  66 }, {   0,  59 },
        {   2,  59 }, {  -3,  81 }, {  -3,  76 }, {  -7,  72 },
        {  -6,  78 }, { -12,  72 }, { -14,  68 }, {  -3,  70 },
        {  -6,  76 }, {  -5,  66 }, {  -5,  62 }, {   0,  57 },
        {  -4,  61 }, {  -9,  60 }, {   1,  54 }, {   2,  58 },
        {  17, -10 }, {  32, -13 }, {  42,  -9 }, {  49,  -5 },
        {  53,   0 }, {  64,   3 }, {  68,  10 }, {  66,  27 },
        {  47,  57 }, {  17, -10 }, {  32, -13 }, {  42,  -9 },
        {  49,  -5 }, {  53,   0 }, {  64,   3 }, {  68,  10 },
        {  66,  27 }, {  47,  57 }, {  -5,  71 }, {   0,  24 },
        {  -1,  36 }, {  -2,  42 }, {  -2,  52 }, {  -9,  57 },
        {  -6,  63 }, {  -4,  65 }, {  -4,  67 }, {  -7,  82 },
        { -21, 126 }, { -23, 124 }, { -20, 110 }, { -26, 126 },
        { -25, 124 }, { -17, 105 }, { -27, 121 }, { -27, 117 },
        { -17, 102 }, { -26, 117 }, { -27, 116 }, { -33, 122 },
        { -10,  95 }, { -14, 100 }, {  -8,  95 }, { -17, 111 },
        { -28, 114 }, {  -6,  89 }, {  -2,  80 }, {  -4,  82 },
        {  -9,  85 }, {  -8,  81 }, {  -1,  72 }, {   5,  64 },
        {   1,  67 }, {   9,  56 }, {   0,  69 }, {   1,  69 },
        {   7,  69 }, {  -7,  69 }, {  -6,  67 }, { -16,  77 },
        {  -2,  64 }, {   2,  61 }, {  -6,  67 }, {  -3,  64 },
        {   2,  57 }, {  -3,  65 }, {  -3,  66 }, {   0,  62 },
        {   9,  51 }, {  -1,  66 }, {  -2,  71 }, {  -2,  75 },
        { -21, 126 }, { -23, 124 }, { -20, 110 }, { -26, 126 },
        { -25, 124 }, { -17, 105 }, { -27, 121 }, { -27, 117 },
        { -17, 102 }, { -26, 117 }, { -27, 116 }, { -33, 122 },
        { -10,  95 }, { -14, 100 }, {  -8,  95 }, { -17, 111 },
        { -28, 114 }, {  -6,  89 }, {  -2,  80 }, {  -4,  82 },
        {  -9,  85 }, {  -8,  81 }, {  -1,  72 }, {   5,  64 },
        {   1,  67 }, {   9,  56 }, {   0,  69 }, {   1,  69 },
        {   7,  69 }, {  -7,  69 }, {  -6,  67 }, { -16,  77 },
        {  -2,  64 }, {   2,  61 }, {  -6,  67 }, {  -3,  64 },
        {   2,  57 }, {  -3,  65 }, {  -3,  66 }, {   0,  62 },
        {   9,  51 }, {  -1,  66 }, {  -2,  71 }, {  -2,  75 },
        {  19,  -6 }, {  18,  -6 }, {  14,   0 }, {  26, -12 },
        {  31, -16 }, {  33, -25 }, {  33, -22 }, {  37, -28 },
        {  39, -30 }, {  42, -30 }, {  47, -42 }, {  45, -36 },
        {  49, -34 }, {  41, -17 }, {  32,   9 }, {  69, -71 },
        {  63, -63 }, {  66, -64 }, {  77, -74 }, {  54, -39 },
        {  52, -35 }, {  41, -10 }, {  36,   0 }, {  40,  -1 },
        {  30,  14 }, {  28,  26 }, {  23,  37 }, {  12,  55 },
        {  11,  65 }, {  37, -33 }, {  39, -36 }, {  40, -37 },
        {  38, -30 }, {  46, -33 }, {  42, -30 }, {  40, -24 },
        {  49, -29 }, {  38, -12 }, {  40, -10 }, {  38,  -3 },
        {  46,  -5 }, {  31,  20 }, {  29,  30 }, {  25,  44 },
        {  19,  -6 }, {  18,  -6 }, {  14,   0 }, {  26, -12 },
        {  31, -16 }, {  33, -25 }, {  33, -22 }, {  37, -28 },
        {  39, -30 }, {  42, -30 }, {  47, -42 }, {  45, -36 },
        {  49, -34 }, {  41, -17 }, {  32,   9 }, {  69, -71 },
        {  63, -63 }, {  66, -64 }, {  77, -74 }, {  54, -39 },
        {  52, -35 }, {  41, -10 }, {  36,   0 }, {  40,  -1 },
        {  30,  14 }, {  28,  26 }, {  23,  37 }, {  12,  55 },
        {  11,  65 }, {  37, -33 }, {  39, -36 }, {  40, -37 },
        {  38, -30 }, {  46, -33 }, {  42, -30 }, {  40, -24 },
        {  49, -29 }, {  38, -12 }, {  40, -10 }, {  38,  -3 },
        {  46,  -5 }, {  31,  20 }, {  29,  30 }, {  25,  44 },
        { -23, 112 }, { -15,  71 }, {  -7,  61 }, {   0,  53 },
        {  -5,  66 }, { -11,  77 }, {  -9,  80 }, {  -9,  84 },
        { -10,  87 }, { -34, 127 }, { -21, 101 }, {  -3,  39 },
        {  -5,  53 }, {  -7,  61 }, { -11,  75 }, { -15,  77 },
        { -17,  91 }, { -25, 107 }, { -25, 111 }, { -28, 122 },
        { -11,  76 }, { -10,  44 }, { -10,  52 }, { -10,  57 },
        {  -9,  58 }, { -16,  72 }, {  -7,  69 }, {  -4,  69 },
        {  -5,  74 }, {  -9,  86 }, { -23, 112 }, { -15,  71 },
        {  -7,  61 }, {   0,  53 }, {  -5,  66 }, { -11,  77 },
        {  -9,  80 }, {  -9,  84 }, { -10,  87 }, { -34, 127 },
        { -21, 101 }, {  -3,  39 }, {  -5,  53 }, {  -7,  61 },
        { -11,  75 }, { -15,  77 }, { -17,  91 }, { -25, 107 },
        { -25, 111 }, { -28, 122 }, { -11,  76 }, { -10,  44 },
        { -10,  52 }, { -10,  57 }, {  -9,  58 }, { -16,  72 },
        {  -7,  69 }, {  -4,  69 }, {  -5,  74 }, {  -9,  86 },
        {  -2,  73 }, { -12, 104 }, {  -9,  91 }, { -31, 127 },
        {  -2,  73 }, { -12, 104 }, {  -9,  91 }, { -31, 127 },
        {  -2,  73 }, { -12, 104 }, {  -9,  91 }, { -31, 127 }
    },

    /* i_cabac_init_idc == 2 */
    {
        /* 0 - 10 */
        {  20, -15 }, {   2,  54 }, {   3,  74 }, {  20, -15 },
        {   2,  54 }, {   3,  74 }, { -28, 127 }, { -23, 104 },
        {  -6,  53 }, {  -1,  54 }, {   7,  51 },

        /* 11 - 23 */
        {  29,  16 }, {  25,   0 }, {  14,   0 }, { -10,  51 },
        {  -3,  62 }, { -27,  99 }, {  26,  16 }, {  -4,  85 },
        { -24, 102 }, {   5,  57 }, {   6,  57 }, { -17,  73 },
        {  14,  57 },

        /* 24 - 39 */
        {  20,  40 }, {  20,  10 }, {  29,   0 }, {  54,   0 },
        {  37,  42 }, {  12,  97 }, { -32, 127 }, { -22, 117 },
        {  -2,  74 }, {  -4,  85 }, { -24, 102 }, {   5,  57 },
        {  -6,  93 }, { -14,  88 }, {  -6,  44 }, {   4,  55 },

        /* 40 - 53 */
        { -11,  89 },{ -15,  103 },{ -21,  116 },{  19,  57 },
        {  20,  58 },{   4,  84 },{   6,  96 },{   1,  63 },
        {  -5,  85 },{ -13,  106 },{   5,  63 },{   6,  75 },
        {  -3,  90 },{  -1,  101 },

        /* 54 - 59 */
        {   3,  55 },{  -4,  79 },{  -2,  75 },{ -12,  97 },
        {  -7,  50 },{   1,  60 },

        /* 60 - 69 */
        { 0, 41 },   { 0, 63 },   { 0, 63 },     { 0, 63 },
        { -9, 83 },  { 4, 86 },   { 0, 97 },     { -7, 72 },
        { 13, 41 },  { 3, 62 },

        /* 70 - 104 */
        {   7,  34 }, {  -9,  88 }, { -20, 127 }, { -36, 127 },
        { -17,  91 }, { -14,  95 }, { -25,  84 }, { -25,  86 },
        { -12,  89 }, { -17,  91 }, { -31, 127 }, { -14,  76 },
        { -18, 103 }, { -13,  90 }, { -37, 127 }, {  11,  80 },
        {   5,  76 }, {   2,  84 }, {   5,  78 }, {  -6,  55 },
        {   4,  61 }, { -14,  83 }, { -37, 127 }, {  -5,  79 },
        { -11, 104 }, { -11,  91 }, { -30, 127 }, {   0,  65 },
        {  -2,  79 }, {   0,  72 }, {  -4,  92 }, {  -6,  56 },
        {   3,  68 }, {  -8,  71 }, { -13,  98 },

        /* 105 -> 165 */
        {  -4,  86 }, { -12,  88 }, {  -5,  82 }, {  -3,  72 },
        {  -4,  67 }, {  -8,  72 }, { -16,  89 }, {  -9,  69 },
        {  -1,  59 }, {   5,  66 }, {   4,  57 }, {  -4,  71 },
        {  -2,  71 }, {   2,  58 }, {  -1,  74 }, {  -4,  44 },
        {  -1,  69 }, {   0,  62 }, {  -7,  51 }, {  -4,  47 },
        {  -6,  42 }, {  -3,  41 }, {  -6,  53 }, {   8,  76 },
        {  -9,  78 }, { -11,  83 }, {   9,  52 }, {   0,  67 },
        {  -5,  90 }, {   1,  67 }, { -15,  72 }, {  -5,  75 },
        {  -8,  80 }, { -21,  83 }, { -21,  64 }, { -13,  31 },
        { -25,  64 }, { -29,  94 }, {   9,  75 }, {  17,  63 },
        {  -8,  74 }, {  -5,  35 }, {  -2,  27 }, {  13,  91 },
        {   3,  65 }, {  -7,  69 }, {   8,  77 }, { -10,  66 },
        {   3,  62 }, {  -3,  68 }, { -20,  81 }, {   0,  30 },
        {   1,   7 }, {  -3,  23 }, { -21,  74 }, {  16,  66 },
        { -23, 124 }, {  17,  37 }, {  44, -18 }, {  50, -34 },
        { -22, 127 },

        /* 166 - 226 */
        {   4,  39 }, {   0,  42 }, {   7,  34 }, {  11,  29 },
        {   8,  31 }, {   6,  37 }, {   7,  42 }, {   3,  40 },
        {   8,  33 }, {  13,  43 }, {  13,  36 }, {   4,  47 },
        {   3,  55 }, {   2,  58 }, {   6,  60 }, {   8,  44 },
        {  11,  44 }, {  14,  42 }, {   7,  48 }, {   4,  56 },
        {   4,  52 }, {  13,  37 }, {   9,  49 }, {  19,  58 },
        {  10,  48 }, {  12,  45 }, {   0,  69 }, {  20,  33 },
        {   8,  63 }, {  35, -18 }, {  33, -25 }, {  28,  -3 },
        {  24,  10 }, {  27,   0 }, {  34, -14 }, {  52, -44 },
        {  39, -24 }, {  19,  17 }, {  31,  25 }, {  36,  29 },
        {  24,  33 }, {  34,  15 }, {  30,  20 }, {  22,  73 },
        {  20,  34 }, {  19,  31 }, {  27,  44 }, {  19,  16 },
        {  15,  36 }, {  15,  36 }, {  21,  28 }, {  25,  21 },
        {  30,  20 }, {  31,  12 }, {  27,  16 }, {  24,  42 },
        {   0,  93 }, {  14,  56 }, {  15,  57 }, {  26,  38 },
        { -24, 127 },

        /* 227 - 275 */
        { -24, 115 }, { -22,  82 }, {  -9,  62 }, {   0,  53 },
        {   0,  59 }, { -14,  85 }, { -13,  89 }, { -13,  94 },
        { -11,  92 }, { -29, 127 }, { -21, 100 }, { -14,  57 },
        { -12,  67 }, { -11,  71 }, { -10,  77 }, { -21,  85 },
        { -16,  88 }, { -23, 104 }, { -15,  98 }, { -37, 127 },
        { -10,  82 }, {  -8,  48 }, {  -8,  61 }, {  -8,  66 },
        {  -7,  70 }, { -14,  75 }, { -10,  79 }, {  -9,  83 },
        { -12,  92 }, { -18, 108 }, {  -4,  79 }, { -22,  69 },
        { -16,  75 }, {  -2,  58 }, {   1,  58 }, { -13,  78 },
        {  -9,  83 }, {  -4,  81 }, { -13,  99 }, { -13,  81 },
        {  -6,  38 }, { -13,  62 }, {  -6,  58 }, {  -2,  59 },
        { -16,  73 }, { -10,  76 }, { -13,  86 }, {  -9,  83 },
        { -10,  87 },

        /* 276 a bit special (not used, x264_cabac_encode_bypass is used instead) */
        { 0, 0 },

        /* 277 - 337 */
        { -22, 127 }, { -25, 127 }, { -25, 120 }, { -27, 127 },
        { -19, 114 }, { -23, 117 }, { -25, 118 }, { -26, 117 },
        { -24, 113 }, { -28, 118 }, { -31, 120 }, { -37, 124 },
        { -10,  94 }, { -15, 102 }, { -10,  99 }, { -13, 106 },
        { -50, 127 }, {  -5,  92 }, {  17,  57 }, {  -5,  86 },
        { -13,  94 }, { -12,  91 }, {  -2,  77 }, {   0,  71 },
        {  -1,  73 }, {   4,  64 }, {  -7,  81 }, {   5,  64 },
        {  15,  57 }, {   1,  67 }, {   0,  68 }, { -10,  67 },
        {   1,  68 }, {   0,  77 }, {   2,  64 }, {   0,  68 },
        {  -5,  78 }, {   7,  55 }, {   5,  59 }, {   2,  65 },
        {  14,  54 }, {  15,  44 }, {   5,  60 }, {   2,  70 },
        {  -2,  76 }, { -18,  86 }, {  12,  70 }, {   5,  64 },
        { -12,  70 }, {  11,  55 }, {   5,  56 }, {   0,  69 },
        {   2,  65 }, {  -6,  74 }, {   5,  54 }, {   7,  54 },
        {  -6,  76 }, { -11,  82 }, {  -2,  77 }, {  -2,  77 },
        {  25,  42 },

        /* 338 - 398 */
        {  17, -13 }, {  16,  -9 }, {  17, -12 }, {  27, -21 },
        {  37, -30 }, {  41, -40 }, {  42, -41 }, {  48, -47 },
        {  39, -32 }, {  46, -40 }, {  52, -51 }, {  46, -41 },
        {  52, -39 }, {  43, -19 }, {  32,  11 }, {  61, -55 },
        {  56, -46 }, {  62, -50 }, {  81, -67 }, {  45, -20 },
        {  35,  -2 }, {  28,  15 }, {  34,   1 }, {  39,   1 },
        {  30,  17 }, {  20,  38 }, {  18,  45 }, {  15,  54 },
        {   0,  79 }, {  36, -16 }, {  37, -14 }, {  37, -17 },
        {  32,   1 }, {  34,  15 }, {  29,  15 }, {  24,  25 },
        {  34,  22 }, {  31,  16 }, {  35,  18 }, {  31,  28 },
        {  33,  41 }, {  36,  28 }, {  27,  47 }, {  21,  62 },
        {  18,  31 }, {  19,  26 }, {  36,  24 }, {  24,  23 },
        {  27,  16 }, {  24,  30 }, {  31,  29 }, {  22,  41 },
        {  22,  42 }, {  16,  60 }, {  15,  52 }, {  14,  60 },
        {   3,  78 }, { -16, 123 }, {  21,  53 }, {  22,  56 },
        {  25,  61 },

        /* 399 -> 435 */
        {  21,  33 }, {  19,  50 }, {  17,  61 },
        {  -3,  78 }, {  -8,  74 }, {  -9,  72 }, { -10,  72 },
        { -18,  75 }, { -12,  71 }, { -11,  63 }, {  -5,  70 },
        { -17,  75 }, { -14,  72 }, { -16,  67 }, {  -8,  53 },
        { -14,  59 }, {  -9,  52 }, { -11,  68 }, {   9,  -2 },
        {  30, -10 }, {  31,  -4 }, {  33,  -1 }, {  33,   7 },
        {  31,  12 }, {  37,  23 }, {  31,  38 }, {  20,  64 },
        {  -9,  71 }, {  -7,  37 }, {  -8,  44 }, { -11,  49 },
        { -10,  56 }, { -12,  59 }, {  -8,  63 }, {  -9,  67 },
        {  -6,  68 }, { -10,  79 },

        /* 436 -> 459 */
        {  -3,  78 }, {  -8,  74 }, {  -9,  72 }, { -10,  72 },
        { -18,  75 }, { -12,  71 }, { -11,  63 }, {  -5,  70 },
        { -17,  75 }, { -14,  72 }, { -16,  67 }, {  -8,  53 },
        { -14,  59 }, {  -9,  52 }, { -11,  68 }, {   9,  -2 },
        {  30, -10 }, {  31,  -4 }, {  33,  -1 }, {  33,   7 },
        {  31,  12 }, {  37,  23 }, {  31,  38 }, {  20,  64 },

        /* 460 - 1024 */
        {  11,  80 }, {   5,  76 }, {   2,  84 }, {   5,  78 },
        {  -6,  55 }, {   4,  61 }, { -14,  83 }, { -37, 127 },
        {  -5,  79 }, { -11, 104 }, { -11,  91 }, { -30, 127 },
        {  11,  80 }, {   5,  76 }, {   2,  84 }, {   5,  78 },
        {  -6,  55 }, {   4,  61 }, { -14,  83 }, { -37, 127 },
        {  -5,  79 }, { -11, 104 }, { -11,  91 }, { -30, 127 },
        {  -4,  86 }, { -12,  88 }, {  -5,  82 }, {  -3,  72 },
        {  -4,  67 }, {  -8,  72 }, { -16,  89 }, {  -9,  69 },
        {  -1,  59 }, {   5,  66 }, {   4,  57 }, {  -4,  71 },
        {  -2,  71 }, {   2,  58 }, {  -1,  74 }, {  -4,  44 },
        {  -1,  69 }, {   0,  62 }, {  -7,  51 }, {  -4,  47 },
        {  -6,  42 }, {  -3,  41 }, {  -6,  53 }, {   8,  76 },
        {  -9,  78 }, { -11,  83 }, {   9,  52 }, {   0,  67 },
        {  -5,  90 }, {   1,  67 }, { -15,  72 }, {  -5,  75 },
        {  -8,  80 }, { -21,  83 }, { -21,  64 }, { -13,  31 },
        { -25,  64 }, { -29,  94 }, {   9,  75 }, {  17,  63 },
        {  -8,  74 }, {  -5,  35 }, {  -2,  27 }, {  13,  91 },
        {  -4,  86 }, { -12,  88 }, {  -5,  82 }, {  -3,  72 },
        {  -4,  67 }, {  -8,  72 }, { -16,  89 }, {  -9,  69 },
        {  -1,  59 }, {   5,  66 }, {   4,  57 }, {  -4,  71 },
        {  -2,  71 }, {   2,  58 }, {  -1,  74 }, {  -4,  44 },
        {  -1,  69 }, {   0,  62 }, {  -7,  51 }, {  -4,  47 },
        {  -6,  42 }, {  -3,  41 }, {  -6,  53 }, {   8,  76 },
        {  -9,  78 }, { -11,  83 }, {   9,  52 }, {   0,  67 },
        {  -5,  90 }, {   1,  67 }, { -15,  72 }, {  -5,  75 },
        {  -8,  80 }, { -21,  83 }, { -21,  64 }, { -13,  31 },
        { -25,  64 }, { -29,  94 }, {   9,  75 }, {  17,  63 },
        {  -8,  74 }, {  -5,  35 }, {  -2,  27 }, {  13,  91 },
        {   4,  39 }, {   0,  42 }, {   7,  34 }, {  11,  29 },
        {   8,  31 }, {   6,  37 }, {   7,  42 }, {   3,  40 },
        {   8,  33 }, {  13,  43 }, {  13,  36 }, {   4,  47 },
        {   3,  55 }, {   2,  58 }, {   6,  60 }, {   8,  44 },
        {  11,  44 }, {  14,  42 }, {   7,  48 }, {   4,  56 },
        {   4,  52 }, {  13,  37 }, {   9,  49 }, {  19,  58 },
        {  10,  48 }, {  12,  45 }, {   0,  69 }, {  20,  33 },
        {   8,  63 }, {  35, -18 }, {  33, -25 }, {  28,  -3 },
        {  24,  10 }, {  27,   0 }, {  34, -14 }, {  52, -44 },
        {  39, -24 }, {  19,  17 }, {  31,  25 }, {  36,  29 },
        {  24,  33 }, {  34,  15 }, {  30,  20 }, {  22,  73 },
        {   4,  39 }, {   0,  42 }, {   7,  34 }, {  11,  29 },
        {   8,  31 }, {   6,  37 }, {   7,  42 }, {   3,  40 },
        {   8,  33 }, {  13,  43 }, {  13,  36 }, {   4,  47 },
        {   3,  55 }, {   2,  58 }, {   6,  60 }, {   8,  44 },
        {  11,  44 }, {  14,  42 }, {   7,  48 }, {   4,  56 },
        {   4,  52 }, {  13,  37 }, {   9,  49 }, {  19,  58 },
        {  10,  48 }, {  12,  45 }, {   0,  69 }, {  20,  33 },
        {   8,  63 }, {  35, -18 }, {  33, -25 }, {  28,  -3 },
        {  24,  10 }, {  27,   0 }, {  34, -14 }, {  52, -44 },
        {  39, -24 }, {  19,  17 }, {  31,  25 }, {  36,  29 },
        {  24,  33 }, {  34,  15 }, {  30,  20 }, {  22,  73 },
        {  -3,  78 }, {  -8,  74 }, {  -9,  72 }, { -10,  72 },
        { -18,  75 }, { -12,  71 }, { -11,  63 }, {  -5,  70 },
        { -17,  75 }, { -14,  72 }, { -16,  67 }, {  -8,  53 },
        { -14,  59 }, {  -9,  52 }, { -11,  68 }, {  -3,  78 },
        {  -8,  74 }, {  -9,  72 }, { -10,  72 }, { -18,  75 },
        { -12,  71 }, { -11,  63 }, {  -5,  70 }, { -17,  75 },
        { -14,  72 }, { -16,  67 }, {  -8,  53 }, { -14,  59 },
        {  -9,  52 }, { -11,  68 }, {   9,  -2 }, {  30, -10 },
        {  31,  -4 }, {  33,  -1 }, {  33,   7 }, {  31,  12 },
        {  37,  23 }, {  31,  38 }, {  20,  64 }, {   9,  -2 },
        {  30, -10 }, {  31,  -4 }, {  33,  -1 }, {  33,   7 },
        {  31,  12 }, {  37,  23 }, {  31,  38 }, {  20,  64 },
        {  -9,  71 }, {  -7,  37 }, {  -8,  44 }, { -11,  49 },
        { -10,  56 }, { -12,  59 }, {  -8,  63 }, {  -9,  67 },
        {  -6,  68 }, { -10,  79 }, {  -3,  78 }, {  -8,  74 },
        {  -9,  72 }, { -10,  72 }, { -18,  75 }, { -12,  71 },
        { -11,  63 }, {  -5,  70 }, { -17,  75 }, { -14,  72 },
        { -16,  67 }, {  -8,  53 }, { -14,  59 }, {  -9,  52 },
        { -11,  68 }, {  -3,  78 }, {  -8,  74 }, {  -9,  72 },
        { -10,  72 }, { -18,  75 }, { -12,  71 }, { -11,  63 },
        {  -5,  70 }, { -17,  75 }, { -14,  72 }, { -16,  67 },
        {  -8,  53 }, { -14,  59 }, {  -9,  52 }, { -11,  68 },
        {   9,  -2 }, {  30, -10 }, {  31,  -4 }, {  33,  -1 },
        {  33,   7 }, {  31,  12 }, {  37,  23 }, {  31,  38 },
        {  20,  64 }, {   9,  -2 }, {  30, -10 }, {  31,  -4 },
        {  33,  -1 }, {  33,   7 }, {  31,  12 }, {  37,  23 },
        {  31,  38 }, {  20,  64 }, {  -9,  71 }, {  -7,  37 },
        {  -8,  44 }, { -11,  49 }, { -10,  56 }, { -12,  59 },
        {  -8,  63 }, {  -9,  67 }, {  -6,  68 }, { -10,  79 },
        { -22, 127 }, { -25, 127 }, { -25, 120 }, { -27, 127 },
        { -19, 114 }, { -23, 117 }, { -25, 118 }, { -26, 117 },
        { -24, 113 }, { -28, 118 }, { -31, 120 }, { -37, 124 },
        { -10,  94 }, { -15, 102 }, { -10,  99 }, { -13, 106 },
        { -50, 127 }, {  -5,  92 }, {  17,  57 }, {  -5,  86 },
        { -13,  94 }, { -12,  91 }, {  -2,  77 }, {   0,  71 },
        {  -1,  73 }, {   4,  64 }, {  -7,  81 }, {   5,  64 },
        {  15,  57 }, {   1,  67 }, {   0,  68 }, { -10,  67 },
        {   1,  68 }, {   0,  77 }, {   2,  64 }, {   0,  68 },
        {  -5,  78 }, {   7,  55 }, {   5,  59 }, {   2,  65 },
        {  14,  54 }, {  15,  44 }, {   5,  60 }, {   2,  70 },
        { -22, 127 }, { -25, 127 }, { -25, 120 }, { -27, 127 },
        { -19, 114 }, { -23, 117 }, { -25, 118 }, { -26, 117 },
        { -24, 113 }, { -28, 118 }, { -31, 120 }, { -37, 124 },
        { -10,  94 }, { -15, 102 }, { -10,  99 }, { -13, 106 },
        { -50, 127 }, {  -5,  92 }, {  17,  57 }, {  -5,  86 },
        { -13,  94 }, { -12,  91 }, {  -2,  77 }, {   0,  71 },
        {  -1,  73 }, {   4,  64 }, {  -7,  81 }, {   5,  64 },
        {  15,  57 }, {   1,  67 }, {   0,  68 }, { -10,  67 },
        {   1,  68 }, {   0,  77 }, {   2,  64 }, {   0,  68 },
        {  -5,  78 }, {   7,  55 }, {   5,  59 }, {   2,  65 },
        {  14,  54 }, {  15,  44 }, {   5,  60 }, {   2,  70 },
        {  17, -13 }, {  16,  -9 }, {  17, -12 }, {  27, -21 },
        {  37, -30 }, {  41, -40 }, {  42, -41 }, {  48, -47 },
        {  39, -32 }, {  46, -40 }, {  52, -51 }, {  46, -41 },
        {  52, -39 }, {  43, -19 }, {  32,  11 }, {  61, -55 },
        {  56, -46 }, {  62, -50 }, {  81, -67 }, {  45, -20 },
        {  35,  -2 }, {  28,  15 }, {  34,   1 }, {  39,   1 },
        {  30,  17 }, {  20,  38 }, {  18,  45 }, {  15,  54 },
        {   0,  79 }, {  36, -16 }, {  37, -14 }, {  37, -17 },
        {  32,   1 }, {  34,  15 }, {  29,  15 }, {  24,  25 },
        {  34,  22 }, {  31,  16 }, {  35,  18 }, {  31,  28 },
        {  33,  41 }, {  36,  28 }, {  27,  47 }, {  21,  62 },
        {  17, -13 }, {  16,  -9 }, {  17, -12 }, {  27, -21 },
        {  37, -30 }, {  41, -40 }, {  42, -41 }, {  48, -47 },
        {  39, -32 }, {  46, -40 }, {  52, -51 }, {  46, -41 },
        {  52, -39 }, {  43, -19 }, {  32,  11 }, {  61, -55 },
        {  56, -46 }, {  62, -50 }, {  81, -67 }, {  45, -20 },
        {  35,  -2 }, {  28,  15 }, {  34,   1 }, {  39,   1 },
        {  30,  17 }, {  20,  38 }, {  18,  45 }, {  15,  54 },
        {   0,  79 }, {  36, -16 }, {  37, -14 }, {  37, -17 },
        {  32,   1 }, {  34,  15 }, {  29,  15 }, {  24,  25 },
        {  34,  22 }, {  31,  16 }, {  35,  18 }, {  31,  28 },
        {  33,  41 }, {  36,  28 }, {  27,  47 }, {  21,  62 },
        { -24, 115 }, { -22,  82 }, {  -9,  62 }, {   0,  53 },
        {   0,  59 }, { -14,  85 }, { -13,  89 }, { -13,  94 },
        { -11,  92 }, { -29, 127 }, { -21, 100 }, { -14,  57 },
        { -12,  67 }, { -11,  71 }, { -10,  77 }, { -21,  85 },
        { -16,  88 }, { -23, 104 }, { -15,  98 }, { -37, 127 },
        { -10,  82 }, {  -8,  48 }, {  -8,  61 }, {  -8,  66 },
        {  -7,  70 }, { -14,  75 }, { -10,  79 }, {  -9,  83 },
        { -12,  92 }, { -18, 108 }, { -24, 115 }, { -22,  82 },
        {  -9,  62 }, {   0,  53 }, {   0,  59 }, { -14,  85 },
        { -13,  89 }, { -13,  94 }, { -11,  92 }, { -29, 127 },
        { -21, 100 }, { -14,  57 }, { -12,  67 }, { -11,  71 },
        { -10,  77 }, { -21,  85 }, { -16,  88 }, { -23, 104 },
        { -15,  98 }, { -37, 127 }, { -10,  82 }, {  -8,  48 },
        {  -8,  61 }, {  -8,  66 }, {  -7,  70 }, { -14,  75 },
        { -10,  79 }, {  -9,  83 }, { -12,  92 }, { -18, 108 },
        {  -5,  79 }, { -11, 104 }, { -11,  91 }, { -30, 127 },
        {  -5,  79 }, { -11, 104 }, { -11,  91 }, { -30, 127 },
        {  -5,  79 }, { -11, 104 }, { -11,  91 }, { -30, 127 }
    }
};

const uint8_t x264_cabac_range_lps[64][4] =
{
    {  2,   2,   2,   2}, {  6,   7,   8,   9}, {  6,   7,   9,  10}, {  6,   8,   9,  11},
    {  7,   8,  10,  11}, {  7,   9,  10,  12}, {  7,   9,  11,  12}, {  8,   9,  11,  13},
    {  8,  10,  12,  14}, {  9,  11,  12,  14}, {  9,  11,  13,  15}, { 10,  12,  14,  16},
    { 10,  12,  15,  17}, { 11,  13,  15,  18}, { 11,  14,  16,  19}, { 12,  14,  17,  20},
    { 12,  15,  18,  21}, { 13,  16,  19,  22}, { 14,  17,  20,  23}, { 14,  18,  21,  24},
    { 15,  19,  22,  25}, { 16,  20,  23,  27}, { 17,  21,  25,  28}, { 18,  22,  26,  30},
    { 19,  23,  27,  31}, { 20,  24,  29,  33}, { 21,  26,  30,  35}, { 22,  27,  32,  37},
    { 23,  28,  33,  39}, { 24,  30,  35,  41}, { 26,  31,  37,  43}, { 27,  33,  39,  45},
    { 29,  35,  41,  48}, { 30,  37,  43,  50}, { 32,  39,  46,  53}, { 33,  41,  48,  56},
    { 35,  43,  51,  59}, { 37,  45,  54,  62}, { 39,  48,  56,  65}, { 41,  50,  59,  69},
    { 43,  53,  63,  72}, { 46,  56,  66,  76}, { 48,  59,  69,  80}, { 51,  62,  73,  85},
    { 53,  65,  77,  89}, { 56,  69,  81,  94}, { 59,  72,  86,  99}, { 62,  76,  90, 104},
    { 66,  80,  95, 110}, { 69,  85, 100, 116}, { 73,  89, 105, 122}, { 77,  94, 111, 128},
    { 81,  99, 117, 135}, { 85, 104, 123, 142}, { 90, 110, 130, 150}, { 95, 116, 137, 158},
    {100, 122, 144, 166}, {105, 128, 152, 175}, {111, 135, 160, 185}, {116, 142, 169, 195},
    {123, 150, 178, 205}, {128, 158, 187, 216}, {128, 167, 197, 227}, {128, 176, 208, 240}
};

const uint8_t x264_cabac_transition[128][2] =
{
    {  0,   0}, {  1,   1}, {  2,  50}, { 51,   3}, {  2,  50}, { 51,   3}, {  4,  52}, { 53,   5},
    {  6,  52}, { 53,   7}, {  8,  52}, { 53,   9}, { 10,  54}, { 55,  11}, { 12,  54}, { 55,  13},
    { 14,  54}, { 55,  15}, { 16,  56}, { 57,  17}, { 18,  56}, { 57,  19}, { 20,  56}, { 57,  21},
    { 22,  58}, { 59,  23}, { 24,  58}, { 59,  25}, { 26,  60}, { 61,  27}, { 28,  60}, { 61,  29},
    { 30,  60}, { 61,  31}, { 32,  62}, { 63,  33}, { 34,  62}, { 63,  35}, { 36,  64}, { 65,  37},
    { 38,  66}, { 67,  39}, { 40,  66}, { 67,  41}, { 42,  66}, { 67,  43}, { 44,  68}, { 69,  45},
    { 46,  68}, { 69,  47}, { 48,  70}, { 71,  49}, { 50,  72}, { 73,  51}, { 52,  72}, { 73,  53},
    { 54,  74}, { 75,  55}, { 56,  74}, { 75,  57}, { 58,  76}, { 77,  59}, { 60,  78}, { 79,  61},
    { 62,  78}, { 79,  63}, { 64,  80}, { 81,  65}, { 66,  82}, { 83,  67}, { 68,  82}, { 83,  69},
    { 70,  84}, { 85,  71}, { 72,  84}, { 85,  73}, { 74,  88}, { 89,  75}, { 76,  88}, { 89,  77},
    { 78,  90}, { 91,  79}, { 80,  90}, { 91,  81}, { 82,  94}, { 95,  83}, { 84,  94}, { 95,  85},
    { 86,  96}, { 97,  87}, { 88,  96}, { 97,  89}, { 90, 100}, {101,  91}, { 92, 100}, {101,  93},
    { 94, 102}, {103,  95}, { 96, 104}, {105,  97}, { 98, 104}, {105,  99}, {100, 108}, {109, 101},
    {102, 108}, {109, 103}, {104, 110}, {111, 105}, {106, 112}, {113, 107}, {108, 114}, {115, 109},
    {110, 116}, {117, 111}, {112, 118}, {119, 113}, {114, 118}, {119, 115}, {116, 122}, {123, 117},
    {118, 122}, {123, 119}, {120, 124}, {125, 121}, {122, 126}, {127, 123}, {124, 127}, {126, 125}
};

const uint8_t x264_cabac_renorm_shift[64] =
{
    6,5,4,4,3,3,3,3,2,2,2,2,2,2,2,2,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

/* -ln2(probability) */
const uint16_t x264_cabac_entropy[128] =
{
    FIX8(0.0273), FIX8(5.7370), FIX8(0.0288), FIX8(5.6618),
    FIX8(0.0303), FIX8(5.5866), FIX8(0.0320), FIX8(5.5114),
    FIX8(0.0337), FIX8(5.4362), FIX8(0.0355), FIX8(5.3610),
    FIX8(0.0375), FIX8(5.2859), FIX8(0.0395), FIX8(5.2106),
    FIX8(0.0416), FIX8(5.1354), FIX8(0.0439), FIX8(5.0602),
    FIX8(0.0463), FIX8(4.9851), FIX8(0.0488), FIX8(4.9099),
    FIX8(0.0515), FIX8(4.8347), FIX8(0.0543), FIX8(4.7595),
    FIX8(0.0572), FIX8(4.6843), FIX8(0.0604), FIX8(4.6091),
    FIX8(0.0637), FIX8(4.5339), FIX8(0.0671), FIX8(4.4588),
    FIX8(0.0708), FIX8(4.3836), FIX8(0.0747), FIX8(4.3083),
    FIX8(0.0788), FIX8(4.2332), FIX8(0.0832), FIX8(4.1580),
    FIX8(0.0878), FIX8(4.0828), FIX8(0.0926), FIX8(4.0076),
    FIX8(0.0977), FIX8(3.9324), FIX8(0.1032), FIX8(3.8572),
    FIX8(0.1089), FIX8(3.7820), FIX8(0.1149), FIX8(3.7068),
    FIX8(0.1214), FIX8(3.6316), FIX8(0.1282), FIX8(3.5565),
    FIX8(0.1353), FIX8(3.4813), FIX8(0.1429), FIX8(3.4061),
    FIX8(0.1510), FIX8(3.3309), FIX8(0.1596), FIX8(3.2557),
    FIX8(0.1686), FIX8(3.1805), FIX8(0.1782), FIX8(3.1053),
    FIX8(0.1884), FIX8(3.0301), FIX8(0.1992), FIX8(2.9549),
    FIX8(0.2107), FIX8(2.8797), FIX8(0.2229), FIX8(2.8046),
    FIX8(0.2358), FIX8(2.7294), FIX8(0.2496), FIX8(2.6542),
    FIX8(0.2642), FIX8(2.5790), FIX8(0.2798), FIX8(2.5038),
    FIX8(0.2964), FIX8(2.4286), FIX8(0.3142), FIX8(2.3534),
    FIX8(0.3331), FIX8(2.2782), FIX8(0.3532), FIX8(2.2030),
    FIX8(0.3748), FIX8(2.1278), FIX8(0.3979), FIX8(2.0527),
    FIX8(0.4226), FIX8(1.9775), FIX8(0.4491), FIX8(1.9023),
    FIX8(0.4776), FIX8(1.8271), FIX8(0.5082), FIX8(1.7519),
    FIX8(0.5412), FIX8(1.6767), FIX8(0.5768), FIX8(1.6015),
    FIX8(0.6152), FIX8(1.5263), FIX8(0.6568), FIX8(1.4511),
    FIX8(0.7020), FIX8(1.3759), FIX8(0.7513), FIX8(1.3008),
    FIX8(0.8050), FIX8(1.2256), FIX8(0.8638), FIX8(1.1504),
    FIX8(0.9285), FIX8(1.0752), FIX8(1.0000), FIX8(1.0000)
};

/*****************************************************************************
 * RDO
 *****************************************************************************/

/* Padded to [64] for easier addressing */
const uint8_t x264_significant_coeff_flag_offset_8x8[2][64] =
{{
    0, 1, 2, 3, 4, 5, 5, 4, 4, 3, 3, 4, 4, 4, 5, 5,
    4, 4, 4, 4, 3, 3, 6, 7, 7, 7, 8, 9,10, 9, 8, 7,
    7, 6,11,12,13,11, 6, 7, 8, 9,14,10, 9, 8, 6,11,
   12,13,11, 6, 9,14,10, 9,11,12,13,11,14,10,12
},{
    0, 1, 1, 2, 2, 3, 3, 4, 5, 6, 7, 7, 7, 8, 4, 5,
    6, 9,10,10, 8,11,12,11, 9, 9,10,10, 8,11,12,11,
    9, 9,10,10, 8,11,12,11, 9, 9,10,10, 8,13,13, 9,
    9,10,10, 8,13,13, 9, 9,10,10,14,14,14,14,14
}};
const uint8_t x264_last_coeff_flag_offset_8x8[63] =
{
    0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 8, 8, 8
};
const uint8_t x264_coeff_flag_offset_chroma_422_dc[7] = { 0, 0, 1, 1, 2, 2, 2 }; /* MIN( i/2, 2 ) */
const uint16_t x264_significant_coeff_flag_offset[2][16] =
{
    { 105+0, 105+15, 105+29, 105+44, 105+47, 402, 484+0, 484+15, 484+29, 660, 528+0, 528+15, 528+29, 718, 0, 0 },
    { 277+0, 277+15, 277+29, 277+44, 277+47, 436, 776+0, 776+15, 776+29, 675, 820+0, 820+15, 820+29, 733, 0, 0 }
};
const uint16_t x264_last_coeff_flag_offset[2][16] =
{
    { 166+0, 166+15, 166+29, 166+44, 166+47, 417, 572+0, 572+15, 572+29, 690, 616+0, 616+15, 616+29, 748, 0, 0 },
    { 338+0, 338+15, 338+29, 338+44, 338+47, 451, 864+0, 864+15, 864+29, 699, 908+0, 908+15, 908+29, 757, 0, 0 }
};
const uint16_t x264_coeff_abs_level_m1_offset[16] =
{
    227+0, 227+10, 227+20, 227+30, 227+39, 426, 952+0, 952+10, 952+20, 708, 982+0, 982+10, 982+20, 766
};
const uint8_t x264_count_cat_m1[14] = {15, 14, 15, 3, 14, 63, 15, 14, 15, 63, 15, 14, 15, 63};

/*****************************************************************************
 * VLC
 *****************************************************************************/

/* [nC] */
const vlc_t x264_coeff0_token[6] =
{
    { 0x1, 1 }, /* str=1 */
    { 0x3, 2 }, /* str=11 */
    { 0xf, 4 }, /* str=1111 */
    { 0x3, 6 }, /* str=000011 */
    { 0x1, 2 }, /* str=01 */
    { 0x1, 1 }, /* str=1 */
};

/* [nC][i_total_coeff-1][i_trailing] */
const vlc_t x264_coeff_token[6][16][4] =
{
    { /* table 0 */
        { /* i_total 1 */
            { 0x5, 6 }, /* str=000101 */
            { 0x1, 2 }, /* str=01 */
        },
        { /* i_total 2 */
            { 0x7, 8 }, /* str=00000111 */
            { 0x4, 6 }, /* str=000100 */
            { 0x1, 3 }, /* str=001 */
        },
        { /* i_total 3 */
            { 0x7, 9 }, /* str=000000111 */
            { 0x6, 8 }, /* str=00000110 */
            { 0x5, 7 }, /* str=0000101 */
            { 0x3, 5 }, /* str=00011 */
        },
        { /* i_total 4 */
            { 0x7, 10 }, /* str=0000000111 */
            { 0x6, 9 },  /* str=000000110 */
            { 0x5, 8 },  /* str=00000101 */
            { 0x3, 6 },  /* str=000011 */
        },
        { /* i_total 5 */
            { 0x7, 11 }, /* str=00000000111 */
            { 0x6, 10 }, /* str=0000000110 */
            { 0x5, 9 },  /* str=000000101 */
            { 0x4, 7 },  /* str=0000100 */
        },
        { /* i_total 6 */
            { 0xf, 13 }, /* str=0000000001111 */
            { 0x6, 11 }, /* str=00000000110 */
            { 0x5, 10 }, /* str=0000000101 */
            { 0x4, 8 },  /* str=00000100 */
        },
        { /* i_total 7 */
            { 0xb, 13 }, /* str=0000000001011 */
            { 0xe, 13 }, /* str=0000000001110 */
            { 0x5, 11 }, /* str=00000000101 */
            { 0x4, 9 },  /* str=000000100 */
        },
        { /* i_total 8 */
            { 0x8, 13 }, /* str=0000000001000 */
            { 0xa, 13 }, /* str=0000000001010 */
            { 0xd, 13 }, /* str=0000000001101 */
            { 0x4, 10 }, /* str=0000000100 */
        },
        { /* i_total 9 */
            { 0xf, 14 }, /* str=00000000001111 */
            { 0xe, 14 }, /* str=00000000001110 */
            { 0x9, 13 }, /* str=0000000001001 */
            { 0x4, 11 }, /* str=00000000100 */
        },
        { /* i_total 10 */
            { 0xb, 14 }, /* str=00000000001011 */
            { 0xa, 14 }, /* str=00000000001010 */
            { 0xd, 14 }, /* str=00000000001101 */
            { 0xc, 13 }, /* str=0000000001100 */
        },
        { /* i_total 14 */
            { 0xf, 15 }, /* str=000000000001111 */
            { 0xe, 15 }, /* str=000000000001110 */
            { 0x9, 14 }, /* str=00000000001001 */
            { 0xc, 14 }, /* str=00000000001100 */
        },
        { /* i_total 12 */
            { 0xb, 15 }, /* str=000000000001011 */
            { 0xa, 15 }, /* str=000000000001010 */
            { 0xd, 15 }, /* str=000000000001101 */
            { 0x8, 14 }, /* str=00000000001000 */
        },
        { /* i_total 13 */
            { 0xf, 16 }, /* str=0000000000001111 */
            { 0x1, 15 }, /* str=000000000000001 */
            { 0x9, 15 }, /* str=000000000001001 */
            { 0xc, 15 }, /* str=000000000001100 */
        },
        { /* i_total 14 */
            { 0xb, 16 }, /* str=0000000000001011 */
            { 0xe, 16 }, /* str=0000000000001110 */
            { 0xd, 16 }, /* str=0000000000001101 */
            { 0x8, 15 }, /* str=000000000001000 */
        },
        { /* i_total 15 */
            { 0x7, 16 }, /* str=0000000000000111 */
            { 0xa, 16 }, /* str=0000000000001010 */
            { 0x9, 16 }, /* str=0000000000001001 */
            { 0xc, 16 }, /* str=0000000000001100 */
        },
        { /* i_total 16 */
            { 0x4, 16 }, /* str=0000000000000100 */
            { 0x6, 16 }, /* str=0000000000000110 */
            { 0x5, 16 }, /* str=0000000000000101 */
            { 0x8, 16 }, /* str=0000000000001000 */
        },
    },
    { /* table 1 */
        { /* i_total 1 */
            { 0xb, 6 }, /* str=001011 */
            { 0x2, 2 }, /* str=10 */
        },
        { /* i_total 2 */
            { 0x7, 6 }, /* str=000111 */
            { 0x7, 5 }, /* str=00111 */
            { 0x3, 3 }, /* str=011 */
        },
        { /* i_total 3 */
            { 0x7, 7 }, /* str=0000111 */
            { 0xa, 6 }, /* str=001010 */
            { 0x9, 6 }, /* str=001001 */
            { 0x5, 4 }, /* str=0101 */
        },
        { /* i_total 4 */
            { 0x7, 8 }, /* str=00000111 */
            { 0x6, 6 }, /* str=000110 */
            { 0x5, 6 }, /* str=000101 */
            { 0x4, 4 }, /* str=0100 */
        },
        { /* i_total 5 */
            { 0x4, 8 }, /* str=00000100 */
            { 0x6, 7 }, /* str=0000110 */
            { 0x5, 7 }, /* str=0000101 */
            { 0x6, 5 }, /* str=00110 */
        },
        { /* i_total 6 */
            { 0x7, 9 }, /* str=000000111 */
            { 0x6, 8 }, /* str=00000110 */
            { 0x5, 8 }, /* str=00000101 */
            { 0x8, 6 }, /* str=001000 */
        },
        { /* i_total 7 */
            { 0xf, 11 }, /* str=00000001111 */
            { 0x6, 9 },  /* str=000000110 */
            { 0x5, 9 },  /* str=000000101 */
            { 0x4, 6 },  /* str=000100 */
        },
        { /* i_total 8 */
            { 0xb, 11 }, /* str=00000001011 */
            { 0xe, 11 }, /* str=00000001110 */
            { 0xd, 11 }, /* str=00000001101 */
            { 0x4, 7 },  /* str=0000100 */
        },
        { /* i_total 9 */
            { 0xf, 12 }, /* str=000000001111 */
            { 0xa, 11 }, /* str=00000001010 */
            { 0x9, 11 }, /* str=00000001001 */
            { 0x4, 9 },  /* str=000000100 */
        },
        { /* i_total 10 */
            { 0xb, 12 }, /* str=000000001011 */
            { 0xe, 12 }, /* str=000000001110 */
            { 0xd, 12 }, /* str=000000001101 */
            { 0xc, 11 }, /* str=00000001100 */
        },
        { /* i_total 11 */
            { 0x8, 12 }, /* str=000000001000 */
            { 0xa, 12 }, /* str=000000001010 */
            { 0x9, 12 }, /* str=000000001001 */
            { 0x8, 11 }, /* str=00000001000 */
        },
        { /* i_total 12 */
            { 0xf, 13 }, /* str=0000000001111 */
            { 0xe, 13 }, /* str=0000000001110 */
            { 0xd, 13 }, /* str=0000000001101 */
            { 0xc, 12 }, /* str=000000001100 */
        },
        { /* i_total 13 */
            { 0xb, 13 }, /* str=0000000001011 */
            { 0xa, 13 }, /* str=0000000001010 */
            { 0x9, 13 }, /* str=0000000001001 */
            { 0xc, 13 }, /* str=0000000001100 */
        },
        { /* i_total 14 */
            { 0x7, 13 }, /* str=0000000000111 */
            { 0xb, 14 }, /* str=00000000001011 */
            { 0x6, 13 }, /* str=0000000000110 */
            { 0x8, 13 }, /* str=0000000001000 */
        },
        { /* i_total 15 */
            { 0x9, 14 }, /* str=00000000001001 */
            { 0x8, 14 }, /* str=00000000001000 */
            { 0xa, 14 }, /* str=00000000001010 */
            { 0x1, 13 }, /* str=0000000000001 */
        },
        { /* i_total 16 */
            { 0x7, 14 }, /* str=00000000000111 */
            { 0x6, 14 }, /* str=00000000000110 */
            { 0x5, 14 }, /* str=00000000000101 */
            { 0x4, 14 }, /* str=00000000000100 */
        },
    },
    { /* table 2 */
        { /* i_total 1 */
            { 0xf, 6 }, /* str=001111 */
            { 0xe, 4 }, /* str=1110 */
        },
        { /* i_total 2 */
            { 0xb, 6 }, /* str=001011 */
            { 0xf, 5 }, /* str=01111 */
            { 0xd, 4 }, /* str=1101 */
        },
        { /* i_total 3 */
            { 0x8, 6 }, /* str=001000 */
            { 0xc, 5 }, /* str=01100 */
            { 0xe, 5 }, /* str=01110 */
            { 0xc, 4 }, /* str=1100 */
        },
        { /* i_total 4 */
            { 0xf, 7 }, /* str=0001111 */
            { 0xa, 5 }, /* str=01010 */
            { 0xb, 5 }, /* str=01011 */
            { 0xb, 4 }, /* str=1011 */
        },
        { /* i_total 5 */
            { 0xb, 7 }, /* str=0001011 */
            { 0x8, 5 }, /* str=01000 */
            { 0x9, 5 }, /* str=01001 */
            { 0xa, 4 }, /* str=1010 */
        },
        { /* i_total 6 */
            { 0x9, 7 }, /* str=0001001 */
            { 0xe, 6 }, /* str=001110 */
            { 0xd, 6 }, /* str=001101 */
            { 0x9, 4 }, /* str=1001 */
        },
        { /* i_total 7 */
            { 0x8, 7 }, /* str=0001000 */
            { 0xa, 6 }, /* str=001010 */
            { 0x9, 6 }, /* str=001001 */
            { 0x8, 4 }, /* str=1000 */
        },
        { /* i_total 8 */
            { 0xf, 8 }, /* str=00001111 */
            { 0xe, 7 }, /* str=0001110 */
            { 0xd, 7 }, /* str=0001101 */
            { 0xd, 5 }, /* str=01101 */
        },
        { /* i_total 9 */
            { 0xb, 8 }, /* str=00001011 */
            { 0xe, 8 }, /* str=00001110 */
            { 0xa, 7 }, /* str=0001010 */
            { 0xc, 6 }, /* str=001100 */
        },
        { /* i_total 10 */
            { 0xf, 9 }, /* str=000001111 */
            { 0xa, 8 }, /* str=00001010 */
            { 0xd, 8 }, /* str=00001101 */
            { 0xc, 7 }, /* str=0001100 */
        },
        { /* i_total 11 */
            { 0xb, 9 }, /* str=000001011 */
            { 0xe, 9 }, /* str=000001110 */
            { 0x9, 8 }, /* str=00001001 */
            { 0xc, 8 }, /* str=00001100 */
        },
        { /* i_total 12 */
            { 0x8, 9 }, /* str=000001000 */
            { 0xa, 9 }, /* str=000001010 */
            { 0xd, 9 }, /* str=000001101 */
            { 0x8, 8 }, /* str=00001000 */
        },
        { /* i_total 13 */
            { 0xd, 10 }, /* str=0000001101 */
            { 0x7, 9 },  /* str=000000111 */
            { 0x9, 9 },  /* str=000001001 */
            { 0xc, 9 },  /* str=000001100 */
        },
        { /* i_total 14 */
            { 0x9, 10 }, /* str=0000001001 */
            { 0xc, 10 }, /* str=0000001100 */
            { 0xb, 10 }, /* str=0000001011 */
            { 0xa, 10 }, /* str=0000001010 */
        },
        { /* i_total 15 */
            { 0x5, 10 }, /* str=0000000101 */
            { 0x8, 10 }, /* str=0000001000 */
            { 0x7, 10 }, /* str=0000000111 */
            { 0x6, 10 }, /* str=0000000110 */
        },
        { /* i_total 16 */
            { 0x1, 10 }, /* str=0000000001 */
            { 0x4, 10 }, /* str=0000000100 */
            { 0x3, 10 }, /* str=0000000011 */
            { 0x2, 10 }, /* str=0000000010 */
        },
    },
    { /* table 3 */
        { /* i_total 1 */
            { 0x0, 6 }, /* str=000000 */
            { 0x1, 6 }, /* str=000001 */
        },
        { /* i_total 2 */
            { 0x4, 6 }, /* str=000100 */
            { 0x5, 6 }, /* str=000101 */
            { 0x6, 6 }, /* str=000110 */
        },
        { /* i_total 3 */
            { 0x8, 6 }, /* str=001000 */
            { 0x9, 6 }, /* str=001001 */
            { 0xa, 6 }, /* str=001010 */
            { 0xb, 6 }, /* str=001011 */
        },
        { /* i_total 4 */
            { 0xc, 6 }, /* str=001100 */
            { 0xd, 6 }, /* str=001101 */
            { 0xe, 6 }, /* str=001110 */
            { 0xf, 6 }, /* str=001111 */
        },
        { /* i_total 5 */
            { 0x10, 6 }, /* str=010000 */
            { 0x11, 6 }, /* str=010001 */
            { 0x12, 6 }, /* str=010010 */
            { 0x13, 6 }, /* str=010011 */
        },
        { /* i_total 6 */
            { 0x14, 6 }, /* str=010100 */
            { 0x15, 6 }, /* str=010101 */
            { 0x16, 6 }, /* str=010110 */
            { 0x17, 6 }, /* str=010111 */
        },
        { /* i_total 7 */
            { 0x18, 6 }, /* str=011000 */
            { 0x19, 6 }, /* str=011001 */
            { 0x1a, 6 }, /* str=011010 */
            { 0x1b, 6 }, /* str=011011 */
        },
        { /* i_total 8 */
            { 0x1c, 6 }, /* str=011100 */
            { 0x1d, 6 }, /* str=011101 */
            { 0x1e, 6 }, /* str=011110 */
            { 0x1f, 6 }, /* str=011111 */
        },
        { /* i_total 9 */
            { 0x20, 6 }, /* str=100000 */
            { 0x21, 6 }, /* str=100001 */
            { 0x22, 6 }, /* str=100010 */
            { 0x23, 6 }, /* str=100011 */
        },
        { /* i_total 10 */
            { 0x24, 6 }, /* str=100100 */
            { 0x25, 6 }, /* str=100101 */
            { 0x26, 6 }, /* str=100110 */
            { 0x27, 6 }, /* str=100111 */
        },
        { /* i_total 11 */
            { 0x28, 6 }, /* str=101000 */
            { 0x29, 6 }, /* str=101001 */
            { 0x2a, 6 }, /* str=101010 */
            { 0x2b, 6 }, /* str=101011 */
        },
        { /* i_total 12 */
            { 0x2c, 6 }, /* str=101100 */
            { 0x2d, 6 }, /* str=101101 */
            { 0x2e, 6 }, /* str=101110 */
            { 0x2f, 6 }, /* str=101111 */
        },
        { /* i_total 13 */
            { 0x30, 6 }, /* str=110000 */
            { 0x31, 6 }, /* str=110001 */
            { 0x32, 6 }, /* str=110010 */
            { 0x33, 6 }, /* str=110011 */
        },
        { /* i_total 14 */
            { 0x34, 6 }, /* str=110100 */
            { 0x35, 6 }, /* str=110101 */
            { 0x36, 6 }, /* str=110110 */
            { 0x37, 6 }, /* str=110111 */
        },
        { /* i_total 15 */
            { 0x38, 6 }, /* str=111000 */
            { 0x39, 6 }, /* str=111001 */
            { 0x3a, 6 }, /* str=111010 */
            { 0x3b, 6 }, /* str=111011 */
        },
        { /* i_total 16 */
            { 0x3c, 6 }, /* str=111100 */
            { 0x3d, 6 }, /* str=111101 */
            { 0x3e, 6 }, /* str=111110 */
            { 0x3f, 6 }, /* str=111111 */
        },
    },
    { /* table 4 */
        { /* i_total 1 */
            { 0x7, 6 }, /* str=000111 */
            { 0x1, 1 }, /* str=1 */
        },
        { /* i_total 2 */
            { 0x4, 6 }, /* str=000100 */
            { 0x6, 6 }, /* str=000110 */
            { 0x1, 3 }, /* str=001 */
        },
        { /* i_total 3 */
            { 0x3, 6 }, /* str=000011 */
            { 0x3, 7 }, /* str=0000011 */
            { 0x2, 7 }, /* str=0000010 */
            { 0x5, 6 }, /* str=000101 */
        },
        { /* i_total 4 */
            { 0x2, 6 }, /* str=000010 */
            { 0x3, 8 }, /* str=00000011 */
            { 0x2, 8 }, /* str=00000010 */
            { 0x0, 7 }, /* str=0000000 */
        },
    },
    { /* table 5 */
        { /* i_total 1 */
            { 0xf, 7 }, /* str=0001111 */
            { 0x1, 2 }, /* str=01 */
        },
        { /* i_total 2 */
            { 0xe, 7 }, /* str=0001110 */
            { 0xd, 7 }, /* str=0001101 */
            { 0x1, 3 }, /* str=001 */
        },
        { /* i_total 3 */
            { 0x7, 9 }, /* str=000000111 */
            { 0xc, 7 }, /* str=0001100 */
            { 0xb, 7 }, /* str=0001011 */
            { 0x1, 5 }, /* str=00001 */
        },
        { /* i_total 4 */
            { 0x6, 9 }, /* str=000000110 */
            { 0x5, 9 }, /* str=000000101 */
            { 0xa, 7 }, /* str=0001010 */
            { 0x1, 6 }, /* str=000001 */
        },
        { /* i_total 5 */
            { 0x7, 10 }, /* str=0000000111 */
            { 0x6, 10 }, /* str=0000000110 */
            { 0x4, 9 },  /* str=000000100 */
            { 0x9, 7 },  /* str=0001001 */
        },
        { /* i_total 6 */
            { 0x7, 11 }, /* str=00000000111 */
            { 0x6, 11 }, /* str=00000000110 */
            { 0x5, 10 }, /* str=0000000101 */
            { 0x8, 7 },  /* str=0001000 */
        },
        { /* i_total 7 */
            { 0x7, 12 }, /* str=000000000111 */
            { 0x6, 12 }, /* str=000000000110 */
            { 0x5, 11 }, /* str=00000000101 */
            { 0x4, 10 }, /* str=0000000100 */
        },
        { /* i_total 8 */
            { 0x7, 13 }, /* str=0000000000111 */
            { 0x5, 12 }, /* str=000000000101 */
            { 0x4, 12 }, /* str=000000000100 */
            { 0x4, 11 }, /* str=00000000100 */
        },
    },
};

/* [i_total_coeff-1][i_total_zeros] */
const vlc_t x264_total_zeros[15][16] =
{
    { /* i_total 1 */
        { 0x1, 1 }, /* str=1 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 3 }, /* str=010 */
        { 0x3, 4 }, /* str=0011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x3, 5 }, /* str=00011 */
        { 0x2, 5 }, /* str=00010 */
        { 0x3, 6 }, /* str=000011 */
        { 0x2, 6 }, /* str=000010 */
        { 0x3, 7 }, /* str=0000011 */
        { 0x2, 7 }, /* str=0000010 */
        { 0x3, 8 }, /* str=00000011 */
        { 0x2, 8 }, /* str=00000010 */
        { 0x3, 9 }, /* str=000000011 */
        { 0x2, 9 }, /* str=000000010 */
        { 0x1, 9 }, /* str=000000001 */
    },
    { /* i_total 2 */
        { 0x7, 3 }, /* str=111 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x5, 4 }, /* str=0101 */
        { 0x4, 4 }, /* str=0100 */
        { 0x3, 4 }, /* str=0011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x3, 5 }, /* str=00011 */
        { 0x2, 5 }, /* str=00010 */
        { 0x3, 6 }, /* str=000011 */
        { 0x2, 6 }, /* str=000010 */
        { 0x1, 6 }, /* str=000001 */
        { 0x0, 6 }, /* str=000000 */
    },
    { /* i_total 3 */
        { 0x5, 4 }, /* str=0101 */
        { 0x7, 3 }, /* str=111 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 4 }, /* str=0100 */
        { 0x3, 4 }, /* str=0011 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x3, 5 }, /* str=00011 */
        { 0x2, 5 }, /* str=00010 */
        { 0x1, 6 }, /* str=000001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x0, 6 }, /* str=000000 */
    },
    { /* i_total 4 */
        { 0x3, 5 }, /* str=00011 */
        { 0x7, 3 }, /* str=111 */
        { 0x5, 4 }, /* str=0101 */
        { 0x4, 4 }, /* str=0100 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 4 }, /* str=0011 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x2, 5 }, /* str=00010 */
        { 0x1, 5 }, /* str=00001 */
        { 0x0, 5 }, /* str=00000 */
    },
    { /* i_total 5 */
        { 0x5, 4 }, /* str=0101 */
        { 0x4, 4 }, /* str=0100 */
        { 0x3, 4 }, /* str=0011 */
        { 0x7, 3 }, /* str=111 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x1, 5 }, /* str=00001 */
        { 0x1, 4 }, /* str=0001 */
        { 0x0, 5 }, /* str=00000 */
    },
    { /* i_total 6 */
        { 0x1, 6 }, /* str=000001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x7, 3 }, /* str=111 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 6 }, /* str=000000 */
    },
    { /* i_total 7 */
        { 0x1, 6 }, /* str=000001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 6 }, /* str=000000 */
    },
    { /* i_total 8 */
        { 0x1, 6 }, /* str=000001 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x3, 3 }, /* str=011 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 6 }, /* str=000000 */
    },
    { /* i_total 9 */
        { 0x1, 6 }, /* str=000001 */
        { 0x0, 6 }, /* str=000000 */
        { 0x1, 4 }, /* str=0001 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x1, 3 }, /* str=001 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 5 }, /* str=00001 */
    },
    { /* i_total 10 */
        { 0x1, 5 }, /* str=00001 */
        { 0x0, 5 }, /* str=00000 */
        { 0x1, 3 }, /* str=001 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 4 }, /* str=0001 */
    },
    { /* i_total 11 */
        { 0x0, 4 }, /* str=0000 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 3 }, /* str=001 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 1 }, /* str=1 */
        { 0x3, 3 }, /* str=011 */
    },
    { /* i_total 12 */
        { 0x0, 4 }, /* str=0000 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 1 }, /* str=1 */
        { 0x1, 3 }, /* str=001 */
    },
    { /* i_total 13 */
        { 0x0, 3 }, /* str=000 */
        { 0x1, 3 }, /* str=001 */
        { 0x1, 1 }, /* str=1 */
        { 0x1, 2 }, /* str=01 */
    },
    { /* i_total 14 */
        { 0x0, 2 }, /* str=00 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 1 }, /* str=1 */
    },
    { /* i_total 15 */
        { 0x0, 1 }, /* str=0 */
        { 0x1, 1 }, /* str=1 */
    },
};

/* [i_total_coeff-1][i_total_zeros] */
const vlc_t x264_total_zeros_2x2_dc[3][4] =
{
    { /* i_total 1 */
        { 0x1, 1 }, /* str=1 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 3 }  /* str=000 */
    },
    { /* i_total 2 */
        { 0x1, 1 }, /* str=1 */
        { 0x1, 2 }, /* str=01 */
        { 0x0, 2 }, /* str=00 */
    },
    { /* i_total 3 */
        { 0x1, 1 }, /* str=1 */
        { 0x0, 1 }, /* str=0 */
    },
};

/* [i_total_coeff-1][i_total_zeros] */
const vlc_t x264_total_zeros_2x4_dc[7][8] =
{
    { /* i_total 1 */
        { 0x1, 1 }, /* str=1 */
        { 0x2, 3 }, /* str=010 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 4 }, /* str=0010 */
        { 0x3, 4 }, /* str=0011 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x0, 5 }, /* str=00000 */
    },
    { /* i_total 2 */
        { 0x0, 3 }, /* str=000 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 3 }, /* str=001 */
        { 0x4, 3 }, /* str=100 */
        { 0x5, 3 }, /* str=101 */
        { 0x6, 3 }, /* str=110 */
        { 0x7, 3 }, /* str=111 */
    },
    { /* i_total 3 */
        { 0x0, 3 }, /* str=000 */
        { 0x1, 3 }, /* str=001 */
        { 0x1, 2 }, /* str=01 */
        { 0x2, 2 }, /* str=10 */
        { 0x6, 3 }, /* str=110 */
        { 0x7, 3 }, /* str=111 */
    },
    { /* i_total 4 */
        { 0x6, 3 }, /* str=110 */
        { 0x0, 2 }, /* str=00 */
        { 0x1, 2 }, /* str=01 */
        { 0x2, 2 }, /* str=10 */
        { 0x7, 3 }, /* str=111 */
    },
    { /* i_total 5 */
        { 0x0, 2 }, /* str=00 */
        { 0x1, 2 }, /* str=01 */
        { 0x2, 2 }, /* str=10 */
        { 0x3, 2 }, /* str=11 */
    },
    { /* i_total 6 */
        { 0x0, 2 }, /* str=00 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 1 }, /* str=1 */
    },
    { /* i_total 7 */
        { 0x0, 1 }, /* str=0 */
        { 0x1, 1 }, /* str=1 */
    }
};

/* [MIN( i_zero_left-1, 6 )][run_before] */
const vlc_t x264_run_before_init[7][16] =
{
    { /* i_zero_left 1 */
        { 0x1, 1 }, /* str=1 */
        { 0x0, 1 }, /* str=0 */
    },
    { /* i_zero_left 2 */
        { 0x1, 1 }, /* str=1 */
        { 0x1, 2 }, /* str=01 */
        { 0x0, 2 }, /* str=00 */
    },
    { /* i_zero_left 3 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x1, 2 }, /* str=01 */
        { 0x0, 2 }, /* str=00 */
    },
    { /* i_zero_left 4 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x1, 2 }, /* str=01 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 3 }, /* str=000 */
    },
    { /* i_zero_left 5 */
        { 0x3, 2 }, /* str=11 */
        { 0x2, 2 }, /* str=10 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 3 }, /* str=001 */
        { 0x0, 3 }, /* str=000 */
    },
    { /* i_zero_left 6 */
        { 0x3, 2 }, /* str=11 */
        { 0x0, 3 }, /* str=000 */
        { 0x1, 3 }, /* str=001 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 3 }, /* str=010 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
    },
    { /* i_zero_left >6 */
        { 0x7, 3 }, /* str=111 */
        { 0x6, 3 }, /* str=110 */
        { 0x5, 3 }, /* str=101 */
        { 0x4, 3 }, /* str=100 */
        { 0x3, 3 }, /* str=011 */
        { 0x2, 3 }, /* str=010 */
        { 0x1, 3 }, /* str=001 */
        { 0x1, 4 }, /* str=0001 */
        { 0x1, 5 }, /* str=00001 */
        { 0x1, 6 }, /* str=000001 */
        { 0x1, 7 }, /* str=0000001 */
        { 0x1, 8 }, /* str=00000001 */
        { 0x1, 9 }, /* str=000000001 */
        { 0x1, 10 }, /* str=0000000001 */
        { 0x1, 11 }, /* str=00000000001 */
    },
};

/* psy_trellis_init() has the largest size requirement of 16*FDEC_STRIDE*SIZEOF_PIXEL */
ALIGNED_64( uint8_t x264_zero[1024] ) = { 0 };
