/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* text.c: Stuff for drawing text on the display */

#include "spettro.h"
#include "text.h"

#include "cacatext.h"
#include "gui.h"
#include "lock.h"

#include <ctype.h>

/* We have our own few-pixels-high font */
#define CHAR_WIDTH 3
#define CHAR_HEIGHT 5
/* How many blank pixels between characters separated by a space? */
#define SPACE_WIDTH 2

static char *digits[] = {
    "000", " 0 ", "00 ", "000", "0 0", "000", "000", "000", "000", "000",
    "0 0", "00 ", "  0", "  0", "0 0", "0  ", "0  ", "  0", "0 0", "0 0",
    "0 0", " 0 ", " 0 ", "000", "000", "00 ", "000", " 0 ", "000", "000",
    "0 0", " 0 ", "0  ", "  0", "  0", "  0", "0 0", "0  ", "0 0", "  0",
    "000", "000", "000", "000", "  0", "00 ", "000", "0  ", "000", "000",
};
static const int digit_stride = 10;

static char *letters_at_G[] = {
    " 0 ", " 0 ", "00 ", " 00", "00 ", "000", "000", " 00",
    "0 0", "0 0", "0 0", "0  ", "0 0", "0  ", "0  ", "0  ",
    "0 0", "0 0", "00 ", "0  ", "0 0", "00 ", "00 ", "0 0",
    "0  ", "000", "0 0", "0  ", "0 0", "0  ", "0  ", "0 0",
    " 00", "0 0", "00 ", " 00", "00 ", "000", "0  ", " 0 ",
};
static const int at_G_stride = 8;
static char *lettersHQ[] = {
    "0 0", "000", "000", "0 0", "0  ", "0 0", "00 ", " 0 ", "00 ", " 0 ",
    "0 0", " 0 ", "  0", "00 ", "0  ", "000", "0 0", "0 0", "0 0", "0 0",
    "000", " 0 ", "  0", "0  ", "0  ", "0 0", "0 0", "0 0", "00 ", "0 0",
    "0 0", " 0 ", "  0", "00 ", "0  ", "0 0", "0 0", "0 0", "0  ", "0 0",
    "0 0", "000", "00 ", "0 0", "000", "0 0", "0 0", " 0 ", "0  ", " 00",
};
static const int HQ_stride = 10;
static char *lettersRZ[] = {
    "00 ", " 00", "000", "0 0", "0 0", "0 0", "0 0", "0 0", "000",
    "0 0", "0  ", " 0 ", "0 0", "0 0", "0 0", "0 0", "0 0", "  0",
    "00 ", " 0 ", " 0 ", "0 0", "0 0", "0 0", " 0 ", " 0 ", " 0 ",
    "0 0", "  0", " 0 ", "0 0", "0 0", "000", "0 0", " 0 ", "0  ",
    "0 0", "00 ", " 0 ", "000", " 0 ", "0 0", "0 0", "0  ", "000",
};
static const int RZ_stride = 9;

/*
 * Return the width of a text in pixels
 */
int
text_width(const char *text)
{
    int width = 0;
    int x;

#if HAVE_LIBCACA
    if (caca_init())
	return caca_text_width(text);
#endif
    /* Calculate the width of the typeset string in pixels,
     * These widths should correspond to those in draw_text()
     */
    for (x = 0; text[x] != '\0'; x++) {
	char c = toupper(text[x]);
	if (isdigit(c) || isupper(c) || c == '@')
	    width += CHAR_WIDTH;
	else switch (c) {
	case '.':
	case ',':
	case ':':
	    width += 1;
	    break;
	case ' ':
	    width += SPACE_WIDTH - 1;
	    break;
	case '+':
	case '-':
	case '=':
	case '/':
	    width += CHAR_WIDTH;
	    break;
	default:
	    fprintf(stderr, "Character value %d is not in the tiny font\n", c);
	    goto no_char; /* No char? No spacing! */
	}
	width += 1;
no_char:
	;
    }
    if (width > 0) width--; /* Not including the trailing blank column */

    return width;
}

int
text_height(const char *text)
{
#if HAVE_LIBCACA
    if (caca_init())
	return caca_text_height(text);
#endif
    return CHAR_HEIGHT;
}

/*
 * Draw the given text at the given coordinates.
 *
 * alignment_y can be
 *	TOP to puts the top pixel of the text at that y coordinate
 *	CENTER to put the middle pixel of the text at that y coordinate
 *	BOTTOM to put the bottom pixel of the text at that y coordinate
 * alignment_x can be
 *	LEFT to put the leftmost pixel of the text at that x coordinate
 *	RIGHT to put the rightmost pixel of the text at that x coordinate
 *	CENTER to centers the text on that coordinate value.
 */
void
draw_text(const char *text, int at_x, int at_y,
		      alignment_t alignment_x, alignment_t alignment_y)
{
    int width, height;
    int x;	/* index into the string */

#if HAVE_LIBCACA
    if (caca_init())
	return caca_draw_text(text, at_x, at_y, alignment_x, alignment_y);
#endif

    width = text_width(text);
    height = text_height(text);

    /* Make at_x and at_y the position of the top left pixel of the text */
    switch (alignment_x) {
    case LEFT:	/* at_x = at_x;	*/	break;
    case RIGHT: at_x -= width - 1;	break;
    case CENTER: at_x -= width/2;	break;
    }

    switch (alignment_y) {
    case TOP:	 /* at_y = at_y; */	break;
    case BOTTOM: at_y += height - 1;	break;
    case CENTER: at_y += height/2;	break;
    }

    /* Draw text at that position */
    gui_lock();
    for (x=0; text[x]; x++) {
	char c = toupper(text[x]);

	if (isdigit(c) || isupper(c) || c == '@') {
	    char **glyphs;
	    int stride;
	    int row, col;
	    int digit;

	    if (isdigit(c)) {
		glyphs = digits;
		stride = digit_stride;
		digit = c - '0';
	    } else if (c >= '@' && c <= 'G') {
		glyphs = letters_at_G;
		stride = at_G_stride;
		digit = c - '@';
	    } else if (c >= 'H' && c <= 'Q') {
		glyphs = lettersHQ;
		stride = HQ_stride;
		digit = c - 'H';
	    } else if (c >= 'R' && c <= 'Z') {
		glyphs = lettersRZ;
		stride = RZ_stride;
		digit = c - 'R';
	    } else {
		fprintf(stderr, "Unknown text character '%c'\n", c);
		gui_unlock();
		return;
	    }
	    /* Paint the character */
	    for (col = 0; col<CHAR_WIDTH; col++)
	        for (row = 0; row<CHAR_HEIGHT; row++)
		   if (glyphs[stride*row + digit][col] == '0')
		       gui_putpixel(at_x + col, at_y - row, green);
	    at_x += CHAR_WIDTH;
	} else {
	    switch (c) {
	    case '.':
		gui_putpixel(at_x, at_y-4, green);
		at_x += 1; break;
	    case ',':
		gui_putpixel(at_x, at_y-3, green);
		gui_putpixel(at_x, at_y-4, green);
		at_x += 1; break;
	    case ':':
		gui_putpixel(at_x,   at_y-3, green);
		gui_putpixel(at_x,   at_y-1, green);
		at_x += 1; break;
	    case '+':
		gui_putpixel(at_x+1, at_y-1, green);	/* top */
		gui_putpixel(at_x,   at_y-2, green);
		gui_putpixel(at_x+1, at_y-2, green);
		gui_putpixel(at_x+2, at_y-2, green);
		gui_putpixel(at_x+1, at_y-3, green);	/* bottom */
		at_x += CHAR_WIDTH; break;
	    case '-':
		gui_putpixel(at_x,   at_y-2, green);
		gui_putpixel(at_x+1, at_y-2, green);
		gui_putpixel(at_x+2, at_y-2, green);
		at_x += CHAR_WIDTH; break;
	    case '=':
		gui_putpixel(at_x,   at_y-1, green);
		gui_putpixel(at_x+1, at_y-1, green);
		gui_putpixel(at_x+2, at_y-1, green);
		gui_putpixel(at_x,   at_y-3, green);
		gui_putpixel(at_x+1, at_y-3, green);
		gui_putpixel(at_x+2, at_y-3, green);
		at_x += CHAR_WIDTH; break;
	    case '/':
		gui_putpixel(at_x+2, at_y, green);
		gui_putpixel(at_x+2, at_y-1, green);
		gui_putpixel(at_x+1, at_y-2, green);
		gui_putpixel(at_x  , at_y-3, green);
		gui_putpixel(at_x  , at_y-4, green);
		at_x += CHAR_WIDTH; break;
	    case ' ':
		at_x += SPACE_WIDTH - 1; break;
	    default:
		fprintf(stderr, "Character '%c' is not in the built-in font\n", c);
		goto no_char2;	/* No char? No spacing! */
	    }
	}
	at_x++;	/* One-pixel spacing between characters */
no_char2:
	;
    }
    gui_unlock();
}
